;====================================================================================================================
;
; UDF Name:        GUISelButton.au3
; Description:     Creates (pseudo) SelButton control to allow multiple selection (by checkbox).
; Requirement(s):  GUIScrollbarsEx.au3 (credit: Melba23)
; Author(s):       G.Sandler
;
; History:         
;                  v1.0
;                  * First public version
;                  
;                  v1.1
;                  * Adde option "Select all" for checkboxes list.
;                  * Few minor fixes.
;                  
;                  v1.2
;                  * Fixed issue when _GUICtrlSelButton_GetSelected was not returned the first selected checkbox.
;                  
;                  v1.3
;                  * Fixed bug when items was not added in case where $sSellAllCBText was not used (empty string) and only one item added for $sData.
;                  * Fixed problem with displaying short item strings.
;====================================================================================================================

#include-once
#include <GUIScrollbarsEx.au3>
#include <WinAPI.au3>

Global Enum $iGCSB_iButtonID, $iGCSB_hParent, $iGCSB_hGUI, $iGCSB_iWidth, $iGCSB_iHeight, $iGCSB_aItems, $iGCSB_bCount, $iGCSB_iCounter, $iGCSB_sSellAllText, _
	$iGCSB_iTotal
Global $ahGCSB_IDs[1][$iGCSB_iTotal]

Global Const $iGCSB_GUIMinWidth = 50
Global Const $iGCSB_GUIMinHeight = 50
Global Const $iGCSB_GUIDefWidth = 100
Global Const $iGCSB_GUIDefHeight = 100

Global $iGCSB_GUIWidth = -1 ;Width taken from SelButton width

; #FUNCTION# ====================================================================================================================
; Name ..........: _GUICtrlSelButton_Create
; Description ...: Creates SelButton control to allow multiple selection (by checkbox).
; Syntax ........: _GUICtrlSelButton_Create($sText, $sData, $iLeft, $iTop[, $iWidth = -1[, $iHeight = -1[, $bCount = True]]])
; Parameters ....: $sText               - Text to disply on selection button.
;                  $sData               - Items data string separated with "|".
;                  $iLeft               - Left position of the control.
;                  $iTop                - Top position of the control.
;                  $iWidth              - [optional] Width of the control. Default is -1, wich is set by $iGCSB_GUIDefWidth (100).
;                  $iHeight             - [optional] Height of the dropdown control. Default is -1, wich is set by $iGCSB_GUIDefHeight (100).
;                  $bCount              - [optional] Defines counter display on the control's label. Default is True - display.
;                  $sSellAllCBText      - [optional] Defines text for "Select All" checkbox. Default is empty string ("") - do not display "Select All" checkbox.
; Return values .: Handle to SelButton control.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _GUICtrlSelButton_Create($sText, $sData, $iLeft, $iTop, $iWidth = -1, $iHeight = -1, $bCount = True, $sSellAllCBText = '')
	$iWidth = ($iWidth = -1 ? $iGCSB_GUIDefWidth : $iWidth)
	$iHeight = ($iHeight = -1 ? $iGCSB_GUIDefHeight : $iHeight)
	
	Local $iW = ($iWidth < $iGCSB_GUIMinWidth ? $iGCSB_GUIMinWidth : $iWidth)
	Local $iH = ($iHeight < $iGCSB_GUIMinHeight ? $iGCSB_GUIMinHeight : $iHeight)
	
	Local $iButton = GUICtrlCreateButton($sText, $iLeft, $iTop, $iW, 20)
	Local $hParent = _WinAPI_GetParent(GUICtrlGetHandle($iButton))
	Local $hGUI = GUICreate('', ($iGCSB_GUIWidth = -1 ? $iW - 4 : (($iGCSB_GUIWidth < $iGCSB_GUIDefWidth) ? $iGCSB_GUIDefWidth : $iGCSB_GUIWidth)), $iH, -1, -1, BitOR($WS_POPUP, $WS_BORDER), -1, $hParent)
	
	GUISetBkColor(_WinAPI_GetSysColor($COLOR_WINDOW), $hGUI)
	
	Local $aItems = StringSplit(($sSellAllCBText ? $sSellAllCBText & '|' : '') & $sData, '|')
	
	If @error And StringStripWS($sSellAllCBText, 3) <> '' Then
		$aItems = 0
	Else
		Local $sBigStr
		
		For $i = 1 To $aItems[0]
			If StringLen($aItems[$i]) > StringLen($sBigStr) Then
				$sBigStr = $aItems[$i]
			EndIf
		Next
		
		Local $aStrSize = __GCSB_StringGetDimensions($hGUI, $sBigStr)
		
		If Not IsArray($aStrSize) Then
			Dim $aStrSize[1] = [$iW - 20]
		EndIf
		
		Local $iT = 5
		
		For $i = 1 To $aItems[0]
			$aItems[$i] = GUICtrlCreateCheckbox($aItems[$i], 10, $iT, $aStrSize[0] + 10, 15)
			$iT += (($sSellAllCBText And $i = 1) ? 40 : 20)
		Next
		
		_GUIScrollbars_Generate($hGUI, $aStrSize[0] + 20, $iT - 10)
	EndIf
	
	GUISwitch($hParent)
	
	If $ahGCSB_IDs[0][0] = 0 Then
		GUIRegisterMsg($WM_COMMAND, '__GCSB_WM_COMMAND')
		GUIRegisterMsg($WM_ACTIVATE, '__GCSB_WM_ACTIVATE')
	EndIf
	
	$ahGCSB_IDs[0][0] += 1
	ReDim $ahGCSB_IDs[$ahGCSB_IDs[0][0] + 1][$iGCSB_iTotal]
	
	$ahGCSB_IDs[$ahGCSB_IDs[0][0]][$iGCSB_iButtonID] = $iButton
	$ahGCSB_IDs[$ahGCSB_IDs[0][0]][$iGCSB_hParent] = $hParent
	$ahGCSB_IDs[$ahGCSB_IDs[0][0]][$iGCSB_hGUI] = $hGUI
	$ahGCSB_IDs[$ahGCSB_IDs[0][0]][$iGCSB_iWidth] = $iW
	$ahGCSB_IDs[$ahGCSB_IDs[0][0]][$iGCSB_iHeight] = $iH
	$ahGCSB_IDs[$ahGCSB_IDs[0][0]][$iGCSB_aItems] = $aItems
	$ahGCSB_IDs[$ahGCSB_IDs[0][0]][$iGCSB_bCount] = $bCount
	$ahGCSB_IDs[$ahGCSB_IDs[0][0]][$iGCSB_iCounter] = 0
	$ahGCSB_IDs[$ahGCSB_IDs[0][0]][$iGCSB_sSellAllText] = $sSellAllCBText
	
	Return $ahGCSB_IDs[0][0]
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GUICtrlSelButton_Delete
; Description ...: Deletes SelButton control created using _GUICtrlSelButton_Create.
; Syntax ........: _GUICtrlSelButton_Delete($hSelButton)
; Parameters ....: $hSelButton          - Handle as returned from _GUICtrlSelButton_Create.
; Return values .: 
;					Success: Deletes the specified control.
;					Failure: 0 and sets @error to 1 if $hSelButton is invalid or not existing control handle.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _GUICtrlSelButton_Delete($hSelButton)
	If $hSelButton > $ahGCSB_IDs[0][0] Or $hSelButton < 0 Then
		Return SetError(1, 0, 0)
	EndIf
	
	GUICtrlDelete($ahGCSB_IDs[$hSelButton][$iGCSB_iButtonID])
	GUICtrlDelete($ahGCSB_IDs[$hSelButton][$iGCSB_hGUI])
	
	Local $aTmp[$ahGCSB_IDs[0][0] + 1][$iGCSB_iTotal]
	
	For $i = $ahGCSB_IDs[0][0] To 1 Step -1
		If $ahGCSB_IDs[$i][$iGCSB_iButtonID] Then
			ExitLoop
		EndIf
		
		For $iGCSB = $iGCSB_iButtonID To $iGCSB_iTotal - 1
			$ahGCSB_IDs[$i][$iGCSB] = 0
		Next
		
		$ahGCSB_IDs[0][0] -= 1
	Next
	
	If $ahGCSB_IDs[0][0] = 0 Then
		GUIRegisterMsg($WM_COMMAND, '')
		GUIRegisterMsg($WM_ACTIVATE, '')
	EndIf
	
	ReDim $ahGCSB_IDs[$ahGCSB_IDs[0][0] + 1][$iGCSB_iTotal]
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GUICtrlSelButton_SetData
; Description ...: Sets data for SelButton control created using _GUICtrlSelButton_Create.
; Syntax ........: _GUICtrlSelButton_SetData($hSelButton, $sData)
; Parameters ....: $hSelButton          - Handle as returned from _GUICtrlSelButton_Create.
;                  $sData               - Items data string separated with "|".
;                  $sSellAllCBText      - [optional] Defines text for "Select All" checkbox. Default is empty string ("") - do not display "Select All" checkbox.
; Return values .: 
;					Success: Sets new data for specified SelButton control.
;					Failure: 0 and sets @error to 1 if $hSelButton is invalid or not existing control handle.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _GUICtrlSelButton_SetData($hSelButton, $sData, $sSellAllCBText = '')
	If $hSelButton > $ahGCSB_IDs[0][0] Or $hSelButton < 0 Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $aItems = $ahGCSB_IDs[$hSelButton][$iGCSB_aItems]
	
	If IsArray($aItems) Then
		For $i = 1 To $aItems[0]
			GUICtrlDelete($aItems[$i])
		Next
	EndIf
	
	$aItems = StringSplit(($sSellAllCBText ? $sSellAllCBText & '|' : '') & $sData, '|')
	
	If @error And StringStripWS($sSellAllCBText, 3) <> '' Then
		$aItems = 0
	Else
		GUISwitch($ahGCSB_IDs[$hSelButton][$iGCSB_hGUI])
		
		Local $sBigStr
		
		For $i = 1 To $aItems[0]
			If StringLen($aItems[$i]) > StringLen($sBigStr) Then
				$sBigStr = $aItems[$i]
			EndIf
		Next
		
		Local $aStrSize = __GCSB_StringGetDimensions($ahGCSB_IDs[$hSelButton][$iGCSB_hGUI], $sBigStr)
		
		If Not IsArray($aStrSize) Then
			Dim $aStrSize[1] = [$ahGCSB_IDs[$hSelButton][$iGCSB_iWidth] - 20]
		EndIf
		
		Local $iT = 5
		
		For $i = 1 To $aItems[0]
			$aItems[$i] = GUICtrlCreateCheckbox($aItems[$i], 10, $iT, $aStrSize[0] + 15, 15)
			$iT += (($sSellAllCBText And $i = 1) ? 40 : 20)
		Next
		
		_GUIScrollbars_Generate($ahGCSB_IDs[$hSelButton][$iGCSB_hGUI], $aStrSize[0] + 20, $iT - 10)
		GUISwitch($ahGCSB_IDs[$hSelButton][$iGCSB_hParent])
	EndIf
	
	$ahGCSB_IDs[$hSelButton][$iGCSB_aItems] = $aItems
	$ahGCSB_IDs[$hSelButton][$iGCSB_iCounter] = 0
	$ahGCSB_IDs[$hSelButton][$iGCSB_sSellAllText] = $sSellAllCBText
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GUICtrlSelButton_SetState
; Description ...: Set state for SelButton control created using _GUICtrlSelButton_Create.
; Syntax ........: _GUICtrlSelButton_SetState($hSelButton, $iState[, $iItem = -1[, $iBttnState = 0]])
; Parameters ....: $hSelButton          - Handle as returned from _GUICtrlSelButton_Create.
;                  $iState              - State to set (the same as used in GUICtrlSetState).
;                  $iItem               - [optional] Item 0-based index to get the state for. Default is -1, gets all items states.
;                  $iBttnState          - [optional] State for the SelButton button control. Default is 0, do not set.
; Return values .: 
;					Success: Set state for specified/all item(s) in SelButton control, or/and for SelButton control itself if $iBttnState <> 0.
;					Failure: 0 and sets @error as following:
;																		1 - $hSelButton is invalid or not existing control handle.
;																		2 - Items not found in $hSelButton control.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _GUICtrlSelButton_SetState($hSelButton, $iState, $iItem = -1, $iBttnState = 0)
	Local $aInfo = _GUICtrlSelButton_GetInfo($hSelButton)
	
	If @error Then
		Return SetError(@error, 0, 0)
	EndIf
	
	If $iBttnState Then
		GUICtrlSetState($ahGCSB_IDs[$hSelButton][$iGCSB_iButtonID], $iBttnState)
		Return 1
	EndIf
	
	If $iState Then
		For $i = 1 To $aInfo[0][0]
			If ($iItem > -1 And $iItem = $aInfo[$i][1]) Or $iItem = -1 Then
				GUICtrlSetState($aInfo[$i][2], BitXOR($iState, $GUI_CHECKED, $GUI_UNCHECKED))
				ControlCommand($ahGCSB_IDs[$hSelButton][$iGCSB_hGUI], '', $aInfo[$i][2], (BitAND($iState, $GUI_CHECKED) ? '' : 'Un') & 'Check')
			EndIf
		Next
	EndIf
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GUICtrlSelButton_GetInfo
; Description ...: Get info data for SelButton control created using _GUICtrlSelButton_Create.
; Syntax ........: _GUICtrlSelButton_GetInfo($hSelButton)
; Parameters ....: $hSelButton          - Handle as returned from _GUICtrlSelButton_Create.
; Return values .: 
;					Success: 2D array of items with their info, as following:
;																		[0][0] - Total items.
;																		[n][0] - Item text.
;																		[n][1] - Item 0-based index.
;																		[n][2] - Item ID.
;																		[n][3] - Item state.
;					Failure: 0 and sets @error as following:
;																		1 - $hSelButton is invalid or not existing control handle.
;																		2 - Items not found in $hSelButton control.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _GUICtrlSelButton_GetInfo($hSelButton)
	If $hSelButton > $ahGCSB_IDs[0][0] Or $hSelButton < 0 Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $aItems = $ahGCSB_IDs[$hSelButton][$iGCSB_aItems]
	
	If Not IsArray($aItems) Then
		Return SetError(2, 0, 0)
	EndIf
	
	Local $aRet[$aItems[0] + 1][4] = [[0]]
	Local $iStart = ($ahGCSB_IDs[$hSelButton][$iGCSB_sSellAllText] ? 2 : 1)
	
	For $i = $iStart To $aItems[0]
		$aRet[0][0] += 1
		$aRet[$aRet[0][0]][0] = GUICtrlRead($aItems[$i], 1) ;Text
		$aRet[$aRet[0][0]][1] = $i - 1 ;Index
		$aRet[$aRet[0][0]][2] = $aItems[$i] ;ID
		$aRet[$aRet[0][0]][3] = GUICtrlRead($aItems[$i]) ;State
	Next
	
	Return $aRet
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GUICtrlSelButton_GetSelected
; Description ...: Get selected items for SelButton control created using _GUICtrlSelButton_Create.
; Syntax ........: _GUICtrlSelButton_GetSelected($hSelButton)
; Parameters ....: $hSelButton          - Handle as returned from _GUICtrlSelButton_Create.
; Return values .: 
;					Success: 2D array of selected items, as following:
;																		[0][0] - Total selected items.
;																		[n][0] - Selected item text.
;																		[n][1] - Selected item 0-based index.
;					Failure: 0 and sets @error as following:
;																		1 - $hSelButton is invalid or not existing control handle.
;																		2 - Items not found in $hSelButton control.
;																		3 - No selected items.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _GUICtrlSelButton_GetSelected($hSelButton)
	Local $aInfo = _GUICtrlSelButton_GetInfo($hSelButton)
	
	If @error Then
		Return SetError(@error, 0, 0)
	EndIf
	
	Local $aRet[$aInfo[0][0] + 1][2] = [[0]]
	
	For $i = 1 To $aInfo[0][0]
		If BitAND($aInfo[$i][3], $GUI_CHECKED) Then
			$aRet[0][0] += 1
			$aRet[$aRet[0][0]][0] = $aInfo[$i][0]
			$aRet[$aRet[0][0]][1] = $aInfo[$i][1]
		EndIf
	Next
	
	If $aRet[0][0] = 0 Then
		Return SetError(3, 0, 0)
	EndIf
	
	ReDim $aRet[$aRet[0][0] + 1][2]
	Return $aRet
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GUICtrlSelButton_GetID
; Description ...: Get SelButton Control ID (the button).
; Syntax ........: _GUICtrlSelButton_GetID($hSelButton)
; Parameters ....: $hSelButton          - Handle as returned from _GUICtrlSelButton_Create.
; Return values .: 
;					Success: SelButton Control ID.
;					Failure: 0 and sets @error to 1 if $hSelButton is invalid or not existing control handle.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _GUICtrlSelButton_GetID($hSelButton)
	If $hSelButton > $ahGCSB_IDs[0][0] Or $hSelButton < 0 Then
		Return SetError(1, 0, 0)
	EndIf
	
	Return $ahGCSB_IDs[$hSelButton][$iGCSB_iButtonID]
EndFunc


; #INTERNAL_USE_ONLY# ===========================================================================================================
; Name ..........: __GCSB_StringGetDimensions
; Description ...: Get string dimensions function
; ===============================================================================================================================
Func __GCSB_StringGetDimensions($hWnd, $sText)
	Local $hDC = _WinAPI_GetDC($hWnd) ; Get the device context handle of the current window.
	Local $hFont = _SendMessage($hWnd, $WM_GETFONT) ; Retrieve the font with which the control is currently drawing its text.
	Local $hSelectObject = _WinAPI_SelectObject($hDC, $hFont) ; Select the object of the context device.
	Local $tSIZE = _WinAPI_GetTextExtentPoint32($hDC, $sText) ; Retrieve the height & width of a string.
	
	_WinAPI_SelectObject($hDC, $hSelectObject)
	_WinAPI_ReleaseDC($hWnd, $hDC) ; Release the device context.
	
	Local $aReturn[2] = [DllStructGetData($tSIZE, 1), DllStructGetData($tSIZE, 2)] ; Set an array with the width & height of the string.
	Return $aReturn
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Name ..........: __GCSB_SetCounter
; Description ...: Sets SelButton counter if needed/defined
; ===============================================================================================================================
Func __GCSB_SetCounter($hSelButton, $iCtrlID)
	If $hSelButton > $ahGCSB_IDs[0][0] Or $hSelButton < 0 Then
		Return SetError(1, 0, 0)
	EndIf
	
	$ahGCSB_IDs[$hSelButton][$iGCSB_iCounter] += (BitAND(GUICtrlRead($iCtrlID), $GUI_CHECKED) ? 1 : -1)
	
	If $ahGCSB_IDs[$hSelButton][$iGCSB_bCount] Then
		Local $sText = StringRegExpReplace(GUICtrlRead($ahGCSB_IDs[$hSelButton][$iGCSB_iButtonID]), '\h*\([0-9-]+\)$', '')
		GUICtrlSetData($ahGCSB_IDs[$hSelButton][$iGCSB_iButtonID], $sText & ($ahGCSB_IDs[$hSelButton][$iGCSB_iCounter] = 0 ? '' : ' (' & $ahGCSB_IDs[$hSelButton][$iGCSB_iCounter] & ')'))
	EndIf
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Name ..........: __GCSB_WM_COMMAND
; Description ...: WM_COMMAND message function
; ===============================================================================================================================
Func __GCSB_WM_COMMAND($hWnd, $iMsg, $wParam, $lParam)
    Local $nID = BitAND($wParam, 0x0000FFFF)
    Local $aItems
	
	For $i = 1 To $ahGCSB_IDs[0][0]
		$aItems = $ahGCSB_IDs[$i][$iGCSB_aItems]
		
		If Not IsArray($aItems) Then
			ContinueLoop
		EndIf
		
		If $aItems[1] = $nID And $ahGCSB_IDs[$i][$iGCSB_sSellAllText] Then
			Local $iChecked = (BitAND(GUICtrlRead($aItems[1]), $GUI_CHECKED) = $GUI_CHECKED)
			$ahGCSB_IDs[$i][$iGCSB_iCounter] = ($iChecked ? 0 : $ahGCSB_IDs[$i][$iGCSB_iCounter])
			
			For $j = 2 To $aItems[0]
				GUICtrlSetState($aItems[$j], ($iChecked ? $GUI_CHECKED : $GUI_UNCHECKED))
				__GCSB_SetCounter($i, $aItems[$j])
			Next
			
			Return $GUI_RUNDEFMSG
		EndIf
		
		For $j = 1 To $aItems[0]
			If $aItems[$j] = $nID Then
				__GCSB_SetCounter($i, $aItems[$j])
				
				If $ahGCSB_IDs[$i][$iGCSB_sSellAllText] And $ahGCSB_IDs[$i][$iGCSB_iCounter] = $aItems[0] - 1 Then
					GUICtrlSetState($aItems[1], $GUI_CHECKED)
				ElseIf $ahGCSB_IDs[$i][$iGCSB_sSellAllText] Then
					GUICtrlSetState($aItems[1], $GUI_UNCHECKED)
				EndIf
				
				Return $GUI_RUNDEFMSG
			EndIf
		Next
		
		Switch $nID
			Case $ahGCSB_IDs[$i][$iGCSB_iButtonID]
				Local $aPos = ControlGetPos($hWnd, '', $nID)
				Local $iX, $iY
				
				If IsArray($aPos) Then
					Local $tPoint = DllStructCreate('int X;int Y')
					
					DllStructSetData($tPoint, 'X', $aPos[0])
					DllStructSetData($tPoint, 'Y', $aPos[1] + $aPos[3])
					
					_WinAPI_ClientToScreen($hWnd, $tPoint)
					
					$iX = DllStructGetData($tPoint, 'X')
					$iY = DllStructGetData($tPoint, 'Y')
					
					If $iY + $ahGCSB_IDs[$i][$iGCSB_iHeight] > @DesktopHeight - 50 Then
						$iY -= ($aPos[3] + $ahGCSB_IDs[$i][$iGCSB_iHeight] + 3)
					EndIf
				EndIf
				
				WinMove($ahGCSB_IDs[$i][$iGCSB_hGUI], '', $iX + 1, $iY)
				GUISetState(@SW_SHOW, $ahGCSB_IDs[$i][$iGCSB_hGUI])
				
				Return $GUI_RUNDEFMSG
		EndSwitch
	Next
	
    Return $GUI_RUNDEFMSG
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Name ..........: __GCSB_WM_ACTIVATE
; Description ...: WM_ACTIVATE message function
; ===============================================================================================================================
Func __GCSB_WM_ACTIVATE($hWnd, $Msg, $wParam, $lParam)
	Local $iActive = BitAND($wParam, 0xFFFF)
	
	For $i = 1 To $ahGCSB_IDs[0][0]
		If $hWnd = $ahGCSB_IDs[$i][$iGCSB_hGUI] And Not $iActive Then
			GUISetState(@SW_HIDE, $ahGCSB_IDs[$i][$iGCSB_hGUI])
			ExitLoop
		EndIf
	Next
	
	Return $GUI_RUNDEFMSG
EndFunc
