#include-once

#Region Header

#CS
	Name:				HtmlSelector.au3
	Description:		UDF library to get html elements using XPath and CSS selectors. Based on HtmlAgilityPack and Htmlayout.
	Author:				Copyright  2015 CreatoR's Lab (G.Sandler), www.creator-lab.ucoz.ru, www.autoit-script.ru. All rights reserved.
	Requirements:		AutoIt 3.3.6.1 - 3.3.12.0 (x86), Win 7.
	UDF version:		0.3
	
	Credits:			
						Simon Mourrier, Jeff Klawiter (for HtmlAgilityPack.dll).
						Terra Informatica Software (for Htmlayout.dll).
	
	History:
						v0.3
						* Changed examples.
						+ Added _HtmlSelector_GetAttribute function to get certain attribute value.
						+ Added user constants to use in result array by _HtmlSelector_SelectBy* and for _HtmlSelector_Init function
							($HSEL_INIT_XPATHSEL, $HSEL_INIT_CSSSEL, $HSEL_INIT_ALLSEL)
							($HSEL_RSLT_TYPE, $HSEL_RSLT_ATTRIBS, $HSEL_RSLT_OUTERHTML, $HSEL_RSLT_INNERHTML, $HSEL_RSLT_INNERTEXT)
							($HSEL_RSLTATTRB_NAME, $HSEL_RSLTATTRB_VALUE).
						+ Added $iFlag parameter to _HTMLSelector_Init function (for details see function description).
						+ Added additional $iIndex and $iRetRslt parameters to _HtmlSelector_SelectBy* functions to get specified data from the selected array
							(for details check examples and function description).
						+ Added additional $fLoadHtml parameter to _HtmlSelector_SelectBy* functions to determine if there is need to reload html each time.
							(for details check examples and function description).
						- Removed $fStripScripts parameter in _HtmlSelector_SelectByCSS function (can be done by user manualy).
						* _HtmlSelector_SelectByCSS now return InnerText in [N][4] element.
						* Speed & memory optimizations.
						
						v0.2
						* Fixed issue with multiple hidden windows creation (for each _HtmlSelector_SelectByCSS call created new window).
						* Changed returned @error values for _HtmlSelector_SelectByCSS.
						
						v0.1
						* First public version.
#CE

#EndRegion

#Region User Constants

Global Enum $HSEL_INIT_XPATHSEL, $HSEL_INIT_CSSSEL, $HSEL_INIT_ALLSEL

Global Enum _
	$HSEL_RSLT_TYPE, $HSEL_RSLT_ATTRIBS, $HSEL_RSLT_OUTERHTML, $HSEL_RSLT_INNERHTML, $HSEL_RSLT_INNERTEXT, _
	$HSEL_RSLT_TOTAL

Global Enum _
	$HSEL_RSLTATTRB_NAME, $HSEL_RSLTATTRB_VALUE, _
	$HSEL_RSLTATTRB_TOTAL

#EndRegion

#Region Global Constants & Variables

Global Const $sHSEL_HAPDLLFile 						= @TempDir & '\HtmlAgilityPack.dll'
Global Const $sHSEL_HLDLLFile 						= @TempDir & '\HtmLayout.dll'

Global $hHSEL_HLDLL									= -1
Global $hHSEL_HLWnd									= 0
Global $iHSEL_HLDefSelElmnts 						= 1000
Global $aHSEL_HLSelElmnts[$iHSEL_HLDefSelElmnts] 	= [0]
Global $oHSEL_HAPHtml 								= 0

; Enumeration used for the __HTMLSelector_DotNet* functions.
Global Enum $DOTNET_PATHS_INDEX, $DOTNET_PATHS_START, $DOTNET_LOADDLL, $DOTNET_UNLOADDLL, $DOTNET_UNLOADDLLALL, $DOTNET_PATHS_MAX = 100

#EndRegion

; #FUNCTION# ====================================================================================================
; Name...........: _HtmlSelector_SelectByXPath
; Description....: Select node(s) from html using XPath query.
; Syntax.........: _HtmlSelector_SelectByXPath($sHtml, $sQuery, $iIndex = -1, $iRetRslt = $HSEL_RSLT_OUTERHTML)
; Parameters.....: $sHtml - Html string to select from.
;                  $sQuery - XPath query.
;                  $iIndex [Optional] - If this is <> -1, then the return is the array element set by $iIndex (depending on $iRetRslt parameter).
;                  $iRetRslt [Optional] - Define the return data by $iIndex (if set), the following are supported:
;                  														$HSEL_RSLT_TYPE(0) - Type (name) of selected node (ex.: $aResult[$iIndex][$HSEL_RSLT_TYPE])
;																		$HSEL_RSLT_ATTRIBS(1) - 2D array with attributes of selected node
;																		$HSEL_RSLT_OUTERHTML(2) - OuterHtml of selected node (default)
;																		$HSEL_RSLT_INNERHTML(3) - InnerHtml of selected node
;																		$HSEL_RSLT_INNERTEXT(4) - InnerText of selected node
;                  $fLoadHtml [Optional] - If False (default is True), the function will use last loaded html.
;                  
; Return values..: Success - Depends on $iIndex and $iRetRslt, if set and valid, returns requested data from the array as set bellow,
;								otherwise the return is 2D array as following:
;													[0][0] - Total number of elements
;													[0][1-4] - Not used
;													[N][$HSEL_RSLT_TYPE] - Type (name) of selected node
;													[N][$HSEL_RSLT_ATTRIBS] - 2D array with attributes of selected node as following:
;																										[0][0] - Total number of attributes
;																										[0][1] - Not used
;																										[N][$HSEL_RSLTATTRB_NAME(0)] - Attribute name
;																										[N][$HSEL_RSLTATTRB_VALUE(1)] - Attribute value
;													[N][$HSEL_RSLT_OUTERHTML] - OuterHtml of selected node
;													[N][$HSEL_RSLT_INNERHTML] - InnerHtml of selected node
;													[N][$HSEL_RSLT_INNERTEXT] - InnerText of selected node
;                  Failure - 0 and set @error as following:
;															1 - Unable to initialize HtmlAgilityPack.HtmlDocument object (check Framework version, 2 or 4).
;															2 - Nodes not found (SelectNodes method did not returned an object).
;															3 - Selected nodes count = 0.
; Author.........: G.Sandler
; Remarks........: Check the XPath syntax here: https://msdn.microsoft.com/ru-ru/library/vstudio/ms256086(v=vs.100).aspx
; Related........: _HtmlSelector_SelectByCSS, _HtmlSelector_Init
; Link(s)........: http://htmlagilitypack.codeplex.com/
; Example........: Yes.
; ===============================================================================================================
Func _HtmlSelector_SelectByXPath($sHtml, $sQuery, $iIndex = -1, $iRetRslt = $HSEL_RSLT_OUTERHTML, $fLoadHtml = True)
	Local $sHtml_File, $sPS_File, $sPS_Code, $hFile, $iPID, $iError, $sRead, $sErrRead
	Local $aRet, $oSelect, $oAttribs, $aAttribs
	
	$sQuery = StringReplace($sQuery, '"', "'")
	
	If Not IsObj($oHSEL_HAPHtml) Then
		Return SetError(1, 0, 0)
	EndIf
	
	If $fLoadHtml Then
		$oHSEL_HAPHtml.LoadHtml($sHtml)
	EndIf
	
	$oSelect = $oHSEL_HAPHtml.DocumentNode.SelectNodes($sQuery)
	
	If Not IsObj($oSelect) Then
		Return SetError(2, 0, 0)
	EndIf
	
	Dim $aRet[$oSelect.Count + 1][5]
	
	For $i = 0 To $oSelect.Count - 1
		$oAttribs = $oSelect($i).Attributes
		
		If IsObj($oAttribs) Then
			Dim $aAttribs[$oAttribs.Count + 1][2]
			
			For $x = 0 To $oAttribs.Count - 1
				$aAttribs[0][0] += 1
				$aAttribs[$aAttribs[0][0]][0] = $oAttribs($x).Name
				$aAttribs[$aAttribs[0][0]][1] = $oAttribs($x).Value
			Next
		Else
			Dim $aAttribs[1][2] = [[0]]
		EndIf
		
		$aRet[0][0] += 1
		$aRet[$aRet[0][0]][0] = $oSelect($i).Name
		$aRet[$aRet[0][0]][1] = $aAttribs
		$aRet[$aRet[0][0]][2] = $oSelect($i).OuterHtml
		$aRet[$aRet[0][0]][3] = $oSelect($i).InnerHtml
		$aRet[$aRet[0][0]][4] = StringRegExpReplace($oSelect($i).InnerText, '<!--(?! // )|(?<! // )-->|<[^>]+>', '')
		
		$aAttribs = 0
	Next
	
	If $aRet[0][0] = 0 Then
		Return SetError(3, 0, 0)
	EndIf
	
	If $iIndex > 0 And $iIndex <= $aRet[0][0] And $iRetRslt >= 0 And $iRetRslt < $HSEL_RSLT_TOTAL Then
		Return $aRet[$iIndex][$iRetRslt]
	EndIf
	
	Return $aRet
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _HtmlSelector_SelectByCSS
; Description....: Select node(s) from html using CSS selector query.
; Syntax.........: _HtmlSelector_SelectByCSS($sHtml, $sQuery, $iIndex = -1, $iRetRslt = $HSEL_RSLT_OUTERHTML)
; Parameters.....: $sHtml - Html string to select from.
;                  $sQuery - CSS selector query, comma separated list of CSS selectors, e.g.: div, #id, div[align="right"].
;                  $iIndex [Optional] - If this is <> -1, then the return is the array element set by $iIndex (depending on $iRetRslt parameter).
;                  $iRetRslt [Optional] - Define the return data by $iIndex (if set), the following are supported:
;                  														$HSEL_RSLT_TYPE(0) - Type (name) of selected node (ex.: $aResult[$iIndex][$HSEL_RSLT_TYPE])
;																		$HSEL_RSLT_ATTRIBS(1) - 2D array with attributes of selected node
;																		$HSEL_RSLT_OUTERHTML(2) - OuterHtml of selected node (default)
;																		$HSEL_RSLT_INNERHTML(3) - InnerHtml of selected node
;																		$HSEL_RSLT_INNERTEXT(4) - InnerText of selected node
;                  $fLoadHtml [Optional] - If False (default is True), the function will use last loaded html.
;
; Return values..: Success - Depends on $iIndex and $iRetRslt, if set and valid, returns requested data from the array as set bellow,
;								otherwise the return is 2D array as following:
;													[0][0] - Total number of elements
;													[0][1-4] - Not used
;													[N][$HSEL_RSLT_TYPE] - Type (name) of selected node
;													[N][$HSEL_RSLT_ATTRIBS] - 2D array with attributes of selected node as following:
;																										[0][0] - Total number of attributes
;																										[0][1] - Not used
;																										[N][$HSEL_RSLTATTRB_NAME(0)] - Attribute name
;																										[N][$HSEL_RSLTATTRB_VALUE(1)] - Attribute value
;													[N][$HSEL_RSLT_OUTERHTML] - OuterHtml of selected node
;													[N][$HSEL_RSLT_INNERHTML] - InnerHtml of selected node
;													[N][$HSEL_RSLT_INNERTEXT] - InnerText of selected node
;                  Failure - 0 and set @error as following:
;														1 - Unable to initialize HTMLayout engine.
;														2 - HTMLayoutLoadHtml error (if $fLoadHtml = True).
;														3 - HTMLayoutGetRootElement error.
;														4 - HTMLayoutSelectElements error.
;														5 - Unable to get elements from callback.
;														6 - Selected elements not found.
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 	* Check the CSS Selector syntax here: http://www.w3schools.com/cssref/css_selectors.asp
;					* If the encoding of input Html string is UTF-8,
;						the output will returned as ANSI, it should be converted using BinaryToString(StringToBinary($sOutput), 4).
; Related........: _HtmlSelector_SelectByXPath, _HtmlSelector_Init
; Link(s)........: http://www.terrainformatica.com/htmlayout/, https://github.com/Erls-Corporation/webinaria-source/blob/master/source/lib/htmlayout/htmlayout_dom.h
; Example........: Yes.
; ===============================================================================================================
Func _HtmlSelector_SelectByCSS($sHtml, $sQuery, $iIndex = -1, $iRetRslt = $HSEL_RSLT_OUTERHTML, $fLoadHtml = True)
	Local $aResult, $hCallback, $aType, $aAttribsCnt, $aAttrib
	Local $aRet, $aAttribs
	
	If $hHSEL_HLDLL = -1 Then
		Return SetError(1, 0, 0)
	EndIf
	
	If $fLoadHtml Then
		$aResult = DllCall($hHSEL_HLDLL, 'BOOL', 'HTMLayoutLoadHtml', 'HWND', $hHSEL_HLWnd, 'str', $sHtml, 'UINT', StringLen($sHtml))
		If @error Or Not $aResult[0] Then Return SetError(2, 0, 0)
	EndIf
	
	$aRoot = DllCall($hHSEL_HLDLL, 'int', 'HTMLayoutGetRootElement', 'HWND', $hHSEL_HLWnd, 'ptr*', '')
	If @error Or $aRoot[0] <> 0 Then Return SetError(3, 0, 0)
	
	$hCallback = DllCallbackRegister('__HTMLSelector_HLCallback', 'BOOL', 'ptr;ptr')
	$aResult = DllCall($hHSEL_HLDLL, 'int', 'HTMLayoutSelectElements', 'ptr', $aRoot[2], 'str', $sQuery, 'ptr', DllCallbackGetPtr($hCallback), 'ptr', '')
	If @error Or $aResult[0] <> 0 Then Return SetError(4, 0, 0)
	
	DllCallbackFree($hCallback)
	
	If $aHSEL_HLSelElmnts[0] = 0 Then
		$aHSEL_HLSelElmnts = 0
		Dim $aHSEL_HLSelElmnts[$iHSEL_HLDefSelElmnts] = [0]
		
		DllCall($hHSEL_HLDLL, 'int', 'HTMLayout_UnuseElement', 'ptr', $aRoot[2])
		DllCall($hHSEL_HLDLL, 'BOOL', 'HTMLayoutLoadHtml', 'HWND', $hHSEL_HLWnd, 'str', '', 'UINT', 0)
		
		Return SetError(5, 0, 0)
	EndIf
	
	Dim $aRet[$aHSEL_HLSelElmnts[0] + 1][5]
	
	For $i = 1 To $aHSEL_HLSelElmnts[0]
		$aRet[0][0] += 1
		
		$aType = DllCall($hHSEL_HLDLL, 'int', 'HTMLayoutGetElementType', 'ptr', $aHSEL_HLSelElmnts[$i], 'str*', '')
		
		If Not @error And $aType[0] = 0 Then
			$aRet[$aRet[0][0]][0] = $aType[2]
		EndIf
		
		$aAttribsCnt = DllCall($hHSEL_HLDLL, 'int', 'HTMLayoutGetAttributeCount', 'ptr', $aHSEL_HLSelElmnts[$i], 'UINT*', '')
		
		If Not @error And $aAttribsCnt[0] = 0 Then
			Dim $aAttribs[$aAttribsCnt[2] + 1][2]
			
			For $x = 0 To $aAttribsCnt[2] - 1
				$aAttrib = DllCall($hHSEL_HLDLL, 'int', 'HTMLayoutGetNthAttribute', 'ptr', $aHSEL_HLSelElmnts[$i], 'UINT', $x, 'str*', '', 'wstr*', '')
				If @error Or $aAttrib[0] <> 0 Then ContinueLoop
				
				$aAttribs[0][0] += 1
				$aAttribs[$aAttribs[0][0]][0] = $aAttrib[3]
				$aAttribs[$aAttribs[0][0]][1] = $aAttrib[4]
			Next
			
			ReDim $aAttribs[$aAttribs[0][0] + 1][2]
			$aRet[$aRet[0][0]][1] = $aAttribs
		EndIf
		
		$aResult = DllCall($hHSEL_HLDLL, 'int', 'HTMLayoutGetElementHtml', 'ptr', $aHSEL_HLSelElmnts[$i], 'str*', '', 'BOOL', True)
		
		If Not @error And $aResult[0] = 0 Then
			$aRet[$aRet[0][0]][2] = StringRegExpReplace($aResult[2], '^[\r\n]+', '')
		EndIf
		
		$aResult = DllCall($hHSEL_HLDLL, 'int', 'HTMLayoutGetElementHtml', 'ptr', $aHSEL_HLSelElmnts[$i], 'str*', '', 'BOOL', False)
		
		If Not @error And $aResult[0] = 0 Then
			$aRet[$aRet[0][0]][3] = StringRegExpReplace($aResult[2], '^[\r\n]+', '')
		EndIf
		
		$aResult = DllCall($hHSEL_HLDLL, 'int', 'HTMLayoutGetElementInnerText', 'ptr', $aHSEL_HLSelElmnts[$i], 'str*', '')
		
		If Not @error And $aResult[0] = 0 Then
			$aRet[$aRet[0][0]][4] = $aResult[2]
		EndIf
	Next
	
	DllCall($hHSEL_HLDLL, 'int', 'HTMLayout_UnuseElement', 'ptr', $aRoot[2])
	;DllCall($hHSEL_HLDLL, 'BOOL', 'HTMLayoutLoadHtml', 'HWND', $hHSEL_HLWnd, 'str', '', 'UINT', 0)
	
	$aHSEL_HLSelElmnts = 0
	Dim $aHSEL_HLSelElmnts[$iHSEL_HLDefSelElmnts] = [0]
	
	ReDim $aRet[$aRet[0][0] + 1][5]
	
	If $aRet[0][0] = 0 Then
		Return SetError(6, 0, 0)
	EndIf
	
	If $iIndex > 0 And $iIndex <= $aRet[0][0] And $iRetRslt >= 0 And $iRetRslt < $HSEL_RSLT_TOTAL Then
		Return $aRet[$iIndex][$iRetRslt]
	EndIf
	
	Return $aRet
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _HtmlSelector_GetAttribute
; Description....: Get attribute value of specified element in the selected array.
; Syntax.........: _HtmlSelector_GetAttribute($aSelResult, $sAttribName, $iIndex = 1)
; Parameters.....: $aSelResult - Selection result array as returned by _HtmlSelector_SelectBy* functions.
;				   $sAttribName - Attribute name to get the value from.
;				   $iIndex [Optional] - Array index to get the attributes from (default is 1).
;                  
; Return values..: Success - Returns requested attribute value.
;                  Failure - Return empty string ('') and set @error as following:
;																	1 - $aSelResult is not array or it's not a 2D array.
;																	2 - $iIndex is less than 1 or grater than total elements in $aSelResult.
;																	3 - Requested attributes not found in $aSelResult.
;																	4 - Attribute with specified name not found.
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: _HtmlSelector_SelectByXPath, _HtmlSelector_SelectByCSS
; Link...........: 
; Example........: Yes.
; ===============================================================================================================
Func _HtmlSelector_GetAttribute($aSelResult, $sAttribName, $iIndex = 1)
	If Not IsArray($aSelResult) Or UBound($aSelResult, 0) <> 2 Then
		Return SetError(1, 0, '')
	EndIf
	
	If $iIndex < 1 Or $iIndex > $aSelResult[0][0] Then
		Return SetError(2, 0, '')
	EndIf
	
	Local $aAttribs = $aSelResult[$iIndex][$HSEL_RSLT_ATTRIBS]
	
	If Not IsArray($aAttribs) Or UBound($aAttribs, 0) <> 2 Then
		Return SetError(3, 0, '')
	EndIf
	
	For $i = 1 To $aAttribs[0][0]
		If $aAttribs[$i][$HSEL_RSLTATTRB_NAME] = $sAttribName Then
			Return $aAttribs[$i][$HSEL_RSLTATTRB_VALUE]
		EndIf
	Next
	
	Return SetError(4, 0, '')
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _HtmlSelector_Init
; Description....: Initialize library engines.
; Syntax.........: _HtmlSelector_Init($iFlag = 2)
; Parameters.....: $iFlag [Optional] - 0 to init only XPath selector, 1 to init only CSS selector, 2 (default) to init both selectors.
;                  
; Return values..: Success - Returns 1.
;                  Failure - Return 0 and set @error as following:
;																	1 - Unable to initialize HtmlAgilityPack.dll (check Framework version, 2 or 4).
;																	2 - Unable to initialize HtmLayout.dll.
;																	3 - Combined @error 1 and 2.
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: _HtmlSelector_SelectByXPath, _HtmlSelector_SelectByCSS
; Link...........: 
; Example........: Yes.
; ===============================================================================================================
Func _HTMLSelector_Init($iFlag = 2)
	Local Static $iInit = False
	
	If $iInit Then
		Return
	EndIf
	
	$iInit = True
	
	OnAutoItExitRegister('__HTMLSelector_OnExit')
	
	Local $iHAP_Init = 0, $aResult
	
	If $iFlag <> 1 Then
		FileInstall('HtmlAgilityPack.dll', $sHSEL_HAPDLLFile, 1)
		$iHAP_Init = __HTMLSelector_DotNetLoad($sHSEL_HAPDLLFile)
		$oHSEL_HAPHtml = ObjCreate('HtmlAgilityPack.HtmlDocument')
		$iHAP_Init = ($iHAP_Init And IsObj($oHSEL_HAPHtml))
	EndIf
	
	If $iFlag <> 0 Then
		FileInstall('HtmLayout.dll', $sHSEL_HLDLLFile, 1)
		
		$hHSEL_HLDLL = DllOpen($sHSEL_HLDLLFile)
		
		If $hHSEL_HLDLL <> -1 Then
			$aResult = DllCall($hHSEL_HLDLL, 'wstr', 'HTMLayoutClassNameW')
			
			If Not @error Then
				$aResult = DllCall('user32.dll', 'hwnd', 'CreateWindowExW', 'dword', 0, 'wstr', $aResult[0], 'wstr', '', _
					'dword', 0, 'int', 0, 'int', 0, 'int', 0, 'int', 0, 'hwnd', 0, 'handle', 0, 'handle', 0, 'ptr', 0)
				
				If Not @error Then
					$hHSEL_HLWnd = $aResult[0]
				EndIf
			EndIf
			
			If Not IsHWnd($hHSEL_HLWnd) Then
				DllClose($hHSEL_HLDLL)
				$hHSEL_HLDLL = -1
			EndIf
		EndIf
	EndIf
	
	If Not $iHAP_Init And $hHSEL_HLDLL = -1 Then
		Return SetError(3, 0, 0)
	EndIf
	
	If Not $iHAP_Init And $iFlag <> 1 Then
		Return SetError(1, 0, 0)
	EndIf
	
	If $hHSEL_HLDLL = -1 And $iFlag <> 0 Then
		Return SetError(2, 0, 0)
	EndIf
	
	Return 1
EndFunc

#Region Internal functions

; #INTERNAL_USE_ONLY# ===========================================================================================
; Name...........: __HTMLSelector_HLCallback()
; Description....: Callback function to get CSS selected elements.
; ===============================================================================================================
Func __HTMLSelector_HLCallback($sSel, $vParam)
	$aHSEL_HLSelElmnts[0] += 1
	
	If $aHSEL_HLSelElmnts[0] >= $iHSEL_HLDefSelElmnts Then
		ReDim $aHSEL_HLSelElmnts[$aHSEL_HLSelElmnts[0] + $iHSEL_HLDefSelElmnts]
	EndIf
	
	$aHSEL_HLSelElmnts[$aHSEL_HLSelElmnts[0]] = $sSel
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================
; Name...........: __HTMLSelector_OnExit
; Description....: On Exit function to release resources.
; ===============================================================================================================
Func __HTMLSelector_OnExit()
	__HTMLSelector_DotNetUnloadAll()
	
	If IsHWnd($hHSEL_HLWnd) Then
		DllCall('user32.dll', 'bool', 'DestroyWindow', 'hwnd', $hHSEL_HLWnd)
	EndIf
	
	If $hHSEL_HLDLL <> -1 Then
		DllClose($hHSEL_HLDLL)
	EndIf
	
	FileDelete($sHSEL_HAPDLLFile)
	FileDelete($sHSEL_HLDLLFile)
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================
; Name ..........: __HTMLSelector_DotNetLoad
; Description ...: Load a .NET compiled dll assembly.
; Syntax ........: __HTMLSelector_DotNetLoad($sDllPath)
; Parameters ....: $sDllPath            - A .NET compiled dll assembly located in the @ScriptDir directory.
; Return values .: Success: True
;                  Failure: False and sets @error to non-zero:
;                       1 = Incorrect filetype aka not a dll.
;                       2 = Dll does not exist in the @ScriptDir location.
;                       3 = .NET RegAsm.exe file not found.
;                       4 = Dll already registered.
; Author ........: guinness
; Example .......: Yes
; ===============================================================================================================
Func __HTMLSelector_DotNetLoad($sDllPath)
	Local $bReturn = ___HTMLSelector_DotNetWrapper($sDllPath, $DOTNET_LOADDLL)
	Return SetError(@error, @extended, $bReturn)
EndFunc ;==>__HTMLSelector_DotNetLoad

; #INTERNAL_USE_ONLY# ===========================================================================================
; Name ..........: __HTMLSelector_DotNetUnload
; Description ...: Unload a previously registered .NET compiled dll assembly.
; Syntax ........: __HTMLSelector_DotNetUnload($sDllPath)
; Parameters ....: $sDllPath            - A .NET compiled dll assembly located in the @ScriptDir directory.
; Return values .: Success: True
;                  Failure: False and sets @error to non-zero:
;                       1 = Incorrect filetype aka not a dll.
;                       2 = Dll does not exist in the @ScriptDir location.
;                       3 = .NET RegAsm.exe file not found.
;                       4 = Dll already registered.
;                       5 = Maximum DLL count prepared to be used reached.
; Author ........: guinness
; Example .......: Yes
; ================================================================================================================
Func __HTMLSelector_DotNetUnload($sDllPath)
	Local $bReturn = ___HTMLSelector_DotNetWrapper($sDllPath, $DOTNET_UNLOADDLL)
	Return SetError(@error, @extended, $bReturn)
EndFunc ;==>__HTMLSelector_DotNetUnload

; #INTERNAL_USE_ONLY# ============================================================================================
; Name ..........: __HTMLSelector_DotNetUnloadAll
; Description ...: Unload all previously registered .NET compiled dll assemblies.
; Syntax ........: __HTMLSelector_DotNetUnloadAll()
; Parameters ....: None
; Return values .: Success: True
;                  Failure: False and sets @error to non-zero:
;                       1 = Incorrect filetype aka not a dll.
;                       2 = Dll does not exist in the @ScriptDir location.
;                       3 = .NET RegAsm.exe file not found.
;                       4 = Dll already registered.
; Author ........: guinness
; Example .......: Yes
; =================================================================================================================
Func __HTMLSelector_DotNetUnloadAll()
	Local $bReturn = ___HTMLSelector_DotNetWrapper(0, $DOTNET_UNLOADDLLALL)
	Return SetError(@error, @extended, $bReturn)
EndFunc ;==>__HTMLSelector_DotNetUnloadAll

; #INTERNAL_USE_ONLY# =============================================================================================
; Name ..........: ___HTMLSelector_DotNetWrapper
; Description ...: A wrapper for the __HTMLSelector_DotNet* functions.
; Syntax ........: ___HTMLSelector_DotNetWrapper($sDllPath, $iType)
; Parameters ....: $sDllPath            - A .NET compiled dll assembly located in the @ScriptDir directory.
;                  $iType               - A $DOTNET_* constant.
; Return values .: Success: True
;                  Failure: False and sets @error to non-zero:
;                       1 = Incorrect filetype aka not a dll.
;                       2 = Dll does not exist in the @ScriptDir location.
;                       3 = .NET RegAsm.exe file not found.
;                       4 = Dll already registered.
;                       5 = Maximum DLL count prepared to be used reached.
; Author ........: guinness
; Remarks .......: ### DO NOT INVOKE, AS THIS IS A WRAPPER FOR THE ABOVE FUNCTIONS. ###
; Related .......: Thanks to Bugfix for the initial idea: http://www.autoitscript.com/forum/topic/129164-create-a-net-class-and-run-it-as-object-from-your-autoit-script/?p=938459
; Example .......: Yes
; =================================================================================================================
Func ___HTMLSelector_DotNetWrapper($sDllPath, $iType)
	Local Static $aDllPaths[$DOTNET_PATHS_MAX][2] = [[0]], $sRegAsmPath = 0
	
	If Not ($iType = $DOTNET_UNLOADDLLALL) Then
		If Not (StringRight($sDllPath, StringLen('dll')) == 'dll') Then ; Check the correct filetype was passed.
			Return SetError(1, 0, False) ; Incorrect filetype.
		EndIf
		
		If Not FileExists($sDllPath) Then ; Check the filepath exists in @ScriptDir.
			Return SetError(2, 0, False) ; Filepath does not exist.
		EndIf
	EndIf
	
	If $sRegAsmPath == 0 Then
		$sRegAsmPath = RegRead('HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\.NETFramework', 'InstallRoot')
		
		If @error Then
			$sRegAsmPath = '' ; Set to an empty string to acknowledge that searching for the path happened.
		Else
			$sRegAsmPath = StringRegExpReplace($sRegAsmPath, '\\+$', '')
			
			Local $aFilePaths = ___HTMLSelector_DotNetFileListToArray($sRegAsmPath, '*', 2)
			Local $sNETFolder = ''
			
			If Not @error Then
				For $i = UBound($aFilePaths) - 1 To 1 Step -1
					If StringRegExp($aFilePaths[$i], '(?:[vV](?:4|2)\.0\.\d+)') Then
						$sNETFolder = $aFilePaths[$i]
						ExitLoop
					EndIf
					
;~ 					If FileExists($sRegAsmPath & '\' & $aFilePaths[$i] & '\RegAsm.exe') Then
;~ 						$sNETFolder = $aFilePaths[$i]
;~ 						ExitLoop
;~ 					EndIf
				Next
			EndIf
			
			$sRegAsmPath &= '\' & $sNETFolder & '\RegAsm.exe'
			
			If FileExists($sRegAsmPath) Then
				;OnAutoItExitRegister('__HTMLSelector_DotNetUnloadAll') ; Register when the AutoIt executable is closed.
			Else
				$sRegAsmPath = '' ; Set to an empty string to acknowledge that searching for the path happened.
			EndIf
		EndIf
	EndIf
	
	If $sRegAsmPath == '' Then
		Return SetError(3, 0, False) ; .NET Framework 2.0 or 4.0 required.
	EndIf
	
	Switch $iType
		Case $DOTNET_LOADDLL
			Local $iIndex = -1
			
			For $i = $DOTNET_PATHS_START To $aDllPaths[$DOTNET_PATHS_INDEX][0]
				If $sDllPath = $aDllPaths[$i][0] Then
					Return SetError(4, 0, False) ; Dll already registered.
				EndIf
				
				If $iIndex = -1 And $aDllPaths[$i][0] == '' Then
					$iIndex = $i
				EndIf
			Next
			
			If $iIndex = -1 Then
				$aDllPaths[$DOTNET_PATHS_INDEX][0] += 1
				$iIndex = $aDllPaths[$DOTNET_PATHS_INDEX][0]
			EndIf
			
			Local Const $iUBound = UBound($aDllPaths, 1)
			
			If $aDllPaths[$DOTNET_PATHS_INDEX][0] >= $iUBound Then
				;ReDim $aDllPaths[Ceiling($iUBound * 1.3)]
				;ReDim of static variables does not work on v3.3.8.1
				$aDllPaths[$DOTNET_PATHS_INDEX][0] -= 1
				Return SetError(5, 0, False) ; Maximum DLL count prepared to be used reached.
			EndIf
			
			$aDllPaths[$iIndex][0] = $sDllPath
			
			RunWait($sRegAsmPath & ' /s /codebase ' & $sDllPath & ' /regfile:' & $sDllPath & '.reg', @ScriptDir, @SW_HIDE) ; Export AsmInfo to .reg file
			___HTMLSelector_DotNetReplaceStringInFile($sDllPath & '.reg', 'HKEY_CLASSES_ROOT', 'HKEY_CURRENT_USER\Software\Classes')
			RunWait('reg import ' & $sDllPath & '.reg', @ScriptDir, @SW_HIDE) ; Import to current users classes
			
			Local $sRegFile = FileRead($sDllPath & '.reg')
			Local $aRegFile = StringSplit(StringStripCR($sRegFile), @LF, 3)
			
			For $i = 0 To UBound($aRegFile) - 1
				If StringLeft($aRegFile[$i], 4) == '@="{' Then
					$aDllPaths[$iIndex][1] = StringTrimRight(StringTrimLeft($aRegFile[$i], 3), 1)
					ExitLoop
				EndIf
			Next
			
			FileDelete($sDllPath & '.reg')
		Case $DOTNET_UNLOADDLL
			For $i = $DOTNET_PATHS_START To $aDllPaths[$DOTNET_PATHS_INDEX][0]
				If $sDllPath = $aDllPaths[$i][0] And Not ($aDllPaths[$i][0] == 0) Then
					RunWait($sRegAsmPath & ' /s /unregister ' & $aDllPaths[$i][0], @ScriptDir, @SW_HIDE) ; Unregister the .NET Dll.
					RegDelete('HKCU\Software\Classes\CLSID\' & $aDllPaths[$i][1]) ;64Bit path
					RegDelete('HKCR\Wow6432Node\CLSID\' & $aDllPaths[$i][1]) ;32Bit path
					$aDllPaths[$i][0] = 0 ; Remove item.
					$aDllPaths[$i][1] = 0 ; Remove item.
				EndIf
			Next
		Case $DOTNET_UNLOADDLLALL
			If $sDllPath == 0 And $aDllPaths[$DOTNET_PATHS_INDEX][0] > 0 Then
				For $i = $DOTNET_PATHS_START To $aDllPaths[$DOTNET_PATHS_INDEX][0]
					If Not ($aDllPaths[$i][0] == 0) Then
						RunWait($sRegAsmPath & ' /s /unregister ' & $aDllPaths[$i][0], @ScriptDir, @SW_HIDE) ; Unregister the .NET Dll.
						RegDelete('HKCU\Software\Classes\CLSID\' & $aDllPaths[$i][1]) ;64Bit path
						RegDelete('HKCR\Wow6432Node\CLSID\' & $aDllPaths[$i][1]) ;32Bit path
						$aDllPaths[$i][0] = 0 ; Remove item.
						$aDllPaths[$i][1] = 0 ; Remove item.
					EndIf
				Next
				$aDllPaths[$DOTNET_PATHS_INDEX][0] = 0 ; Reset the count.
			EndIf
	EndSwitch
	
	Return True
EndFunc

; #INTERNAL_USE_ONLY# =============================================================================================
; Name ..........: ___HTMLSelector_DotNetFileListToArray
; Description ...: Lists files and\or folders in a specified folder (Similar to using Dir with the /B Switch).
; Author ........: AutoIt Team
; =================================================================================================================
Func ___HTMLSelector_DotNetFileListToArray($sFilePath, $sFilter = "*", $iFlag = 0, $fReturnPath = False)
	Local $sDelimiter = "|", $sFileList = "", $sFileName = "", $sFullPath = ""
	
	; Check parameters for the Default keyword or they meet a certain criteria
	$sFilePath = StringRegExpReplace($sFilePath, "[\\/]+$", "") & "\" ; Ensure a single trailing backslash
	If $iFlag = Default Then $iFlag = 0
	If $fReturnPath Then $sFullPath = $sFilePath
	If $sFilter = Default Then $sFilter = "*"
	
	; Check if the directory exists
	If Not FileExists($sFilePath) Then Return SetError(1, 0, 0)
	If StringRegExp($sFilter, "[\\/:><\|]|(?s)^\s*$") Then Return SetError(2, 0, 0)
	If Not ($iFlag = 0 Or $iFlag = 1 Or $iFlag = 2) Then Return SetError(3, 0, 0)
	Local $hSearch = FileFindFirstFile($sFilePath & $sFilter)
	If @error Then Return SetError(4, 0, 0)
	
	While 1
		$sFileName = FileFindNextFile($hSearch)
		If @error Then ExitLoop
		If ($iFlag + @extended = 2) Then ContinueLoop
		$sFileList &= $sDelimiter & $sFullPath & $sFileName
	WEnd
	
	FileClose($hSearch)
	If $sFileList = "" Then Return SetError(4, 0, 0)
	
	Return StringSplit(StringTrimLeft($sFileList, 1), $sDelimiter)
EndFunc

; #INTERNAL_USE_ONLY# =============================================================================================
; Name ..........: ___HTMLSelector_DotNetReplaceStringInFile
; Description ...: Replaces substrings in a file.
; Author ........: AutoIt Team
; =================================================================================================================
Func ___HTMLSelector_DotNetReplaceStringInFile($sFilePath, $sSearchString, $sReplaceString, $iCaseSensitive = 0, $iOccurance = 1)
	If StringInStr(FileGetAttrib($sFilePath), "R") Then Return SetError(1, 0, -1)
	
	; Open the file for reading.
	Local $hFileOpen = FileOpen($sFilePath, 0)
	If $hFileOpen = -1 Then Return SetError(2, 0, -1)
	
	; Read the contents of the file and stores in a variable
	Local $sFileRead = FileRead($hFileOpen)
	FileClose($hFileOpen) ; Close the open file after reading
	
	; Set the default parameters
	If $iCaseSensitive = Default Then $iCaseSensitive = 0
	If $iOccurance = Default Then $iOccurance = 1
	
	; Replace strings
	$sFileRead = StringReplace($sFileRead, $sSearchString, $sReplaceString, 1 - $iOccurance, $iCaseSensitive)
	Local $iReturn = @extended
	
	; If there are replacements then overwrite the file
	If $iReturn Then
		; Retrieve the file encoding
		Local $iFileEncoding = FileGetEncoding($sFilePath)
		
		; Open the file for writing and set the overwrite flag
		$hFileOpen = FileOpen($sFilePath, $iFileEncoding + 2)
		If $hFileOpen = -1 Then Return SetError(3, 0, -1)
		
		; Write to the open file
		FileWrite($hFileOpen, $sFileRead)
		FileClose($hFileOpen) ; Close the open file after writing
	EndIf
	
	Return $iReturn
EndFunc

#EndRegion
