#include-once

#Region Header

#CS Info
	Title:          TCPInet UDF
    Filename:       TCPInet.au3
    Description:    Inet* analog functions. Allows to download files with more options (such as pause background downloads, callback, download to variable etc.).
    Author:         G.Sandler a.k.a (Mr)CreatoR (CreatoR's Lab - www.creator-lab.ucoz.ru, www.autoit-script.ru)
    Version:        0.3
    Requirements:   AutoIt v3.3.6.1 - v3.3.12.0, Developed/Tested on Win 7 x64 (rus).
	
	Notes:			* This library does not support cache and ssl (https).
					* This library sets Opt('TCPTimeOut', 10000) (every time _TCPInet_GetFile or _TCPInet_GetSize is called)
					  !!! It's important for TCP functions to work properly !!!
					* When using TCPShutdown, do not forget to set TCPStartup before using functions from this library.
					* Use background mode with caution, if downloading many large files at once, it can consume large portion of memory
					 (make queue and use _TCPInet_SetMaxBufferLen to correlate the memory consumption).
	
	Credits:		Yashied
	
	History:
	                v0.3
					* Fixed issue with FileOpen/Write, now used WinAPI to write to the file.
					* Fixed issue with error 3 when some hosts does not allow empty Referer ($sReferer = '').
					* Changed return values for _TCPInet_GetFile and _TCPInet_GetInfo (see functions description).
					* Increased TCPTimeOut to 10000.
					
					v0.2
					* Now file can be downloaded to memory (see _TCPInet_GetFile description).
					* Minor fixes.
					* Examples changed.
					
					v0.1
					* First public version.
	
	ToDo:
					* Check for FileWrite errors and return them as errors.
					* Add optional parameter to force close socket (when the download completed).
					* Add cache and https support (perhaps use WinHTTP.au3).
					* Perhaps try to make the same "handles logic" as in InetGet function (handles from InetGet are growing, never inserted instead of empty one).
#CE

#EndRegion

#Region User Constants

Global Const $TCPINET_DOWNLOADWAIT 			= 0
Global Const $TCPINET_DOWNLOADBACKGROUND 	= 1

Global Const $TCPINET_DOWNLOADREAD 			= 0
Global Const $TCPINET_DOWNLOADSIZE 			= 1
Global Const $TCPINET_DOWNLOADCOMPLETE		= 2
Global Const $TCPINET_DOWNLOADSUCCESS 		= 3
Global Const $TCPINET_DOWNLOADERROR 		= 4
Global Const $TCPINET_DOWNLOADPAUSED 		= 5
Global Const $TCPINET_DOWNLOADFILE 			= 6
Global Const $TCPINET_DOWNLOADDATA 			= 7
Global Const $INET_DOWNLOADPAUSED 			= $TCPINET_DOWNLOADPAUSED

Global Const $TCPINET_CB_CONTINUE			= 0
Global Const $TCPINET_CB_PAUSE				= 1
Global Const $TCPINET_CB_RESUME				= 2
Global Const $TCPINET_CB_ABORT				= 3

#EndRegion

#Region Global Variables

Global Enum _
	$iTI_DATA_iSOCKET, $iTI_DATA_hFILE, $iTI_DATA_sFILE, $iTI_DATA_bDATA, $iTI_DATA_iREAD, $iTI_DATA_iSIZE, _
	$iTI_DATA_fCOMPLETE, $iTI_DATA_fSUCCESS, $iTI_DATA_fERROR, $iTI_DATA_fPAUSED, _
	$iTI_DATA_fHEADERSTRIPPED, $iTI_DATA_bBUFFER, $iTI_DATA_sCALLBACK, $iTI_DATA_vCBDATA, _
	$iTI_DATA_iTOTAL

Global $fTCPINET_BUSY						= False
Global $aTCPINET_TIMER[2]					= [DllCallbackRegister('__TCPInet_SetDataProc', 'none', 'hwnd;int;uint_ptr;dword'), 0]
Global $aTCPINET_DATA[1][1]					= [[0]]

Global Const $sTCPINET_USERAGENT			= 'Mozilla/5.0 (Windows NT 6.1; rv:27.0) Gecko/20100101 Firefox/27.0'
Global Const $sTCPINET_HTTPVER				= '1.1'
Global Const $iTCPINET_PORT					= 80				;Default port to use on http functions
Global Const $iTCPINET_CBELAPSE				= 200 				;Callback timer delay (time in ms between each function call - affecting on download speed)
Global Const $iTCPINET_MAXBYTES				= 1 * 1024 * 1024 	;1 MB - Maximum bytes to get with TCPRecv
Global $iTCPINET_MAXBUFFERLEN 				= 16 * 1024 * 1024	;16 MB - Maximum buffer lenght (for FileWrite chunks)

TCPStartup()
OnAutoItExitRegister('__TCPInet_OnExit')

#EndRegion

#Region Public Functions

; #FUNCTION# ====================================================================================================
; Name...........: _TCPInet_GetFile
; Description....: Downloads a file from the internet using the HTTP protocol.
; Syntax.........: _TCPInet_GetFile($sURL, $sFile, $iBackground = $INET_DOWNLOADWAIT)
; Parameters.....: $sURL - URL of the file to download.
;                  $sFile [Optional] - Local filename/path to download to.
;									   If $sFile = '' (default), then data downloaded to memory (see Return value for details).
;                  $fOverwrite [Optional] - If True, will overwrite the local file if it's already exists (default is False).
;                  $iBackground [Optional] - 
;                  							$INET_DOWNLOADWAIT (0) = Wait until the download is complete before continuing (default).
;											$INET_DOWNLOADBACKGROUND (1) = return immediately and download in the background (see remarks).
;                  $sCallBack_Func [Optional] - Callback function to call when using $INET_DOWNLOADBACKGROUND. This parameter ignored on $INET_DOWNLOADWAIT.
;												* When the download is paused or it's completed, the callback function not called (only once when download completed)
;												* Callback function will stop main script execution until it returns.
;												* The function always called with 2 parameters - _MyCallback($hInet, $vCBData = ''):
;																$hInet - Handle from _TCPInet_GetFile.
;																$vCBData - User defined data.
;                  $vCBData [Optional] - User defined data parameter to pass to Callback function ($sCallBack_Func).
;                  
; Return values..: Success - the return value changes depending on if the download is in the background:
;								Background: a handle is returned.
;								Wait: If filename/path specified ($sFile <> ''), the number of bytes downloaded,
;										otherwise downloaded binary data and @extended set to number of bytes downloaded.
;                  Failure - Background/Wait: 0 and set @error as following:
;								1 - Failed to connect to the host.
;								2 - Failed to get request data (GET).
;								3 - Server did not returned 200 OK status. In this case @extended set to status code returned from HTTP request.
;								[The following 2 errors are set only if filename/path specified ($sFile <> '')]:
;								4 - $fOverwrite parameter is False and destination file ($sFile) exists.
;								5 - Unable to open destination file ($sFile).
;							 Wait: 0 and set @error as following:
;								1 - Downloaded file/Binary data is incomplete (filesize <> content-length).
;								2 - File write error.
;							  	Other than 1 and 2 - error code from WSAGetLastError (http://msdn.microsoft.com/en-us/library/windows/desktop/ms741580(v=vs.85).aspx).
;
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 	Use the returned handle with _TCPInet_GetInfo() to determine if there was an error with the download.
;					The returned handle must be closed with _TCPInet_Close().
;					The URL parameter should be in the form "http://www.somesite.com/path/file.html". https does not supported.
;
;					* Notes about the "background" Parameter *
;					By default the function waits until the download has finished before returning.
;					If the background parameter is set to $INET_DOWNLOADBACKGROUND (1) the function returns immediately and the download continues in the background.
;					The function _TCPInet_GetInfo() can be used to check the status of the download or retrieve binary data (if filename/path not specified).
;					  It takes the handle returned from _TCPInet_GetFile().
;
;					* Notes about the "memory download" mode *
;					If filename/path does not specified ($sFile = ''), the download binary data stored in memory (variable).
;					Make sure that it will be less than 2,147,483,647 bytes (~2 Gb), it's an AutoIt MAX_BINARYSIZE limitation.
;
;					Multiple downloads are supported if they are started in background mode.
;					To abort a download call _TCPInet_Close() and pass it the handle returned by _TCPInet_GetFile().
;					To pause a download call _TCPInet_Pause() and pass it the handle returned by _TCPInet_GetFile().
; Related........: 
; Link...........: 
; Example........: Yes.
; ===============================================================================================================
Func _TCPInet_GetFile($sURL, $sFile = '', $fOverwrite = False, $iBackground = $TCPINET_DOWNLOADWAIT, $sCallBack_Func = '', $vCBData = '')
	Local $aHostPage = __TCPInet_SplitURL($sURL)
	Local $sHost = $aHostPage[0], $sPage = $aHostPage[1], $iPort = $aHostPage[2], $sUserName = $aHostPage[4], $sPassword = $aHostPage[5]
	Local $sHeader = '', $sLocation = '', $sUserNamePass = ''
	Local $hFile = 0
	
	Opt('TCPTimeOut', 10000)
	
	;Try 3 times to get redirected url
	For $i = 1 To 3
		If $sUserName <> '' Then
			$sUserNamePass = $sUserName & ':' & $sPassword
		EndIf
		
		$sHeader = __TCPInet_GetHeader($sHost, $sPage, $iPort, -1, $sUserNamePass)
		$sLocation = __TCPInet_GetTagHeader($sHeader, 'Location')
		
		If $sLocation = '' Then
			ExitLoop
		EndIf
		
		$sURL = $sLocation
		$aHostPage = __TCPInet_SplitURL($sURL)
		Dim $sHost = $aHostPage[0], $sPage = $aHostPage[1], $iPort = $aHostPage[2], $sUserName = $aHostPage[4], $sPassword = $aHostPage[5]
	Next
	
	Local $iSocket = __TCPInet_Connect($sHost, $iPort)
	
	If @error Then
		Return SetError(1, 0, 0)
	EndIf
	
	__TCPInet_Request($sHost, $sPage, $iSocket, 'GET', $sUserNamePass)
	
	If @error Then
		TCPCloseSocket($iSocket)
		Return SetError(2, 0, 0)
	EndIf
	
	If $sLocation = '' Then
		$sHeader = __TCPInet_GetHeader($sHost, $sPage, $iPort, -1, $sUserNamePass)
	EndIf
	
	If Not StringRegExp($sHeader, '^HTTP/\d.\d (20[0-2]|401).*') Then ;401 is Authorization Required (we set it in http request)
		Return SetError(3, Number(StringRegExpReplace($sHeader, '^HTTP/\d.\d (\d{3}).*', '\1')), 0)
	EndIf
	
	Local $iContent_Length = Number(__TCPInet_GetTagHeader($sHeader, 'Content-Length'))
	
	If $iContent_Length = 0 Then
		$iContent_Length = InetGetSize($sURL)
	EndIf
	
	If $sFile <> '' Then
		__TCPInet_SetURLFile($sURL, $sHeader, $sFile)
		
		If Not $fOverwrite And FileExists($sFile) Then
			Return SetError(4, 0, 0)
		EndIf
		
		$hFile = __TCPInet_FileOpenEx($sFile)
		
		If @error Or Not $hFile Then
			Return SetError(5, 0, 0)
		EndIf
	EndIf
	
	If $iBackground = $TCPINET_DOWNLOADBACKGROUND Then
		Local $hTCPInet = 0
		
		$fTCPINET_BUSY = True
		
		;Search for closed (empty) handle element
		For $i = 1 To $aTCPINET_DATA[0][0]
			If $aTCPINET_DATA[$i][$iTI_DATA_iSOCKET] = 0 Then
				;If found, set it instead
				$hTCPInet = $i
				ExitLoop
			EndIf
		Next
		
		If Not $hTCPInet Then
			$aTCPINET_DATA[0][0] += 1
			ReDim $aTCPINET_DATA[$aTCPINET_DATA[0][0] + 1][$iTI_DATA_iTOTAL]
			$hTCPInet = $aTCPINET_DATA[0][0]
		EndIf
		
		$aTCPINET_DATA[$hTCPInet][$iTI_DATA_iSOCKET] = $iSocket
		$aTCPINET_DATA[$hTCPInet][$iTI_DATA_hFILE] = $hFile
		$aTCPINET_DATA[$hTCPInet][$iTI_DATA_sFILE] = $sFile
		$aTCPINET_DATA[$hTCPInet][$iTI_DATA_bDATA] = Binary('')
		$aTCPINET_DATA[$hTCPInet][$iTI_DATA_iREAD] = 0
		$aTCPINET_DATA[$hTCPInet][$iTI_DATA_iSIZE] = $iContent_Length
		$aTCPINET_DATA[$hTCPInet][$iTI_DATA_fCOMPLETE] = False
		$aTCPINET_DATA[$hTCPInet][$iTI_DATA_fSUCCESS] = False
		$aTCPINET_DATA[$hTCPInet][$iTI_DATA_fERROR] = 0
		$aTCPINET_DATA[$hTCPInet][$iTI_DATA_fPAUSED] = False
		$aTCPINET_DATA[$hTCPInet][$iTI_DATA_fHEADERSTRIPPED] = False
		$aTCPINET_DATA[$hTCPInet][$iTI_DATA_bBUFFER] = Binary('')
		$aTCPINET_DATA[$hTCPInet][$iTI_DATA_sCALLBACK] = $sCallBack_Func
		$aTCPINET_DATA[$hTCPInet][$iTI_DATA_vCBDATA] = $vCBData
		
		$fTCPINET_BUSY = False
		
		__TCPInet_SetTimer()
		Return $hTCPInet
	EndIf
	
	Local $bData = 0, $iError = 0, $iExtended = 0, $vRet = 0, $bHeader_Stripped = False
	Local $bBuff = Binary('')
	
	While 1
		$bData = TCPRecv($iSocket, $iTCPINET_MAXBYTES, 1)
		
		If @error <> 0 Then
			Local $aWS2_32_Err = DllCall('ws2_32.dll', 'int', 'WSAGetLastError')
			If Not @error Then $aWS2_32_Err = $aWS2_32_Err[0]
			
			$iError = $aWS2_32_Err
			ExitLoop
		EndIf
		
		If Not $bHeader_Stripped Then
			$bHeader_Stripped = __TCPInet_TrimHeader($bData)
		EndIf
		
		If $hFile Then
			$bBuff += $bData
			$vRet += BinaryLen($bData)
			
			If BinaryLen($bBuff) >= $iTCPINET_MAXBUFFERLEN Then
				__TCPInet_FileWriteEx($hFile, $bBuff)
				
				If @error Then
					$iError = 2
					ExitLoop
				EndIf
				
				$bBuff = Binary('')
			EndIf
		Else
			$vRet += $bData
			$iExtended += BinaryLen($bData)
		EndIf
	WEnd
	
	TCPCloseSocket($iSocket)
	
	If $hFile Then
		If BinaryLen($bBuff) Then
			__TCPInet_FileWriteEx($hFile, $bBuff)
			
			If @error Then
				$iError = 2
			EndIf
		EndIf
		
		__TCPInet_FileCloseEx($hFile)
		
		If $iError = 0 And ($iContent_Length > 0 And $vRet <> $iContent_Length) Then
			$iError = 1
		EndIf
		
		If $iError > 0 Then
			$vRet = 0
		EndIf
	Else
		If $iError = 0 And ($iContent_Length > 0 And $iExtended <> $iContent_Length) Then
			$iError = 1
		EndIf
		
		If $iError > 0 Then
			$iExtended = 0
		EndIf
	EndIf
	
	Return SetError($iError, $iExtended, $vRet)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _TCPInet_GetSize
; Description....: Returns the size (in bytes) of a file located on the internet.
; Syntax.........: _TCPInet_GetSize($sURL)
; Parameters.....: $sURL - URL of the file to get size.
;                  
;                 
; Return values..: Success - the size of the file in bytes.
;                  Failure - 0 and sets the @error as following:
;								1 - Failed to connect to the host.
;								2 - Failed to get Content-Length header.
;
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 	The URL parameter should be in the form "http://www.somesite.com/path/file.html" - just like an address you would type into your web browser.
;					Not all servers will correctly give the file size, especially when using a proxy server.
; Related........: 
; Link...........: 
; Example........: Yes.
; ===============================================================================================================
Func _TCPInet_GetSize($sURL)
	Local $sUserNamePass = ''
	Local $aHostPage = __TCPInet_SplitURL($sURL)
	
	Opt('TCPTimeOut', 10000)
	
	If $aHostPage[4] <> '' Then
		$sUserNamePass = $aHostPage[4] & ':' & $aHostPage[5]
	EndIf
	
	Local $sHeader = __TCPInet_GetHeader($aHostPage[0], $aHostPage[1], $aHostPage[2], -1, $sUserNamePass)
	
	If @error Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $iContent_Length = __TCPInet_GetTagHeader($sHeader, 'Content-Length')
	
	If @error Then
		Return SetError(2, 0, 0)
	EndIf
	
	Return SetError(0, 0, Number($iContent_Length))
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _TCPInet_GetInfo
; Description....: Returns detailed data for a handle returned from _TCPInet_GetFile().
; Syntax.........: _TCPInet_GetInfo($hTCPInet, $iIndex = -1)
; Parameters.....: $hTCPInet [Optional] - A handle returned from _TCPInet_GetFile().
;                  $iIndex [Optional] - The index for the data to retrieve. If this value is -1 an array containing all of the below data will be returned.
;								$TCPINET_DOWNLOADREAD (0) - Bytes read so far (this is updated while the download progresses).
;								$TCPINET_DOWNLOADSIZE (1) - The size of the download in bytes (this may not always be present).
;								$TCPINET_DOWNLOADCOMPLETE (2) - Set to True if the download is complete, False if the download is still ongoing.
;								$TCPINET_DOWNLOADSUCCESS (3) - True if the download was successful (can be False if there was some problem to get file size).
;								$TCPINET_DOWNLOADERROR (4) - The error value for the download:
;															(testing that the value is non-zero is sufficient for determining if an error occurred)
;																1 - Downloaded file is incomplete (filesize <> content-length).
;							  									2 - File write error.
;																Other than 1 and 2 - error code from WSAGetLastError.
;								$TCPINET_DOWNLOADPAUSED (5) - True if the download is currently paused.
;								$TCPINET_DOWNLOADFILE (6) - Full path to local downloaded file (usefull in callback function).
;								$TCPINET_DOWNLOADDATA (7) - Downloaded binary data in case file was not specified (data is released, see remarks).
;
; Return values..: Success - the request data.
;                  Failure - an empty string and sets the @error as following:
;								1 - Invalid/Closed $hTCPInet handle.
;								2 - Invalid $iIndex.
;
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 	* If called with no arguments then the total number of active downloads will be returned.
;					* This function can be called in a loop to query the number of bytes download or to pause until a download is complete.
;					* !!! If $iIndex = -1 or if it's set to $TCPINET_DOWNLOADDATA, then binary data released, make sure to store it in your script !!!
; Related........: 
; Link...........: 
; Example........: Yes.
; ===============================================================================================================
Func _TCPInet_GetInfo($hTCPInet = 0, $iIndex = -1)
	If @NumParams = 0 Then
		Local $iRet = 0
		
		For $i = 1 To $aTCPINET_DATA[0][0]
			$iRet += Not $aTCPINET_DATA[$i][$iTI_DATA_fCOMPLETE]
		Next
		
		Return $iRet
	EndIf
	
	If $hTCPInet > $aTCPINET_DATA[0][0] Or $hTCPInet < 1 Or $aTCPINET_DATA[$hTCPInet][$iTI_DATA_iSOCKET] = 0 Then
		Return SetError(1, 0, '')
	EndIf
	
	Local $aRet[8] = _
		[ _
			$aTCPINET_DATA[$hTCPInet][$iTI_DATA_iREAD], _
			$aTCPINET_DATA[$hTCPInet][$iTI_DATA_iSIZE], _
			$aTCPINET_DATA[$hTCPInet][$iTI_DATA_fCOMPLETE], _
			$aTCPINET_DATA[$hTCPInet][$iTI_DATA_fSUCCESS], _
			$aTCPINET_DATA[$hTCPInet][$iTI_DATA_fERROR], _
			$aTCPINET_DATA[$hTCPInet][$iTI_DATA_fPAUSED], _
			$aTCPINET_DATA[$hTCPInet][$iTI_DATA_sFILE], _
			$aTCPINET_DATA[$hTCPInet][$iTI_DATA_bDATA] _
		]
	
	If $iIndex = -1 Then
		$aTCPINET_DATA[$hTCPInet][$iTI_DATA_bDATA] = Binary('')
		Return $aRet
	Else
		If $iIndex >= 0 And $iIndex <= UBound($aRet) Then
			If $iIndex = $iTI_DATA_bDATA Then
				$aTCPINET_DATA[$hTCPInet][$iTI_DATA_bDATA] = Binary('')
			EndIf
			
			Return $aRet[$iIndex]
		EndIf
	EndIf
	
	Return SetError(2, 0, '')
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _TCPInet_GetOpenHandles
; Description....: Get all opened TCPInet handles (opened by _TCPInet_GetFile).
; Syntax.........: _TCPInet_GetOpenHandles()
; Parameters.....: None.
; Return values..: Array of opened TCPInet handles ([0] - total number of handles).
;
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: Yes.
; ===============================================================================================================
Func _TCPInet_GetOpenHandles()
	Local $aRet[$aTCPINET_DATA[0][0] + 1] = [0]
	
	For $i = 1 To $aTCPINET_DATA[0][0]
		If $aTCPINET_DATA[$i][$iTI_DATA_iSOCKET] Then
			$aRet[0] += 1
			$aRet[$aRet[0]] = $i
		EndIf
	Next
	
	ReDim $aRet[$aRet[0] + 1]
	Return $aRet
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _TCPInet_Close
; Description....: Closes a handle returned from _TCPInet_GetFile().
; Syntax.........: _TCPInet_Close($hTCPInet)
; Parameters.....: $hTCPInet - A handle returned from _TCPInet_GetFile().
;
; Return values..: Success - True if the handle was found and closed.
;                  Failure - 0 and sets @error to 1 if passed invalid/closed $hTCPInet handle.
;
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: Closing the handle to a download still in progress will cancel the download.
;					(new handle can be inserted instead of closed one, therfore, closed handle should not be used again)
; Related........: 
; Link...........: 
; Example........: Yes.
; ===============================================================================================================
Func _TCPInet_Close($hTCPInet)
	If $hTCPInet > $aTCPINET_DATA[0][0] Or $hTCPInet < 1 Or $aTCPINET_DATA[$hTCPInet][$iTI_DATA_iSOCKET] = 0 Then
		Return SetError(1, 0, 0)
	EndIf
	
	$fTCPINET_BUSY = True
	
	TCPCloseSocket($aTCPINET_DATA[$hTCPInet][$iTI_DATA_iSOCKET])
	
	If $aTCPINET_DATA[$hTCPInet][$iTI_DATA_hFILE] Then
		__TCPInet_FileCloseEx($aTCPINET_DATA[$hTCPInet][$iTI_DATA_hFILE])
	EndIf
	
	$aTCPINET_DATA[$hTCPInet][$iTI_DATA_iSOCKET] = 0
	$aTCPINET_DATA[$hTCPInet][$iTI_DATA_bBUFFER] = Binary('')
	$aTCPINET_DATA[$hTCPInet][$iTI_DATA_bDATA] = Binary('')
	$aTCPINET_DATA[$hTCPInet][$iTI_DATA_vCBDATA] = ''
	
	Local $iClose = 0
	
	For $i = $aTCPINET_DATA[0][0] To 1 Step -1
		If $aTCPINET_DATA[$i][$iTI_DATA_iSOCKET] <> 0 Then
			ExitLoop
		EndIf
		
		$iClose += 1
	Next
	
	;ReDim if the rest of the downloads are closed
	$aTCPINET_DATA[0][0] -= $iClose
	ReDim $aTCPINET_DATA[$aTCPINET_DATA[0][0] + 1][$iTI_DATA_iTOTAL]
	
	;Reset if no more active downloads
	If $aTCPINET_DATA[0][0] = 0 Then
		__TCPInet_KillTimer()
	EndIf
	
	$fTCPINET_BUSY = False
	
	Return True
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _TCPInet_Pause
; Description....: Pauses specified download.
; Syntax.........: _TCPInet_Pause($hTCPInet, $bPause = True)
; Parameters.....: $hTCPInet - A handle returned from _TCPInet_GetFile().
;                  $bPause [Optional] - True for pause, False to resume the download.
;
; Return values..: Success - Pauses download.
;                  Failure - 0 and set @error to 1 if $hTCPInet is invalid/closed.
;
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: When paused, the callback function in _TCPInet_GetFile() does not called.
; Related........: 
; Link...........: 
; Example........: Yes.
; ===============================================================================================================
Func _TCPInet_Pause($hTCPInet, $bPause = True)
	If $hTCPInet > $aTCPINET_DATA[0][0] Or $hTCPInet < 1 Or $aTCPINET_DATA[$hTCPInet][$iTI_DATA_iSOCKET] = 0 Then
		Return SetError(1, 0, 0)
	EndIf
	
	$fTCPINET_BUSY = True
	
	If BinaryLen($aTCPINET_DATA[$hTCPInet][$iTI_DATA_bBUFFER]) Then
		If $aTCPINET_DATA[$hTCPInet][$iTI_DATA_hFILE] Then
			__TCPInet_FileWriteEx($aTCPINET_DATA[$hTCPInet][$iTI_DATA_hFILE], $aTCPINET_DATA[$hTCPInet][$iTI_DATA_bBUFFER])
			
			If @error Then
				$aTCPINET_DATA[$hTCPInet][$iTI_DATA_fCOMPLETE] = True
				$aTCPINET_DATA[$hTCPInet][$iTI_DATA_fSUCCESS] = False
				$aTCPINET_DATA[$hTCPInet][$iTI_DATA_fERROR] = 2
			EndIf
		Else
			$aTCPINET_DATA[$hTCPInet][$iTI_DATA_bDATA] += $aTCPINET_DATA[$hTCPInet][$iTI_DATA_bBUFFER]
		EndIf
	EndIf
	
	$aTCPINET_DATA[$hTCPInet][$iTI_DATA_fPAUSED] = $bPause
	
	$fTCPINET_BUSY = False
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _TCPInet_SetMaxBufferLen
; Description....: Set maximum buffer length (FileWrite chunks). This function does not applied for download to memory mode.
; Syntax.........: _TCPInet_SetMaxBufferLen($iLen)
; Parameters.....: $iLen - Maximum buffer length in MBytes (1 to 16).
;
; Return values..: Success - Sets $iTCPINET_MAXBUFFERLEN to specified length.
;                  Failure - 0 and set @error to 1 if $iLen is invalid (< 1 or > 16).
;
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: Yes.
; ===============================================================================================================
Func _TCPInet_SetMaxBufferLen($iLen)
	If $iLen < 1 Or $iLen > 16 Then
		Return SetError(1, 0, 0)
	EndIf
	
	$iTCPINET_MAXBUFFERLEN = $iLen * 1024 * 1024
EndFunc

#EndRegion

#Region INTERNAL FUNCTIONS

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_SetDataProc
; Description....: Internal function to set background downloads data (called by __TCPInet_SetTimer every 250 ms).
; Author.........: G.Sandler
; ===============================================================================================================
Func __TCPInet_SetDataProc($hWnd, $iMsg, $iIDTimer, $dwTime)
	#forceref $hWnd, $iMsg, $iIDTimer, $dwTime
	
	If $fTCPINET_BUSY Then
		Return
	EndIf
	
	Local $iSocket, $hFile, $bData, $iDataLen, $iCall_Ret
	
	For $i = 1 To $aTCPINET_DATA[0][0]
		If $i > $aTCPINET_DATA[0][0] Then ;Array changed
			ExitLoop
		EndIf
		
		If Not $aTCPINET_DATA[$i][$iTI_DATA_iSOCKET] Or $aTCPINET_DATA[$i][$iTI_DATA_fCOMPLETE] Then
			ContinueLoop
		EndIf
		
		If Not $aTCPINET_DATA[$i][$iTI_DATA_fPAUSED] Then
			$iSocket = $aTCPINET_DATA[$i][$iTI_DATA_iSOCKET]
			$hFile = $aTCPINET_DATA[$i][$iTI_DATA_hFILE]
			
			$bData = TCPRecv($iSocket, $iTCPINET_MAXBYTES, 1)
			
			If @error <> 0 Then
				Local $aWS2_32_Err = DllCall('ws2_32.dll', 'uint', 'WSAGetLastError')
				
				If Not @error Then
					$aWS2_32_Err = $aWS2_32_Err[0]
				Else
					$aWS2_32_Err = 0
				EndIf
				
				$aTCPINET_DATA[$i][$iTI_DATA_fCOMPLETE] = True
				$aTCPINET_DATA[$i][$iTI_DATA_fSUCCESS] = ($aWS2_32_Err = 0)
				$aTCPINET_DATA[$i][$iTI_DATA_fERROR] = $aWS2_32_Err
				
				If $aWS2_32_Err = 0 Then
					;If File not downloaded completely...
					If $aTCPINET_DATA[$i][$iTI_DATA_iSIZE] > 0 And $aTCPINET_DATA[$i][$iTI_DATA_iREAD] <> $aTCPINET_DATA[$i][$iTI_DATA_iSIZE] Then
						$aTCPINET_DATA[$i][$iTI_DATA_fSUCCESS] = False
						$aTCPINET_DATA[$i][$iTI_DATA_fERROR] = 1
					EndIf
				EndIf
				
				TCPCloseSocket($iSocket)
				
				If $hFile Then
					If BinaryLen($aTCPINET_DATA[$i][$iTI_DATA_bBUFFER]) Then
						__TCPInet_FileWriteEx($hFile, $aTCPINET_DATA[$i][$iTI_DATA_bBUFFER])
						
						If @error Then
							$aTCPINET_DATA[$i][$iTI_DATA_fCOMPLETE] = True
							$aTCPINET_DATA[$i][$iTI_DATA_fSUCCESS] = False
							$aTCPINET_DATA[$i][$iTI_DATA_fERROR] = 2
						EndIf
					EndIf
					
					$aTCPINET_DATA[$i][$iTI_DATA_bBUFFER] = Binary('')
					__TCPInet_FileCloseEx($hFile)
				EndIf
			Else
				If Not $aTCPINET_DATA[$i][$iTI_DATA_fHEADERSTRIPPED] Then
					$aTCPINET_DATA[$i][$iTI_DATA_fHEADERSTRIPPED] = __TCPInet_TrimHeader($bData)
				EndIf
				
				$iDataLen = BinaryLen($bData)
				
				If $iDataLen = 0 Then
					ContinueLoop
				EndIf
				
				$aTCPINET_DATA[$i][$iTI_DATA_iREAD] += $iDataLen
				
				If $hFile Then
					$aTCPINET_DATA[$i][$iTI_DATA_bBUFFER] += $bData
					
					If BinaryLen($aTCPINET_DATA[$i][$iTI_DATA_bBUFFER]) >= $iTCPINET_MAXBUFFERLEN Then
						__TCPInet_FileWriteEx($hFile, $aTCPINET_DATA[$i][$iTI_DATA_bBUFFER])
						
						If @error Then
							$aTCPINET_DATA[$i][$iTI_DATA_fCOMPLETE] = True
							$aTCPINET_DATA[$i][$iTI_DATA_fSUCCESS] = False
							$aTCPINET_DATA[$i][$iTI_DATA_fERROR] = 2
						EndIf
						
						$aTCPINET_DATA[$i][$iTI_DATA_bBUFFER] = Binary('')
					EndIf
				Else
					$aTCPINET_DATA[$i][$iTI_DATA_bDATA] += $bData
				EndIf
			EndIf
		EndIf
		
		If $aTCPINET_DATA[$i][$iTI_DATA_sCALLBACK] <> '' Then
			$iCall_Ret = Call($aTCPINET_DATA[$i][$iTI_DATA_sCALLBACK], $i, $aTCPINET_DATA[$i][$iTI_DATA_vCBDATA]) ;$i - It's a handle ($hInet)
			
			If Not (@error = 0xDEAD And @extended = 0xBEEF) Then
				Switch $iCall_Ret
					Case $TCPINET_CB_CONTINUE
						;No action
					Case $TCPINET_CB_PAUSE
						$aTCPINET_DATA[$i][$iTI_DATA_fPAUSED] = True
					Case $TCPINET_CB_RESUME
						$aTCPINET_DATA[$i][$iTI_DATA_fPAUSED] = False
					Case $TCPINET_CB_ABORT
						_TCPInet_Close($i)
				EndSwitch
			EndIf
		EndIf
	Next
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_SetTimer
; Description....: Internal function to set timer.
; Author.........: G.Sandler
; ===============================================================================================================
Func __TCPInet_SetTimer()
	Local $iID = $aTCPINET_TIMER[1] ;Return from SetTimer early
	Local $pTimer = DllCallbackGetPtr($aTCPINET_TIMER[0])
	Local $aResult = DllCall('user32.dll', 'uint_ptr', 'SetTimer', 'hwnd', 0, 'uint_ptr', $iID, 'uint', $iTCPINET_CBELAPSE, 'ptr', $pTimer)
	
	If @error Or $aResult[0] = 0 Then
		Return SetError(1, 0, 0)
	EndIf
	
	$aTCPINET_TIMER[1] = $aResult[0]
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_KillTimer
; Description....: Internal function to kill previously set timer with __TCPInet_SetTimer.
; Author.........: G.Sandler
; ===============================================================================================================
Func __TCPInet_KillTimer()
	If $aTCPINET_TIMER[1] Then
		DllCall('user32.dll', 'bool', 'KillTimer', 'hwnd', 0, 'uint_ptr', $aTCPINET_TIMER[1])
		$aTCPINET_TIMER[1] = 0
	EndIf
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_Connect
; Description....: Internal function to connect to specified host.
; Syntax.........: __TCPInet_Connect($sHost)
; Parameters.....: $sHost - Url host name part.
;                  $sPort - [Optional] - Port to use in TCPConnect. Default is $iTCPINET_PORT.
; Return values..: Success - Connection socket.
;                  Failure - -1 and set @error to 1 if unable to connect.
; Author.........: G.Sandler
; ===============================================================================================================
Func __TCPInet_Connect($sHost, $iPort = '')
	If Not StringIsDigit($iPort) Then
		$iPort = $iTCPINET_PORT
	EndIf
	
	Local $sName_To_IP = TCPNameToIP($sHost)
	Local $iSocket = TCPConnect($sName_To_IP, $iPort)
	
	If $iSocket = -1 Or $iSocket = 0 Then
		Return SetError(1, 0, -1)
	EndIf
	
	Return $iSocket
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_Request
; Description....: Internal function to send HTTP request.
; Syntax.........: __TCPInet_Request($sHost, $sPage, $iSocket, $sRequest = 'GET', $sUserNamePass = '', $sReferer = '')
; Parameters.....: $sHost - Url host name part.
;                  $sPage - Url page part.
;                  $iSocket - Connection socket returned from __TCPInet_Connect().
;                  $sRequest [Optional] - Request type (GET, HEAD, etc.). Default is GET.
;                  $sUserNamePass [Optional] - UserName and Password in "name:pass" format.
;                  $sReferer [Optional] - Referer for HTTP request.
; Return values..: Success - Number of bytes sent during request.
;                  Failure - 0 and sets @error to 1 if no bytes sent, @extended set to @error from TCPSend.
; Author.........: G.Sandler
; ===============================================================================================================
Func __TCPInet_Request($sHost, $sPage, $iSocket, $sRequest = 'GET', $sUserNamePass = '', $sReferer = '')
	Local $sCommand = ''
	
	$sCommand &= $sRequest & ' ' & $sPage & ' HTTP/' & $sTCPINET_HTTPVER & @CRLF
	$sCommand &= 'Host: ' & $sHost & @CRLF
	$sCommand &= 'User-Agent: ' & $sTCPINET_USERAGENT & @CRLF
	
	If $sReferer Then
		$sCommand &= 'Referer: ' & $sReferer & @CRLF
	EndIf
	
	;$sCommand &= 'Cache-Control: no-cache' & @CRLF
	;$sCommand &= 'Accept: */*' & @CRLF
	
	If $sUserNamePass Then
		$sCommand &= 'Authorization: Basic ' & StringStripWS(__TCPInet_Base64Encode($sUserNamePass), 3) & @CRLF
	EndIf
	
	$sCommand &= 'Connection: close' & @CRLF & @CRLF
	
	Local $iBytesSent = TCPSend($iSocket, $sCommand)
	Local $iError = @error
	
	Return SetError(Number($iBytesSent = 0), $iError, $iBytesSent)
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_GetHeader
; Description....: Internal function to get header.
; Syntax.........: __TCPInet_GetHeader($sHost, $sPage, $iPort = $iTCPINET_PORT, $iSocket = -1, $sUserNamePass = '', $sReferer = '')
; Parameters.....: $sHost - Url host name part.
;                  $sPage - Url page part.
;                  $sPort [Optional] - Port to use in __TCPInet_Connect. Default is $iTCPINET_PORT.
;                  $iSocket [Optional] - Socket as returned from __TCPInet_Connect. Default is 0 (open new socket).
;										  ($sHost and $sPage must be the same as used in __TCPInet_Request, $iPort ignored in this case)
;                  $sUserNamePass [Optional] - UserName and Password in "name:pass" format.
;                  $sReferer [Optional] - Referer for HTTP request.
; Return values..: Success - Header data.
;                  Failure - Empty string and sets @error as following:
;								1 - Failed to connect.
;								2 - Failed to get HEAD request.
; Author.........: G.Sandler
; ===============================================================================================================
Func __TCPInet_GetHeader($sHost, $sPage, $iPort = $iTCPINET_PORT, $iSocket = -1, $sUserNamePass = '', $sReferer = '')
	Local $sData = ''
	
	If $iSocket = -1 Or $iSocket = 0 Then
		$iSocket = __TCPInet_Connect($sHost, $iPort)
		
		If @error Then
			Return SetError(1, 0, '')
		EndIf
		
		__TCPInet_Request($sHost, $sPage, $iSocket, 'HEAD', $sUserNamePass, $sReferer)
		
		If @error Then
			TCPCloseSocket($iSocket)
			Return SetError(2, 0, '')
		EndIf
	EndIf
	
	Local $iTimer = TimerInit()
	
	While 1
		$sData &= TCPRecv($iSocket, $iTCPINET_MAXBYTES)
		
		If @error <> 0 Or TimerDiff($iTimer) >= 5000 Then
			ExitLoop
		EndIf
	WEnd
	
	TCPCloseSocket($iSocket)
	
	Return $sData
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_GetTagHeader
; Description....: Internal function to get header tag.
; Syntax.........: __TCPInet_GetTagHeader($sHeader, $sTag)
; Parameters.....: $sHeader - Header data returned from __TCPInet_GetHeader().
;                  $sTag - Tag to retrieve ("Content-Length" for example).
; Return values..: Success - Header tag value string.
;                  Failure - Empty string and sets @error to 1 if tag not found.
; Author.........: G.Sandler
; ===============================================================================================================
Func __TCPInet_GetTagHeader($sHeader, $sTag)
	Local $aRet = StringRegExp($sHeader, '(?i)' & $sTag & ': ([^\r\n]+)', 3)
	
	If @error Or Not IsArray($aRet) Then
		Return SetError(1, 0, '')
	EndIf
	
	Return $aRet[0]
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_TrimHeader
; Description....: Internal function to remove header from data.
; Syntax.........: __TCPInet_TrimHeader($bData)
; Parameters.....: $bData - Binary data with header.
; Return values..: Binary data without header.
; Author.........: Yashied
; ===============================================================================================================
Func __TCPInet_TrimHeader(ByRef $bData)
	Local $iCRLF_Pos = StringInStr(BinaryMid($bData, 1, 4096), '0D0A0D0A', 2)
	
	If $iCRLF_Pos Then
		$bData = BinaryMid($bData, ($iCRLF_Pos + 7) / 2)
		Return True
	EndIf
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_SplitURL
; Description....: Internal function to split URL to Host and Page.
; Syntax.........: __TCPInet_SplitURL($sURL)
; Parameters.....: $sURL - URL to split.
; Return values..: Array with 3 elements: [0] - Host part, [1] - Page part, [2] - Port part.
; Author.........: G.Sandler
; ===============================================================================================================
Func __TCPInet_SplitURL($sURL)
	Local Const $URL_PART_SCHEME = 1, $URL_PART_HOSTNAME = 2, $URL_PART_USERNAME = 3, $URL_PART_PASSWORD = 4, $URL_PART_PORT = 5, $URL_PART_QUERY = 6
	
	Local $sScheme = __TCPInet_UrlGetPart($sURL, $URL_PART_SCHEME)
	Local $sHost = __TCPInet_UrlGetPart($sURL, $URL_PART_HOSTNAME)
	Local $sUserName = __TCPInet_UrlGetPart($sURL, $URL_PART_USERNAME)
	Local $sPassword = __TCPInet_UrlGetPart($sURL, $URL_PART_PASSWORD)
	Local $sPort = __TCPInet_UrlGetPart($sURL, $URL_PART_PORT)
	Local $sPage = '/' & StringRegExpReplace($sURL, '(?is)^' & $sScheme & '://.*?\Q' & $sHost & '\E(?::' & $sPort & ')?(?:/+)?(.*?)$', '\1')
	
	Local $aRet[6] = [$sHost, $sPage, $sPort, $sScheme, $sUserName, $sPassword]
	Return $aRet
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_PathIsDirectory
; Description....: Internal function to check if certain path is directory.
; Syntax.........: __TCPInet_PathIsDirectory($sPath)
; Parameters.....: $sPath - Path to check.
; Return values..: True if given path is directory, otherwise False.
; Author.........: Yashied
; ===============================================================================================================
Func __TCPInet_PathIsDirectory($sPath)
	Local $Ret = DllCall('shlwapi.dll', 'bool', 'PathIsDirectoryW', 'wstr', $sPath)
	If @error Then Return SetError(@error, @extended, False)
	
	Return $Ret[0]
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_PathIsFileSpec
; Description....: Internal function to check a path for any path-delimiting characters.
; Syntax.........: __TCPInet_PathIsFileSpec($sPath)
; Parameters.....: $sPath - Path to check.
; Return values..: 	True if there are no path-delimiting characters (":" or "\") within the path. 
;					False: there are path-delimiting characters.
; Author.........: Yashied
; ===============================================================================================================
Func __TCPInet_PathIsFileSpec($sPath)
	Local $Ret = DllCall('shlwapi.dll', 'bool', 'PathIsFileSpecW', 'wstr', $sPath)
	If @error Then Return SetError(@error, @extended, False)
	
	Return $Ret[0]
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_UrlGetPart
; Description....: Internal function to retrieves a specified part from the URL.
; Syntax.........: __TCPInet_UrlGetPart($sUrl, $iPart)
; Parameters.....: $sUrl - The Url.
;				   $iPart - The part of the URL to retrieve. It can be one of the following values.
;				   		$URL_PART_HOSTNAME
;				   		$URL_PART_PASSWORD
;				   		$URL_PART_PORT
;				   		$URL_PART_QUERY
;				   		$URL_PART_SCHEME
;				   		$URL_PART_USERNAME
; Return values..: The part of the URL.
; Author.........: Yashied
; ===============================================================================================================
Func __TCPInet_UrlGetPart($sUrl, $iPart)
	Local $Ret = DllCall('shlwapi.dll', 'long', 'UrlGetPartW', 'wstr', $sUrl, 'wstr', '', 'dword*', 4096, 'dword', $iPart, 'dword', 0)
	If @error Then Return SetError(@error, @extended, '')
	If $Ret[0] Then Return SetError(10, $Ret[0], '')
	
	Return $Ret[2]
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_SetURLFile
; Description....: Internal function to set file full path (with filename).
; Syntax.........: __TCPInet_SetURLFile($sURL, $sHeader, ByRef $sFile)
; Parameters.....: $sURL - The URL.
;                  $sHeader - Header from HEAD request.
;                  $sFile - ByRef variable representing file path.
; Return values..: Sets $sFile to full proper file with filename.
; Author.........: G.Sandler
; ===============================================================================================================
Func __TCPInet_SetURLFile($sURL, $sHeader, ByRef $sFile)
	Local $iContent_Type = __TCPInet_GetTagHeader($sHeader, 'Content-Type')
	Local $iContent_Disposition = __TCPInet_GetTagHeader($sHeader, 'Content-Disposition')
	
	Local $sName = StringRegExpReplace($iContent_Disposition, '(?i).*?filename=(.*)', '\1')
	
	If @extended = 0 Then
		$sName = StringRegExpReplace($sURL, '^.*/', '')
	EndIf
	
	$sName = __TCPInet_GetValidFileName($sName)
	Local $sExt = '.' & __TCPInet_GetFileExtByMIMEType($iContent_Type)
	If $sExt = '.' Then $sExt = ''
	
	If $sName = '' Then
		$sName = 'file'
	EndIf
	
	If Not StringRegExp($sName, '\.[^\.]+$') Then
		$sName &= $sExt
	EndIf
	
	If __TCPInet_PathIsDirectory($sFile) Then
		$sFile &= '\' & $sName
	ElseIf __TCPInet_PathIsFileSpec($sFile) Then
		$sFile = $sName
	EndIf
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_FileOpenEx
; Description....: Internal function to open file using WinAPI.
; Syntax.........: __TCPInet_FileOpenEx($sFileName)
; Parameters.....: $sFileName - File to open.
; Return values..: On success returns file handle, on failure returns False and @error is set to 1.
; Author.........: G.Sandler
; ===============================================================================================================
Func __TCPInet_FileOpenEx($sFileName)
	Local $iDA, $iCD, $aResult
	
	$iDA = 1073741824 ;$GENERIC_WRITE
	$iCD = 2 ;$CREATE_ALWAYS
	
	$aResult = DllCall("kernel32.dll", "handle", "CreateFileW", "wstr", $sFileName, "dword", $iDA, "dword", 0, "ptr", 0, "dword", $iCD, "dword", 0, "ptr", 0)
	
	If @error Or ($aResult[0] = Ptr(-1)) Then ;$INVALID_HANDLE_VALUE
		Return SetError(1, 0, False)
	EndIf
	
	Return $aResult[0]
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_FileWriteEx
; Description....: Internal function to write to the file using WinAPI.
; Syntax.........: __TCPInet_FileWriteEx($hFile, $vData)
; Parameters.....: $hFile - File handle as returned by __TCPInet_FileOpenEx.
;                  $vData - Data to write.
; Return values..: On success returns True and @extended set to number of written bytes, on failure returns False and @error is set to 1.
; Author.........: G.Sandler
; ===============================================================================================================
Func __TCPInet_FileWriteEx($hFile, $vData)
	Local $iLen, $tBuffer, $pBuffer, $aResult
	
	If IsBinary($vData) Then
		$iLen = BinaryLen($vData)
	Else
		$iLen = StringLen($vData)
	EndIf
	
	$tBuffer = DllStructCreate('byte[' & $iLen & ']')
	DllStructSetData($tBuffer, 1, $vData)
	$pBuffer = DllStructGetPtr($tBuffer)
	
	$aResult = DllCall('kernel32.dll', 'bool', 'WriteFile', 'handle', $hFile, 'ptr', $pBuffer, 'dword', $iLen, 'dword*', 0, 'ptr', 0)
	
	If @error Or Not $aResult[0] Then
		Return SetError(1, 0, False)
	EndIf
	
	Return SetExtended($aResult[4], $aResult[0])
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_FileCloseEx
; Description....: Internal function to close file handle.
; Syntax.........: __TCPInet_FileCloseEx($hFile)
; Parameters.....: $hFile - File handle as returned by __TCPInet_FileOpenEx.
; Return values..: On success returns True, on failure returns False and @error is set to 1.
; Author.........: G.Sandler
; ===============================================================================================================
Func __TCPInet_FileCloseEx($hFile)
	Local $aResult = DllCall("kernel32.dll", "bool", "CloseHandle", "handle", $hFile)
	
	If @error Then
		Return SetError(1, 0, False)
	EndIf
	
	Return $aResult[0]
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_GetValidFileName
; Description....: Internal function to get valid local file name.
; Syntax.........: __TCPInet_GetValidFileName($sString, $sPattern = '[*?\/|:<>"]', $sReplace = '_')
; Parameters.....: $sString - String to convert to valid file name string.
;                  $sPattern [Optional] - RegExp Pattern to check (default is '[*?\\/|:<>"]').
;                  $sReplace [Optional] - Replace string of invalid chars (default is '_').
; Return values..: Valid file name string.
; Author.........: G.Sandler
; ===============================================================================================================
Func __TCPInet_GetValidFileName($sString, $sPattern = '[*?\\/|:<>"]', $sReplace = '_')
	If StringStripWS($sString, 8) = '' Then
		Return $sString
	EndIf
	
	If StringLen($sString) > 200 Then
		$sString = StringLeft($sString, 98) & '...' & StringRight($sString, 99)
	EndIf
	
	$sString = StringRegExpReplace($sString, $sPattern, $sReplace)
	Return SetExtended(@extended, StringRegExpReplace($sString, '(' & $sReplace & '+)', $sReplace))
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_GetFileExtByMIMEType
; Description....: Internal function to get file extension by MIME Type.
; Syntax.........: __TCPInet_GetFileExtByMIMEType($sMimeType)
; Parameters.....: $sMimeType - MIME Type.
; Return values..: File extension associated with specified MIME Type.
; Author.........: G.Sandler
; ===============================================================================================================
Func __TCPInet_GetFileExtByMIMEType($sMimeType)
    Local $sTypes = ''
    
	$sTypes &= "|ai;application/postscript|aif;audio/x-aiff|aifc;audio/x-aiff|aiff;audio/x-aiff|asc;text/plain|atom;application/atom+xml|au;audio/basic|avi;video/x-msvideo|bcpio;application/x-bcpio|bin;application/octet-stream|bmp;image/bmp|cdf;application/x-netcdf|cgm;image/cgm|class;application/octet-stream|cpio;application/x-cpio|cpt;application/mac-compactpro|csh;application/x-csh|css;text/css|dcr;application/x-director|dif;video/x-dv|dir;application/x-director|djv;image/vnd.djvu|djvu;image/vnd.djvu|dll;application/octet-stream|dmg;application/octet-stream|dms;application/octet-stream|doc;application/msword|dtd;application/xml-dtd|dv;video/x-dv|dvi;application/x-dvi|dxr;application/x-director|eps;application/postscript|etx;text/x-setext|exe;application/octet-stream|ez;application/andrew-inset|gif;image/gif|gram;application/srgs|grxml;application/srgs+xml|gtar;application/x-gtar|hdf;application/x-hdf|hqx;application/mac-binhex40|htm;text/html|html;text/html|ice;x-conference/x-cooltalk|ico;image/x-icon|ics;text/calendar|ief;image/ief|ifb;text/calendar|iges;model/iges|igs;model/iges|jnlp;application/x-java-jnlp-file|jp2;image/jp2|jpe;image/jpeg|jpeg;image/jpeg|jpg;image/jpeg|js;application/x-javascript|kar;audio/midi|latex;application/x-latex|lha;application/octet-stream|lzh;application/octet-stream|m3u;audio/x-mpegurl|m4a;audio/mp4a-latm|m4b;audio/mp4a-latm|m4p;audio/mp4a-latm|m4u;video/vnd.mpegurl|m4v;video/x-m4v|mac;image/x-macpaint|man;application/x-troff-man|mathml;application/mathml+xml|me;application/x-troff-me|mesh;model/mesh|mid;audio/midi|midi;audio/midi|mif;application/vnd.mif|mov;video/quicktime|movie;video/x-sgi-movie|mp4;video/mp4|mpe;video/mpeg|mpeg;video/mpeg|mpg;video/mpeg|ms;application/x-troff-ms|msh;model/mesh|mxu;video/vnd.mpegurl|nc;application/x-netcdf|oda;application/oda|ogg;application/ogg|pbm;image/x-portable-bitmap|pct;image/pict|mp3;audio/mpeg|"
	$sTypes &= "pdb;chemical/x-pdb|pdf;application/pdf|pgm;image/x-portable-graymap|pgn;application/x-chess-pgn|pic;image/pict|pict;image/pict|png;image/png|pnm;image/x-portable-anymap|pnt;image/x-macpaint|pntg;image/x-macpaint|ppm;image/x-portable-pixmap|ppt;application/vnd.ms-powerpoint|ps;application/postscript|qt;video/quicktime|qti;image/x-quicktime|qtif;image/x-quicktime|ra;audio/x-pn-realaudio|ram;audio/x-pn-realaudio|ras;image/x-cmu-raster|rdf;application/rdf+xml|rgb;image/x-rgb|rm;application/vnd.rn-realmedia|roff;application/x-troff|rtf;text/rtf|rtx;text/richtext|sgm;text/sgml|sgml;text/sgml|sh;application/x-sh|shar;application/x-shar|silo;model/mesh|sit;application/x-stuffit|skd;application/x-koan|skm;application/x-koan|skp;application/x-koan|skt;application/x-koan|smi;application/smil|smil;application/smil|snd;audio/basic|so;application/octet-stream|spl;application/x-futuresplash|src;application/x-wais-source|sv4cpio;application/x-sv4cpio|sv4crc;application/x-sv4crc|svg;image/svg+xml|swf;application/x-shockwave-flash|t;application/x-troff|tar;application/x-tar|tcl;application/x-tcl|tex;application/x-tex|texi;application/x-texinfo|texinfo;application/x-texinfo|tif;image/tiff|tiff;image/tiff|tr;application/x-troff|tsv;text/tab-separated-values|txt;text/plain|ustar;application/x-ustar|vcd;application/x-cdlink|vrml;model/vrml|vxml;application/voicexml+xml|wav;audio/x-wav|wbmp;image/vnd.wap.wbmp|wbmxl;application/vnd.wap.wbxml|wml;text/vnd.wap.wml|wmlc;application/vnd.wap.wmlc|wmls;text/vnd.wap.wmlscript|wmlsc;application/vnd.wap.wmlscriptc|wrl;model/vrml|xbm;image/x-xbitmap|xht;application/xhtml+xml|xhtml;application/xhtml+xml|xls;application/vnd.ms-excel|xml;application/xml|xpm;image/x-xpixmap|xsl;application/xml|xslt;application/xslt+xml|xul;application/vnd.mozilla.xul+xml|xwd;image/x-xwindowdump|xyz;chemical/x-xyz|zip;application/zip|"
	$sTypes &= ".123;application/vnd.lotus-1-2-3|3dml;text/vnd.in3d.3dml|3g2;video/3gpp2|3gp;video/3gpp|aab;application/x-authorware-bin|aac;audio/x-aac|aam;application/x-authorware-map|a;application/octet-stream|aas;application/x-authorware-seg|abw;application/x-abiword|acc;application/vnd.americandynamics.acc|ace;application/x-ace-compressed|acu;application/vnd.acucobol|acutc;application/vnd.acucorp|adp;audio/adpcm|aep;application/vnd.audiograph|afm;application/x-font-type1|afp;application/vnd.ibm.modcap|air;application/vnd.adobe.air-application-installer-package+zip|ami;application/vnd.amiga.ami|apk;application/vnd.android.package-archive|application;application/x-ms-application|apr;application/vnd.lotus-approach|asc;application/pgp-signature|asc;text/plain|asf;video/x-ms-asf|asm;text/x-asm|aso;application/vnd.accpac.simply.aso|asx;video/x-ms-asf|atc;application/vnd.acucorp|atomcat;application/atomcat+xml|atomsvc;application/atomsvc+xml|atx;application/vnd.antix.game-component|aw;application/applixware|azf;application/vnd.airzip.filesecure.azf|azs;application/vnd.airzip.filesecure.azs|azw;application/vnd.amazon.ebook|bat;application/x-msdownload|bdf;application/x-font-bdf|bdm;application/vnd.syncml.dm+wbxml|bh2;application/vnd.fujitsu.oasysprs|bmi;application/vnd.bmi|book;application/vnd.framemaker|box;application/vnd.previewsystems.box|boz;application/x-bzip2|bpk;application/octet-stream|btif;image/prs.btif|bz2;application/x-bzip2|bz;application/x-bzip|c4d;application/vnd.clonk.c4group|c4f;application/vnd.clonk.c4group|c4g;application/vnd.clonk.c4group|c4p;application/vnd.clonk.c4group|c4u;application/vnd.clonk.c4group|cab;application/vnd.ms-cab-compressed|car;application/vnd.curl.car|cat;application/vnd.ms-pki.seccat|cct;application/x-director|cc;text/x-c|ccxml;application/ccxml+xml|cdbcmsg;application/vnd.contact.cmsg|cdkey;application/vnd.mediastation.cdkey|cdx;chemical/x-cdx|cdxml;application/vnd.chemdraw+xml|cdy;application/vnd.cinderella|cer;application/pkix-cert|chat;application/x-chat|chm;application/vnd.ms-htmlhelp|"
	$sTypes &= "chrt;application/vnd.kde.kchart|cif;chemical/x-cif|cii;application/vnd.anser-web-certificate-issue-initiation|cil;application/vnd.ms-artgalry|cla;application/vnd.claymore|clkk;application/vnd.crick.clicker.keyboard|clkp;application/vnd.crick.clicker.palette|clkt;application/vnd.crick.clicker.template|clkw;application/vnd.crick.clicker.wordbank|clkx;application/vnd.crick.clicker|clp;application/x-msclip|cmc;application/vnd.cosmocaller|cmdf;chemical/x-cmdf|cml;chemical/x-cml|cmp;application/vnd.yellowriver-custom-menu|cmx;image/x-cmx|cod;application/vnd.rim.cod|com;application/x-msdownload|conf;text/plain|cpp;text/x-c|crd;application/x-mscardfile|crl;application/pkix-crl|crt;application/x-x509-ca-cert|csml;chemical/x-csml|csp;application/vnd.commonspace|cst;application/x-director|csv;text/csv|c;text/x-c|cu;application/cu-seeme|curl;text/vnd.curl|cww;application/prs.cww|cxt;application/x-director|cxx;text/x-c|daf;application/vnd.mobius.daf|dataless;application/vnd.fdsn.seed|davmount;application/davmount+xml|dcurl;text/vnd.curl.dcurl|dd2;application/vnd.oma.dd2+xml|ddd;application/vnd.fujixerox.ddd|deb;application/x-debian-package|def;text/plain|deploy;application/octet-stream|der;application/x-x509-ca-cert|dfac;application/vnd.dreamfactory|dic;text/x-c|diff;text/plain|dif;video/x-dv|dis;application/vnd.mobius.dis|dist;application/octet-stream|distz;application/octet-stream|dna;application/vnd.dna|docm;application/vnd.ms-word.document.macroenabled.12|docx;application/vnd.openxmlformats-officedocument.wordprocessingml.document|dot;application/msword|dotm;application/vnd.ms-word.template.macroenabled.12|dotx;application/vnd.openxmlformats-officedocument.wordprocessingml.template|dp;application/vnd.osgi.dp|dpg;application/vnd.dpgraph|dsc;text/prs.lines.tag|dtb;application/x-dtbook+xml|dts;audio/vnd.dts|dtshd;audio/vnd.dts.hd|dump;application/octet-stream|dv;video/x-dv|dwf;model/vnd.dwf|dwg;image/vnd.dwg|dxf;image/vnd.dxf|dxp;application/vnd.spotfire.dxp|ecma;application/ecmascript|edm;application/vnd.novadigm.edm|"
	$sTypes &= "edx;application/vnd.novadigm.edx|efif;application/vnd.picsel|ei6;application/vnd.pg.osasli|elc;application/octet-stream|eml;message/rfc822|emma;application/emma+xml|eol;audio/vnd.digital-winds|eot;application/vnd.ms-fontobject|epub;application/epub+zip|es3;application/vnd.eszigno3+xml|esf;application/vnd.epson.esf|et3;application/vnd.eszigno3+xml|ext;application/vnd.novadigm.ext|ez2;application/vnd.ezpix-album|ez3;application/vnd.ezpix-package|f4v;video/x-f4v|f77;text/x-fortran|f90;text/x-fortran|fbs;image/vnd.fastbidsheet|fdf;application/vnd.fdf|fe_launch;application/vnd.denovo.fcselayout-link|fg5;application/vnd.fujitsu.oasysgp|fgd;application/x-director|fh4;image/x-freehand|fh5;image/x-freehand|fh7;image/x-freehand|fhc;image/x-freehand|fh;image/x-freehand|fig;application/x-xfig|fli;video/x-fli|flo;application/vnd.micrografx.flo|flv;video/x-flv|flw;application/vnd.kde.kivio|flx;text/vnd.fmi.flexstor|fly;text/vnd.fly|fm;application/vnd.framemaker|fnc;application/vnd.frogans.fnc|for;text/x-fortran|fpx;image/vnd.fpx|frame;application/vnd.framemaker|fsc;application/vnd.fsc.weblaunch|fst;image/vnd.fst|ftc;application/vnd.fluxtime.clip|f;text/x-fortran|fti;application/vnd.anser-web-funds-transfer-initiation|fvt;video/vnd.fvt|fzs;application/vnd.fuzzysheet|g3;image/g3fax|gac;application/vnd.groove-account|gdl;model/vnd.gdl|geo;application/vnd.dynageo|gex;application/vnd.geometry-explorer|ggb;application/vnd.geogebra.file|ggt;application/vnd.geogebra.tool|ghf;application/vnd.groove-help|gim;application/vnd.groove-identity-message|gmx;application/vnd.gmx|gnumeric;application/x-gnumeric|gph;application/vnd.flographit|gqf;application/vnd.grafeq|gqs;application/vnd.grafeq|gre;application/vnd.geometry-explorer|grv;application/vnd.groove-injector|gsf;application/x-font-ghostscript|gtm;application/vnd.groove-tool-message|gtw;model/vnd.gtw|gv;text/vnd.graphviz|gz;application/x-gzip|h261;video/h261|h263;video/h263|h264;video/h264|hbci;application/vnd.hbci|hh;text/x-c|hlp;application/winhlp|hpgl;application/vnd.hp-hpgl|"
	$sTypes &= "hpid;application/vnd.hp-hpid|hps;application/vnd.hp-hps|h;text/x-c|htke;application/vnd.kenameaapp|hvd;application/vnd.yamaha.hv-dic|hvp;application/vnd.yamaha.hv-voice|hvs;application/vnd.yamaha.hv-script|icc;application/vnd.iccprofile|icm;application/vnd.iccprofile|ifm;application/vnd.shana.informed.formdata|igl;application/vnd.igloader|igx;application/vnd.micrografx.igx|iif;application/vnd.shana.informed.interchange|imp;application/vnd.accpac.simply.imp|ims;application/vnd.ms-ims|in;text/plain|ipk;application/vnd.shana.informed.package|irm;application/vnd.ibm.rights-management|irp;application/vnd.irepository.package+xml|iso;application/octet-stream|itp;application/vnd.shana.informed.formtemplate|ivp;application/vnd.immervision-ivp|ivu;application/vnd.immervision-ivu|jad;text/vnd.sun.j2me.app-descriptor|jam;application/vnd.jam|jar;application/java-archive|java;text/x-java-source|jisp;application/vnd.jisp|jlt;application/vnd.hp-jlyt|joda;application/vnd.joost.joda-archive|jp2;image/jp2|jpgm;video/jpm|jpgv;video/jpeg|jpm;video/jpm|json;application/json|karbon;application/vnd.kde.karbon|kfo;application/vnd.kde.kformula|kia;application/vnd.kidspiration|kil;application/x-killustrator|kml;application/vnd.google-earth.kml+xml|kmz;application/vnd.google-earth.kmz|kne;application/vnd.kinar|knp;application/vnd.kinar|kon;application/vnd.kde.kontour|kpr;application/vnd.kde.kpresenter|kpt;application/vnd.kde.kpresenter|ksh;text/plain|ksp;application/vnd.kde.kspread|ktr;application/vnd.kahootz|ktz;application/vnd.kahootz|kwd;application/vnd.kde.kword|kwt;application/vnd.kde.kword|lbd;application/vnd.llamagraphics.life-balance.desktop|lbe;application/vnd.llamagraphics.life-balance.exchange+xml|les;application/vnd.hhe.lesson-player|link66;application/vnd.route66.link66+xml|list3820;application/vnd.ibm.modcap|listafp;application/vnd.ibm.modcap|list;text/plain|log;text/plain|lostxml;application/lost+xml|lrf;application/octet-stream|lrm;application/vnd.ms-lrm|ltf;application/vnd.frogans.ltf|lvp;audio/vnd.lucent.voice|"
	$sTypes &= "lwp;application/vnd.lotus-wordpro|m13;application/x-msmediaview|m14;application/x-msmediaview|m1v;video/mpeg|m2v;video/mpeg|m4a;audio/mp4a-latm|m4b;audio/mp4a-latm|m4p;audio/mp4a-latm|ma;application/mathematica|mac;image/x-macpaint|mag;application/vnd.ecowin.chart|maker;application/vnd.framemaker|man;application/x-troff-man|man;text/troff|mb;application/mathematica|mbk;application/vnd.mobius.mbk|mbox;application/mbox|mc1;application/vnd.medcalcdata|mcd;application/vnd.mcd|mcurl;text/vnd.curl.mcurl|mdb;application/x-msaccess|mdi;image/vnd.ms-modi|me;application/x-troff-me|me;text/troff|mfm;application/vnd.mfmp|mgz;application/vnd.proteus.magazine|mht;message/rfc822|mhtml;message/rfc822|mime;message/rfc822|mj2;video/mj2|mjp2;video/mj2|mlp;application/vnd.dolby.mlp|mmd;application/vnd.chipnuts.karaoke-mmd|mmf;application/vnd.smaf|mmr;image/vnd.fujixerox.edmics-mmr|mny;application/x-msmoney|mobi;application/x-mobipocket-ebook|mp4a;audio/mp4|mp4s;application/mp4|mp4v;video/mp4|mpa;video/mpeg|mpc;application/vnd.mophun.certificate|mpg4;video/mp4|mpkg;application/vnd.apple.installer+xml|mpm;application/vnd.blueice.multipass|mpn;application/vnd.mophun.application|mpp;application/vnd.ms-project|mpt;application/vnd.ms-project|mpy;application/vnd.ibm.minipay|mqy;application/vnd.mobius.mqy|mrc;application/marc|ms;application/x-troff-ms|mscml;application/mediaservercontrol+xml|mseed;application/vnd.fdsn.mseed|mseq;application/vnd.mseq|msf;application/vnd.epson.msf|msi;application/x-msdownload|msl;application/vnd.mobius.msl|ms;text/troff|msty;application/vnd.muvee.style|mts;model/vnd.mts|mus;application/vnd.musician|musicxml;application/vnd.recordare.musicxml+xml|mvb;application/x-msmediaview|mwf;application/vnd.mfer|mxf;application/mxf|mxl;application/vnd.recordare.musicxml|mxml;application/xv+xml|mxs;application/vnd.triscape.mxs|nb;application/mathematica|ncx;application/x-dtbncx+xml|n-gage;application/vnd.nokia.n-gage.symbian.install|ngdat;application/vnd.nokia.n-gage.data|"
	$sTypes &= "nlu;application/vnd.neurolanguage.nlu|nml;application/vnd.enliven|nnd;application/vnd.noblenet-directory|nns;application/vnd.noblenet-sealer|nnw;application/vnd.noblenet-web|npx;image/vnd.net-fpx|nsf;application/vnd.lotus-notes|nws;message/rfc822|oa2;application/vnd.fujitsu.oasys2|oa3;application/vnd.fujitsu.oasys3|o;application/octet-stream|oas;application/vnd.fujitsu.oasys|obd;application/x-msbinder|obj;application/octet-stream|odb;application/vnd.oasis.opendocument.database|odc;application/vnd.oasis.opendocument.chart|odf;application/vnd.oasis.opendocument.formula|odft;application/vnd.oasis.opendocument.formula-template|odg;application/vnd.oasis.opendocument.graphics|odi;application/vnd.oasis.opendocument.image|odp;application/vnd.oasis.opendocument.presentation|ods;application/vnd.oasis.opendocument.spreadsheet|odt;application/vnd.oasis.opendocument.text|oga;audio/ogg|ogv;video/ogg|ogx;application/ogg|onepkg;application/onenote|onetmp;application/onenote|opf;application/oebps-package+xml|oprc;application/vnd.palm|org;application/vnd.lotus-organizer|osf;application/vnd.yamaha.openscoreformat|osfpvg;application/vnd.yamaha.openscoreformat.osfpvg+xml|otc;application/vnd.oasis.opendocument.chart-template|otf;application/x-font-otf|otg;application/vnd.oasis.opendocument.graphics-template|oth;application/vnd.oasis.opendocument.text-web|oti;application/vnd.oasis.opendocument.image-template|otm;application/vnd.oasis.opendocument.text-master|otp;application/vnd.oasis.opendocument.presentation-template|ots;application/vnd.oasis.opendocument.spreadsheet-template|ott;application/vnd.oasis.opendocument.text-template|oxt;application/vnd.openofficeorg.extension|p10;application/pkcs10|p12;application/x-pkcs12|p7b;application/x-pkcs7-certificates|p7c;application/pkcs7-mime|p7m;application/pkcs7-mime|p7r;application/x-pkcs7-certreqresp|p7s;application/pkcs7-signature|pas;text/x-pascal|pbd;application/vnd.powerbuilder6|pcf;application/x-font-pcf|pcl;application/vnd.hp-pcl|pclxl;application/vnd.hp-pclxl|"
	$sTypes &= "pcurl;application/vnd.curl.pcurl|pcx;image/x-pcx|pdb;application/vnd.palm|pdb;chemical/x-pdb|pfa;application/x-font-type1|pfb;application/x-font-type1|pfm;application/x-font-type1|pfr;application/font-tdpfr|pfx;application/x-pkcs12|pgp;application/pgp-encrypted|php;text/x-php|pict;image/pict|pkg;application/octet-stream|pki;application/pkixcmp|pkipath;application/pkix-pkipath|plb;application/vnd.3gpp.pic-bw-large|plc;application/vnd.mobius.plc|plf;application/vnd.pocketlearn|pls;application/pls+xml|pl;text/plain|pml;application/vnd.ctc-posml|pntg;image/x-macpaint|pnt;image/x-macpaint|portpkg;application/vnd.macports.portpkg|pot;application/vnd.ms-powerpoint|potm;application/vnd.ms-powerpoint.template.macroenabled.12|potx;application/vnd.openxmlformats-officedocument.presentationml.template|ppa;application/vnd.ms-powerpoint|ppam;application/vnd.ms-powerpoint.addin.macroenabled.12|ppd;application/vnd.cups-ppd|pps;application/vnd.ms-powerpoint|ppsm;application/vnd.ms-powerpoint.slideshow.macroenabled.12|ppsx;application/vnd.openxmlformats-officedocument.presentationml.slideshow|pptm;application/vnd.ms-powerpoint.presentation.macroenabled.12|pptx;application/vnd.openxmlformats-officedocument.presentationml.presentation|pqa;application/vnd.palm|prc;application/x-mobipocket-ebook|pre;application/vnd.lotus-freelance|prf;application/pics-rules|psb;application/vnd.3gpp.pic-bw-small|psd;image/vnd.adobe.photoshop|psf;application/x-font-linux-psf|p;text/x-pascal|ptid;application/vnd.pvi.ptid1|pub;application/x-mspublisher|pvb;application/vnd.3gpp.pic-bw-var|pwn;application/vnd.3m.post-it-notes|pwz;application/vnd.ms-powerpoint|pya;audio/vnd.ms-playready.media.pya|pyc;application/x-python-code|pyo;application/x-python-code|py;text/x-python|pyv;video/vnd.ms-playready.media.pyv|qam;application/vnd.epson.quickanime|qbo;application/vnd.intu.qbo|qfx;application/vnd.intu.qfx|qps;application/vnd.publishare-delta-tree|qtif;image/x-quicktime|qti;image/x-quicktime|qwd;application/vnd.quark.quarkxpress|qwt;application/vnd.quark.quarkxpress|"
	$sTypes &= "qxb;application/vnd.quark.quarkxpress|qxd;application/vnd.quark.quarkxpress|qxl;application/vnd.quark.quarkxpress|qxt;application/vnd.quark.quarkxpress|rar;application/x-rar-compressed|rcprofile;application/vnd.ipunplugged.rcprofile|rdz;application/vnd.data-vision.rdz|rep;application/vnd.businessobjects|res;application/x-dtbresource+xml|rif;application/reginfo+xml|rl;application/resource-lists+xml|rlc;image/vnd.fujixerox.edmics-rlc|rld;application/resource-lists-diff+xml|rmi;audio/midi|rmp;audio/x-pn-realaudio-plugin|rms;application/vnd.jcp.javame.midlet-rms|rnc;application/relax-ng-compact-syntax|rpm;application/x-rpm|rpss;application/vnd.nokia.radio-presets|rpst;application/vnd.nokia.radio-preset|rq;application/sparql-query|rs;application/rls-services+xml|rsd;application/rsd+xml|rss;application/rss+xml|rtf;application/rtf|rtf;text/rtf|saf;application/vnd.yamaha.smaf-audio|sbml;application/sbml+xml|sc;application/vnd.ibm.secure-container|scd;application/x-msschedule|scm;application/vnd.lotus-screencam|scq;application/scvp-cv-request|scs;application/scvp-cv-response|scurl;text/vnd.curl.scurl|sda;application/vnd.stardivision.draw|sdc;application/vnd.stardivision.calc|sdd;application/vnd.stardivision.impress|sdkd;application/vnd.solent.sdkm+xml|sdkm;application/vnd.solent.sdkm+xml|sdp;application/sdp|sdw;application/vnd.stardivision.writer|see;application/vnd.seemail|seed;application/vnd.fdsn.seed|sema;application/vnd.sema|semd;application/vnd.semd|semf;application/vnd.semf|ser;application/java-serialized-object|setpay;application/set-payment-initiation|setreg;application/set-registration-initiation|sfd-hdstx;application/vnd.hydrostatix.sof-data|sfs;application/vnd.spotfire.sfs|sgl;application/vnd.stardivision.writer-global|shf;application/shf+xml|sic;application/vnd.wap.sic|sig;application/pgp-signature|sis;application/vnd.symbian.install|sisx;application/vnd.symbian.install|si;text/vnd.wap.si|sitx;application/x-stuffitx|slc;application/vnd.wap.slc|sldm;application/vnd.ms-powerpoint.slide.macroenabled.12|"
	$sTypes &= "sldx;application/vnd.openxmlformats-officedocument.presentationml.slide|slt;application/vnd.epson.salt|sl;text/vnd.wap.sl|smf;application/vnd.stardivision.math|snf;application/x-font-snf|spc;application/x-pkcs7-certificates|spf;application/vnd.yamaha.smaf-phrase|spot;text/vnd.in3d.spot|spp;application/scvp-vp-response|spq;application/scvp-vp-request|spx;audio/ogg|srx;application/sparql-results+xml|sse;application/vnd.kodak-descriptor|ssf;application/vnd.epson.ssf|ssml;application/ssml+xml|stc;application/vnd.sun.xml.calc.template|std;application/vnd.sun.xml.draw.template|s;text/x-asm|stf;application/vnd.wt.stf|sti;application/vnd.sun.xml.impress.template|stk;application/hyperstudio|stl;application/vnd.ms-pki.stl|str;application/vnd.pg.format|stw;application/vnd.sun.xml.writer.template|sus;application/vnd.sus-calendar|susp;application/vnd.sus-calendar|svd;application/vnd.svd|svgz;image/svg+xml|swa;application/x-director|swi;application/vnd.arastra.swi|sxc;application/vnd.sun.xml.calc|sxd;application/vnd.sun.xml.draw|sxg;application/vnd.sun.xml.writer.global|sxi;application/vnd.sun.xml.impress|sxm;application/vnd.sun.xml.math|sxw;application/vnd.sun.xml.writer|tao;application/vnd.tao.intent-module-archive|t;application/x-troff|tcap;application/vnd.3gpp2.tcap|teacher;application/vnd.smart.teacher|text;text/plain|tfm;application/x-tex-tfm|tgz;application/x-gzip|tmo;application/vnd.tmobile-livetv|torrent;application/x-bittorrent|tpl;application/vnd.groove-tool-template|tpt;application/vnd.trid.tpt|tra;application/vnd.trueapp|tr;application/x-troff|trm;application/x-msterminal|tr;text/troff|ttc;application/x-font-ttf|t;text/troff|ttf;application/x-font-ttf|twd;application/vnd.simtech-mindmapper|twds;application/vnd.simtech-mindmapper|txd;application/vnd.genomatix.tuxedo|txf;application/vnd.mobius.txf|u32;application/x-authorware-bin|udeb;application/x-debian-package|ufd;application/vnd.ufdl|ufdl;application/vnd.ufdl|umj;application/vnd.umajin|unityweb;application/vnd.unity|uoml;application/vnd.uoml+xml|"
	$sTypes &= "uris;text/uri-list|uri;text/uri-list|urls;text/uri-list|utz;application/vnd.uiq.theme|uu;text/x-uuencode|vcf;text/x-vcard|vcg;application/vnd.groove-vcard|vcs;text/x-vcalendar|vcx;application/vnd.vcx|vis;application/vnd.visionary|viv;video/vnd.vivo|vor;application/vnd.stardivision.writer|vox;application/x-authorware-bin|vsd;application/vnd.visio|vsf;application/vnd.vsf|vss;application/vnd.visio|vst;application/vnd.visio|vsw;application/vnd.visio|vtu;model/vnd.vtu|w3d;application/x-director|wad;application/x-doom|wax;audio/x-ms-wax|wbmxl;application/vnd.wap.wbxml|wbs;application/vnd.criticaltools.wbs+xml|wbxml;application/vnd.wap.wbxml|wcm;application/vnd.ms-works|wdb;application/vnd.ms-works|wiz;application/msword|wks;application/vnd.ms-works|wma;audio/x-ms-wma|wmd;application/x-ms-wmd|wmf;application/x-msmetafile|wm;video/x-ms-wm|wmv;video/x-ms-wmv|wmx;video/x-ms-wmx|wmz;application/x-ms-wmz|wpd;application/vnd.wordperfect|wpl;application/vnd.ms-wpl|wps;application/vnd.ms-works|wqd;application/vnd.wqd|wri;application/x-mswrite|wsdl;application/wsdl+xml|wspolicy;application/wspolicy+xml|wtb;application/vnd.webturbo|wvx;video/x-ms-wvx|x32;application/x-authorware-bin|x3d;application/vnd.hzn-3d-crossword|xap;application/x-silverlight-app|xar;application/vnd.xara|xbap;application/x-ms-xbap|xbd;application/vnd.fujixerox.docuworks.binder|xdm;application/vnd.syncml.dm+xml|xdp;application/vnd.adobe.xdp+xml|xdw;application/vnd.fujixerox.docuworks|xenc;application/xenc+xml|xer;application/patch-ops-error+xml|xfdf;application/vnd.adobe.xfdf|xfdl;application/vnd.xfdl|xhvml;application/xv+xml|xif;image/vnd.xiff|xla;application/vnd.ms-excel|xlam;application/vnd.ms-excel.addin.macroenabled.12|xlb;application/vnd.ms-excel|xlc;application/vnd.ms-excel|xlm;application/vnd.ms-excel|xlsb;application/vnd.ms-excel.sheet.binary.macroenabled.12|xlsm;application/vnd.ms-excel.sheet.macroenabled.12|xlsx;application/vnd.openxmlformats-officedocument.spreadsheetml.sheet|xlt;application/vnd.ms-excel|"
	$sTypes &= "xltm;application/vnd.ms-excel.template.macroenabled.12|xltx;application/vnd.openxmlformats-officedocument.spreadsheetml.template|xlw;application/vnd.ms-excel|xo;application/vnd.olpc-sugar|xop;application/xop+xml|xpdl;application/xml|xpi;application/x-xpinstall|xpr;application/vnd.is-xpr|xps;application/vnd.ms-xpsdocument|xpw;application/vnd.intercon.formnet|xpx;application/vnd.intercon.formnet|xsm;application/vnd.syncml+xml|xspf;application/xspf+xml|xvm;application/xv+xml|xvml;application/xv+xml|zaz;application/vnd.zzazz.deck+xml|zir;application/vnd.zul|zirz;application/vnd.zul|zmm;application/vnd.handheld-entertainment+xml|7z;application/x-7z-compressed|"
	
	Local $aRet = StringRegExp($sTypes, '(?i)\|([^;]+);' & $sMimeType & '\|', 3)
	
	If IsArray($aRet) Then
		Return $aRet[0]
	EndIf
	
	Return SetError(1, 0, '')
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_Base64Encode
; Description....: Internal function to encode string Base64.
; Syntax.........: __TCPInet_Base64Encode($sData)
; Parameters.....: $sData - Data to encode.
; Return values..: Base64 encoded string.
; Author.........: G.Sandler
; ===============================================================================================================
Func __TCPInet_Base64Encode($sData)
	$sData = Binary($sData)
	
	Local $struct = DllStructCreate("byte[" & BinaryLen($sData) & "]")
	
	DllStructSetData($struct, 1, $sData)
	
	Local $strc = DllStructCreate("int")
	
	Local $a_Call = DllCall("Crypt32.dll", "int", "CryptBinaryToString", _
			"ptr", DllStructGetPtr($struct), _
			"int", DllStructGetSize($struct), _
			"int", 1, _
			"ptr", 0, _
			"ptr", DllStructGetPtr($strc))
	
	If @error Or Not $a_Call[0] Then
		Return SetError(1, 0, "") ; error calculating the length of the buffer needed
	EndIf
	
	Local $a = DllStructCreate("char[" & DllStructGetData($strc, 1) & "]")
	
	$a_Call = DllCall("Crypt32.dll", "int", "CryptBinaryToString", _
			"ptr", DllStructGetPtr($struct), _
			"int", DllStructGetSize($struct), _
			"int", 1, _
			"ptr", DllStructGetPtr($a), _
			"ptr", DllStructGetPtr($strc))
	
	If @error Or Not $a_Call[0] Then
		Return SetError(2, 0, ""); error encoding
	EndIf
	
	Return BinaryToString(DllStructGetData($a, 1))
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TCPInet_OnExit
; Description....: Internal function to shutdown TCP services when autoit exits.
; Syntax.........: __TCPInet_OnExit()
; Author.........: G.Sandler
; ===============================================================================================================
Func __TCPInet_OnExit()
	__TCPInet_KillTimer()
	DllCallbackFree($aTCPINET_TIMER[0])
	TCPShutdown()
EndFunc

#EndRegion
