#Region Header

#CS
	Name:				Container UDF
	Author:				Copyright  2011-2014 CreatoR's Lab (G.Sandler), www.creator-lab.ucoz.ru, www.autoit-script.ru. All rights reserved.
	AutoIt version:		3.3.6.1+
	UDF version:		1.3
	Requiremets:		Windows 2000+, (?)Internet Explorer 7+(?).
	Remarks:			Only for AutoIt version bellow 3.3.7.9:
							- Arrays transfer limited to 2D array, due to fixing AutoIt bug with "ptr" variable type set via COM interface (ticket #1410).
	
	History:
						[1.3]
						* Removed "Ptr" check for AutoIt versions above 3.3.7.9 (changed $bContainer_CheckPtrVal global variable).
						* Changed "...Structs Transfer" example to close opened process handle.
						
						[1.2]
						* Removed IE version verification. Now used "InternetExplorer.Application", it should solve the IE version issue.
						* Now the UDF fixes the issue with transfering "ptr" variable type via COM interface (ticket #1410).
						  But this "fix" sets an array limitation for _Container_PutProperty/GetProperty, it's supports only 2D array.
						  Can be disabled with $bContainer_CheckPtrVal global variable, set it to False to disable this fix.
						* Added example of transfering structure pointer and reading that structures from receiver process.
						
						[1.1]
						* Added IE version verification. This library requierd IE 7 (and windows 2000+).
						* Better errors handling.
						
						[1.0]
						* First version.
	
#CE

#include-once

#EndRegion Header

#Region Global Variables

Global $oContainer_COMErr

;Fixes the issue with transfering "ptr" via COM interface (ticket #1410).
;Can be slow if $vValue is an array, and only 2D arrays are supported, so this "fix" sets an array limitation.
Global $bContainer_CheckPtrVal = (Number(StringReplace(@AutoItVersion, '.', '')) < 3379)

#EndRegion Global Variables

#Region Public Functions

; #FUNCTION# ====================================================================================================
; Name...........:	_Container_Open
; Description....:	Creates global object for interaction between processes.
; Syntax.........:	_Container_Open($sName, $iMode)
; Parameters.....:	$sName     - Container name to use for data transfer.
; 	                $iMode     - Defines container mode:
;                                 0  - open to recieve data.
;                                 1  - open for sending data.
;
; Return values..:	Success - Container object to use later with PutProperty/GetProperty.
;                   Failure - Returns 0 and set @error as following:
;                                                                    [NOT Relevant] -1 - IExplore version is < 7.
;                                                                     1 - If $iMode = 0 and the container object is not found.
;                                                                     2 - Unable to create Shell.Application object.
;                                                                     3 - Unable to get container object.
;
; Author.........:	G.Sandler (CreatoR)
; Modified.......:	
; Remarks........:	After the data has been transfered, you should close the container using _Container_Close function, either by the Sender script, or by the Receiver script.
; Related........:	_Container_Close, _Container_PutProperty, _Container_GetProperty
; Link...........:	
; Example........:	Yes.
; ===============================================================================================================
Func _Container_Open($sName, $iMode)
	Local $oShell, $oShellWindow, $oShellWindows, $oContainer
	
	;If Int(StringLeft(FileGetVersion(@ProgramFilesDir & "\Internet Explorer\IExplore.exe"), 1)) < 7 Then
	;	Return SetError(-1, 0, 0)
	;EndIf
	
	If Not ObjEvent("AutoIt.Error") Then
		$oContainer_COMErr = ObjEvent("AutoIt.Error", "__Container_COMErrHandler")
	EndIf
	
	$oShell = ObjCreate("Shell.Application")
	
	If Not IsObj($oShell) Then
		Return SetError(2, 0, 0)
	EndIf
	
	$oShellWindows = $oShell.Windows
	
	For $oShellWindow In $oShellWindows
		If $oShellWindow.StatusText = $sName Then
			Return $oShellWindow
		EndIf
		
		If @error Then
			ExitLoop
		EndIf
	Next
	
	If $iMode = 0 Then
		Return SetError(1, 0, 0)
	EndIf
	
	;$oContainer = ObjGet("new:{C08AFD90-F2A1-11D1-8455-00A0C91F3880}")
	$oContainer = ObjCreate("InternetExplorer.Application")
   
	If Not IsObj($oContainer) Then
		Return SetError(3, 0, 0)
	EndIf
	
	$oContainer.StatusBar = True ; IMPORTANT!!
	$oContainer.StatusText = $sName
	
	Return $oContainer
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........:	_Container_Close
; Description....:	Closes the container object.
; Syntax.........:	_Container_Close($oContainer)
; Parameters.....:	$oContainer - Container object as returned by _Container_Open.
;					
; Return values..:	Success - Returns 1.
;                   Failure - Returns 0 and set @error to 1.
;
; Author.........:	G.Sandler (CreatoR)
; Modified.......:	
; Remarks........:	
; Related........:	_Container_Open.
; Link...........:	
; Example........:	
; ===============================================================================================================
Func _Container_Close($oContainer)
	;$oContainer_COMErr = 0
	;$oContainer.StatusText = ''
	
	Local $iRet = 0
	Local $hContainer = HWnd($oContainer.HWnd)
	
	If $hContainer Then
		ProcessClose(WinGetProcess($hContainer))
		$iRet = WinClose($hContainer)
	EndIf
	
	Return SetError(Number($iRet <> 1), 0, $iRet)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........:	_Container_PutProperty
; Description....:	Puts data into the Container.
; Syntax.........:	_Container_PutProperty($oContainer, $sName, $vValue)
; Parameters.....:	$oContainer    - Container object as returned by _Container_Open.
;					$sVarName      - Variable name to set.
;					$vValue        - Value for $sVarName.
;					
; Return values..:	None.
;
; Author.........:	G.Sandler (CreatoR)
; Modified.......:	
; Remarks........:	$sVarName is case sensitive, e.g. "var" is not the same as "Var"!
; Related........:	_Container_Open, _Container_GetProperty
; Link...........:	
; Example........:	
; ===============================================================================================================
Func _Container_PutProperty($oContainer, $sVarName, $vValue)
	If $bContainer_CheckPtrVal Then
		If IsPtr($vValue) Then
			$vValue = "__PTR_SIG_" & String($vValue)
		ElseIf IsArray($vValue) Then
			Local $iDims = UBound($vValue, 0)
			
			If $iDims > 2 Then
				Return SetError(1)
			EndIf
			
			For $i = 0 To UBound($vValue) - 1
				If $iDims > 1 Then
					For $j = 0 To UBound($vValue, 0) - 1
						If IsPtr($vValue[$i][$j]) Then
							$vValue[$i][$j] = "__PTR_SIG_" & String($vValue[$i][$j])
						EndIf
					Next
				Else
					If IsPtr($vValue[$i]) Then
						$vValue[$i] = "__PTR_SIG_" & String($vValue[$i])
					EndIf
				EndIf
			Next
		EndIf
	EndIf
	
	$oContainer.PutProperty($sVarName, $vValue)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........:	_Container_GetProperty
; Description....:	Get data from the Container.
; Syntax.........:	_Container_GetProperty($oContainer, $sVarName)
; Parameters.....:	$oContainer - Container object as returned by _Container_Open.
;					$sVarName   - Variable name to get the data from.
;					
; Return values..:	The value from $sVarName variable.
;
; Author.........:	G.Sandler (CreatoR)
; Modified.......:	
; Remarks........:	$sVarName is case sensitive, e.g. "var" is not the same as "Var"!
; Related........:	_Container_Open, _Container_PutProperty
; Link...........:	
; Example........:	
; ===============================================================================================================
Func _Container_GetProperty($oContainer, $sVarName)
	Local $vValue = $oContainer.GetProperty($sVarName)
	
	If $bContainer_CheckPtrVal Then
		If IsString($vValue) And StringLeft($vValue, 10) = "__PTR_SIG_" Then
			$vValue = Ptr(StringTrimLeft($vValue, 10))
		ElseIf IsArray($vValue) Then
			Local $iDims = UBound($vValue, 0)
			
			If $iDims > 2 Then
				Return SetError(1)
			EndIf
			
			For $i = 0 To UBound($vValue) - 1
				If $iDims > 1 Then
					For $j = 0 To UBound($vValue, 0) - 1
						If IsString($vValue[$i][$j]) And StringLeft($vValue[$i][$j], 10) = "__PTR_SIG_" Then
							$vValue[$i][$j] = Ptr(StringTrimLeft($vValue[$i][$j], 10))
						EndIf
					Next
				Else
					If IsString($vValue[$i]) And StringLeft($vValue[$i], 10) = "__PTR_SIG_" Then
						$vValue[$i] = Ptr(StringTrimLeft($vValue[$i], 10))
					EndIf
				EndIf
			Next
		EndIf
	EndIf
	
	Return $vValue
EndFunc

#EndRegion Public Functions

#Region Internal Functions

;Helper function to handle COM Errors
Func __Container_COMErrHandler()
	Msgbox(48, "COM Error", "We intercepted a COM Error !"          & @CRLF  & @CRLF & _
		"err.description is: "    & @TAB & $oContainer_COMErr.description    & @CRLF & _
		"err.windescription:"     & @TAB & $oContainer_COMErr.windescription & @CRLF & _
		"err.number is: "         & @TAB & Hex($oContainer_COMErr.number, 8) & @CRLF & _
		"err.lastdllerror is: "   & @TAB & $oContainer_COMErr.lastdllerror   & @CRLF & _
		"err.scriptline is: "     & @TAB & $oContainer_COMErr.scriptline     & @CRLF & _
		"err.source is: "         & @TAB & $oContainer_COMErr.source         & @CRLF & _
		"err.helpfile is: "       & @TAB & $oContainer_COMErr.helpfile       & @CRLF & _
		"err.helpcontext is: "    & @TAB & $oContainer_COMErr.helpcontext _
	)
	
	SetError(1)
Endfunc

#EndRegion Internal Functions
