#include-once
#include <GUIConstantsEx.au3>
#include <EditConstants.au3>
#include <ButtonConstants.au3>
#include <WindowsConstants.au3>

#Include "GUIResizingLimit.au3"

;===============================================================================
;
; Function Name:    _InputBox()
; Description:      Custom InputBox Function (GUI based).
; Parameter(s):     $sTitle    - The title of the Input Box.
;                   $sPrompt   - A message to the user indicating what kind of input is expected.
;                   $sDefault  - [Optional] The value that the input box starts with.
;                   $iLimit    - [optional] Limit the input for accept N characters.
;                   $iW        - [optional] The width of the window. Use -1 for default width (200).
;                   $iH        - [optional] The height of the window. Use -1 for default height (150).
;                   $iLeft     - [optional] The left side of the input box. By default (-1), the box is centered.
;                   $iTop      - [optional] The top of the input box. By default (-1), the box is centered.
;                   $iTimeOut  - [optional] How many seconds to wait before automatically cancelling the InputBox.
;                   $hWnd      - [Optional] Window handle of the parent.
;                   $iFlags    - [optional] Combination of the following flags:
;                                                                     1 - Typed characters will be replaced with password characters (*).
;                                                                     2 - Only numbers can by typed to the input.
;                                                                     4 - The inputbox window will have a "Top Most" attribute.
;                   $sChkBxTxt - [Optional] If this parameter is a string, the inputbox will have additional checkbox with the text of this string.
; Return Value(s):  Success: Returns the string that was entered.
;                   Failure: Returns "" (blank string) and sets @error as follows:
;                      @Error 0 = The string returned is valid.
;                             1 = The Cancel button was pushed.
;                             2 = The Timeout time was reached.
;                             3 = The InputBox failed to open. This is usually caused by bad arguments.
;
; Requirement(s):   None.
; Note(s):          None.
; Author(s):        G.Sandler (a.k.a CreatoR).
;
;===============================================================================
Func _InputBox($sTitle, $sPrompt, $sDefault = "", $iLimit = -1, $iW = -1, $iH = -1, $iLeft = -1, $iTop = -1, $iTimeOut = 0, $hWnd = 0, $iFlags = 0, $sChkBxTxt = 0)
	Local $OldOpt_GOEM = Opt("GUIOnEventMode", 0)
	Local $InputGui, $OKButton, $CancelButton, $InputBoxID, $AdditionCheckBox = -1
	Local $RetValue, $RetErr = 0, $RetExtnd = 0, $InputMsg, $iTimerStart
	Local $InputStyle = $ES_AUTOHSCROLL, $GUIExStyle = -1
	Local $iMinWidth = 300, $iMinHeight = 140, $iMaxWidth = 700, $iMaxHeight = 600
	
	If $iW < 300 Then
		$iW = 300
	EndIf
	
	If $iH < 150 Then
		$iH = 150
	EndIf
	
	If BitAND($iFlags, 1) = 1 Then
		$InputStyle = BitOR($InputStyle, $ES_PASSWORD)
	EndIf
	
	If BitAND($iFlags, 2) = 2 Then
		$InputStyle = BitOR($InputStyle, $ES_NUMBER)
	EndIf
	
	If BitAND($iFlags, 4) = 4 Then
		$GUIExStyle = $WS_EX_TOPMOST
	EndIf
	
	WinSetState($hWnd, "", @SW_DISABLE)
	$InputGui = GUICreate($sTitle, $iW, $iH, $iLeft, $iTop, BitOR($WS_CAPTION, $WS_POPUP, $WS_SYSMENU, $WS_SIZEBOX, $WS_CLIPCHILDREN), $GUIExStyle, $hWnd)
	
	GUICtrlCreateLabel($sPrompt, 15, 5, $iW, -1)
	GUICtrlSetResizing(-1, BitOR($GUI_DOCKSIZE, $GUI_DOCKLEFT, $GUI_DOCKTOP))
	_GUIResizingSetLimit($InputGui, $iMinWidth, $iMinHeight, $iMaxWidth, $iMaxHeight)
	
	$OKButton = GUICtrlCreateButton("OK", ($iW / 2) - 100, $iH - 33, 75, 23, $BS_DEFPUSHBUTTON)
	GUICtrlSetResizing(-1, BitOR($GUI_DOCKSIZE, $GUI_DOCKBOTTOM, $GUI_DOCKVCENTER))
	GUICtrlSetState(-1, $GUI_ONTOP)
	
	$CancelButton = GUICtrlCreateButton("Cancel", ($iW / 2) + 30, $iH - 33, 75, 23)
	GUICtrlSetResizing(-1, BitOR($GUI_DOCKSIZE, $GUI_DOCKBOTTOM, $GUI_DOCKVCENTER))
	GUICtrlSetState(-1, $GUI_ONTOP)
	
	If IsString($sChkBxTxt) Then
		$AdditionCheckBox = GUICtrlCreateCheckbox($sChkBxTxt, 10, $iH - 85, $iW - 20, 20)
		GUICtrlSetResizing(-1, BitOR($GUI_DOCKSIZE, $GUI_DOCKBOTTOM, $GUI_DOCKLEFT, $GUI_DOCKRIGHT))
	EndIf
	
	$InputBoxID = GUICtrlCreateInput($sDefault, 10, $iH - 60, $iW - 20, 20, $InputStyle)
	GUICtrlSetResizing(-1, BitOR($GUI_DOCKSIZE, $GUI_DOCKBOTTOM, $GUI_DOCKLEFT, $GUI_DOCKRIGHT))
	GUICtrlSetState(-1, $GUI_FOCUS)
	
	If $iLimit > 0 Then
		GUICtrlSetLimit(-1, $iLimit)
	EndIf
	
	GUISetState(@SW_SHOW, $InputGui)
	
	If Not BitAND(WinGetState($InputGui), 2) Then
		Return SetError(3, 0, "")
	EndIf
	
	If $iTimeOut > 0 Then
		$iTimerStart = TimerInit()
	EndIf
	
	While 1
		$InputMsg = GUIGetMsg()
		
		Switch $InputMsg
			Case $GUI_EVENT_CLOSE, $CancelButton
				$RetValue = ""
				$RetErr = 1
				ExitLoop
			Case $OKButton
				$RetValue = GUICtrlRead($InputBoxID)
				$RetExtnd = Number(GUICtrlRead($AdditionCheckBox) = $GUI_CHECKED)
				$RetErr = 0
				ExitLoop
		EndSwitch
		
		If $iTimeOut > 0 And Round(TimerDiff($iTimerStart) / 1000) = $iTimeOut Then
			$RetValue = GUICtrlRead($InputBoxID)
			$RetErr = 2
			ExitLoop
		EndIf
	WEnd
	
	WinSetState($hWnd, "", @SW_ENABLE)
	GUIDelete($InputGui)
	GUISwitch($hWnd)
	
	Opt("GUIOnEventMode", $OldOpt_GOEM)
	Return SetError($RetErr, $RetExtnd, $RetValue)
EndFunc
