#Region Header

#CS
	Name:				GUICtrlSetResizing UDF
	Description:		Allows you to resize ("on the fly") GUI controls by dragging them (by the edges).
	Author:				Copyright  2011 CreatoR's Lab (G.Sandler), www.creator-lab.ucoz.ru, www.autoit-script.ru. All rights reserved.
	AutoIt version:		3.3.6.1
	UDF version:		1.2
	
	History:
						[1.2]
						* Flags constants renamed from $iGCSOR_*_EDGE to $iGCSOR_RESIZE_*.
						+ Added flags to set corners resizing mode:
							$iGCSOR_RESIZE_LEFTTOPCORNER, $iGCSOR_RESIZE_LEFTBOTTOMCORNER
							$iGCSOR_RESIZE_RIGHTTOPCORNER, $iGCSOR_RESIZE_RIGHTBOTTOMCORNER
							- To include all corners modes, use $iGCSOR_RESIZE_ALLCORNERS.
						+ Added $iGCSOR_RESIZE_ALL flag (the same as -1).
						* Fixed an issue when controls was resized to they original size after the parent window resized (with $WS_SIZEBOX style).
						* Now the controls are stricted to the parent window size (they can't be resized/dragged out of the window).
						* Example updated.
						
						[1.1]
						* 3.3.6.1 compatibility.
						+ Added ability to set resize mode.
						* General improvements.
						
						[1.0]
						* First version.
	
#CE

;Includes
#include-once
#include <GUIConstantsEx.au3>
#include <WindowsConstants.au3>
#include <WinAPI.au3>

#EndRegion Header

#Region Global Variables

Global Const $SM_CXFIXEDFRAME 			= 7 ;Window border size

Global $iGCSOR_WndCaptHeight 			= _WinAPI_GetSystemMetrics($SM_CYCAPTION)
Global $iGCSOR_WndBorder 				= _WinAPI_GetSystemMetrics($SM_CXFIXEDFRAME)
Global $iGCSOR_CurIsSet 				= 0
Global $iGCSOR_CtrlEdgeSize 			= 5 	;Size of the control's edge
Global $iGCSOR_DefCtrlMinSize 			= 20	;Default resizing limit

Global $aGCSOR_Array[1][1]

#EndRegion Global Variables

#Region Public Constants

Global Const $iGCSOR_RESIZE_LEFT				= 1
Global Const $iGCSOR_RESIZE_RIGHT				= 8
Global Const $iGCSOR_RESIZE_TOP					= 16
Global Const $iGCSOR_RESIZE_BOTTOM				= 32
Global Const $iGCSOR_RESIZE_LEFTTOPCORNER		= 64
Global Const $iGCSOR_RESIZE_LEFTBOTTOMCORNER	= 128
Global Const $iGCSOR_RESIZE_RIGHTTOPCORNER		= 256
Global Const $iGCSOR_RESIZE_RIGHTBOTTOMCORNER	= 512
Global Const $iGCSOR_RESIZE_ALLCORNERS			= 960 ;64 + 128 + 256 + 512
Global Const $iGCSOR_RESIZE_ALL					= -1 ;57 ;1 + 8 + 16 + 32

#EndRegion Public Constants

#Region Options

AdlibRegister("__GCSOR_Handler", 50)

#EndRegion Options

#Region Example

#CS

#include <GUIConstantsEx.au3>
#include <WindowsConstants.au3>
#include "GUICtrlSetResizing.au3"

$hGUI = GUICreate("_GUICtrl_SetOnResize - Example", 700, 480) ;, -1, -1, BitOr($GUI_SS_DEFAULT_GUI, $WS_SIZEBOX))

$Edit = GUICtrlCreateEdit("Resize only Left edge", 40, 40, 280, 200)
_GUICtrl_SetOnResize($hGUI, $Edit, 10, $iGCSOR_DefCtrlMinSize, $iGCSOR_RESIZE_LEFT) ;Resize only Left edge (-1 or $iGCSOR_RESIZE_ALL to resize all)

$ListView = GUICtrlCreateListView("ListView Column", 340, 40, 300, 200)
GUICtrlCreateListViewItem("Item (Top + Bottom)", $ListView)
_GUICtrl_SetOnResize($hGUI, $ListView, 10, $iGCSOR_DefCtrlMinSize, BitOR($iGCSOR_RESIZE_TOP, $iGCSOR_RESIZE_BOTTOM)) ;Resize Top + Bottom

$Checkbox = GUICtrlCreateCheckbox("Some Checkbox (Left + Right)", 40, 270)
GUICtrlSetBkColor(-1, 0xFFFFFF)
_GUICtrl_SetOnResize($hGUI, $Checkbox, 10, $iGCSOR_DefCtrlMinSize, BitOR($iGCSOR_RESIZE_LEFT, $iGCSOR_RESIZE_RIGHT)) ;Resize Left + Right

$Label = GUICtrlCreateLabel("Some Label (Top + Right)", 40, 320, -1, 15)
GUICtrlSetBkColor(-1, 0x0000FF)
GUICtrlSetColor(-1, 0xFFFFFF)
_GUICtrl_SetOnResize($hGUI, $Label, 10, 15, BitOR($iGCSOR_RESIZE_TOP, $iGCSOR_RESIZE_RIGHT)) ;Resize Top + Right

$Button = GUICtrlCreateButton("Some Button (All corners)", 40, 370, 160, 40)
_GUICtrl_SetOnResize($hGUI, $Button, 10, $iGCSOR_DefCtrlMinSize, $iGCSOR_RESIZE_ALLCORNERS) ;Resize All Corners

GUISetState(@SW_SHOW, $hGUI)

While 1
	$nMsg = GUIGetMsg()
	
	Switch $nMsg
		Case $GUI_EVENT_CLOSE
			Exit
	EndSwitch
WEnd

#CE

#EndRegion Example

#Region Public Functions

; #FUNCTION# ====================================================================================================
; Name...........:	_GUICtrl_SetOnResize
; Description....:	Set resizing mode for specific control - allows you to resize ("on the fly") GUI controls by dragging them (by the edges).
; Syntax.........:	_GUICtrl_SetOnResize($hWnd [, $nCtrlID = -1 [, $iWait = 10 [, $iCtrlMinSize = $iGCSOR_DefCtrlMinSize [, $iResizeEdgeFlags = -1]]]])
; Parameters.....:	$hWnd 				- Window Handle (returned by GUICreate()).
;					$nCtrlID 			- [Optional] Control Identifier to set the resizing (can be -1 as last control id).
;					$iWait 				- [Optional] Time to wait between each resize attempts (in milliseconds).
;					$iCtrlMinSize 		- [Optional] Control resizing limit (the minimum size).
;					$iResizeEdgeFlags 	- [Optional] Flags for the resizing mode, see the flags constants (default -1, resize all).
;					
; Return values..:	Always return 1.
; Author.........:	G.Sandler (CreatoR).
; Modified.......:	
; Remarks........:	To unset resizing for specific control, just call the function like this:
;					_GUICtrl_SetOnResize(-1, $iCtrlID)
; Related........:	
; Link...........:	http://www.autoitscript.com/forum/topic/75723-guictrlsetresizing-udf
; Example........:	Yes.
; ===============================================================================================================
Func _GUICtrl_SetOnResize($hWnd, $nCtrlID = -1, $iWait = 10, $iCtrlMinSize = $iGCSOR_DefCtrlMinSize, $iResizeEdgeFlags = -1)
	If $nCtrlID = -1 Then
		$nCtrlID = __GCSOR_GUIGetLastCtrlID()
	EndIf
	
	If Not IsHWnd($hWnd) Then
		Local $aTmpSetResArr[1][1]
		
		For $i = 1 To $aGCSOR_Array[0][0]
			If $nCtrlID <> $aGCSOR_Array[$i][1] Then
				$aTmpSetResArr[0][0] += 1
				ReDim $aTmpSetResArr[$aTmpSetResArr[0][0]+1][5]
				$aTmpSetResArr[$aTmpSetResArr[0][0]][0] = $aGCSOR_Array[$i][0]
				$aTmpSetResArr[$aTmpSetResArr[0][0]][1] = $aGCSOR_Array[$i][1]
				$aTmpSetResArr[$aTmpSetResArr[0][0]][2] = $aGCSOR_Array[$i][2]
				$aTmpSetResArr[$aTmpSetResArr[0][0]][3] = $aGCSOR_Array[$i][3]
				$aTmpSetResArr[$aTmpSetResArr[0][0]][4] = $aGCSOR_Array[$i][4]
			EndIf
		Next
		
		GUICtrlSetCursor($nCtrlID, -1)
		$aGCSOR_Array = $aTmpSetResArr
		
		Return 1
	EndIf
	
	$aGCSOR_Array[0][0] += 1
	ReDim $aGCSOR_Array[$aGCSOR_Array[0][0]+1][5]
	$aGCSOR_Array[$aGCSOR_Array[0][0]][0] = $hWnd
	$aGCSOR_Array[$aGCSOR_Array[0][0]][1] = $nCtrlID
	$aGCSOR_Array[$aGCSOR_Array[0][0]][2] = $iWait
	$aGCSOR_Array[$aGCSOR_Array[0][0]][3] = $iCtrlMinSize
	$aGCSOR_Array[$aGCSOR_Array[0][0]][4] = $iResizeEdgeFlags
	
	Return 1
EndFunc

#EndRegion Public Functions

#Region Internal Functions

;Handler to call the set functions (for all controls that _GUICtrl_SetOnResize() is set)
Func __GCSOR_Handler()
	If $aGCSOR_Array[0][0] = 0 Then
		Return
	EndIf
	
	For $i = 1 To $aGCSOR_Array[0][0]
		__GCSOR_ResizeProc($aGCSOR_Array[$i][0], $aGCSOR_Array[$i][1], $aGCSOR_Array[$i][2], $aGCSOR_Array[$i][3], $aGCSOR_Array[$i][4])
	Next
EndFunc

;Main Resizing Control Function
Func __GCSOR_ResizeProc($hWnd, $nCtrlID, $iWait = 10, $iCtrlMinSize = $iGCSOR_DefCtrlMinSize, $iResizeEdgeFlags = -1)
;~ 	Local $aNumClassList = StringSplit(__GCSOR_WinGetNumeratedClassList($hWnd), @CRLF, 1)
	
	Local $aCurInfo = GUIGetCursorInfo($hWnd)
	If @error Then Return
	
	Local $aCtrlInfo = ControlGetPos($hWnd, "", $nCtrlID)
	If @error Then Return
	
	Local $aWndPos = WinGetPos($hWnd)
	If @error Then Return
	
	If $iResizeEdgeFlags = -1 Then
		$iResizeEdgeFlags = BitOR($iGCSOR_RESIZE_LEFT, $iGCSOR_RESIZE_TOP, $iGCSOR_RESIZE_RIGHT, $iGCSOR_RESIZE_BOTTOM)
	EndIf
	
	Local $iCursorID
	Local $iCheckFlag = -1
	Local $nOld_Opt_GOEM = Opt("GUIOnEventMode", 0)
	
	#Region Check the resizing mode (Left, Right, Top, Bottom)
	If ($aCurInfo[0] > $aCtrlInfo[0] - $iGCSOR_CtrlEdgeSize And $aCurInfo[0] < $aCtrlInfo[0] + $iGCSOR_CtrlEdgeSize) And _
		($aCurInfo[1] >= $aCtrlInfo[1] And $aCurInfo[1] <= $aCtrlInfo[1] + $aCtrlInfo[3]) And _
		BitAND($iResizeEdgeFlags, $iGCSOR_RESIZE_LEFT) = $iGCSOR_RESIZE_LEFT Then 								;Left
		
		$iCheckFlag = 1
		$iCursorID = 13
	EndIf
	
	If ($aCurInfo[0] > ($aCtrlInfo[0] + $aCtrlInfo[2]) - $iGCSOR_CtrlEdgeSize And _
		$aCurInfo[0] < ($aCtrlInfo[0] + $aCtrlInfo[2]) + $iGCSOR_CtrlEdgeSize) And _
		($aCurInfo[1] >= $aCtrlInfo[1] And $aCurInfo[1] <= $aCtrlInfo[1] + $aCtrlInfo[3]) And _
		BitAND($iResizeEdgeFlags, $iGCSOR_RESIZE_RIGHT) = $iGCSOR_RESIZE_RIGHT Then 							;Right
		
		$iCheckFlag = 2
		$iCursorID = 13
	EndIf
	
	If ($aCurInfo[1] > $aCtrlInfo[1] - $iGCSOR_CtrlEdgeSize And $aCurInfo[1] < $aCtrlInfo[1] + $iGCSOR_CtrlEdgeSize) And _
		($aCurInfo[0] >= $aCtrlInfo[0] And $aCurInfo[0] <= $aCtrlInfo[0] + $aCtrlInfo[2]) And _
		BitAND($iResizeEdgeFlags, $iGCSOR_RESIZE_TOP) = $iGCSOR_RESIZE_TOP Then 								;Top
		
		If $iCheckFlag = 1 Then 		;Left+Top
			$iCheckFlag = 5
			$iCursorID = 12
		ElseIf $iCheckFlag = 2 Then 	;Right+Top
			$iCheckFlag = 7
			$iCursorID = 10
		Else 							;Just Top
			$iCheckFlag = 3
			$iCursorID = 11
		EndIf
	EndIf
	
	If ($aCurInfo[1] > ($aCtrlInfo[1] + $aCtrlInfo[3]) - $iGCSOR_CtrlEdgeSize And _
		$aCurInfo[1] < ($aCtrlInfo[1] + $aCtrlInfo[3]) + $iGCSOR_CtrlEdgeSize) And _
		($aCurInfo[0] >= $aCtrlInfo[0] And $aCurInfo[0] <= $aCtrlInfo[0] + $aCtrlInfo[2]) And _
		BitAND($iResizeEdgeFlags, $iGCSOR_RESIZE_BOTTOM) = $iGCSOR_RESIZE_BOTTOM Then 							;Bottom
		
		If $iCheckFlag = 1 Then 		;Left+Bottom
			$iCheckFlag = 6
			$iCursorID = 10
		ElseIf $iCheckFlag = 2 Then 	;Right+Bottom
			$iCheckFlag = 8
			$iCursorID = 12
		Else 							;Just Bottom
			$iCheckFlag = 4
			$iCursorID = 11
		EndIf
	EndIf
	#EndRegion
	
	#Region Check the resizing mode (Corners)
	If ($aCurInfo[1] > $aCtrlInfo[1] - $iGCSOR_CtrlEdgeSize And $aCurInfo[1] < $aCtrlInfo[1] + $iGCSOR_CtrlEdgeSize) And _
		($aCurInfo[0] >= $aCtrlInfo[0] And $aCurInfo[0] <= $aCtrlInfo[0] + $aCtrlInfo[2]) And _
		($aCurInfo[0] >= $aCtrlInfo[0] And $aCurInfo[0] <= $aCtrlInfo[0] + $iGCSOR_CtrlEdgeSize) And _
		BitAND($iResizeEdgeFlags, $iGCSOR_RESIZE_LEFTTOPCORNER) = $iGCSOR_RESIZE_LEFTTOPCORNER Then				;Left-Top corner
		
		$iCheckFlag = 5
		$iCursorID = 12
	EndIf
	
	If ($aCurInfo[1] > ($aCtrlInfo[1] + $aCtrlInfo[3]) - $iGCSOR_CtrlEdgeSize And _
		$aCurInfo[1] < ($aCtrlInfo[1] + $aCtrlInfo[3]) + $iGCSOR_CtrlEdgeSize) And _
		($aCurInfo[0] >= $aCtrlInfo[0] And $aCurInfo[0] <= $aCtrlInfo[0] + $iGCSOR_CtrlEdgeSize) And _
		BitAND($iResizeEdgeFlags, $iGCSOR_RESIZE_LEFTBOTTOMCORNER) = $iGCSOR_RESIZE_LEFTBOTTOMCORNER Then		;Left-Bottom corner
		
		$iCheckFlag = 6
		$iCursorID = 10
	EndIf
	
	If ($aCurInfo[0] > ($aCtrlInfo[0] + $aCtrlInfo[2]) - $iGCSOR_CtrlEdgeSize And _
		$aCurInfo[0] < ($aCtrlInfo[0] + $aCtrlInfo[2]) + $iGCSOR_CtrlEdgeSize) And _
		($aCurInfo[1] >= $aCtrlInfo[1] And $aCurInfo[1] <= $aCtrlInfo[1] + $iGCSOR_CtrlEdgeSize) And _
		BitAND($iResizeEdgeFlags, $iGCSOR_RESIZE_RIGHTTOPCORNER) = $iGCSOR_RESIZE_RIGHTTOPCORNER Then			;Right-Top corner
		
		$iCheckFlag = 7
		$iCursorID = 10
	EndIf
	
	If ($aCurInfo[0] > ($aCtrlInfo[0] + $aCtrlInfo[2]) - $iGCSOR_CtrlEdgeSize And _
		$aCurInfo[0] < ($aCtrlInfo[0] + $aCtrlInfo[2]) + $iGCSOR_CtrlEdgeSize) And _
		($aCurInfo[1] >= ($aCtrlInfo[1] + $aCtrlInfo[3]) - $iGCSOR_CtrlEdgeSize And $aCurInfo[1] <= ($aCtrlInfo[1] + $aCtrlInfo[3]) + $iGCSOR_CtrlEdgeSize) And _
		BitAND($iResizeEdgeFlags, $iGCSOR_RESIZE_RIGHTBOTTOMCORNER) = $iGCSOR_RESIZE_RIGHTBOTTOMCORNER Then		;Right-Bottom corner
		
		$iCheckFlag = 8
		$iCursorID = 12
	EndIf
	#EndRegion
	
	If $iCheckFlag = -1 Then
		If ($aCurInfo[4] = 0 Or $aCurInfo[4] = $nCtrlID) And Not $iGCSOR_CurIsSet Then
			If $aCurInfo[4] = $nCtrlID Then $iGCSOR_CurIsSet = 1
			GUISetCursor(-1, 0, $hWnd)
			GUICtrlSetCursor($nCtrlID, -1)
		EndIf
		
		Return
	Else
		$iGCSOR_CurIsSet = 0
		GUICtrlSetCursor($nCtrlID, $iCursorID)
		GUISetCursor($iCursorID, 0, $hWnd)
	EndIf
	
	While $aCurInfo[2] = 1
		;This loop is to prevent control movement while the mouse is not moving
		While GUIGetMsg() <> $GUI_EVENT_MOUSEMOVE
			Sleep(10)
		WEnd
		
		$aCurInfo = GUIGetCursorInfo($hWnd)
		If @error Then ExitLoop
		
		$aCtrlInfo = ControlGetPos($hWnd, "", $nCtrlID)
		If @error Then ExitLoop
		
		#Region Check if the control is resized out of the main window - do not allow this
		If $aCurInfo[0] < 0 Then
			GUICtrlSetPos($nCtrlID, 0, $aCtrlInfo[1], $aCtrlInfo[2] + $aCtrlInfo[0])
			ContinueLoop
		EndIf
		
		If $aCurInfo[0] > $aWndPos[2] Then
			Local $iEdgeDiff = ($aWndPos[2] - ($aCtrlInfo[0] + $aCtrlInfo[2]))
			GUICtrlSetPos($nCtrlID, $aWndPos[2] - $aCtrlInfo[2] - $iEdgeDiff, $aCtrlInfo[1], $aCtrlInfo[2] + $iEdgeDiff - ($iGCSOR_WndBorder * 2))
			ContinueLoop
		EndIf
		
		If $aCurInfo[1] < 0 Then
			GUICtrlSetPos($nCtrlID, $aCtrlInfo[0], 0, $aCtrlInfo[2], $aCtrlInfo[3] + $aCtrlInfo[1])
			ContinueLoop
		EndIf
		
		If $aCurInfo[1] > $aWndPos[3] Then
			Local $iEdgeDiff = ($aWndPos[3] - ($aCtrlInfo[1] + $aCtrlInfo[3]))
			GUICtrlSetPos($nCtrlID, $aCtrlInfo[0], $aWndPos[3] - $aCtrlInfo[3] - $iEdgeDiff, $aCtrlInfo[2], _
				$aCtrlInfo[3] + $iEdgeDiff - ($iGCSOR_WndCaptHeight + ($iGCSOR_WndBorder * 2)))
			ContinueLoop
		EndIf
		#EndRegion
		
		; $iCheckFlag Values:
		;1 = Left
		;2 = Right
		;3 = Top
		;4 = Bottom
		;5 = Left + Top
		;6 = Left + Bottom
		;7 = Right + Top
		;8 = Right + Bottom
		
;~ 		For $nCtrl = 1 To $aNumClassList[0]
;~ 			If ControlGetHandle($hWnd, "", $aNumClassList[$nCtrl]) = GUICtrlGetHandle($nCtrlID) Then
;~ 				ContinueLoop
;~ 			EndIf
;~ 			
;~ 			$aOtherCtrlPos = ControlGetPos($hWnd, "", $aNumClassList[$nCtrl])
;~ 			
;~ 			If $aCtrlInfo[0] < $aOtherCtrlPos[0] + $aOtherCtrlPos[2] Then ;Or $aCtrlInfo[0] + $aCtrlInfo[2] > $aOtherCtrlPos[0] Then
;~ 				Switch $iCheckFlag
;~ 					Case 1
;~ 						Local $iEdgeDiff = $aCtrlInfo[0] + $aCtrlInfo[2]
;~ 						Local $iLeft = $aOtherCtrlPos[0] + $aOtherCtrlPos[2] + 3
;~ 						GUICtrlSetPos($nCtrlID, $iLeft, $aCtrlInfo[1], $iEdgeDiff - $iLeft)
;~ 					Case 2
;~ 						
;~ 					Case 3
;~ 						
;~ 						
;~ 				EndSwitch
;~ 				
;~ 				GUISetCursor(-1, 0, $hWnd)
;~ 				GUICtrlSetCursor($nCtrlID, -1)
;~ 				ExitLoop 2
;~ 			EndIf
;~ 		Next
		
		#Region Check the set flags and move the controls
		If $iCheckFlag = 1 Or $iCheckFlag = 5 Or $iCheckFlag = 6 Then
			If $aCtrlInfo[2] - ($aCurInfo[0] - $aCtrlInfo[0]) > $iCtrlMinSize Then 					;Move from Left
				$aCtrlInfo[2] = $aCtrlInfo[2] - ($aCurInfo[0] - $aCtrlInfo[0])
				$aCtrlInfo[0] = $aCurInfo[0]
				
				GUICtrlSetPos($nCtrlID, $aCtrlInfo[0], $aCtrlInfo[1], $aCtrlInfo[2])
				;ControlMove($hWnd, "", $nCtrlID, $aCtrlInfo[0], $aCtrlInfo[1], $aCtrlInfo[2])
			EndIf
		EndIf
		
		If $iCheckFlag = 2 Or $iCheckFlag = 7 Or $iCheckFlag = 8 Then
			If $aCurInfo[0] - $aCtrlInfo[0] > $iCtrlMinSize Then 									;Move from Right
				$aCtrlInfo[2] = $aCurInfo[0] - $aCtrlInfo[0]
				GUICtrlSetPos($nCtrlID, $aCtrlInfo[0], $aCtrlInfo[1], $aCtrlInfo[2])
				;ControlMove($hWnd, "", $nCtrlID, $aCtrlInfo[0], $aCtrlInfo[1], $aCtrlInfo[2])
			EndIf
		EndIf
		
		If $iCheckFlag = 3 Or $iCheckFlag = 5 Or $iCheckFlag = 7 Then
			If $aCtrlInfo[3] - ($aCurInfo[1] - $aCtrlInfo[1]) > $iCtrlMinSize Then 					;Move from Top
				$aCtrlInfo[3] = $aCtrlInfo[3] - ($aCurInfo[1] - $aCtrlInfo[1])
				$aCtrlInfo[1] = $aCurInfo[1]
				GUICtrlSetPos($nCtrlID, $aCtrlInfo[0], $aCtrlInfo[1], $aCtrlInfo[2], $aCtrlInfo[3])
				;ControlMove($hWnd, "", $nCtrlID, $aCtrlInfo[0], $aCtrlInfo[1], $aCtrlInfo[2], $aCtrlInfo[3])
			EndIf
		EndIf
		
		If $iCheckFlag = 4 Or $iCheckFlag = 6 Or $iCheckFlag = 8 Then
			If $aCurInfo[1] - $aCtrlInfo[1] > $iCtrlMinSize Then 									;Move from Bottom
				$aCtrlInfo[3] = $aCurInfo[1] - $aCtrlInfo[1]
				GUICtrlSetPos($nCtrlID, $aCtrlInfo[0], $aCtrlInfo[1], $aCtrlInfo[2], $aCtrlInfo[3])
				;ControlMove($hWnd, "", $nCtrlID, $aCtrlInfo[0], $aCtrlInfo[1], $aCtrlInfo[2], $aCtrlInfo[3])
			EndIf
		EndIf
		#EndRegion
		
		Sleep($iWait)
	WEnd
	
	Opt("GUIOnEventMode", $nOld_Opt_GOEM)
EndFunc

;Function to get last CtrlID (in case that user will pass -1 as $iCtrlID)
Func __GCSOR_GUIGetLastCtrlID()
	Local $aRet = DllCall("user32.dll", "int", "GetDlgCtrlID", "hwnd", GUICtrlGetHandle(-1))
	
	If @error Then
		Return SetError(1)
	EndIf
	
	Return $aRet[0]
EndFunc

Func __GCSOR_WinGetNumeratedClassList($sTitle, $sText = "", $bOnlyRootCtrl = True)
	Local $hWnd = WinGetHandle($sTitle, $sText)
	Local $sClassList = WinGetClassList($sTitle, $sText)
	Local $aClassList = StringSplit($sClassList, @CRLF)
	Local $sRetClassList = "", $sHold_List = "|"
	Local $aiInHold, $iInHold
	
	For $i = 1 To UBound($aClassList) - 1
		If $aClassList[$i] = "" Then
			ContinueLoop
		EndIf
		
		If StringRegExp($sHold_List, "\|" & $aClassList[$i] & "~(\d+)\|") Then
			$aiInHold = StringRegExp($sHold_List, ".*\|" & $aClassList[$i] & "~(\d+)\|.*", 1)
			$iInHold = Number($aiInHold[UBound($aiInHold)-1])
			
			If $iInHold = 0 Then $iInHold += 1
			
			$aClassList[$i] &= "~" & $iInHold + 1
			$sHold_List &= $aClassList[$i] & "|"
			
			$sRetClassList &= $aClassList[$i] & @CRLF
		Else
			$aClassList[$i] &= "~1"
			$sHold_List &= $aClassList[$i] & "|"
			$sRetClassList &= $aClassList[$i] & @CRLF
		EndIf
	Next
	
	$sRetClassList = StringReplace($sRetClassList, "~", "")
	
	If $bOnlyRootCtrl Then
		$aCL = StringSplit($sRetClassList, @CRLF)
		$sRetClassList = ""
		
		For $i = 1 To $aCL[0]
			If _WinAPI_GetParent(ControlGetHandle($hWnd, "", $aCL[$i])) = $hWnd Then
				$sRetClassList &= $aCL[$i] & @CRLF
			EndIf
		Next
	EndIf
	
	Return StringStripWS($sRetClassList, 3)
EndFunc

#EndRegion Internal Functions
