#Region Header

#CS
    Title:          Syntax Highlighter for AutoIt v3 Code
    Filename:       Au3SyntaxHighlight.au3
    Description:    Allows to highlight AutoIt v3 syntax to html/bbcode format.
    Author:         G.Sandler (a.k.a (Mr)CreatoR), www.creator-lab.ucoz.ru, www.autoit-script.ru
    Version:        1.1
    Requirements:   AutoIt v3.3.x.x, Developed/Tested on WinXP (Service Pack 3)/Win 7/Win10.
    Uses:           
    Notes:          *Idea* based on similar old project: http://www.autoitscript.com/forum/index.php?showtopic=34236
	
	
	History:
					[1.1] @THIS IS SCRIPT BREAKING VERSION@
					* $iAu3SH_AddURLs renamed to $AU3SH_ADD_URLS.
					* $iAu3SH_AbortHighlight renamed to $AU3SH_HIGHLIGHT_ABORT.
						ATTENTION: If this is set to True, then it's should be reset to False after the "_Au3_SyntaxHighlight" function returns.
					+ Added new user variables to set urls when using $AU3SH_ADD_URLS: $AU3SH_FUNCS_URL, $AU3SH_UDFS_URL, $AU3SH_KEYWORDS_URL, $AU3SH_MACROS_URL, $AU3SH_PREPROC_URL.
					+ Added $AU3SH_KEYWORDS_FILE user variable to specify path to au3.keywords.properties file, if not found used one from Resources dir (can be outdated).
					+ Added $AU3SH_API_FILE user variable to specify path to au3.api file, if not found used one from Resources dir (can be outdated).
					* _Au3_SyntaxHighlight_Proc renamed to _Au3_SyntaxHighlight.
					* Function names was shorten.
					* Speed optimization.
					- Removed debug system ($fAu3SH_Debug_RE_Patterns and $fAu3SH_Debug_RE_WriteLog also have been removed).
					
					[1.0]
					* Speed optimization.
					
					[0.9]
					+ Compatibility with AutoIt 3.3.12.0.
					+ Added "Add missing keywords.au3" script to add missing keywords from new AutoIt version.
					* $iDebug_RegExp_Patterns user global variable renamed to $fAu3SH_Debug_RE_Patterns.
					* $iDebug_RegExp_WriteLog user global variable renamed to $fAu3SH_Debug_RE_WriteLog.
					* Examples changed.
					
					[0.8]
					* Fixed bug with highlighting variables inside special keywords (#region, #forceref etc.).
					
					[0.7]
					* Fixed issue with wrong parsing when Add Urls option is enabled.
					
					[0.6]
					+ Compatibility with AutoIt 3.3.8.1.
					* Fixed issue with wrong restored strings.
					* Fixed bug with not proper UDF naming for url (UDFs links does not recognized due to not proper case, i.e: _stringrepeat instead of _StringRepeat).
					
					[0.5]
					* Fixed bug with highlighting literal strings inside comment blocks.
					* Fixed few issues with highlighting special keywords.
					
					[0.4]
					* Fixed bug with (re)highlighting keywords and functions inside COM object methods.
					* Docs updated.
					
					[0.3]
					* Fixed bug when strings and send keys was highlighted inside commented lines.
					* Few optimizations to the code.
					
					[0.2]
					+ Added Global parameter ($iAu3SH_AbortHighlight) to abort the highlighting process (break the function execution).
					   - If this variable set to 1, the function will return the original Au3 Code and set @error to -1.
					+ Added "#White space" style support (now you can set it in the au3.styles.properties file).
					+ Added "#Background" style support, used only for highlighting with Html tags mode (now you can set it in the au3.styles.properties file).
					* Styles header classes renamed with "au3_" prefix.
					* Fixed bug with highlighting COM objects.
					* Fixed bug with highlighting keywords inside variables.
					* Fixed bug with highlighting more than one preprocessor when instead of <Include.au3> used double/single qoutes ("Include.au3").
					* Fixed bug with highlighting commented lines/comment blocks inside literal strings.
					* Fixed bug with not (properly) highlighting commented lines.
					* Fixed issue with converting to BBCode format.
					* Now the PreProcessor and Special keywords are correctly highlighted (all included tags removed).
					
					[0.1]
					* First public release.

================ ToDo: ================

#CE

#include-once

#EndRegion Header

#Region Global Variables

Global $sAu3SH_Styles_File 			= @ScriptDir & "\Resources\au3.styles.properties"

Global $fAu3SH_Get_Kwrds_From_Api	= True

Global Enum _
	$iAu3SH_WSpace_Style = 0, $iAu3SH_Cmnt_Style, $iAu3SH_CmntBlck_Style, $iAu3SH_Nmbr_Style, $iAu3SH_Fnc_Style, $iAu3SH_Kwd_Style, $iAu3SH_Mcro_Style, $iAu3SH_Str_Style, _
	$iAu3SH_Oprtr_Style, $iAu3SH_Var_Style, $iAu3SH_SntKys_Style, $iAu3SH_PrPrc_Style, $iAu3SH_Spec_Style, $iAu3SH_Abrv_Style, $iAu3SH_COM_Style, $iAu3SH_Udf_Style

Global $aAu3SH_Styles[$iAu3SH_Udf_Style + 1]

For $i = $iAu3SH_WSpace_Style To $iAu3SH_Udf_Style
	$aAu3SH_Styles[$i] 				= '<span class="au3_S' & $i & '">'
Next

Global $sAu3SH_CloseTag 			= '</span>'

#EndRegion Global Variables

#Region User Variables

Global $AU3SH_KEYWORDS_FILE 		= @ScriptDir & "\Resources\au3.keywords.properties"
Global $AU3SH_API_FILE 				= @ScriptDir & "\Resources\au3.api"

;Abort _Au3_SyntaxHighlight
;ATTENTION: If this is set to True, then it's should be reset to False after the "_Au3_SyntaxHighlight" function returns.
Global $AU3SH_HIGHLIGHT_ABORT 		= False

Global $AU3SH_ADD_URLS 				= True

Global $AU3SH_FUNCS_URL 			= "http://www.autoitscript.com/autoit3/docs/functions/%s.htm"
Global $AU3SH_UDFS_URL 				= "http://www.autoitscript.com/autoit3/docs/libfunctions/%s.htm"
Global $AU3SH_KEYWORDS_URL 			= "http://www.autoitscript.com/autoit3/docs/keywords.htm#%s"
Global $AU3SH_MACROS_URL 			= "http://www.autoitscript.com/autoit3/docs/macros.htm#%s"
Global $AU3SH_PREPROC_URL 			= "http://www.autoitscript.com/autoit3/docs/keywords.htm#%s"

#EndRegion User Variables

#Region Public Functions

; #FUNCTION# ====================================================================================================
; Name...........:	_Au3_SyntaxHighlight
; Description....:	Allows to highlight AutoIt v3 syntax to html/bbcode format.
; Syntax.........:	_Au3_SyntaxHighlight($sAu3Code [, $iOutput = -1])
; Parameters.....:	$sAu3Code - AutoIt v3 plain code.
;					$iOutput  - [Optional] Sets the output format:
;					                                      -1 - (Default) Return CSS classes header and the AutoIt Syntax Highlighted code as string (wrapped with code tags)
;					                                       1 - Return CSS classes header and AutoIt Syntax Highlighted code as array, where...
;					                                            [0] = CSS styles.
;					                                            [1] = AutoIt Syntax Highlighted code.
;					                                       2 - Return the result as Html formatted string.
;					                                       3 - Return the result as BBCode formatted string (html tags replaced with bbcode tags).
;					
; Return values..:	Success - Returns AutoIt Syntax Highlighted code (see $iOutput parameter above for more details).
;					Failure - Sets @error to:
;												1 	- The $sAu3Code is empty string.
;											   -1 	- The process is aborted by user parameter $AU3SH_HIGHLIGHT_ABORT, and the return value is the original $sAu3Code.
;												2 	- Keywords or styles file not found (check the @ScriptDir & "\Resources").
;
; Author.........:	G.Sandler (a.k.a (Mr)CreatoR), www.creator-lab.ucoz.ru, www.autoit-script.ru
; Modified.......:	
; Remarks........:	*Idea* based on similar old project: http://www.autoitscript.com/forum/index.php?showtopic=34236
; Related........:	
; Link...........:	
; Example........:	Yes (see #Region Header).
; ===============================================================================================================
Func _Au3_SyntaxHighlight($sAu3Code, $iOutput = -1)
	;************************** I M P O R T A N T *************************
	;   Highlighting elements order must be preserved (DO NOT CHANGE IT)   
	;************************** I M P O R T A N T *************************
	
	Local $sOrigin_Au3Code = $sAu3Code
	
	If StringStripWS($sAu3Code, 8) = '' Then
		Return SetError(1, 0, 0)
	EndIf
	
	If Not $AU3SH_KEYWORDS_FILE Or Not FileExists($AU3SH_KEYWORDS_FILE) Then
		$AU3SH_KEYWORDS_FILE = @ScriptDir & "\Resources\au3.keywords.properties"
	EndIf
	
	If Not $AU3SH_API_FILE Or Not FileExists($AU3SH_API_FILE) Then
		$AU3SH_API_FILE = @ScriptDir & "\Resources\au3.api"
	EndIf
	
	If Not FileExists($AU3SH_KEYWORDS_FILE) Or Not FileExists($sAu3SH_Styles_File) Or Not FileExists($AU3SH_API_FILE) Then
		Return SetError(2, 0, 0)
	EndIf
	
	Local $sPattern1, $sPattern2, $sPattern3
	Local $sReplace1, $sReplace2, $sReplace3
	
	Local $sUnique_Str_Quote = __Au3_SH_GetUniqueStr($sAu3Code, '%@~@%')
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	Local $sUnique_Str_Include = __Au3_SH_GetUniqueStr($sAu3Code, '!~@%@~!') ; "!" must be the first character
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	Local $sUnique_Str_CmntBlk = __Au3_SH_GetUniqueStr($sAu3Code, '%~@~%~@~%')
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	Local $sUnique_Str_Cmnt = __Au3_SH_GetUniqueStr($sAu3Code, '%~@%@~%')
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	; Get the comment blocks to array
	Local $aCmntBlk_Strings = __Au3_SH_GetCmntBlcks($sAu3Code)
	
	; Replace all the comment blocks with unique marks (must be done before we get the literal strings)
	For $i = 1 To $aCmntBlk_Strings[0]
		$sAu3Code = StringReplace($sAu3Code, $aCmntBlk_Strings[$i], $sUnique_Str_CmntBlk, 0, 2)
	Next
	
	; Get all strings to array (all between "", '' and <> chars), so we can replace later with unique marks
	$sPattern1 = '("[^"\r\n]*?"|''[^''\r\n]*?'')'
	$sPattern2 = "(?i)#include\s+?(<[^\>]*>).*?"
	
	Local $aQuote_Strings = StringRegExp($sAu3Code, $sPattern1, 3)
	Local $aInclude_Strings = StringRegExp($sAu3Code, $sPattern2, 3)
	
	; Replace all the strings with unique marks
	$sPattern1 = '(?s)("|'')([^\1\r\n])*?(\1)'
	$sPattern2 = "(?si)(#include\s+?)<[^\>]*>(.*?)"
	
	$sReplace1 = $sUnique_Str_Quote
	$sReplace2 = '\1' & $sUnique_Str_Include & '\2'
	
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern2, $sReplace2)
	
	; Get all commented lines to array (must be done AFTER "Replace all the strings unique marks")
	$sPattern1 = '(;.*)'
	
	Local $aCmnt_Strings = StringRegExp($sAu3Code, $sPattern1, 3)
	
	; Replace all the commented lines with unique marks
	$sPattern1 = '(;.*)'
	$sReplace1 = $sUnique_Str_Cmnt
	
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	$sPattern1 = '([\(\)\[\]\<\>\.\*\+\-\=\&\^\,\/])'
	$sReplace1 = $aAu3SH_Styles[$iAu3SH_Oprtr_Style] & '\1' & $sAu3SH_CloseTag
	
	; Highlight the operators, brakets, commas (must be done before all other parsers)
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	$sPattern1 = '(\W+)(_)(\W+)'
	$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Oprtr_Style] & '\2' & $sAu3SH_CloseTag & '\3'
	
	; Highlight the line braking character, wich is the underscore (must be done before all other parsers)
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	$sPattern1 = '((?:\s+)?' & $aAu3SH_Styles[$iAu3SH_Oprtr_Style] & '\.' & $sAu3SH_CloseTag & '|(?:\s+)?\.)([^\d\$<]\w+)'
	$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_COM_Style] & '\2' & $sAu3SH_CloseTag
	
	; Highlight the COM Objects (must be done before all other parsers)
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	$sPattern1 = '([^\w#@])(\d+)([^\w]|\R|$)'
	$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Nmbr_Style] & '\2' & $sAu3SH_CloseTag & '\3'
	
	; Highlight the number
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	; Highlight the function
	__Au3_SH_ParseFuncs($sAu3Code)
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	; Highlight the UDFs
	__Au3_SH_ParseUDFs($sAu3Code)
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	; Highlight the keyword
	__Au3_SH_ParseKwrds($sAu3Code)
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	; Highlight the macros
	__Au3_SH_ParseMacros($sAu3Code)
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	; Highlight the PreProcessor
	__Au3_SH_ParsePreProc($sAu3Code)
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	; Highlight special keywords
	__Au3_SH_ParseSpec($sAu3Code)
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	$sPattern1 = '([^\w#@]|\R)((?i)0x[abcdef\d]+)([^abcdef\d])'
	$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Nmbr_Style] & '\2' & $sAu3SH_CloseTag & '\3'
	
	; Highlight the hexadecimal number
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	$sPattern1 = '\$(\w+)?'
	$sReplace1 = $aAu3SH_Styles[$iAu3SH_Var_Style] & '$\1' & $sAu3SH_CloseTag
	
	; Highlight variables (also can be just the dollar sign)
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	; Highlight send keys (must be done before the strings been restored, because send keys should be highlighted inside the strings)
	__Au3_SH_ParseSendKeys($aQuote_Strings)
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	$sPattern1 = '(\w+)(\h*' & $aAu3SH_Styles[$iAu3SH_Oprtr_Style] & '\()'
	$sReplace1 = $aAu3SH_Styles[$iAu3SH_WSpace_Style] & '\1' & $sAu3SH_CloseTag & '\2'
	
	; Highlight finaly the '#White space' (only user defined functions)
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	; Replace back the unique marks with the original one and wrap them with "string" tags
	For $i = 1 To UBound($aCmntBlk_Strings) - 1
		$sAu3Code = StringReplace($sAu3Code, $sUnique_Str_CmntBlk, $aAu3SH_Styles[$iAu3SH_CmntBlck_Style] & $aCmntBlk_Strings[$i] & $sAu3SH_CloseTag, 1, 2)
		
		If $AU3SH_HIGHLIGHT_ABORT = 1 Then
			Return SetError(-1, 0, $sOrigin_Au3Code)
		EndIf
	Next
	
	For $i = 0 To UBound($aCmnt_Strings) - 1
		$sAu3Code = StringReplace($sAu3Code, $sUnique_Str_Cmnt, $aAu3SH_Styles[$iAu3SH_Cmnt_Style] & $aCmnt_Strings[$i] & $sAu3SH_CloseTag, 1, 2)
		
		If $AU3SH_HIGHLIGHT_ABORT = 1 Then
			Return SetError(-1, 0, $sOrigin_Au3Code)
		EndIf
	Next
	
	For $i = 0 To UBound($aQuote_Strings) - 1
		$sAu3Code = StringReplace($sAu3Code, $sUnique_Str_Quote, $aAu3SH_Styles[$iAu3SH_Str_Style] & $aQuote_Strings[$i] & $sAu3SH_CloseTag, 1, 2)
		
		If $AU3SH_HIGHLIGHT_ABORT = 1 Then
			Return SetError(-1, 0, $sOrigin_Au3Code)
		EndIf
	Next
	
	For $i = 0 To UBound($aInclude_Strings) - 1
		$aInclude_Strings[$i] = StringReplace($aInclude_Strings[$i], '<', '&lt;', 0, 2)
		$aInclude_Strings[$i] = StringReplace($aInclude_Strings[$i], '>', '&gt;', 0, 2)
		
		$sAu3Code = StringReplace($sAu3Code, $sUnique_Str_Include, $aAu3SH_Styles[$iAu3SH_Str_Style] & $aInclude_Strings[$i] & $sAu3SH_CloseTag, 1, 2)
		
		If $AU3SH_HIGHLIGHT_ABORT = 1 Then
			Return SetError(-1, 0, $sOrigin_Au3Code)
		EndIf
	Next
	
	; Strip tags from "string" inside commented lines and regions (special keywords)
	Do
		$sAu3Code = StringRegExpReplace($sAu3Code, _
			'(.*?(?:' & $aAu3SH_Styles[$iAu3SH_Cmnt_Style] & ';|' & $aAu3SH_Styles[$iAu3SH_Spec_Style] & '#).*?)' & _
			'(?:' & $aAu3SH_Styles[$iAu3SH_Str_Style] & '|' & $aAu3SH_Styles[$iAu3SH_SntKys_Style] & '|' & $aAu3SH_Styles[$iAu3SH_Var_Style] & ')(.*?)' & $sAu3SH_CloseTag & '(.*?)', _
			'\1\2\3')
	Until @extended = 0
	
	; Replace tabs with 4 spaces, in IE the tabs not looking good :(.
	$sAu3Code = StringReplace($sAu3Code, @TAB, '    ', 0, 2) ;'&nbsp;&nbsp;&nbsp;&nbsp;'
	
	; Get the CSS styles header (to use on return)
	Local $sStyles_Header = __Au3_SH_GetKwrdStlsHdr()
	
	; Check the $iOutput flag
	Switch $iOutput
		; Return CSS classes header and AutoIt Syntax Highlighted code as array ([0] = CSS styles, [1] = Au3 Code)
		Case 1
			Local $aRet[2] = [$sStyles_Header, $sAu3Code]
			Return $aRet
		; Return AutoIt Syntax Highlighted code as string
		Case 2
			Return $sAu3Code
		; Return the result as BBCode formatted string
		Case 3
			__Au3_SH_HtmlToBBCTags($sAu3Code)
			
			If @error = -1 Then
				Return SetError(-1, 0, $sOrigin_Au3Code)
			EndIf
			
			Return $sAu3Code
		; Return CSS classes header and the AutoIt Syntax Highlighted code as Html formatted string
		Case Else
			Return $sStyles_Header & @CRLF & '<pre class="au3_codebox"><span>' & $sAu3Code & '</span></pre>'
	EndSwitch
EndFunc

#EndRegion Public Functions

#Region Internal Functions

Func __Au3_SH_ParseFuncs(ByRef $sAu3Code)
	Local $oFuncs = __Au3_SH_PropertyRead($AU3SH_KEYWORDS_FILE, "au3.keywords.functions")
	Local $sPattern1, $sReplace1
	
	For $sFunc In $oFuncs
		If $AU3SH_HIGHLIGHT_ABORT = 1 Then
			Return SetError(-1)
		EndIf
		
		If Not StringInStr($sAu3Code, $sFunc, 2) Then
			ContinueLoop
		EndIf
		
		$sPattern1 = '([^\w\$]+|\A)(?<!' & $aAu3SH_Styles[$iAu3SH_COM_Style] & ')((?i)' & $sFunc & ')(\W+|$)'
		
		If $AU3SH_ADD_URLS Then
			$sReplace1 = '\1<a href="' & StringFormat($AU3SH_FUNCS_URL, __Au3_SH_KeywordGetProperName($AU3SH_API_FILE, $sFunc)) & '">' & $aAu3SH_Styles[$iAu3SH_Fnc_Style] & '\2' & $sAu3SH_CloseTag & '</a>\3'
		Else
			$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Fnc_Style] & '\2' & $sAu3SH_CloseTag & '\3'
		EndIf
		
		$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	Next
EndFunc

Func __Au3_SH_ParseUDFs(ByRef $sAu3Code)
	Local $oUDFs = __Au3_SH_PropertyRead($AU3SH_KEYWORDS_FILE, "au3.keywords.udfs")
	
	If @error Then
		Return 0
	EndIf
	
	Local $sPattern1, $sReplace1
	
	For $sUDF In $oUDFs
		If $AU3SH_HIGHLIGHT_ABORT = 1 Then
			Return SetError(-1)
		EndIf
		
		If Not StringInStr($sAu3Code, $sUDF, 2) Then
			ContinueLoop
		EndIf
		
		$sPattern1 = '([^\w\$]+|\A)(?<!' & $aAu3SH_Styles[$iAu3SH_COM_Style] & ')((?i)' & $sUDF & ')(\W+|$)'
		
		If $AU3SH_ADD_URLS Then
			$sReplace1 = '\1<a href="' & StringFormat($AU3SH_UDFS_URL, __Au3_SH_KeywordGetProperName($AU3SH_API_FILE, $sUDF)) & '">' & $aAu3SH_Styles[$iAu3SH_Udf_Style] & '\2' & $sAu3SH_CloseTag & '</a>\3'
		Else
			$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Udf_Style] & '\2' & $sAu3SH_CloseTag & '\3'
		EndIf
		
		$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	Next
EndFunc

Func __Au3_SH_ParseKwrds(ByRef $sAu3Code)
	Local $oKwrds = __Au3_SH_PropertyRead($AU3SH_KEYWORDS_FILE, "au3.keywords.keywords")
	Local $sPattern1, $sReplace1
	
	For $sKwrd In $oKwrds
		If $AU3SH_HIGHLIGHT_ABORT = 1 Then
			Return SetError(-1)
		EndIf
		
		If Not StringInStr($sAu3Code, $sKwrd, 2) Then
			ContinueLoop
		EndIf
		
		$sPattern1 = '([^\w\$@]|\R|\A)(?<!' & $aAu3SH_Styles[$iAu3SH_COM_Style] & ')((?i)' & $sKwrd & ')(\W|$)'
		
		If $AU3SH_ADD_URLS Then
			$sReplace1 = '\1<a href="' & StringFormat($AU3SH_KEYWORDS_URL, __Au3_SH_KeywordGetProperName($AU3SH_API_FILE, $sKwrd)) & '">' & $aAu3SH_Styles[$iAu3SH_Kwd_Style] & '\2' & $sAu3SH_CloseTag & '</a>\3'
		Else
			$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Kwd_Style] & '\2' & $sAu3SH_CloseTag & '\3'
		EndIf
		
		$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	Next
EndFunc

Func __Au3_SH_ParseMacros(ByRef $sAu3Code)
	Local $oMcrs = __Au3_SH_PropertyRead($AU3SH_KEYWORDS_FILE, "au3.keywords.macros")
	Local $sPattern1, $sReplace1
	
	For $sMcr In $oMcrs
		If $AU3SH_HIGHLIGHT_ABORT = 1 Then
			Return SetError(-1)
		EndIf
		
		If Not StringInStr($sAu3Code, $sMcr, 2) Then
			ContinueLoop
		EndIf
		
		$sPattern1 = '(\W+|\R|\A)((?i)' & $sMcr & ')(\W+|$)'
		
		If $AU3SH_ADD_URLS Then
			$sReplace1 = '\1<a href="' & StringFormat($AU3SH_MACROS_URL, __Au3_SH_KeywordGetProperName($AU3SH_API_FILE, $sMcr)) & '">' & $aAu3SH_Styles[$iAu3SH_Mcro_Style] & '\2' & $sAu3SH_CloseTag & '</a>\3'
		Else
			$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Mcro_Style] & '\2' & $sAu3SH_CloseTag & '\3'
		EndIf
		
		$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	Next
EndFunc

Func __Au3_SH_ParsePreProc(ByRef $sAu3Code)
	Local $oPrPrc = __Au3_SH_PropertyRead($AU3SH_KEYWORDS_FILE, "au3.keywords.preprocessor")
	Local $sPattern1, $sPattern2, $sReplace1, $sReplace2
	
	For $sPrPrc In $oPrPrc
		If $AU3SH_HIGHLIGHT_ABORT = 1 Then
			Return SetError(-1)
		EndIf
		
		If Not StringInStr($sAu3Code, $sPrPrc, 2) Then
			ContinueLoop
		EndIf
		
		$sPattern1 = '(\W+|\R|\A)((?i)' & $sPrPrc & '.*)<(?:span|a href=).*?>(.*)</(?:span|a)>'
		$sPattern2 = '(\W+|\R|\A)((?i)' & $sPrPrc & ')([^\w!%-]+?|\Z)'
		
		$sReplace1 = '\1\2\3'
		
		If $AU3SH_ADD_URLS Then
			$sReplace2 = '\1<a href="' & StringFormat($AU3SH_PREPROC_URL, __Au3_SH_KeywordGetProperName($AU3SH_API_FILE, $sPrPrc)) & '">' & $aAu3SH_Styles[$iAu3SH_PrPrc_Style] & '\2' & $sAu3SH_CloseTag & '</a>\3'
		Else
			$sReplace2 = '\1' & $aAu3SH_Styles[$iAu3SH_PrPrc_Style] & '\2' & $sAu3SH_CloseTag & '\3'
		EndIf
		
		Do
			$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
		Until @extended = 0
		
		$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern2, $sReplace2)
	Next
EndFunc

Func __Au3_SH_ParseSpec(ByRef $sAu3Code)
	Local $oSpcl = __Au3_SH_PropertyRead($AU3SH_KEYWORDS_FILE, "au3.keywords.special")
	Local $sPattern1, $sPattern2, $sReplace1, $sReplace2
	
	For $sSpcl In $oSpcl
		If $AU3SH_HIGHLIGHT_ABORT = 1 Then
			Return SetError(-1)
		EndIf
		
		If Not StringInStr($sAu3Code, $sSpcl, 2) Then
			ContinueLoop
		EndIf
		
		$sPattern1 = '(\W+|\R|\A)((?i)' & $sSpcl & '.*)<(?:span|a href=).*?>(.*)</(?:span|a)>'
		$sPattern2 = '(\W+|\R|\A)((?i)' & $sSpcl & '.*)'
		$sReplace1 = '\1\2\3'
		$sReplace2 = '\1' & $aAu3SH_Styles[$iAu3SH_Spec_Style] & '\2' & $sAu3SH_CloseTag
		
		Do
			$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
		Until @extended = 0
		
		$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern2, $sReplace2)
	Next
	
	$sAu3Code = StringReplace($sAu3Code, @CR & '</span>', '</span>', 0, 2)
EndFunc

Func __Au3_SH_ParseSendKeys(ByRef $aStrs)
	Local $oSndKs = __Au3_SH_PropertyRead($AU3SH_KEYWORDS_FILE, "au3.keywords.sendkeys")
	Local $sPattern1, $sPattern2, $sReplace1, $sReplace2
	
	$sReplace1 = $aAu3SH_Styles[$iAu3SH_SntKys_Style] & '\1' & $sAu3SH_CloseTag
	$sPattern2 = '(?i)([\^+!#]*?{\S(?:[\h]+?[\h\d]*?)?})'
	$sReplace2 = $sReplace1
	
	For $i = 0 To UBound($aStrs) - 1
		If $AU3SH_HIGHLIGHT_ABORT = 1 Then
			Return SetError(-1)
		EndIf
		
		For $sSndKy In $oSndKs
			If Not StringInStr($aStrs[$i], $sSndKy, 2) Then
				ContinueLoop 2
			EndIf
		Next
		
		$sPattern1 = '(?i)([\^+!#]*?{' & $sSndKy & '(?:[\h]+?[\h\d]*?)?})'
		
		$aStrs[$i] = StringRegExpReplace($aStrs[$i], $sPattern1, $sReplace1)
		$aStrs[$i] = StringRegExpReplace($aStrs[$i], $sPattern2, $sReplace2)
	Next
EndFunc

Func __Au3_SH_GetUniqueStr(ByRef $sAu3Code, $sUnique_Str)
	While StringInStr($sAu3Code, $sUnique_Str, 2)
		$sUnique_Str &= '1' ;Random(10000, 99999)
		
		If $AU3SH_HIGHLIGHT_ABORT = 1 Then
			Return SetError(-1)
		EndIf
	WEnd
	
	Return $sUnique_Str
EndFunc

Func __Au3_SH_GetCmntBlcks($sAu3Code)
	Local $aSplit_Code = StringSplit(StringStripCR($sAu3Code), @LF)
	Local $sCmmntsEnd, $sCmntBlk, $iSubCmntOpen, $sSubCmntEnd, $aCmmnt_Blocks[1] = [0]
	
	For $i = 1 To $aSplit_Code[0]
		$sCmntBlk = ''
		
		If StringRegExp($aSplit_Code[$i], '(?i)^\h*#(cs|comments-start)') Then
			$sCmmntsEnd = 'comments-end'
			
			If StringRegExp($aSplit_Code[$i], '(?i)^\h*#cs') Then
				$sCmmntsEnd = 'ce'
			EndIf
			
			$sCmntBlk &= $aSplit_Code[$i] & @CRLF
			
			For $j = $i + 1 To $aSplit_Code[0]
				If StringRegExp($aSplit_Code[$j], '(?i)^\h*#(cs|comments-start)') Then
					$iSubCmntOpen = 1
					$sSubCmntEnd = 'comments-end'
					
					If StringRegExp($aSplit_Code[$j], '(?i)^\h*#cs') Then
						$sSubCmntEnd = 'ce'
					EndIf
				EndIf
				
				$sCmntBlk &= $aSplit_Code[$j] & @CRLF
				
				If Not $iSubCmntOpen And StringRegExp($aSplit_Code[$j], '(?i)^\h*#' & $sCmmntsEnd) Then
					ExitLoop
				EndIf
				
				If $iSubCmntOpen And StringRegExp($aSplit_Code[$j], '(?i)^\h*#' & $sSubCmntEnd) Then
					$iSubCmntOpen = 0
				EndIf
			Next
			
			$i = $j + 1
		EndIf
		
		If $sCmntBlk <> '' Then
			$aCmmnt_Blocks[0] += 1
			ReDim $aCmmnt_Blocks[$aCmmnt_Blocks[0] + 1]
			$aCmmnt_Blocks[$aCmmnt_Blocks[0]] = $sCmntBlk
		EndIf
	Next
	
	Return $aCmmnt_Blocks
EndFunc

Func __Au3_SH_GetKwrdStlsHdr()
	Local $sStyle
	
	Local $sFontBkColor = IniRead($sAu3SH_Styles_File, 'Styles', 'style.au3.32', '#f0f5fa')
	Local $aColor = StringSplit(IniRead($sAu3SH_Styles_File, 'Styles', 'style.au3.0', '#000000'), ',')
	
	Local $sFontColor = $aColor[1]
	Local $sFontWeight = ($aColor[0] > 1 ? $aColor[2] : 'normal')
	Local $sFontFamily = ($aColor[0] > 2 ? $aColor[3] : 'Courier New')
	
	Local $sRet = _
		'<head>' & @CRLF & _
		'<style type="text/css">' & @CRLF & @CRLF & _
		'.au3_codebox' & @CRLF & _
		'{' & @CRLF & _
		'	BORDER-BOTTOM: #AAAAAA 1px solid;' & @CRLF & _
		'	BORDER-LEFT: #AAAAAA 1px solid;' & @CRLF & _
		'	BORDER-RIGHT: #AAAAAA 1px solid;' & @CRLF & _
		'	BORDER-TOP: #AAAAAA 1px solid;' & @CRLF & _
		'	PADDING-RIGHT: 8px;' & @CRLF & _
		'	PADDING-LEFT: 8px;' & @CRLF & _
		'	PADDING-BOTTOM: 8px;' & @CRLF & _
		'	PADDING-TOP: 8px;' & @CRLF & _
		'	FONT-SIZE: 12px;' & @CRLF & _
		'	FONT-FAMILY: Courier New, Verdana, Courier, Arial;' & @CRLF & _
		'	FONT-WEIGHT: ' & $sFontWeight & ';' & @CRLF & _
		'	BACKGROUND-COLOR: ' & $sFontBkColor & ';' & @CRLF & _
		'	COLOR: ' & $sFontColor & ';' & @CRLF & _
		'	WHITE-SPACE: pre;' & @CRLF & _
		'}' & @CRLF & @CRLF & _
		'a' & @CRLF & _
		'{' & @CRLF & _
		'	text-decoration:none;' & @CRLF & _
		'}' & @CRLF & @CRLF & _
		'a:hover' & @CRLF & _
		'{' & @CRLF & _
		'	text-decoration:underline;' & @CRLF & _
		'}' & @CRLF & @CRLF & _
		'pre' & @CRLF & _
		'{' & @CRLF & _
		'	font-family: Verdana, Arial, Helvetica, sans-serif, "MS sans serif";' & @CRLF & _
		'	line-height: normal;' & @CRLF & _
		'	margin-top: 0.5em;' & @CRLF & _
		'	margin-bottom: 0.5em;' & @CRLF & _
		'}' & @CRLF & @CRLF & _
		'span' & @CRLF & _
		'{' & @CRLF & _
		'	font-family: ' & $sFontFamily & ';' & @CRLF & _
		'	font-weight: ' & $sFontWeight & ';' & @CRLF & _
		'	color: ' & $sFontColor & ';' & @CRLF & _
		'}' & @CRLF & @CRLF
	
	For $i = 0 To 15
		If $AU3SH_HIGHLIGHT_ABORT = 1 Then
			Return SetError(-1)
		EndIf
		
		$sStyle = IniRead($sAu3SH_Styles_File, 'Styles', 'style.au3.' & $i, '')
		$sRet &= '.au3_S' & $i
		
		; Check if it's Special keywords, then we need to override the other style
		If $i = 11 Or $i = 12 Then
			$sRet &= ', .au3_S' & $i & ' span'
		EndIf
		
		$sRet &= @CRLF & '{' & @CRLF
		
		If StringInStr($sStyle, 'bold', 2) Then
			$sRet &= '	font-weight: bold;' & @CRLF
		EndIf
		
		If StringInStr($sStyle, 'normal', 2) Then
			$sRet &= '	font-weight: normal;' & @CRLF
		EndIf
		
		If StringInStr($sStyle, 'italics', 2) Then
			$sRet &= '	font-style: italic;' & @CRLF
		EndIf
		
		If StringInStr($sStyle, '#', 2) Then
			$sRet &= '	color: ' & StringRegExpReplace($sStyle, '.*((?i)#[a-z0-9]+).*', '\1') & ';' & @CRLF
		EndIf
		
		$sRet &= '}' & @CRLF & @CRLF
	Next
	
	Return $sRet & '</style></head>' & @CRLF
EndFunc

Func __Au3_SH_HtmlToBBCTags(ByRef $sAu3HtmlCode)
	Local $sStyle, $sTags_Start, $sTags_End, $sRet
	
	For $i = 0 To 15
		If $AU3SH_HIGHLIGHT_ABORT = 1 Then
			Return SetError(-1)
		EndIf
		
		$sStyle = IniRead($sAu3SH_Styles_File, 'Styles', 'style.au3.' & $i, '')
		
		$sTags_Start = ''
		$sTags_End = ''
		
		If StringInStr($sStyle, 'bold', 2) Then
			$sTags_Start = '[b]'
			$sTags_End = '[/b]'
		EndIf
		
		If StringInStr($sStyle, 'italics', 2) Then
			$sTags_Start = '[i]' & $sTags_Start
			$sTags_End &= '[/i]'
		EndIf
		
		If StringInStr($sStyle, '#', 2) Then
			$sTags_Start = '[color=' & StringRegExpReplace($sStyle, '.*((?i)#[a-z0-9]+).*', '\1') & ']' & $sTags_Start
			$sTags_End &= '[/color]'
		EndIf
		
		; Replace <a href=> tag with bbcode [url=] tag...
		$sAu3HtmlCode = StringRegExpReplace($sAu3HtmlCode, '(?si)(.*?)<a href="([^"]*)">(.*?)</a>(.*?)', '\1[url=\2]\3[/url]\4')
		
		; Replace all the styles with bbcode
		$sAu3HtmlCode = StringRegExpReplace($sAu3HtmlCode, _
			'(?si)(.*?)' & $aAu3SH_Styles[$i] & '(.*?)' & $sAu3SH_CloseTag & '(.*?)', '\1' & $sTags_Start & '\2' & $sTags_End & '\3')
		
		While StringRegExp($sAu3HtmlCode, $aAu3SH_Styles[$i] & '(.*?)' & $sAu3SH_CloseTag)
			If $AU3SH_HIGHLIGHT_ABORT = 1 Then
				Return SetError(-1)
			EndIf
			
			$sAu3HtmlCode = StringRegExpReplace($sAu3HtmlCode, _
				'(?si)(.*?)' & $aAu3SH_Styles[$i] & '(.*?)' & $sAu3SH_CloseTag & '(.*?)', '\1' & $sTags_Start & '\2' & $sTags_End & '\3')
		WEnd
	Next
	
	; Replace all the html entities and <br>s
	$sAu3HtmlCode = StringReplace($sAu3HtmlCode, '<br>', @CRLF, 0, 2)
	$sAu3HtmlCode = StringReplace($sAu3HtmlCode, '&lt;', '<', 0, 2)
	$sAu3HtmlCode = StringReplace($sAu3HtmlCode, '&gt;', '>', 0, 2)
	$sAu3HtmlCode = StringReplace($sAu3HtmlCode, '&nbsp;', ' ', 0, 2)
EndFunc

Func __Au3_SH_PropertyRead($sFile, $sProperty, $sDefault = '', $fRetObj = True)
	Local Static $sFileRead = StringStripCR(FileRead($sFile))
	Local Static $sOld_File = $sFile
	
	If $sOld_File <> $sFile Then
		$sOld_File = $sFile
		$sFileRead = StringStripCR(FileRead($sFile))
	EndIf
	
	Local $oRet = ObjCreate('Scripting.Dictionary')
	
	If Not IsObj($oRet) Or Not $sFileRead Then
		Return SetError(1, 0, 0)
	EndIf
	
	$oRet.CompareMode = 1 ;Text mode (not case sensitive)
	
	Local $aProp = StringRegExp(@CRLF & $sFileRead, '(?i)\r?\n\Q' & $sProperty & '\E\h*=\h*([\s\S]+?)(?:\r?\n[^\r\n]+=|$)', 3)
	
	If @error Then
		Return SetError(2, 0, $sDefault)
	EndIf
	
	$aProp = StringRegExp($aProp[0], '([^\s\\]+)', 3)
	
	If @error Then
		Return SetError(3, 0, $sDefault)
	EndIf
	
	Local $sRet
	
	For $sProp In $aProp
		If Not $oRet.Exists($sProp) Then
			If $fRetObj Then
				$oRet.Add($sProp, $sProp)
			Else
				$sRet &= ($sRet ? ' ' : '') & $sProp
			EndIf
		EndIf
	Next
	
	If ($fRetObj And $oRet.Count = 0) Or (Not $fRetObj And $sRet = '') Then
		Return $sDefault
	EndIf
	
	Return ($fRetObj ? $oRet : $sRet)
EndFunc

Func __Au3_SH_KeywordGetProperName($sAPI_File, $sKeyword)
	Local Static $aRead = StringRegExp(FileRead($sAPI_File), '(\r?\n|^)([#@]?\w+)\W', 3)
	Local Static $oArr = ObjCreate('Scripting.Dictionary')
	Local Static $sOld_File = $sAPI_File
	
	If $sOld_File <> $sAPI_File Then
		$aRead = StringRegExp(FileRead($sAPI_File), '(\r?\n|^)([#@]?\w+)\W', 3)
		$sOld_File = $sAPI_File
	EndIf
	
	If $oArr.Count = 0 Then
		$oArr.CompareMode = 1
		
		For $i = 1 To UBound($aRead) - 1
			If Not $oArr.Exists($aRead[$i]) Then
				$oArr.Add($aRead[$i], $aRead[$i])
			EndIf
		Next
	EndIf
	
	Return $oArr($sKeyword)
EndFunc

#EndRegion Internal Functions
