#include-once
#include <FileConstants.au3>

; #INDEX# =======================================================================================================================
; Title .........: IniEx
; AutoIt Version : 3.3.14.5
; Description ...: Extended Ini functions to allow working with strings instead of files and other built-in missing functions.
; Author(s) .....: G.Sandler ((Mr)CreatoR)
; Dll ...........: None.
; ===============================================================================================================================

; #CONSTANTS# ===================================================================================================================
Global Const $INIEX_TMPINIFILE = @TempDir & '\~IniEx.tmp'
; ===============================================================================================================================

; #CURRENT# =====================================================================================================================
;_IniEx_Count
;_IniEx_Delete
;_IniEx_Exists
;_IniEx_Read
;_IniEx_ReadSection
;_IniEx_ReadSectionNames
;_IniEx_RenameKey
;_IniEx_RenameKeyValue
;_IniEx_RenameSection
;_IniEx_Write
;_IniEx_WriteSection
; ===============================================================================================================================

; #INTERNAL_USE_ONLY# ===========================================================================================================
;__IniEx_InitVirtual
;__IniEx_RetVirtual
; ===============================================================================================================================

; #FUNCTION# ====================================================================================================================
; Name ..........: _IniEx_Count
; Description ...: Count keys or sections names in file or in ini string.
; Syntax ........: _IniEx_Count(Byref $sFile[, $sSect = Default])
; Parameters ....: $sFile       - [in/out] File or string representing ini content.
;                  $sSect       - [optional] Section name. Default is Default keyword.
; Return values .: Success      - If $sSect is specified, the count of keys in section, otherwise the count of sections.
;                  Failure      - 0 and sets @error to 1 if unable to get keys/sections count.
;                  |@error can be set to -1 if unable to initialize virtual ini
;                  |(in case $sFile is a string and $INIEX_TMPINIFILE is not writable).
; Author ........: G.Sandler (CreatoR)
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _IniEx_Count(ByRef $sFile, $sSect = Default)
	Local $fVirtual = __IniEx_InitVirtual($sFile)
	
	If @error Then
		Return SetError(-1, 0, 0)
	EndIf
	
	Local $iRet, $aRead
	
	If $sSect == Default Then
		$aRead = IniReadSectionNames($sFile)
	Else
		$aRead = IniReadSection($sFile, $sSect)
	EndIf
	
	Local $iErr = @error
	$iRet = UBound($aRead) - 1
	
	__IniEx_RetVirtual($sFile, $fVirtual)
	Return SetError(($iErr ? 1 : 0), 0, $iRet)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _IniEx_Delete
; Description ...: Deletes a value from a standard format .ini file or to ini string.
; Syntax ........: _IniEx_Delete(Byref $sFile, $sSect[, $sKeyVal = Default[, $fByVal = False]])
; Parameters ....: $sFile       - [in/out] File or string representing ini content.
;                  $sSect       - Section name.
;                  $sKeyVal     - [optional] The key name to delete, or a value by which the key will be deleted.
;                  |If the key name or a value is not given the entire section is deleted.
;                  |The Default keyword may also be used which will cause the section to be deleted.
;                  $fByVal      - [optional] If True, $sKeyVal considered as value, 
;                  |so the key deleted (the first one) is the one that found by value. Default is False.
; Return values .: Success      - 1. If $sFile is a string representing ini content, the result will be written to this string.
;                  Failure      - 0 if the INI file does not exist or if the file is read-only.
;                  |@error can be set to -1 if unable to initialize virtual ini
;                  |(in case $sFile is a string and $INIEX_TMPINIFILE is not writable).
; Author ........: G.Sandler (CreatoR)
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _IniEx_Delete(ByRef $sFile, $sSect, $sKeyVal = Default, $fByVal = False)
	Local $fVirtual = __IniEx_InitVirtual($sFile)
	
	If @error Then
		Return SetError(-1, 0, 0)
	EndIf
	
	Local $iRet
	
	If Not ($sKeyVal == Default) And $fByVal Then
		Local $aSect = IniReadSection($sFile, $sSect)
		
		If Not @error Then
			Local $aTmp[$aSect[0][0] + 1][2]
			
			For $i = 1 To $aSect[0][0]
				;Make sure we delete only first key of the found value
				If Not $iRet And $sKeyVal = $aSect[$i][1] Then
					$iRet = 1
				Else
					$aTmp[0][0] += 1
					$aTmp[$aTmp[0][0]][0] = $aSect[$i][0]
					$aTmp[$aTmp[0][0]][1] = $aSect[$i][1]
				EndIf
			Next
			
			ReDim $aTmp[$aTmp[0][0] + 1][2]
			$iRet = IniWriteSection($sFile, $sSect, $aTmp)
		EndIf
	Else
		$iRet = IniDelete($sFile, $sSect, $sKeyVal)
	EndIf
	
	__IniEx_RetVirtual($sFile, $fVirtual)
	Return $iRet
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _IniEx_Exists
; Description ...: Checks if a given key or section exists in a standard format .ini file or ini string.
; Syntax ........: _IniEx_Exists($sFile, $sSect[, $sKey = Default])
; Parameters ....: $sFile       - [in/out] File or string representing ini content.
;                  $sSect       - Section name.
;                  $sKey        - [optional] The key name to check. Default is Default keyword.
; Return values .: Success      - True if a key or section (if $sKey is not specified) exists.
;                  Failure      - False and sets @error to 1 if key or section is not found.
;                  |@error can be set to -1 if unable to initialize virtual ini
;                  |(in case $sFile is a string and $INIEX_TMPINIFILE is not writable).
; Author ........: G.Sandler (CreatoR)
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _IniEx_Exists(ByRef $sFile, $sSect, $sKey = Default)
	Local $fVirtual = __IniEx_InitVirtual($sFile)
	
	If @error Then
		Return SetError(-1, 0, 0)
	EndIf
	
	Local $vRet = Default
	Local $iErr = 0
	
	If Not ($sKey == Default) Then
		$vRet = IniRead($sFile, $sSect, $sKey, Default)
		$iErr = @error
	Else
		Local $aNames = IniReadSectionNames($sFile)
		$iErr = @error
		
		If Not $iErr Then
			For $i = 1 To $aNames[0]
				If $sSect = $aNames[$i] Then
					$vRet = 1
					ExitLoop
				EndIf
			Next
		EndIf
	EndIf
	
	__IniEx_RetVirtual($sFile, $fVirtual)
	Return SetError(($iErr ? 1 : 0), 0, Not ($vRet == Default))
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _IniEx_Read
; Description ...: Reads a value or a key from a standard format .ini file or from ini string.
; Syntax ........: _IniEx_Read(Byref $sFile, $sSect, $sKeyVal[, $sDefault = ''[, $fByVal = False[, $iCount = -1]]])
; Parameters ....: $sFile       - [in/out] File or string representing ini content.
;                  $sSect       - Section name.
;                  $sKeyVal     - Key or key value.
;                  $sDefault    - [optional] The default value to return if the requested key/value is not found. Default is ''.
;                  $fByVal      - [optional] If True, $sKeyVal considered as value. Default is False.
;                  $iCount      - [optional] Key/value count to return if more than one is found. Default is -1, the first found.
; Return values .: Success      - Depends on $fByVal, if True, the return will be the key of found value, 
;                  |otherwise the value of found key.
;                  Failure      - The default value to return if the requested key/value is not found
;                  |@error can be set to -1 if unable to initialize virtual ini
;                  |(in case $sFile is a string and $INIEX_TMPINIFILE is not writable).
; Author ........: G.Sandler (CreatoR)
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _IniEx_Read(ByRef $sFile, $sSect, $sKeyVal, $sDefault = '', $fByVal = False, $iCount = -1)
	Local $fVirtual = __IniEx_InitVirtual($sFile)
	
	If @error Then
		Return SetError(-1, 0, $sDefault)
	EndIf
	
	Local $sRet = IniRead($sFile, $sSect, $sKeyVal, $sDefault)
	
	If $iCount <= 0 And Not $fByVal Then
		__IniEx_RetVirtual($sFile, $fVirtual)
		Return $sRet
	EndIf
	
	Local $aSect = IniReadSection($sFile, $sSect)
	Local $iErr = @error
	
	If Not $iErr Then
		$sRet = $sDefault
		Local $iC = 0, $iFound = 0
		
		For $i = 1 To $aSect[0][0]
			If $aSect[$i][($fByVal ? 1 : 0)] = $sKeyVal Then
				;Get only the first found item
				If Not $iFound Then
					$iFound = $i
				EndIf
				
				$iC += 1
				
				If $iC = $iCount Then
					$sRet = $aSect[$i][($fByVal ? 0 : 1)]
					ExitLoop
				EndIf
			EndIf
		Next
		
		;Make sure that we get the key/val if count is grater than actualy found item
		If $sRet = $sDefault And $iFound Then
			$sRet = $aSect[$iFound][($fByVal ? 0 : 1)]
		EndIf
	EndIf
	
	__IniEx_RetVirtual($sFile, $fVirtual)
	Return SetError(($iErr ? 1 : 0), 0, $sRet)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _IniEx_ReadSection
; Description ...: Reads all key/value pairs from a section in a standard format .ini file or from ini string.
; Syntax ........: _IniEx_ReadSection(Byref $sFile, $sSect)
; Parameters ....: $sFile       - [in/out] File or string representing ini content.
;                  $sSect       - Section name.
; Return values .: Success      - A 2 dimensional array where element[n][0] is the key and element[n][1] is the value.
;                  Failure      - Sets the @error flag to non-zero if unable to read the section
;                  |(The INI file may not exist or the section may not exist or is empty).
;                  |@error can be set to -1 if unable to initialize virtual ini
;                  |(in case $sFile is a string and $INIEX_TMPINIFILE is not writable).
; Author ........: G.Sandler (CreatoR)
; Modified ......: 
; Remarks .......: The number of elements returned will be in $aArray[0][0]. If an @error occurs, no array is created.
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _IniEx_ReadSection(ByRef $sFile, $sSect)
	Local $fVirtual = __IniEx_InitVirtual($sFile)
	
	If @error Then
		Return SetError(-1, 0, 0)
	EndIf
	
	Local $aRet = IniReadSection($sFile, $sSect)
	Local $iErr = @error
	
	__IniEx_RetVirtual($sFile, $fVirtual)
	Return SetError(($iErr ? 1 : 0), 0, $aRet)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _IniEx_ReadSectionNames
; Description ...: Reads all section names in a standard format .ini file or from ini string.
; Syntax ........: _IniEx_ReadSectionNames(Byref $sFile)
; Parameters ....: $sFile       - [in/out] File or string representing ini content.
; Return values .: Success      - An array of all section names in the INI file.
;                  Failure      - Sets the @error flag to non-zero.
;                  |@error can be set to -1 if unable to initialize virtual ini
;                  |(in case $sFile is a string and $INIEX_TMPINIFILE is not writable).
; Author ........: G.Sandler (CreatoR)
; Modified ......: 
; Remarks .......: The number of elements returned will be in $aArray[0]. If an @error occurs, no array is returned.
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _IniEx_ReadSectionNames(ByRef $sFile)
	Local $fVirtual = __IniEx_InitVirtual($sFile)
	
	If @error Then
		Return SetError(-1, 0, 0)
	EndIf
	
	Local $aRet = IniReadSectionNames($sFile)
	Local $iErr = @error
	
	__IniEx_RetVirtual($sFile, $fVirtual)
	Return SetError(($iErr ? 1 : 0), 0, $aRet)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _IniEx_RenameKey
; Description ...: Renames a key in a standard format .ini file or in ini string.
; Syntax ........: _IniEx_RenameKey(Byref $sFile, $sSect, $sOld_Key, $sNew_Key[, $iFlag = 0])
; Parameters ....: $sFile       - [in/out] File or string representing ini content.
;                  $sSect       - Section name.
;                  $sOld_Key    - Key to rename.
;                  $sNew_Key    - New key.
;                  $iFlag       - [optional] 
;                  |0 (default) - Fail if "new key" already exists.
;                  |1 - Rename to "new key" even if it's already exists.
; Return values .: Success      - 1.
;                  Failure      - 0 and sets the @error to 1 if renaming failed because the key already exists (if $iFlag = 0).
;                  |@error can be set to -1 if unable to initialize virtual ini
;                  |(in case $sFile is a string and $INIEX_TMPINIFILE is not writable).
; Author ........: G.Sandler (CreatoR)
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _IniEx_RenameKey(ByRef $sFile, $sSect, $sOld_Key, $sNew_Key, $iFlag = 0)
	Local $fVirtual = __IniEx_InitVirtual($sFile)
	
	If @error Then
		Return SetError(-1, 0, 0)
	EndIf
	
	Local $iErr = 0, $iRet = 0, $fWrite = True, $fFound = False
	Local $aSect = IniReadSection($sFile, $sSect)
	
	If Not @error Then
		For $i = 1 To $aSect[0][0]
			If $iFlag = 0 And $aSect[$i][0] = $sNew_Key Then
				$fWrite = False
				ExitLoop
			EndIf
			
			If $aSect[$i][0] = $sOld_Key Then
				$aSect[$i][0] = $sNew_Key
				$fFound = True
			EndIf
		Next
		
		If $fWrite And $fFound Then
			$iRet = IniWriteSection($sFile, $sSect, $aSect)
		ElseIf Not $fWrite Then
			$iErr = 1
		EndIf
	EndIf
	
	__IniEx_RetVirtual($sFile, $fVirtual)
	Return SetError(($iErr ? 1 : 0), 0, $iRet)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _IniEx_RenameKeyValue
; Description ...: Renames a key=value pair in a standard format .ini file or in ini string.
; Syntax ........: _IniEx_RenameKeyValue(Byref $sFile, $sSect, $sOld_Key, $sOld_Val, $sNew_Key, $sNew_Val[, $iFlag = 0])
; Parameters ....: $sFile       - [in/out] File or string representing ini content.
;                  $sSect       - Section name.
;                  $sOld_Key    - Key to rename.
;                  $sOld_Value  - Value pair to check in a key.
;                  $sNew_Key    - New key.
;                  $sNew_Val    - New value to be written to new key.
;                  $iFlag       - [optional] 
;                  |0 (default) - Fail if "new key=new value" already exists.
;                  |1 - Rename to "new key=new value" even if it's already exists.
; Return values .: Success      - 1.
;                  Failure      - 0 and sets the @error to 1 if renaming failed because the key=value already exists (if $iFlag = 0).
;                  |@error can be set to -1 if unable to initialize virtual ini
;                  |(in case $sFile is a string and $INIEX_TMPINIFILE is not writable).
; Author ........: G.Sandler (CreatoR)
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _IniEx_RenameKeyValue(ByRef $sFile, $sSect, $sOld_Key, $sOld_Val, $sNew_Key, $sNew_Val, $iFlag = 0)
	Local $fVirtual = __IniEx_InitVirtual($sFile)
	
	If @error Then
		Return SetError(-1, 0, 0)
	EndIf
	
	Local $iErr = 0, $iRet = 0, $fWrite = True, $fFound = False
	Local $aSect = IniReadSection($sFile, $sSect)
	
	If Not @error Then
		For $i = 1 To $aSect[0][0]
			If $iFlag = 0 And $aSect[$i][0] = $sNew_Key And $aSect[$i][1] = $sNew_Val Then
				$fWrite = False
				ExitLoop
			EndIf
			
			If $aSect[$i][0] = $sOld_Key And $aSect[$i][1] = $sOld_Val Then
				$aSect[$i][0] = $sNew_Key
				$aSect[$i][1] = $sNew_Val
				$fFound = True
			EndIf
		Next
		
		If $fWrite And $fFound Then
			$iRet = IniWriteSection($sFile, $sSect, $aSect)
		ElseIf Not $fWrite Then
			$iErr = 1
		EndIf
	EndIf
	
	__IniEx_RetVirtual($sFile, $fVirtual)
	Return SetError(($iErr ? 1 : 0), 0, $iRet)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _IniEx_RenameSection
; Description ...: Renames a section in a standard format .ini file or in ini string.
; Syntax ........: _IniEx_RenameSection(Byref $sFile, $sOld_Sect, $sNew_Sect[, $iFlag = 0])
; Parameters ....: $sFile       - [in/out] File or string representing ini content.
;                  $sOld_Sect   - Section name to rename.
;                  $sNew_Sect   - New section name.
;                  $iFlag       - [optional] 
;                  |$FC_NOOVERWRITE(0) = (default) Fail if "new section" already exists.
;                  |$FC_OVERWRITE(1) = Overwrite "new section". This will erase any existing keys in "new section".
;                  |
;                  |Constants are defined in FileConstants.au3.
; Return values .: Success      - 1.
;                  Failure      - 0 and may sets the @error flag to non-zero, if renaming failed because the 
;                  |section already exists (only when flag = 0).
;                  |@error can be set to -1 if unable to initialize virtual ini
;                  |(in case $sFile is a string and $INIEX_TMPINIFILE is not writable).
; Author ........: G.Sandler (CreatoR)
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _IniEx_RenameSection(ByRef $sFile, $sOld_Sect, $sNew_Sect, $iFlag = 0)
	Local $fVirtual = __IniEx_InitVirtual($sFile)
	
	If @error Then
		Return SetError(-1, 0, 0)
	EndIf
	
	Local $iRet = IniRenameSection($sFile, $sOld_Sect, $sNew_Sect, $iFlag)
	Local $iErr = @error
	
	__IniEx_RetVirtual($sFile, $fVirtual)
	Return SetError(($iErr ? 1 : 0), 0, $iRet)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _IniEx_Write
; Description ...: Writes a value to a standard format .ini file or to ini string.
; Syntax ........: _IniEx_Write(Byref $sFile, $sSect, $sKey, $sValue[, $fReplace = True[, $iCount = -1[, $iPos = -1]]])
; Parameters ....: $sFile       - [in/out] File or string representing ini content.
;                  $sSect       - Section name.
;                  $sKey        - Key name.
;                  $sValue      - Value to write.
;                  $fReplace    - [optional] If True (default), the key will be rewriten, otherwise will add a new key.
;                  $iCount      - [optional] 1-based item that defines to what key to write the value if found several key names.
;                               |Default is -1, write to first found key name.
;                               |This parameter forces $fReplace = True, which means the next parameter is ignored as well.
;                  $iPos        - [optional] 1-based position where to insert the key. Default is -1, insert to the end.
;                               |This parameter ignored if $fReplace = True.
; Return values .: Success      - 1.
;                  Failure      - 0 if unable to write to the file.
;                  |@error can be set to -1 if unable to initialize virtual ini
;                  |(in case $sFile is a string and $INIEX_TMPINIFILE is not writable).
; Author ........: G.Sandler (CreatoR)
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _IniEx_Write(ByRef $sFile, $sSect, $sKey, $sValue, $fReplace = True, $iCount = -1, $iPos = -1)
	Local $fVirtual = __IniEx_InitVirtual($sFile)
	
	If @error Then
		Return SetError(-1, 0, 0)
	EndIf
	
	Local $iRet
	
	If $fReplace Or $iCount > 0 Then
		Local $fFound = False
		Local $aSect = IniReadSection($sFile, $sSect)
		
		If Not @error Then
			Local $iC = 0
			
			For $i = 1 To $aSect[0][0]
				If $aSect[$i][0] = $sKey Then
					$iC += 1
					
					If $iC = $iCount Then
						$fFound = True
						 $aSect[$i][1] = $sValue
						 ExitLoop
					EndIf
				EndIf
			Next
		EndIf
		
		If $fFound Then
			$iRet = IniWriteSection($sFile, $sSect, $aSect)
		Else
			$iRet = IniWrite($sFile, $sSect, $sKey, $sValue)
		EndIf
		
		__IniEx_RetVirtual($sFile, $fVirtual)
		Return $iRet
	EndIf
	
	Local $aSect = IniReadSection($sFile, $sSect)
	
	If @error Then
		Dim $aSect[1][2] = [[0]]
	EndIf
	
	ReDim $aSect[$aSect[0][0] + 2][2]
	$aSect[0][0] += 1
	
	If $iPos <= 0 Or $iPos = $aSect[0][0] Then
		$aSect[$aSect[0][0]][0] = $sKey
		$aSect[$aSect[0][0]][1] = $sValue
	Else
		If $iPos > $aSect[0][0] Then
			$iPos = $aSect[0][0]
		EndIf
		
		For $i = $aSect[0][0] To $iPos Step -1
			$aSect[$i][0] = $aSect[$i-1][0]
			$aSect[$i][1] = $aSect[$i-1][1]
		Next
		
		$aSect[$iPos][0] = $sKey
		$aSect[$iPos][1] = $sValue
	EndIf
	
	$iRet = IniWriteSection($sFile, $sSect, $aSect)
	
	__IniEx_RetVirtual($sFile, $fVirtual)
	Return $iRet
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _IniEx_WriteSection
; Description ...: Writes a section to a standard format .ini file or to ini string.
; Syntax ........: _IniEx_WriteSection(Byref $sFile, $sSect, $vData[, $iIndex = 1])
; Parameters ....: $sFile       - [in/out] File or string representing ini content.
;                  $sSect       - Section name.
;                  $vData       - The data to write. The data can either be a string or an array.
;                  |If the data is a string, then each key=value pair must be delimited by @LF.
;                  |If the data is an array, the array must be 2-dimensional and the second dimension must be 2 elements.
;                  $iIndex              - [optional] If an array is passed as data,
;                  |this specifies the index to start writing from. By default, this is 1 so that the return value of 
;                  |_IniEx_ReadSection() can be used immediately. For manually created arrays, this value need to be different 
;                  |depending on how the array was created. This parameter is ignored if a string is passed as data.
; Return values .: Success      - 1. If $sFile is a string representing ini content, the result will be written to this string.
;                  Failure      - 0. The function will sets the @error flag to 1 if the data format is invalid.
;                  |@error can be set to -1 if unable to initialize virtual ini
;                  |(in case $sFile is a string and $INIEX_TMPINIFILE is not writable).
; Author ........: G.Sandler (CreatoR)
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _IniEx_WriteSection(ByRef $sFile, $sSect, $vData, $iIndex = 1)
	Local $fVirtual = __IniEx_InitVirtual($sFile)
	
	If @error Then
		Return SetError(-1, 0, 0)
	EndIf
	
	Local $iRet = IniWriteSection($sFile, $sSect, $vData, $iIndex)
	Local $iErr = @error
	
	__IniEx_RetVirtual($sFile, $fVirtual)
	Return SetError(($iErr ? 1 : 0), 0, $iRet)
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Name ..........: __IniEx_InitVirtual
; Description ...: Initialize Ini virtual (to work with string instead of file).
; Syntax ........: __IniEx_InitVirtual(Byref $sFile)
; Parameters ....: $sFile       - [in/out] File name to check if it's ini file or ini string.
; Return values .: Success      - True.
;                  Failure      - False and sets @error to 1 if unable to write to tmp file ($INIEX_TMPINIFILE).
; Author ........: G.Sandler (CreatoR)
; Modified ......: 
; Remarks .......: If $sFile is not recognised as file, 
;                  |the function will write to temporary file ($INIEX_TMPINIFILE) the ini content and set $sFile to temp path.
; Related .......: All public functions.
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func __IniEx_InitVirtual(ByRef $sFile)
	If StringRegExp($sFile, '(?m)^\h*\[.*\][^\r\n]*\r?\n\s*^[^=]+=.*$') Then
		Local $hFile = FileOpen($INIEX_TMPINIFILE, $FO_OVERWRITE)
		
		If $hFile = -1 Then
			Return SetError(1, 0, 0)
		EndIf
		
		FileWrite($hFile, $sFile)
		FileClose($hFile)
		
		$sFile = $INIEX_TMPINIFILE
		Return True
	EndIf
	
	Return False
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Name ..........: __IniEx_RetVirtual
; Description ...: Checks if it's a virtual ini, and sets it's content to the return value.
; Syntax ........: __IniEx_RetVirtual(Byref $vRet, $fVirtual)
; Parameters ....: $vRet        - [in/out] Return value to set.
;                  $fVirtual    - Flag indicates virtual ini as returned by __IniEx_InitVirtual.
; Return values .: Success      - 0 and sets virtual ini to $vRet variable and deletes tmp file ($INIEX_TMPINIFILE).
;                  Failure      - 0.
; Author ........: G.Sandler (CreatoR)
; Modified ......: 
; Remarks .......: 
; Related .......: __IniEx_InitVirtual
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func __IniEx_RetVirtual(ByRef $vRet, $fVirtual)
	If $fVirtual Then
		$vRet = StringStripWS(FileRead($INIEX_TMPINIFILE), 3)
		FileDelete($INIEX_TMPINIFILE)
	 EndIf
	 
	 Return 0
EndFunc
