#Region Header

#CS
	Name:				GDIPlusPic
	Description:		Library designed to create picture control using GDI+ that supports more image formats and faster handle on big size images.
	Author:				Copyright  2019 CreatoR's Lab (G.Sandler), www.creator-lab.ucoz.ru, www.autoit-script.ru. All rights reserved.
	AutoIt version:		3.3.1x.x
	UDF version:		
	
	History:
	                    1.1
						+ Added URL support for $sFile in _GDIPlusPic_Create and _GDIPlusPic_SetInfo.
						* $GPP_EVENT_HOVER renamed to $GPP_EVENT_ENTERHOVER.
						* $GPP_EVENT_ENDHOVER renamed to $GPP_EVENT_EXITHOVER.
						* Fixed wrong callback handling.
						* Changed some functions logic.
						* Example changed.
						
						1.0
						* First public version
#CE

#include-once
#include <WindowsConstants.au3>
#include <GDIPlus.au3>
#include <WinAPISys.au3>
#include <WinAPIvkeysConstants.au3>

#EndRegion Header

#Region Global Variables and startup

Global Enum $iGPP_File, $iGPP_Parent, $iGPP_Pic, $iGPP_Image, $iGPP_Left, $iGPP_Top, $iGPP_Width, $iGPP_Height, $iGPP_Callback, $iGPP_Callback_Flag, _
	$iGPP_Total
Global $aGPP_Data[1][$iGPP_Total]

_GDIPlus_Startup()
OnAutoItExitRegister('__GPP_OnExit')
AdlibRegister('__GPP_Callback', 10)

#EndRegion Global Variables and startup

#Region User Variables

;Set/GetInfo constants (for SetInfo first two ignored)
Global Enum $GPP_INFO_GUI, $GPP_INFO_PIC, $GPP_INFO_FILE, $GPP_INFO_LEFT, $GPP_INFO_TOP, $GPP_INFO_WIDTH, $GPP_INFO_HEIGHT, $GPP_INFO_CALLBACK, _
	$GPP_INFO_TOTAL

Global Enum $GPP_EVENT_ENTERHOVER = 1, $GPP_EVENT_EXITHOVER, $GPP_EVENT_LBUTTONDOWN, $GPP_EVENT_LBUTTONUP

#EndRegion User Variables

#Region Public Functions

; #FUNCTION# ====================================================================================================================
; Name ..........: _GDIPlusPic_Create
; Description ...: Creates picture control using GDIPlus.
; Syntax ........: _GDIPlusPic_Create($hWnd, $sFile, $iL, $iT, $iW, $iH[, $bShow = True[, $sCallback = '']])
; Parameters ....: $hWnd                - Handle to main GUI window.
;                  $sFile               - File name or URL of the picture to be loaded. Supported types: BMP, JPG, GIF (not animated), PNG, TIFF, EMF.
;                  $iL                  - The left side of the control.
;                  $iT                  - The top of the control.
;                  $iW                  - The width of the control.
;                  $iH                  - The height of the control.
;                  $bShow               - [optional] If true, shows the control (see remarks). Default is True.
;                  $sCallback           - [optional] Callback function to handle control events, function called with two forced parameters:
;                                                                                                              $iPic - Internal ID to picture control.
;                                                                                                              $iEvent - Control event that return the following user constants:
;                                                                                                                                                 $GPP_EVENT_ENTERHOVER - Mouse pointer hovering the control (called constantly).
;                                                                                                                                                 $GPP_EVENT_EXITHOVER - Mouse pointer is not hovering the control anymore (called once).
;                                                                                                                                                 $GPP_EVENT_LBUTTONDOWN - Mouse left button down.
;                                                                                                                                                 $GPP_EVENT_LBUTTONUP - Mouse left button up
; Return values .: Success: Handle (internal ID actualy) to created picture control.
;                  Failure: 0 and set @error to 1 if unable to create picture control.
; Author ........: G.Sandler
; Remarks .......: * $bShow parameter can be used to prevent displaying the control before the main gui is shown, use _GDIPlusPic_Show to display it later.
;                  * For URL that used as $sFile only http(s) protocol is supported.
; Related .......: _GDIPlusPic_SetInfo, _GDIPlusPic_GetInfo, _GDIPlusPic_Show
; Example .......: Yes
; ===============================================================================================================================
Func _GDIPlusPic_Create($hWnd, $sFile, $iL, $iT, $iW, $iH, $bShow = True, $sCallback = '')
	Local $iGPP_ID = -1
	Local $oDict = ObjCreate('Scripting.Dictionary')
	Local $hPic = GUICreate('', $iW, $iH, $iL, $iT, BitOR($WS_POPUP, $WS_DISABLED), BitOR($WS_EX_LAYERED, $WS_EX_MDICHILD), $hWnd)
	
	If Not IsObj($oDict) Or Not IsHWnd($hPic) Then
		Return SetError(1, 0, 0)
	EndIf
	
	For $i = 1 To $aGPP_Data[0][0]
		If $aGPP_Data[$i][$iGPP_Pic] = '' Then
			$iGPP_ID = $i
			ExitLoop
		EndIf
	Next
	
	If $iGPP_ID = -1 Then
		$aGPP_Data[0][0] += 1
		ReDim $aGPP_Data[$aGPP_Data[0][0] + 1][$iGPP_Total]
		$iGPP_ID = $aGPP_Data[0][0]
	EndIf
	
	$aGPP_Data[$iGPP_ID][$iGPP_Parent] = $hWnd
	$aGPP_Data[$iGPP_ID][$iGPP_Pic] = $hPic
	$aGPP_Data[$iGPP_ID][$iGPP_Left] = $iL
	$aGPP_Data[$iGPP_ID][$iGPP_Top] = $iT
	$aGPP_Data[$iGPP_ID][$iGPP_Width] = $iW
	$aGPP_Data[$iGPP_ID][$iGPP_Height] = $iH
	$aGPP_Data[$iGPP_ID][$iGPP_Callback] = $sCallback
	$aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag] = $oDict
	
	Local $aInfo[1][2] = [[$GPP_INFO_FILE, $sFile]]
	
	_GDIPlusPic_SetInfo($iGPP_ID, $aInfo)
	_GDIPlusPic_Show($iGPP_ID, $bShow)
	
	Return $iGPP_ID
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GDIPlusPic_Show
; Description ...: Show or hide the picture control.
; Syntax ........: _GDIPlusPic_Show($iGPP_ID[, $bShow = True])
; Parameters ....: $iGPP_ID             - ID as returned from _GDIPlusPic_Create.
;                  $bShow               - [optional] True - shows the control, False - hides the control. Default is True.
; Return values .: None
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: _GDIPlusPic_Create, _GDIPlusPic_SetInfo
; Example .......: Yes
; ===============================================================================================================================
Func _GDIPlusPic_Show($iGPP_ID, $bShow = True)
	GUISetState(($bShow ? @SW_SHOWNOACTIVATE : @SW_HIDE), $aGPP_Data[$iGPP_ID][$iGPP_Pic])
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GDIPlusPic_Delete
; Description ...: Deletes picture control.
; Syntax ........: _GDIPlusPic_Delete($iGPP_ID)
; Parameters ....: $iGPP_ID             - ID as returned from _GDIPlusPic_Create.
; Return values .: Success: 1
;                  Failure: 0 ($iGPP_ID is not valid ID)
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: _GDIPlusPic_Create
; Example .......: Yes
; ===============================================================================================================================
Func _GDIPlusPic_Delete($iGPP_ID)
	If $iGPP_ID < 1 Or $iGPP_ID > $aGPP_Data[0][0] Or $aGPP_Data[$iGPP_ID][$iGPP_Pic] = '' Then
		Return 0
	EndIf
	
	GUIDelete($aGPP_Data[$iGPP_ID][$iGPP_Pic])
	
	If $aGPP_Data[$iGPP_ID][$iGPP_Image] Then
		_GDIPlus_ImageDispose($aGPP_Data[$iGPP_ID][$iGPP_Image])
	EndIf
	
	$aGPP_Data[$iGPP_ID][$iGPP_File] = ''
	$aGPP_Data[$iGPP_ID][$iGPP_Parent] = ''
	$aGPP_Data[$iGPP_ID][$iGPP_Pic] = ''
	$aGPP_Data[$iGPP_ID][$iGPP_Image] = ''
	$aGPP_Data[$iGPP_ID][$iGPP_Left] = ''
	$aGPP_Data[$iGPP_ID][$iGPP_Top] = ''
	$aGPP_Data[$iGPP_ID][$iGPP_Width] = ''
	$aGPP_Data[$iGPP_ID][$iGPP_Height] = ''
	$aGPP_Data[$iGPP_ID][$iGPP_Callback] = ''
	$aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag] = ''
	
	For $i = 1 To $aGPP_Data[0][0]
		If $aGPP_Data[$i][$iGPP_Pic] <> '' Then
			Return 1
		EndIf
	Next
	
	$aGPP_Data = 0
	Dim $aGPP_Data[1][$iGPP_Total]
	
	Return 1
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GDIPlusPic_SetInfo
; Description ...: Sets info for picture control.
; Syntax ........: _GDIPlusPic_SetInfo($iGPP_ID, $aGPP_INFO)
; Parameters ....: $iGPP_ID             - ID as returned from _GDIPlusPic_Create.
;                  $aGPP_INFO           - 2D array where the following data can be used:
;                                                                                         [0] element = [$GPP_INFO_FILE, 'filename'] - Sets new file name (or url) and it's image for picture control.
;                                                                                         [1] element = [$GPP_INFO_LEFT, N] - Sets new left position for picture control.
;                                                                                         [N] element = [$GPP_INFO_TOP, N] - Sets new top position for picture control.
;                                                                                         [N] element = [$GPP_INFO_WIDTH, N] - Sets new width for picture control.
;                                                                                         [N] element = [$GPP_INFO_HEIGHT, N] - Sets new height for picture control.
;                                                                                         [N] element = [$GPP_INFO_CALLBACK, 'Callback_Func'] - Sets new event callback function for picture control.
; Return values .: Success: 1
;                  Failure: 0 and sets @error to:
;                                                 1 - $iGPP_ID is not valid ID.
;                                                 2 - $aGPP_INFO is a wrong array (not 2D).
;                                                 3 - Unable to load image (wrong $GPP_INFO_FILE path/url).
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: _GDIPlusPic_Create, _GDIPlusPic_GetInfo
; Example .......: Yes
; ===============================================================================================================================
Func _GDIPlusPic_SetInfo($iGPP_ID, $aGPP_INFO)
	If $iGPP_ID < 1 Or $iGPP_ID > $aGPP_Data[0][0] Or $aGPP_Data[$iGPP_ID][$iGPP_Pic] = '' Then
		Return SetError(1, 0, 0)
	EndIf
	
	If UBound($aGPP_INFO, 0) <> 2 Then
		Return SetError(2, 0, 0)
	EndIf
	
	Local $hImage
	Local $tPoint = DllStructCreate('int X;int Y')
	
	Local $sFound_FILE = __GPP_FindInfo($aGPP_INFO, $GPP_INFO_FILE)
	Local $sFound_LEFT = __GPP_FindInfo($aGPP_INFO, $GPP_INFO_LEFT)
	Local $sFound_TOP = __GPP_FindInfo($aGPP_INFO, $GPP_INFO_TOP)
	Local $sFound_WIDTH = __GPP_FindInfo($aGPP_INFO, $GPP_INFO_WIDTH)
	Local $sFound_HEIGHT = __GPP_FindInfo($aGPP_INFO, $GPP_INFO_HEIGHT)
	Local $sFound_CALLBACK = __GPP_FindInfo($aGPP_INFO, $GPP_INFO_CALLBACK)
	
	If $sFound_FILE Then
		If $aGPP_Data[$iGPP_ID][$iGPP_Image] Then
			_GDIPlus_ImageDispose($aGPP_Data[$iGPP_ID][$iGPP_Image])
			$aGPP_Data[$iGPP_ID][$iGPP_Image] = ''
		EndIf
		
		If StringRegExp($sFound_FILE, '^https?://') Then
			$hImage = _GDIPlus_BitmapCreateFromMemory(InetRead($sFound_FILE, BitOR(1, 2)))
		Else
			$hImage = _GDIPlus_ImageLoadFromFile($sFound_FILE)
		EndIf
		
		If @error Then
			Return SetError(3, 0, 0)
		EndIf
		
		$aGPP_Data[$iGPP_ID][$iGPP_Image] = __GPP_ImageResize($hImage, ($sFound_WIDTH ? $sFound_WIDTH : $aGPP_Data[$iGPP_ID][$iGPP_Width]), ($sFound_HEIGHT ? $sFound_HEIGHT : $aGPP_Data[$iGPP_ID][$iGPP_Height]))
		_WinAPI_UpdateLayeredWindowEx($aGPP_Data[$iGPP_ID][$iGPP_Pic], -1, -1, _GDIPlus_BitmapCreateHBITMAPFromBitmap($aGPP_Data[$iGPP_ID][$iGPP_Image]), 255, True)
	EndIf
	
	If $sFound_LEFT Or $sFound_TOP Then
		$tPoint.X = $sFound_LEFT
		$tPoint.Y = $sFound_TOP
		
		_WinAPI_ClientToScreen($aGPP_Data[$iGPP_ID][$iGPP_Parent], $tPoint)
		WinMove($aGPP_Data[$iGPP_ID][$iGPP_Pic], '', $tPoint.X, $tPoint.Y, Default, Default)
	EndIf
	
	If $sFound_WIDTH Or $sFound_HEIGHT Then
		If $sFound_FILE Then
			$aGPP_Data[$iGPP_ID][$iGPP_Image] = __GPP_ImageResize($aGPP_Data[$iGPP_ID][$iGPP_Image], $sFound_WIDTH, $sFound_HEIGHT)
			_WinAPI_UpdateLayeredWindowEx($aGPP_Data[$iGPP_ID][$iGPP_Pic], -1, -1, _GDIPlus_BitmapCreateHBITMAPFromBitmap($aGPP_Data[$iGPP_ID][$iGPP_Image]), 255, True)
		EndIf
		
		WinMove($aGPP_Data[$iGPP_ID][$iGPP_Pic], '', Default, Default, $sFound_WIDTH, $sFound_HEIGHT)
	EndIf
	
	$aGPP_Data[$iGPP_ID][$iGPP_File] = ($sFound_FILE ? $sFound_FILE : $aGPP_Data[$iGPP_ID][$iGPP_File])
	
	$aGPP_Data[$iGPP_ID][$iGPP_Left] = ($sFound_LEFT ? $sFound_LEFT : $aGPP_Data[$iGPP_ID][$iGPP_Left])
	$aGPP_Data[$iGPP_ID][$iGPP_Top] = ($sFound_TOP ? $sFound_TOP : $aGPP_Data[$iGPP_ID][$iGPP_Top])
	$aGPP_Data[$iGPP_ID][$iGPP_Width] = ($sFound_WIDTH ? $sFound_WIDTH : $aGPP_Data[$iGPP_ID][$iGPP_Width])
	$aGPP_Data[$iGPP_ID][$iGPP_Height] = ($sFound_HEIGHT ? $sFound_HEIGHT : $aGPP_Data[$iGPP_ID][$iGPP_Height])
	
	$aGPP_Data[$iGPP_ID][$iGPP_Callback] = ($sFound_CALLBACK ? $sFound_CALLBACK : $aGPP_Data[$iGPP_ID][$iGPP_Callback])
	;$aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag] = ($sFound_CALLBACK ? '' : $aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag])
	
	Return 1
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GDIPlusPic_GetInfo
; Description ...: Gets picture control info.
; Syntax ........: _GDIPlusPic_GetInfo($iGPP_ID)
; Parameters ....: $iGPP_ID             - ID as returned from _GDIPlusPic_Create.
; Return values .: 1D array with the following data (8 elements):
;                                                     $aArray[$GPP_INFO_GUI]      - Main GUI handle as passed to _GDIPlusPic_Create function.
;                                                     $aArray[$GPP_INFO_PIC]      - Picture control handle.
;                                                     $aArray[$GPP_INFO_FILE]     - Picture control file name as passed to _GDIPlusPic_Create or _GDIPlusPic_SetInfo function.
;                                                     $aArray[$GPP_INFO_LEFT]     - Left position of the picture control.
;                                                     $aArray[$GPP_INFO_TOP]      - Top position of the picture control.
;                                                     $aArray[$GPP_INFO_WIDTH]    - Width of the picture control.
;                                                     $aArray[$GPP_INFO_HEIGHT]   - Height of the picture control.
;                                                     $aArray[$GPP_INFO_CALLBACK] - Callback function of the picture control as passed to _GDIPlusPic_Create function.
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: _GDIPlusPic_Create
; Example .......: Yes
; ===============================================================================================================================
Func _GDIPlusPic_GetInfo($iGPP_ID)
	If $iGPP_ID < 1 Or $iGPP_ID > $aGPP_Data[0][0] Or $aGPP_Data[$iGPP_ID][$iGPP_Pic] = '' Then
		Return 0
	EndIf
	
	Local $aRet[$GPP_INFO_TOTAL]
	
	$aRet[$GPP_INFO_GUI] = $aGPP_Data[$iGPP_ID][$iGPP_Parent]
	$aRet[$GPP_INFO_PIC] = $aGPP_Data[$iGPP_ID][$iGPP_Pic]
	$aRet[$GPP_INFO_FILE] = $aGPP_Data[$iGPP_ID][$iGPP_File]
	$aRet[$GPP_INFO_LEFT] = $aGPP_Data[$iGPP_ID][$iGPP_Left]
	$aRet[$GPP_INFO_TOP] = $aGPP_Data[$iGPP_ID][$iGPP_Top]
	$aRet[$GPP_INFO_WIDTH] = $aGPP_Data[$iGPP_ID][$iGPP_Width]
	$aRet[$GPP_INFO_HEIGHT] = $aGPP_Data[$iGPP_ID][$iGPP_Height]
	$aRet[$GPP_INFO_CALLBACK] = $aGPP_Data[$iGPP_ID][$iGPP_Callback]
	
	Return $aRet
EndFunc

#EndRegion Public Functions

#Region Internal Functions

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Description ...: Callback wrapper to handle picture control events.
; Syntax ........: __GPP_Callback()
; Author ........: G.Sandler
; ===============================================================================================================================
Func __GPP_Callback()
	For $iGPP_ID = 1 To $aGPP_Data[0][0]
		If $aGPP_Data[$iGPP_ID][$iGPP_Callback] Then
			If __GPP_WinIsUnderMouse($aGPP_Data[$iGPP_ID][$iGPP_Pic]) Then
				If BitAND(_WinAPI_GetAsyncKeyState($VK_LBUTTON), 0x8000) And Not $aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag].Exists('LBttnDwn') And Not $aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag].Exists('LBttnDwnOut') Then
					$aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag]('LBttnDwn') = 1
					Call($aGPP_Data[$iGPP_ID][$iGPP_Callback], $iGPP_ID, $GPP_EVENT_LBUTTONDOWN)
				ElseIf Not BitAND(_WinAPI_GetAsyncKeyState($VK_LBUTTON), 0x8000) Then
					If $aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag].Exists('LBttnDwnOut') Then
						$aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag].Remove('LBttnDwnOut')
					EndIf
					
					If $aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag].Exists('LBttnDwn') Then
						$aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag].Remove('LBttnDwn')
						Call($aGPP_Data[$iGPP_ID][$iGPP_Callback], $iGPP_ID, $GPP_EVENT_LBUTTONUP)
					EndIf
				EndIf
				
				If Not $aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag].Exists('EntrHvr') Then
					$aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag]('EntrHvr') = 1
					Call($aGPP_Data[$iGPP_ID][$iGPP_Callback], $iGPP_ID, $GPP_EVENT_ENTERHOVER)
				EndIf
			ElseIf $aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag].Exists('EntrHvr') Then
				$aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag].Remove('EntrHvr')
				
				If $aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag].Exists('LBttnDwn') Then
					$aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag].Remove('LBttnDwn')
				EndIf
				
				Call($aGPP_Data[$iGPP_ID][$iGPP_Callback], $iGPP_ID, $GPP_EVENT_EXITHOVER)
			ElseIf BitAND(_WinAPI_GetAsyncKeyState($VK_LBUTTON), 0x8000) Then
				$aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag]('LBttnDwnOut') = 1
			Else
				If $aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag].Exists('LBttnDwnOut') Then
					$aGPP_Data[$iGPP_ID][$iGPP_Callback_Flag].Remove('LBttnDwnOut')
				EndIf
			EndIf
		EndIf
	Next
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Description ...: Finds specified info in SetInfo array.
; Syntax ........: __GPP_FindInfo($aInfo, $iInfo)
; Author ........: G.Sandler
; ===============================================================================================================================
Func __GPP_FindInfo($aInfo, $iInfo)
	For $i = 0 To UBound($aInfo) - 1
		If $aInfo[$i][0] == $iInfo Then
			Return $aInfo[$i][1]
		EndIf
	Next
	
	Return 0
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Description ...: Checks if a specified window handle is under the mouse pointer.
; Syntax ........: __GPP_WinUnderMouse()
; Author ........: G.Sandler
; ===============================================================================================================================
Func __GPP_WinIsUnderMouse($hWin)
	Local $tPoint = _WinAPI_GetMousePos()
	Return (_WinAPI_WindowFromPoint($tPoint) = $hWin)
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Description ...: Resize image wrapper.
; Syntax ........: __GPP_ImageResize($hImage, $iW, $iH)
; Author ........: G.Sandler
; ===============================================================================================================================
Func __GPP_ImageResize($hImage, $iW, $iH)
	If Not IsPtr($hImage) Then
		Return $hImage
	EndIf
	
	Local $hImage_Resized = _GDIPlus_ImageResize($hImage, $iW, $iH)
	_GDIPlus_ImageDispose($hImage)
	
	Return $hImage_Resized
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Description ...: Cleanup function, called on script exit.
; Syntax ........: __GPP_OnExit()
; Author ........: G.Sandler
; ===============================================================================================================================
Func __GPP_OnExit()
	For $iGPP_ID = 1 To $aGPP_Data[0][0]
		_GDIPlusPic_Delete($iGPP_ID)
	Next
	
	_GDIPlus_Shutdown()
EndFunc

#EndRegion Internal Functions
