#Region Header

#CS
	Name:				Registry.au3
	Description:		Registry extended functions.
	Author:				Copyright  2009-2012 wraithdu, MrCreatoR (G.Sandler), www.creator-lab.ucoz.ru, www.autoit-script.ru. All rights reserved.
	AutoIt version:		3.3.6.1
	UDF version:		1.1
	
	History:
	
#CE

;Includes
#include-once

#EndRegion Header

#Region Global Variables

Global Const $g_REGTYPE_REG_NONE = 'REG_NONE' ;0
Global Const $g_REGTYPE_REG_SZ = 'REG_SZ' ;1
Global Const $g_REGTYPE_REG_EXPAND_SZ = 'REG_EXPAND_SZ' ;2
Global Const $g_REGTYPE_REG_BINARY = 'REG_BINARY' ;3
Global Const $g_REGTYPE_REG_DWORD = 'REG_DWORD' ;4
Global Const $g_REGTYPE_REG_DWORD_BIG_ENDIAN = 'REG_DWORD_BIG_ENDIAN' ;5
Global Const $g_REGTYPE_REG_LINK = 'REG_LINK' ;6
Global Const $g_REGTYPE_REG_MULTI_SZ = 'REG_MULTI_SZ' ;7
Global Const $g_REGTYPE_REG_RESOURCE_LIST = 'REG_RESOURCE_LIST' ;8
Global Const $g_REGTYPE_REG_FULL_RESOURCE_DESCRIPTOR = 'REG_FULL_RESOURCE_DESCRIPTOR' ;9
Global Const $g_REGTYPE_REG_RESOURCE_REQUIREMENTS_LIST = 'REG_RESOURCE_REQUIREMENTS_LIST' ;10
Global Const $g_REGTYPE_REG_QWORD = 'REG_QWORD' ;11 ; Vista

#EndRegion Global Variables

#Region Public Functions

Func _RegCopyKey($s_Key, $d_Key, $bDelete = False)
	Local $i, $sVal, $sData, $sType, $sKey

	RegWrite($d_Key) ; write dest Key, in case Key empty
	If @error <> 0 Then Return @error ; some error

	; Value loop
	$i = 0

	While 1
		$i += 1
		
		$sVal = RegEnumVal($s_Key, $i)
		If @error <> 0 Then ExitLoop ; no more Values
		
		$sData = RegRead($s_Key, $sVal)
		If @error <> 0 Then ContinueLoop ; some error reading Value, skip it
		
		$sType = _RegGetExtendedType(@extended)
		RegWrite($d_Key, $sVal, $sType, $sData) ; write new Value
	WEnd
	
	; Key loop
	$i = 0
	
	While 1
		$i += 1
		$Key = RegEnumKey($s_Key, $i)
		If @error <> 0 Then ExitLoop ; no more Keys
		
		_RegCopyKey($s_Key & "\" & $sKey, $d_Key & "\" & $sKey) ; recurse
	WEnd
	
	; move Key
	If $bDelete Then RegDelete($s_Key)
EndFunc   ;==>_RegCopyKey

Func _RegCopyValue($s_Key, $s_Val, $d_Key, $d_Val, $bDelete = False)
	Local $sData, $sType
	
	$sData = RegRead($s_Key, $s_Val)
	If @error <> 0 Then Return SetError(1, 0, 0) ; some error reading Value, skip it
	
	$sType = _RegGetExtendedType(@extended)
	
	RegWrite($d_Key, $d_Val, $sType, $sData)
	If $bDelete Then RegDelete($s_Key, $s_Val)
EndFunc   ;==>_RegCopyValue

;Delete registry key by it's value
Func _RegDeleteEx($s_Key, $s_Val, $i_UseRegExp = 0)
	Local $sCurrent_ValName, $sRegRead, $iCount = 1

	While 1
		$sCurrent_ValName = RegEnumVal($s_Key, $iCount)
		If @error <> 0 Then ExitLoop

		$sRegRead = String(RegRead($s_Key, $sCurrent_ValName))

		If ($i_UseRegExp > 0 And StringRegExp($sRegRead, $s_Val)) Or ($i_UseRegExp = 0 And $sRegRead = $s_Val) Then
			Return RegDelete($s_Key, $sCurrent_ValName)
		EndIf

		$iCount += 1
	WEnd

	Return @error
EndFunc   ;==>_RegDeleteEx

;===============================================================================
; Function Name:    _RegEnumKeys2Array()
; Version:          v0.3
; Description:      Create an Array of all registry keys and sub-keys at the pointed location
; Parameter(s):     $sRegKey        - Start Key to enumerate including any sub-keys
; Return Value(s):  On Success  - Returns an array with RegKeys Name, Value & Type
;                   On Failure  - No error checking yet...
; Author(s):        KaFu (Mod. by MrCreatoR).
; URL:              http://www.autoitscript.com/forum/index.php?showtopic=85661
; Last modified:    2008-Dec-22
;
;===============================================================================
Func _RegEnumKeys2Array($sRegKey)
	Local $iArrBuff = 1000
	Local $aResult[$iArrBuff][4], $aResult_sub, $i, $y
	Local $sRegKey_Name, $sRegKey_Value, $sRegKey_Type
	
	;If StringRight($sRegKey, 1) <> "\" Then
	;$sRegKey &= "\"
	;EndIf

	$aResult[0][0] = 0
	$i = 1

	While 1
		$sRegKey_Name = RegEnumVal($sRegKey, $i)
		If @error Then ExitLoop

		$aResult[0][0] += 1

		If $aResult[0][0] >= $iArrBuff Then
			$iArrBuff *= 2
			ReDim $aResult[$iArrBuff][4]
		EndIf

		$aResult[$aResult[0][0]][0] = $sRegKey
		$aResult[$aResult[0][0]][1] = $sRegKey_Name

		$sRegKey_Value = RegRead($sRegKey, $sRegKey_Name)
		$sRegKey_Type = _RegGetExtendedType(@extended)

		$aResult[$aResult[0][0]][2] = $sRegKey_Type
		$aResult[$aResult[0][0]][3] = $sRegKey_Value

		$i += 1
	WEnd

	$i = 1

	While 1
		$sRegKey_Name = RegEnumKey($sRegKey, $i)

		If @error Then
			ExitLoop
		EndIf

		$aResult_sub = _RegEnumKeys2Array($sRegKey & $sRegKey_Name)

		If UBound($aResult_sub) > 1 Then
			For $y = 1 To $aResult_sub[0][0]
				$aResult[0][0] += 1

				If $aResult[0][0] >= $iArrBuff Then
					$iArrBuff *= 2
					ReDim $aResult[$iArrBuff][4]
				EndIf

				$aResult[$aResult[0][0]][0] = $aResult_sub[$y][0]
				$aResult[$aResult[0][0]][1] = $aResult_sub[$y][1]
				$aResult[$aResult[0][0]][2] = $aResult_sub[$y][2]
				$aResult[$aResult[0][0]][3] = $aResult_sub[$y][3]
			Next
		EndIf

		$i += 1
	WEnd
	
	ReDim $aResult[$aResult[0][0]+1][4]
	Return $aResult
EndFunc   ;==>_RegEnumKeys2Array

Func _RegExport($sRegFile, $sKeyName, $sValueName = "")
	If $sValueName <> "" Then
		Local $sRegRead = RegRead($sKeyName, $sValueName)
		Local $hOpen = FileOpen($sRegFile, 2)

		FileWrite($hOpen, 'Windows Registry Editor Version 5.00' & @CRLF & @CRLF & _
				"[" & $sKeyName & "]" & @CRLF & _
				'"' & $sValueName & '"="' & StringRegExpReplace($sRegRead, '([\\"])', '\\\1') & '"' & @CRLF)

		FileClose($hOpen)
	Else
		Run(@ComSpec & ' /c Reg Export "' & $sKeyName & '" "' & $sRegFile & '"', '', @SW_HIDE)
	EndIf
EndFunc   ;==>_RegExport

Func _RegGetExtendedType($iExtended)
	Local $aRegTypeArr[8] = [7, _
		$g_REGTYPE_REG_SZ, $g_REGTYPE_REG_EXPAND_SZ, $g_REGTYPE_REG_BINARY, $g_REGTYPE_REG_DWORD, _
		$g_REGTYPE_REG_DWORD_BIG_ENDIAN, $g_REGTYPE_REG_LINK, $g_REGTYPE_REG_MULTI_SZ]

	For $i = 1 To 7
		If $iExtended = $i Then
			Return $aRegTypeArr[$i]
		EndIf
	Next

	Return $g_REGTYPE_REG_SZ
EndFunc   ;==>_RegGetExtendedType

Func _RegImport($sRegFile, $iWait = 0)
	If $iWait Then
		RunWait('RegEdit.exe /s "' & $sRegFile & '"')
	Else
		Run('RegEdit.exe /s "' & $sRegFile & '"')
	EndIf
EndFunc   ;==>_RegImport

Func _RegKeyEmpty($s_Key)
	Local $i_Error1 = 0, $i_Error2 = 0

	; check for Keys
	RegEnumKey($s_Key, 1)
	If @error <> 0 Then $i_Error1 = 1
	; check for Values
	RegEnumVal($s_Key, 1)
	If @error <> 0 Then $i_Error2 = 1

	; set return
	If $i_Error1 And $i_Error2 Then Return 1 ; empty

	Return 0
EndFunc   ;==>_RegKeyEmpty

Func _RegKeyExists($s_Key)
	RegRead($s_Key, "")
	If @error <= 0 Then Return 1 ; Key exists
	Return 0
EndFunc   ;==>_RegKeyExists

Func _RegMoveKey($s_Key, $d_Key)
	_RegCopyKey($s_Key, $d_Key, True)
EndFunc   ;==>_RegMoveKey

Func _RegMoveValue($s_Key, $s_Val, $d_Key, $d_Val)
	_RegCopyValue($s_Key, $s_Val, $d_Key, $d_Val, True)
EndFunc   ;==>_RegMoveValue

;Reads registry key, and return $sDefault if the key is empty ("")
Func _RegReadEx($sKeyName, $sValueName, $sDefault = "")
	Local $sRet_Value = RegRead($sKeyName, $sValueName)
	If $sRet_Value = "" Then $sRet_Value = $sDefault
	Return SetError(@error, @extended, $sRet_Value)
EndFunc   ;==>_RegReadEx

Func _RegReadKeyValueToArray($s_Key, $iKey_Value_Mode = 0)
	Local $aKeysListArr[1], $aSubKeysListArr, $iInstance = 0, $sEnum_KeyVal, $sCurrentKeyPath

	If Not _RegKeyExists($s_Key) Then Return SetError(1, 0, $aKeysListArr)

	While 1
		$iInstance += 1

		If $iKey_Value_Mode = 0 Then
			$sEnum_KeyVal = RegEnumKey($s_Key, $iInstance)
		Else
			$sEnum_KeyVal = RegEnumVal($s_Key, $iInstance)
		EndIf

		If @error <> 0 Then ExitLoop

		$sCurrentKeyPath = $s_Key & "\" & $sEnum_KeyVal

		$aKeysListArr[0] += 1
		ReDim $aKeysListArr[$aKeysListArr[0] + 1]
		$aKeysListArr[$aKeysListArr[0]] = $sCurrentKeyPath

		$aSubKeysListArr = _RegReadKeyValueToArray($sCurrentKeyPath)

		For $j = 1 To $aSubKeysListArr[0]
			$aKeysListArr[0] += 1

			ReDim $aKeysListArr[$aKeysListArr[0] + 1]
			$aKeysListArr[$aKeysListArr[0]] = $aSubKeysListArr[$j]
		Next
	WEnd

	If $aKeysListArr[0] < 1 Then SetError(1)

	Return $aKeysListArr
EndFunc   ;==>_RegReadKeyValueToArray

Func _RegRunAdd($sAppName, $sAppPath, $sParams = "", $bUseLocalMachine = False, $sRunKeyName = "Run")
	Local $sRegKey = "HKEY_CURRENT_USER\Software\Microsoft\Windows\CurrentVersion\" & $sRunKeyName
	If $bUseLocalMachine Then $sRegKey = "HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\" & $sRunKeyName

	If $sParams <> "" Then
		$sAppPath = '"' & $sAppPath & '" "' & $sParams & '"'
	EndIf

	Return RegWrite($sRegKey, $sAppName, $g_REGTYPE_REG_SZ, $sAppPath)
EndFunc   ;==>_RegRunAdd

Func _RegRunDelete($sAppName, $bUseLocalMachine = False, $sRunKeyName = "Run")
	Local $sRegKey = "HKEY_CURRENT_USER\Software\Microsoft\Windows\CurrentVersion\" & $sRunKeyName
	If $bUseLocalMachine Then $sRegKey = "HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\" & $sRunKeyName

	Return RegDelete($sRegKey, $sAppName)
EndFunc   ;==>_RegRunDelete

Func _RegSetEnvironment($sEnv_Name, $sEnv_Value, $iReplace = 0, $iKeyVal = 0, $iEnv_Update = 0)
	If $sEnv_Name = "" Then Return SetError(1, 0, 0)
	If $sEnv_Value = "" Then Return SetError(2, 0, 0)
	If $iReplace <> 0 And $iReplace <> 1 Then Return SetError(3, 0, 0)

	Local $iRet = 0, $iError = 0
	Local $sSystemRegKey = "HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Control\Session Manager\Environment"
	Local $sUserRegKey = "HKEY_CURRENT_USER\Environment"

	Local $sRegKey = $sSystemRegKey
	If $iKeyVal = 1 Then $sRegKey = $sUserRegKey

	If $iReplace = 1 Then
		$iRet = RegWrite($sRegKey, $sEnv_Name, $g_REGTYPE_REG_SZ, $sEnv_Value)
		$iError = @error
	ElseIf RegRead($sRegKey, $sEnv_Name) = "" Then
		$iRet = RegWrite($sRegKey, $sEnv_Name, $g_REGTYPE_REG_SZ, $sEnv_Value)
		$iError = @error
	EndIf

	If $iEnv_Update Then EnvUpdate()

	Return SetError($iError, 0, $iRet)
EndFunc   ;==>_RegSetEnvironment

;==============================================================================================
;
; Description:		_RegSetFileExt($sExt, $sCommand, $sVerb[, $iDefault[, $sIcon = ""[, $sDescription = ""]]])
;					Registers a file type in Explorer
; Parameter(s):		$sExt - 	File Extension without period eg. "zip"
;					$sCommand - 	Program path with arguments eg. '
;							(%1 is 1st argument, %2 is 2nd, etc.)
;					$sVerb - Name of action to perform on file
;							eg. "Open with ProgramName" or "Extract Files"
;					$iDefault - 	Action is the default action for this filetype
;							(1 for true 0 for false)
;							If the file is not already associated, this will be the default.
;					$sIcon - Default icon for filetype including resource # if needed
;							eg. "C:\test\testprog.exe,0" or "C:\test\filetype.ico"
;					$sDescription - File Description eg. "Zip File" or "ProgramName Document"
;
;===============================================================================================
Func _RegSetFileExt($sExt, $sCommand, $sVerb, $iDefault = 0, $sIcon = "", $sDescription = "")
	Local $sExtRead = RegRead("HKCR\." & $sExt, "")

	If @error Then
		RegWrite("HKCR\." & $sExt, "", $g_REGTYPE_REG_SZ, $sExt & "file")
		$sExtRead = $sExt & "file"
	EndIf

	Local $sCurrentDescription = RegRead("HKCR\" & $sExtRead, "")

	If @error Then
		If $sDescription <> "" Then RegWrite("HKCR\" & $sExtRead, "", $g_REGTYPE_REG_SZ, $sDescription)
	Else
		If $sDescription <> "" And $sCurrentDescription <> $sDescription Then
			RegWrite("HKCR\" & $sExtRead, "", $g_REGTYPE_REG_SZ, $sDescription)
			RegWrite("HKCR\" & $sExtRead, "olddesc", $g_REGTYPE_REG_SZ, $sCurrentDescription)
		EndIf

		If $sCurrentDescription = "" And $sDescription <> "" Then RegWrite("HKCR\" & $sExtRead, "", $g_REGTYPE_REG_SZ, $sDescription)
	EndIf

	Local $sCurrentVerb = RegRead("HKCR\" & $sExtRead & "\shell", "")
	Local $iError = @error

	If $iDefault = 1 Then
		RegWrite("HKCR\" & $sExtRead & "\shell\Open\Command", "", $g_REGTYPE_REG_EXPAND_SZ, $sCommand)

		If $iError Then
			RegWrite("HKCR\" & $sExtRead & "\shell", "", $g_REGTYPE_REG_SZ, $sVerb)
		Else
			RegWrite("HKCR\" & $sExtRead & "\shell", "", $g_REGTYPE_REG_SZ, $sVerb)
			RegWrite("HKCR\" & $sExtRead & "\shell", "oldverb", $g_REGTYPE_REG_SZ, $sCurrentVerb)
		EndIf
	EndIf

	Local $sCurrentCommand = RegRead("HKCR\" & $sExtRead & "\shell\" & $sVerb & "\command", "")

	If Not @error Then
		RegRead("HKCR\" & $sExtRead & "\shell\" & $sVerb & "\command", "oldcmd")
		If @error Then RegWrite("HKCR\" & $sExtRead & "\shell\" & $sVerb & "\command", "oldcmd", $g_REGTYPE_REG_SZ, $sCurrentCommand)
	EndIf

	RegWrite("HKCR\" & $sExtRead & "\shell\" & $sVerb & "\command", "", $g_REGTYPE_REG_SZ, $sCommand)

	If $sIcon <> "" Then
		Local $sCurrentIcon = RegRead("HKCR\" & $sExtRead & "\DefaultIcon", "")

		If @error Then
			RegWrite("HKCR\" & $sExtRead & "\DefaultIcon", "", $g_REGTYPE_REG_SZ, $sIcon)
		Else
			RegWrite("HKCR\" & $sExtRead & "\DefaultIcon", "", $g_REGTYPE_REG_SZ, $sIcon)
			RegWrite("HKCR\" & $sExtRead & "\DefaultIcon", "oldicon", $g_REGTYPE_REG_SZ, $sCurrentIcon)
		EndIf
	EndIf
EndFunc   ;==>_RegSetFileExt

#CS
-  :
$s_Key -   ( )     .
$s_Search -  .
$i_Mode -  : 0 ( ) ->    (  ), 1 ->    , 2 ->    .
$i_Partial -   : 0 ->    , 1 ( ) ->  .
$i_Case -   : 0 ( ) ->     , 1 ->     .
$i_RetFirstPos -   : 0 ( ) ->      , 1 ->     .

-   , ...
[0][0] -    (   ).
[N][0] -     .
[N][1] -       .
#CE
Func _RegSearch($s_Key, $s_Search, $i_Mode = 0, $i_Partial = 1, $i_Case = 0, $i_RetFirstPos = 0)
	Local $i = 1, $sKey, $sSubKey, $sVal, $aRet[1][1]
	
	While 1
		Switch $i_Mode
			Case 0 ;Search in Keys
				$sKey = RegEnumKey($s_Key, $i)
				If @error <> 0 Then ExitLoop
				
				If ($i_Partial And StringInStr($sKey, $s_Search, $i_Case)) Or (Not $i_Partial And $sKey = $s_Search) Then
					$aRet[0][0] += 1
					ReDim $aRet[$aRet[0][0] + 1][2]
					
					$aRet[$aRet[0][0]][0] = $sKey
					
					If $i_RetFirstPos Then
						Return $aRet
					EndIf
				EndIf
			Case 1, 2 ;Search in SubKeys, Search in SubKeys Value
				$sSubKey = RegEnumVal($s_Key, $i)
				If @error <> 0 Then ExitLoop
				
				$sVal = $sSubKey
				
				If $i_Mode = 2 Then
					$sVal = RegRead($s_Key, $sSubKey)
				EndIf
				
				If ($i_Partial And StringInStr($sVal, $s_Search, $i_Case)) Or (Not $i_Partial And $sVal = $s_Search) Then
					If $i_Mode <> 2 Then $sVal = ''
					
					$aRet[0][0] += 1
					ReDim $aRet[$aRet[0][0] + 1][2]
					
					$aRet[$aRet[0][0]][0] = $sSubKey
					$aRet[$aRet[0][0]][1] = $sVal
					
					If $i_RetFirstPos Then
						Return $aRet
					EndIf
				EndIf
		EndSwitch
		
		$i += 1
	WEnd
	
	If $aRet[0][0] > 0 Then
		Return $aRet
	EndIf
	
	Return SetError(1, 0, 0)
EndFunc   ;==>_RegSearch

;===============================================================================
;
; Description:		_RegUnSetFileExt($sExt, $sVerb)
;					UnRegisters a verb for a file type in Explorer
; Parameter(s):		$sExt - File Extension without period eg. "zip"
;					$sVerb [Optional] - Name of file action to remove, eg. "Open with ProgramName" or "Extract Files"
;                     If $sVerb = "" (default) then this fuunction will remove the file type from registry completely.
;
;===============================================================================
Func _RegUnSetFileExt($sExt, $sVerb = "")
	Local $sExtRead = RegRead("HKCR\." & $sExt, "")

	If Not @error Then
		If $sVerb = "" Then
			RegDelete("HKCR\." & $sExt)
			RegDelete("HKCR\" & $sExtRead)

			Return 1
		EndIf

		Local $sOldIcon = RegRead("HKCR\" & $sExtRead & "\shell", "oldicon")

		If Not @error Then
			RegWrite("HKCR\" & $sExtRead & "\DefaultIcon", "", $g_REGTYPE_REG_SZ, $sOldIcon)
		Else
			RegDelete("HKCR\" & $sExtRead & "\DefaultIcon", "")
		EndIf

		Local $sOldVerb = RegRead("HKCR\" & $sExtRead & "\shell", "oldverb")

		If Not @error Then
			RegWrite("HKCR\" & $sExtRead & "\shell", "", $g_REGTYPE_REG_SZ, $sOldVerb)
		Else
			RegDelete("HKCR\" & $sExtRead & "\shell", "")
		EndIf

		Local $sOldDesc = RegRead("HKCR\" & $sExtRead, "olddesc")

		If Not @error Then
			RegWrite("HKCR\" & $sExtRead, "", $g_REGTYPE_REG_SZ, $sOldDesc)
		Else
			RegDelete("HKCR\" & $sExtRead, "")
		EndIf

		Local $sOldCmd = RegRead("HKCR\" & $sExtRead & "\shell\" & $sVerb & "\command", "oldcmd")

		If Not @error Then
			RegWrite("HKCR\" & $sExtRead & "\shell\" & $sVerb & "\command", "", $g_REGTYPE_REG_SZ, $sOldCmd)
			RegDelete("HKCR\" & $sExtRead & "\shell\" & $sVerb & "\command", "oldcmd")
		Else
			RegDelete("HKCR\" & $sExtRead & "\shell\" & $sVerb)
		EndIf
	EndIf
EndFunc   ;==>_RegUnSetFileExt

Func _RegValueExists($s_Key, $s_Val)
	RegRead($s_Key, $s_Val)
	; @error == -2 is 'type not supported', implying Value exists
	If @error = -1 Or @error >= 1 Then Return 0 ; Value does not exist
	Return 1
EndFunc   ;==>_RegValueExists

Func _RegGetFavorites()
	Local $aKeysListArr = _RegReadKeyValueToArray("HKEY_CURRENT_USER\Software\Microsoft\Windows\CurrentVersion\Applets\Regedit\Favorites", 1)
	Local $aRet[$aKeysListArr[0]+1] = [$aKeysListArr[0]]
	Local $sKeyPath, $sKeyName
	
	For $i = 1 To $aKeysListArr[0]
		$sKeyPath = StringLeft($aKeysListArr[$i], StringInStr($aKeysListArr[$i], "\", 2, -1))
		$sKeyName = StringTrimLeft($aKeysListArr[$i], StringInStr($aKeysListArr[$i], "\", 2, -1))
		
		$aRet[$i] = RegRead($sKeyPath, $sKeyName)
		$aRet[$i] = StringRegExpReplace($aRet[$i], "^.*?\\", "")
	Next
	
	Return $aRet
EndFunc

Func _RegGetMostUsedKeys()
	Local $aKeys[13] = _
		[12, _
			"HKEY_CURRENT_USER\Control Panel\Desktop", _ ;Desktop settings (wallpapers, tooltips, etc.)
			"HKEY_CURRENT_USER\Software\Microsoft\Windows\CurrentVersion\Explorer", _ ;Explorer settings
			"HKEY_CURRENT_USER\Software\Microsoft\Windows\CurrentVersion\Explorer\Advanced", _ ;Explorer advanced settings (extensions, show hidden files, etc.)
			"HKEY_CURRENT_USER\Software\Microsoft\Windows\CurrentVersion\Applets\Regedit\Favorites", _ ;Registry Favorites section
			"HKEY_CURRENT_USER\Software\Microsoft\Windows\CurrentVersion\Explorer\ComDlg32\OpenSaveMRU", _ ;Open/Save MRu List
			"HKEY_CURRENT_USER\Software\Microsoft\Windows\CurrentVersion\Run", _ ;"Run with windows" in CURRENT_USER section
			"HKEY_CURRENT_USER\Software\Microsoft\Internet Explorer\MenuExt", _ ;Context menu items for IE (http://msdn.microsoft.com/en-us/library/aa753589(v=vs.85).aspx)
			"HKEY_CURRENT_USER\Software", _ ;Softwares in CURRENT_USER section
			"HKEY_LOCAL_MACHINE\SOFTWARE", _;Softwares in LOCAL_MACHINE section
			"HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows NT\CurrentVersion\AeDebug", _ ;Configuring Automatic Debugging for Application Crashes, Disable dr.watson
			"HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\Run", _ ;"Run with windows" in LOCAL_MACHINE section
			"HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall" _ ;Uninstal section
		]
	
	Return $aKeys
EndFunc

;Author: Yashied
Func _RegJump($sKey)
	Local $Root, $Text = StringSplit($sKey, '\', 2)
	
	If IsArray($Text) Then
		$Text = $Text[0]
	Else
		$Text = $sKey
	EndIf
	
	Switch $Text
		Case 'HKEY_CLASSES_ROOT', 'HKEY_CURRENT_USER', 'HKEY_LOCAL_MACHINE', 'HKEY_USERS', 'HKEY_CURRENT_CONFIG'
			$Root = $Text
		Case 'HKCR'
			$Root = 'HKEY_CLASSES_ROOT'
		Case 'HKCU'
			$Root = 'HKEY_CURRENT_USER'
		Case 'HKLM'
			$Root = 'HKEY_LOCAL_MACHINE'
		Case 'HKU'
			$Root = 'HKEY_USERS'
		Case 'HKCC'
			$Root = 'HKEY_CURRENT_CONFIG'
		Case Else
			Return 0
	EndSwitch
	
	Local $Class = '[CLASS:RegEdit_RegEdit]', $Delay = Opt('WinWaitDelay', 0)
	Local $Prev, $Result = 1
	
	If WinExists($Class) Then
		WinClose($Class)
		
		If Not WinWaitClose($Class, '', 5) Then
			$Result = 0
		EndIf
	EndIf
	
	If $Result Then
		$Prev = RegRead('HKCU\Software\Microsoft\Windows\CurrentVersion\Applets\Regedit', 'Lastkey')
		
		If @error Then
			$Prev = 0
		EndIf
		
		If Not RegWrite('HKCU\Software\Microsoft\Windows\CurrentVersion\Applets\Regedit', 'Lastkey', 'REG_SZ', StringReplace($sKey, $Text, $Root, 1)) Then
			$Result = 0
		Else
			If Not Run('regedit.exe') Then
				$Result = 0
				
				If IsString($Prev) Then
					RegWrite('HKCU\Software\Microsoft\Windows\CurrentVersion\Applets\Regedit', 'Lastkey', 'REG_SZ', $Prev)
				EndIf
			EndIf
		EndIf
	EndIf
	
	Opt('WinWaitDelay', $Delay)
	
	Return $Result
EndFunc

#EndRegion Public Functions
