#include-once
#include <FileConstants.au3>
#include <StringConstants.au3>

; #INDEX# =======================================================================================================================
; Title .........: Encoding
; AutoIt Version : 3.3.14.5
; Description ...: Collection of functions that used to work with different string encoding.
; Author(s) .....: G.Sandler ((Mr)CreatoR), amel27, LEX1, trancexx, Ward, rafzak, Latoid, LazyCat (Loopback)
; Dll ...........: user32.dll, kernel32.dll
; ===============================================================================================================================

; #CURRENT# =====================================================================================================================
; _Encoding_1252ToUTF8
; _Encoding_866To1251
; _Encoding_ANSIToOEM
; _Encoding_ANSIToUTF8
; _Encoding_Base64Decode
; _Encoding_Base64Encode
; _Encoding_CyrillicTo1251
; _Encoding_GetCyrillicANSIEncoding
; _Encoding_GetFileEncoding
; _Encoding_ISO88591To1251
; _Encoding_ISO8859To1251
; _Encoding_IsUTF8
; _Encoding_JavaUnicodeDecode
; _Encoding_JavaUnicodeEncode
; _Encoding_KOI8To1251
; _Encoding_OEM2ANSI
; _Encoding_QuotedPrintableToANSI
; _Encoding_StringToUTF8
; _Encoding_URIDecode
; _Encoding_URIEncode
; _Encoding_UTF8BOMDecode
; _Encoding_UTF8ToANSI
; _Encoding_UTF8ToUnicode
; ===============================================================================================================================

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_1252ToUTF8
; Description ...: Converts 1252 encoded string to utf8.
; Syntax ........: _Encoding_1252ToUTF8($sString)
; Parameters ....: $sString		- String to encode.
; Return values .: Encoded string.
; Author ........: rafzak
; Modified ......: G.Sandler
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_1252ToUTF8($sString)
	$sString = StringToBinary($sString, 4)
	
	$sString = StringReplace($sString, 'E282AC', 'C280')
	$sString = StringReplace($sString, 'E2809A', 'C282')
	$sString = StringReplace($sString, 'C692', 'C283')
	$sString = StringReplace($sString, 'E2809E', 'C284')
	$sString = StringReplace($sString, 'E280A6', 'C285')
	$sString = StringReplace($sString, 'E280A0', 'C286')
	$sString = StringReplace($sString, 'E280A1', 'C287')
	$sString = StringReplace($sString, 'CB86', 'C288')
	$sString = StringReplace($sString, 'E280B0', 'C289')
	$sString = StringReplace($sString, 'C5A0', 'C28A')
	$sString = StringReplace($sString, 'E280B9', 'C28B')
	$sString = StringReplace($sString, 'C592', 'C28C')
	$sString = StringReplace($sString, 'C5BD', 'C28E')
	$sString = StringReplace($sString, 'E28098', 'C291')
	$sString = StringReplace($sString, 'E28099', 'C292')
	$sString = StringReplace($sString, 'E2809C', 'C293')
	$sString = StringReplace($sString, 'E2809D', 'C294')
	$sString = StringReplace($sString, 'E280A2', 'C295')
	$sString = StringReplace($sString, 'E28093', 'C296')
	$sString = StringReplace($sString, 'E28094', 'C297')
	$sString = StringReplace($sString, 'CB9C', 'C298')
	$sString = StringReplace($sString, 'E284A2', 'C299')
	$sString = StringReplace($sString, 'C5A1', 'C29A')
	$sString = StringReplace($sString, 'E280BA', 'C29B')
	$sString = StringReplace($sString, 'C593', 'C29C')
	$sString = StringReplace($sString, 'C5BE', 'C29E')
	$sString = StringReplace($sString, 'C5B8', 'C29F')
	
	$sString =  BinaryToString($sString, 4)
	$sString = StringToASCIIArray($sString, 0, Default, 0)
	$sString = StringFromASCIIArray($sString, 0, Default, 2)
	
    Return $sString
EndFunc ;==>_Encoding_1252ToUTF8

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_866To1251
; Description ...: Converts cyrillic string from IBM 866 codepage to Microsoft 1251 codepage.
; Syntax ........: _Encoding_866To1251($sString)
; Parameters ....: $sString		- String to encode.
; Return values .: Encoded string.
; Author ........: Latoid
; Modified ......: G.Sandler
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_866To1251($sString)
	Local $sRet = '', $iCode
	Local $aSplit = StringSplit($sString, '')
	
	For $i = 1 To $aSplit[0]
		$iCode = Asc($aSplit[$i])
		
		Switch $iCode
			Case 128 To 175
				$aSplit[$i] = Chr($iCode + 64)
			Case 224 To 239
				$aSplit[$i] = Chr($iCode + 16)
			Case 240
				$aSplit[$i] = Chr(168)
			Case 241
				$aSplit[$i] = Chr(184)
			Case 252
				$aSplit[$i] = Chr(185)
		EndSwitch
		
		$sRet &= $aSplit[$i]
	Next
	
	Return $sRet
EndFunc ;==>_Encoding_866To1251

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_ANSIToOEM
; Description ...: Converts ANSI string to OEM encoded string.
; Syntax ........: _Encoding_ANSIToOEM($sString)
; Parameters ....: $sString  	- String to convert.
; Return values .: Success		- Encoded string.
;				   Failure		- Empty string and sets the @error flag:
;                               |1 - DLL error.
;                               |2 - CharToOem failure.
; Author ........: amel27
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_ANSIToOEM($sString)
	Local $sBuffer = DllStructCreate('char[' & StringLen($sString) + 1 & ']')
	Local $aRet = DllCall('User32.dll', 'int', 'CharToOem', 'str', $sString, 'ptr', DllStructGetPtr($sBuffer))
	
	If Not IsArray($aRet) Then
		Return SetError(1, 0, '') ; DLL error
	EndIf
	
	If $aRet[0] = 0 Then
		Return SetError(2, 0, '') ; Function error
	EndIf
	
	Return DllStructGetData($sBuffer, 1)
EndFunc ;==>_Encoding_ANSIToOEM

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_ANSIToUTF8
; Description ...: Converts ANSI string to UTF-8 encoding.
; Syntax ........: _Encoding_ANSIToUTF8($sString)
; Parameters ....: $sString     - String to convert.
; Return values .: Encoded string.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_ANSIToUTF8($sString)
	Return BinaryToString(StringToBinary($sString, 4))
EndFunc ;==>_Encoding_ANSIToUTF8

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_Base64Decode
; Description ...: Decode string from Base64 data format.
; Syntax ........: _Encoding_Base64Decode($sString)
; Parameters ....: $sString     - String to decode.
; Return values .: Success      - Decoded string.
;				   Failure		- Empty string and sets the @error flag:
;								|1 - CallWindowProc error.
;								|2 - Output error.
; Author ........: trancexx, Ward
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _Encoding_Base64Decode($sString)
    Local $sOpcode = "0xC81000005356578365F800E8500000003EFFFFFF3F3435363738393A3B3C3DFFFFFF00FFFFFF000102030405060708090A0B0C0D0E0F10111213141516171819FFFFFFFFFFFF1A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132338F45F08B7D0C8B5D0831D2E9910000008365FC00837DFC047D548A034384C0750383EA033C3D75094A803B3D75014AB00084C0751A837DFC047D0D8B75FCC64435F400FF45FCEBED6A018F45F8EB1F3C2B72193C7A77150FB6F083EE2B0375F08A068B75FC884435F4FF45FCEBA68D75F4668B06C0E002C0EC0408E08807668B4601C0E004C0EC0208E08847018A4602C0E00624C00A46038847028D7F038D5203837DF8000F8465FFFFFF89D05F5E5BC9C21000"
    
    Local $stCodeBuffer = DllStructCreate("byte[" & BinaryLen($sOpcode) & "]")
    DllStructSetData($stCodeBuffer, 1, $sOpcode)
	
    Local $stOuput = DllStructCreate("byte[" & BinaryLen($sString) & "]")
    Local $aRet = DllCall("user32.dll", "int", "CallWindowProc", "ptr", DllStructGetPtr($stCodeBuffer), "str", $sString, "ptr", DllStructGetPtr($stOuput), "int", 0, "int", 0)
	
	If Not IsArray($aRet) Then
		Return SetError(1, 0, '') ; DLL error
	EndIf
	
	If $aRet[0] = 0 Then
		Return SetError(2, 0, '') ; Function error
	EndIf
	
    Return BinaryToString(BinaryMid(DllStructGetData($stOuput, 1), 1, $aRet[0]))
EndFunc ;==>_Encoding_Base64Decode

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_Base64Encode
; Description ...: Encode string to Base64 data format.
; Syntax ........: _Encoding_Base64Encode($sString)
; Parameters ....: $sString     - String to encode.
; Return values .: Success      - Encoded string.
;				   Failure		- Empty string and sets @error to non-zero.
; Author ........: trancexx, Ward
; Modified ......: G.Sandler
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _Encoding_Base64Encode($sString)
    Local $sOpcode = "0x5589E5FF7514535657E8410000004142434445464748494A4B4C4D4E4F505152535455565758595A6162636465666768696A6B6C6D6E6F707172737475767778797A303132333435363738392B2F005A8B5D088B7D108B4D0CE98F0000000FB633C1EE0201D68A06880731C083F901760C0FB6430125F0000000C1E8040FB63383E603C1E60409C601D68A0688470183F90176210FB6430225C0000000C1E8060FB6730183E60FC1E60209C601D68A06884702EB04C647023D83F90276100FB6730283E63F01D68A06884703EB04C647033D8D5B038D7F0483E903836DFC04750C8B45148945FC66B80D0A66AB85C90F8F69FFFFFFC607005F5E5BC9C21000"
	
    Local $stCodeBuffer = DllStructCreate("byte[" & BinaryLen($sOpcode) & "]")
    DllStructSetData($stCodeBuffer, 1, $sOpcode)
	
    $sString = Binary($sString)
    Local $stInput = DllStructCreate("byte[" & BinaryLen($sString) & "]")
    DllStructSetData($stInput, 1, $sString)
	
    Local $iLineBreak = Floor(76 / 4) * 4
    Local $iOputputSize = Ceiling(BinaryLen($sString) * 4 / 3) 
    $iOputputSize += Ceiling($iOputputSize / $iLineBreak) * 2 + 4
	
    Local $stOuput = DllStructCreate("char[" & $iOputputSize & "]")
    Local $aRet = DllCall("user32.dll", "none", "CallWindowProc", "ptr", DllStructGetPtr($stCodeBuffer), "ptr", DllStructGetPtr($stInput), "int", BinaryLen($sString), "ptr", DllStructGetPtr($stOuput), "uint", $iLineBreak)
	
	If Not IsArray($aRet) Then
		Return SetError(1, 0, '') ; DLL error
	EndIf
	
    Return DllStructGetData($stOuput, 1)
EndFunc ;==>_Encoding_Base64Encode

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_CyrillicTo1251
; Description ...: Converts cyrillic string of any encoding to Microsoft 1251 codepage.
; Syntax ........: _Encoding_CyrillicTo1251($sString)
; Parameters ....: $sString     - String to convert
; Return values .: Encoded string.
; Author ........: Latoid
; Modified ......: 
; Remarks .......: 
; Related .......: _Encoding_QuotedPrintableToANSI, _Encoding_IsUTF8, _Encoding_GetCyrillicANSIEncoding, _Encoding_866To1251,
;                  _Encoding_KOI8To1251, _Encoding_ISO8859To1251
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_CyrillicTo1251($sString)
	If StringRegExp($sString, '(=[A-Fa-f0-9]{2}=[A-Fa-f0-9]{2})') Then
		$sString = _Encoding_QuotedPrintableToANSI($sString)
	EndIf
	
	If _Encoding_IsUTF8($sString) Then
		$sString = BinaryToString(StringToBinary($sString), 4)
		
		If Asc(StringLeft($sString, 1)) = 63 Then
			$sString = StringTrimLeft($sString, 1)
		EndIf
	EndIf
	
	Local $sEncoding = _Encoding_GetCyrillicANSIEncoding($sString)
	
	If $sEncoding = 'IBM-866' Then
		Return _Encoding_866To1251($sString)
	ElseIf $sEncoding = 'KOI8-R' Then
		Return _Encoding_KOI8To1251($sString)
	ElseIf $sEncoding = 'ISO-8859-5' Then
		Return _Encoding_ISO8859To1251($sString)
	Else
		Return $sString
	EndIf
EndFunc ;==>_Encoding_CyrillicTo1251

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_GetCyrillicANSIEncoding
; Description ...: Finds out the ANSI encoding of cyrillic string. Needs at least 3-4 proper cyrillic words 
;                  for certain definition.
; Syntax ........: _Encoding_GetCyrillicANSIEncoding($sString)
; Parameters ....: $sString     - String to encode.
; Return values .: Success      - The name of ANSI encoding (KOI8-R, WINDOWS-1251, IBM-866 or ISO-8859-5).
;				   Failure		- Empty string and sets @error to non-zero.
; Author ........: Latoid
; Modified ......: G.Sandler
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_GetCyrillicANSIEncoding($sString)
	Local $iWIN = 0, $iDOS = 0, $iKOI = 0, $iISO = 0
	Local $aStr = StringSplit($sString, '')
	
	For $i = 1 To $aStr[0]
		Switch Asc($aStr[$i])
			Case 192, 224, 200, 232, 206, 238, 210, 242
				$iWIN += 1
			Case 128, 160, 136, 168, 142, 174, 146
				$iDOS += 1
			Case 225, 193, 233, 201, 239, 207, 244, 212
				$iKOI += 1
			Case 176, 208, 184, 216, 190, 222, 194
				$iISO += 1
			Case 226
				$iDOS += 1
				$iISO += 1
		EndSwitch
	Next
	
	If $iKOI >= $iWIN And $iKOI > $iDOS And $iKOI > $iISO Then
		Return 'KOI8-R'
	ElseIf $iWIN > $iKOI And $iWIN > $iDOS And $iWIN > $iISO Then
		Return 'WINDOWS-1251'
	ElseIf $iDOS > $iKOI And $iDOS > $iWIN And $iDOS > $iISO Then
		Return 'IBM-866'
	ElseIf $iISO > $iWIN And $iISO > $iDOS And $iISO > $iKOI Then
		Return 'ISO-8859-5'
	EndIf
	
	Return SetError(1, 0, '')
EndFunc ;==>_Encoding_GetCyrillicANSIEncoding

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_GetFileEncoding
; Description ...: Gets the encoding type of specified file.
; Syntax ........: _Encoding_GetFileEncoding($sFile[, $iReadSize = -1[, $fRetTypeName = False]])
; Parameters ....: $sFile        - File path to check.
;                  $iReadSize    - [optional] Size to read the file for checking. Default is -1 - read whole file.
;                  $fRetTypeName - [optional] If this parameter is True, the return will be the name of encoding type 
;                                |(default is False - return encoding type).
; Return values .: Success       - If $fRetTypeName = False, the return is encoding type 
;                                |(512 = ANSI, 256 = UTF-8 (no BOM), 128 = UTF-8 (with BOM), 32 = UTF-16, 64 = UTF-32), 
;                                |otherwise the return is a string representing encoding type.
;				   Failure		 - -1 and sets the @error flag:
;								 |1 - unable to read the file.
;								 |2 - unable to determine file encoding.
; Author ........: Lazycat (Loopback)
; Modified ......: G.Sandler
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_GetFileEncoding($sFile, $iReadSize = -1, $fRetTypeName = False)
	Local $hFile, $sRead, $iUTF8 = 256, $sChar
	Local $nByte1, $nByte2, $nByte3, $nByte4
	Local $vRet = -1
	
	Local $hFile = FileOpen($sFile, $FO_BINARY)
	
	If $hFile = -1 Then
		Return SetError(1, 0, -1)
	EndIf
	
	Local $sRead = FileRead($hFile, 4)
	FileClose($hFile)
	
	Local $nByte1 = BinaryMid($sRead, 1, 1)
	Local $nByte2 = BinaryMid($sRead, 2, 1)
	Local $nByte3 = BinaryMid($sRead, 3, 1)
	Local $nByte4 = BinaryMid($sRead, 4, 1)
	
	Select
		Case ($nByte1 = 0xFF) And ($nByte2 = 0xFE) And ($nByte3 = 0) And ($nByte4 = 0)
			$vRet = 64 ;Unicode UTF-32
		Case ($nByte1 = 0) And ($nByte2 = 0) And ($nByte3 = 0xFE) And ($nByte4 = 0xFF)
			$vRet = 64 ;Unicode UTF-32
		Case (($nByte1 = 0xFF) And ($nByte2 = 0xFE)) Or (($nByte1 = 0xFE) And ($nByte2 = 0xFF))
			$vRet = 32 ;Unicode UTF-16
		Case ($nByte1 = 0xEF) And ($nByte2 = 0xBB) And ($nByte3 = 0xBF)
			$vRet = 128 ;Unicode UTF-8 with BOM
	EndSelect
	
	; If no BOMs found, try to check if text is UTF-8 without BOM...
	If $vRet = -1 Then
		If $iReadSize = -1 Then
			$iReadSize = FileGetSize($sFile)
		EndIf
		
		$hFile = FileOpen($sFile, $FO_BINARY)
		$sRead = FileRead($hFile, $iReadSize)
		FileClose($hFile)
		
		For $i = 1 To $iReadSize
			$sChar = BinaryMid($sRead, $i, 1)
			
			If $sChar < 0x80 Then
				ContinueLoop
			EndIf
			
			$iUTF8 = 256
			
			If BitAND($sChar, 0xE0) = 0xC0 Then
				If $iReadSize - $i < 2 Then
					$vRet = 512
				Else
					$sChar = BinaryMid($sRead, $i + 1, 1)
					$iBit = BitShift($sChar, 6)
					
					If BitAND($sChar, 0xC0) <> 0x80 Then
						$vRet = 512
					Else
						$iBit = BitNOT(BitAND($sChar, 0x3F))
						
						If $iBit < 0 Then
							$iBit = 0xFF - $iBit
						EndIf
						
						If $iBit < 0x80 Then
							$vRet = 512
						EndIf
					EndIf
				EndIf
			ElseIf BitAND($sChar, 0xF0) = 0xE0 Then
				Return SetError(2, 0, -1) ; ATM - 3-bytes symbols not supported
			EndIf
		Next
	EndIf
	
	If $vRet = -1 Then
		$vRet = $iUTF8 ;ASCII
	EndIf
	
	If $fRetTypeName Then
		Switch $vRet
			Case 512
				$vRet = _Encoding_GetCyrillicANSIEncoding(FileRead($sFile, 256))
			Case 32
				$vRet = 'UTF16 Little Endian'
			Case 64
				$vRet = 'UTF16 Big Endian'
			Case 128
				$vRet = 'UTF8 (with BOM)'
			Case 256
				$vRet = 'UTF8 (without BOM)'
		EndSwitch
	EndIf
	
	Return $vRet
EndFunc ;==>_Encoding_GetFileEncoding

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_ISO88591To1251
; Description ...: Converts cyrillic string from ISO-8859-1 to Microsoft 1251 codepage.
; Syntax ........: _Encoding_ISO88591To1251($sString)
; Parameters ....: $sString     - String to convert.
; Return values .: Success      - Encoded string.
;				   Failure		- Empty string and sets @error to non-zero.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_ISO88591To1251($sString)
	Local $sRet = StringFromASCIIArray(StringToASCIIArray($sString, 0, Default, $SE_UTF16), 0, Default, $SE_UTF8)
	Return SetError((@error ? 1 : 0), 0, (@error ? '' : $sRet))
EndFunc ;==>_Encoding_ISO88591To1251

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_ISO8859To1251
; Description ...: Converts cyrillic string from ISO-8859-5 to Microsoft 1251 codepage.
; Syntax ........: _Encoding_ISO8859To1251($sString)
; Parameters ....: $sString     - String to convert.
; Return values .: Success      - Encoded string.
;				   Failure		- Empty string and sets @error to non-zero.
; Author ........: Latoid
; Modified ......: G.Sandler
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_ISO8859To1251($sString)
	Local $sRet = '', $iCode
	Local $aSplit = StringSplit($sString, '')
	
	If @error Then
		Return SetError(1, 0, '')
	EndIf
	
	For $i = 1 To $aSplit[0]
		$iCode = Asc($aSplit[$i])
		
		Switch $iCode
			Case 176 To 239
				$aSplit[$i] = Chr($iCode + 16)
			Case 161
				$aSplit[$i] = Chr(168)
			Case 241
				$aSplit[$i] = Chr(184)
			Case 240
				$aSplit[$i] = Chr(185)
		EndSwitch
		
		$sRet &= $aSplit[$i]
	Next
	
	Return $sRet
EndFunc ;==>_Encoding_ISO8859To1251

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_IsUTF8
; Description ...: Checks if a given string is stored in UTF-8 encoding.
; Syntax ........: _Encoding_IsUTF8($sString[, $fCheckASCIICode = False])
; Parameters ....: $sString         - String to check.
;                  $fCheckASCIICode - [optional] Check string as ASCII Code. Default is False.
; Return values .: Success          - True
;				   Failure		    - False
; Author ........: amel27
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_IsUTF8($sText, $fCheckASCIICode = False)
	Local $iAsc, $iExt, $iLen = StringLen($sText), $fLess128 = True
	
	For $i = 1 To $iLen
		$iAsc = Asc(StringMid($sText, $i, 1))
		
		If $fCheckASCIICode And $iAsc > 128 Then
			$fLess128 = False
		EndIf
		
		If Not BitAND($iAsc, 0x80) Then
			ContinueLoop
		ElseIf Not BitXOR(BitAND($iAsc, 0xE0), 0xC0) Then
			$iExt = 1
		ElseIf Not (BitXOR(BitAND($iAsc, 0xF0), 0xE0)) Then
			$iExt = 2
		ElseIf Not BitXOR(BitAND($iAsc, 0xF8), 0xF0) Then
			$iExt = 3
		Else
			Return False
		EndIf
		
		If $i + $iExt > $iLen Then
			Return False
		EndIf
		
		For $j = $i + 1 To $i + $iExt
			$iAsc = Asc(StringMid($sText, $j, 1))
			
			If BitXOR(BitAND($iAsc, 0xC0), 0x80) Then
				Return False
			EndIf
		Next
		
		$i += $iExt
	Next
	
	If $fCheckASCIICode Then
		Return ($fLess128 = False)
	EndIf
	
	Return True
EndFunc ;==>_Encoding_IsUTF8

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_JavaUnicodeDecode
; Description ...: Decode string from Java Unicode format.
; Syntax ........: _Encoding_JavaUnicodeDecode($sString)
; Parameters ....: $sString     - String to decode.
; Return values .: Success      - Decoded string.
;				   Failure		- Empty string and sets @error to non-zero.
; Author ........: amel27
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_JavaUnicodeDecode($sString)
	Local $sRet = ''
	Local $aString = StringRegExp($sString, "(\\\\|\\'|\\u[[:xdigit:]]{4}|[[:ascii:]])", 3)
	
	If @error Then
		Return SetError(1, 0, '')
	EndIf
	
	For $i = 0 To UBound($aString) - 1
		Switch StringLen($aString[$i])
			Case 1
				$sRet &= $aString[$i]
			Case 2
				$sRet &= StringRight($aString[$i], 1)
			Case 6
				$sRet &= ChrW(Dec(StringRight($aString[$i], 4)))
		EndSwitch
	Next
	
	Return $sRet
EndFunc ;==>_Encoding_JavaUnicodeDecode

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_JavaUnicodeEncode
; Description ...: Encode string to Java Unicode format.
; Syntax ........: _Encoding_JavaUnicodeEncode($sString)
; Parameters ....: $sString     - String to encode.
; Return values .: Success      - Encoded string.
;				   Failure		- Empty string and sets @error to non-zero.
; Author ........: amel27
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_JavaUnicodeEncode($sString)
	Local $sRet = '', $sChr, $iAsc
	Local $iLen = StringLen($sString)
	
	If $iLen = 0 Then
		Return SetError(1, 0, '')
	EndIf
	
	Local $stChr = DllStructCreate('wchar[' & $iLen + 1 & ']')
	Local $stAsc = DllStructCreate('ushort[' & $iLen + 1 & ']', DllStructGetPtr($stChr))
	DllStructSetData($stChr, 1, $sString)
	
	For $i = 1 To $iLen
		$sChr = DllStructGetData($stChr, 1, $i)
		$iAsc = DllStructGetData($stAsc, 1, $i)
		
		If $sChr = '\' Or $sChr = "'" Then
			$sRet &= '\' & $sChr
		ElseIf $iAsc < 128 Then
			$sRet &= $sChr
		Else
			$sRet &= '\u' & Hex($iAsc, 4)
		EndIf
	Next
	
	Return $sRet
EndFunc ;==>_Encoding_JavaUnicodeEncode

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_KOI8To1251
; Description ...: Converts cyrillic string from KOI8-R to Microsoft 1251 codepage.
; Syntax ........: _Encoding_KOI8To1251($sString)
; Parameters ....: $sString     - String to convert.
; Return values .: Success      - Encoded string.
;				   Failure		- Empty string and sets @error to non-zero.
; Author ........: Latoid
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_KOI8To1251($sString)
	Local $sRet = '', $iCode, $aRange
	Local $aSplit = StringSplit($sString, '')
	
	If @error Then
		Return SetError(1, 0, '')
	EndIf
	
	;[N][0] - Ascii Code (can be a range (ex.: '100-110')
	;[N][1] - Chr (if Ascii above it's a range, then this is a char number that substructed from ascii code)
	Local $aTable[][2] = [[63, 185], [163, 184], [179, 168], ['233-240', 33], ['242-245', 34], ['201-208', 31], ['210-213', 30], ['225-226', 33], ['228-229', 32], ['193-194', 31], _
			[247, 194], [194, 231], [231, 195], [195, 246], [246, 198], [198, 250], [250, 199], [199, 230], [230, 212], [212, 232], [232, 213], [213, 227], [227, 214], _
			[214, 254], [254, 215], [215, 251], [251, 216], [216, 253], [253, 217], [217, 255], [255, 218], [218, 249], [249, 219], [219, 248], [248, 220], [220, 252], _
			[252, 221], [221, 224], [224, 222], [222, 241], [241, 223], [223, 215], [215, 226], [226, 199], [199, 227], [227, 196], [196, 228], [228, 197], [197, 229], _
			[229, 214], [214, 230], [230, 218], [218, 231], [231, 198], [198, 244], [244, 200], [200, 245], [245, 195], [195, 246], [246, 222], [222, 247], [247, 219], _
			[219, 248], [248, 221], [221, 249], [249, 223], [223, 250], [250, 217], [217, 251], [251, 216], [216, 252], [252, 220], [220, 253], [253, 192], [192, 254], _
			[254, 209], [209, 255]]
	
	Local $iUbnd = UBound($aTable) - 1
	
	For $i = 1 To $aSplit[0]
		$iCode = Asc($aSplit[$i])
		
		For $x = 0 To $iUbnd
			If StringInStr($aTable[$x][0], '-') Then
				$aRange = StringRegExp($aTable[$x][0], '(\d+)-(\d+)', 3)
				
				For $iRange = Int($aRange[0]) To Int($aRange[1])
					If $iCode = $iRange Then
						$aSplit[$i] = Chr($iCode - $aTable[$x][1])
						ExitLoop
					EndIf
				Next
			Else
				If $iCode = $aTable[$x][0] Then
					$aSplit[$i] = Chr($aTable[$x][1])
				EndIf
			EndIf
		Next
		
		$sRet &= $aSplit[$i]
	Next
	
	Return $sRet
EndFunc ;==>_Encoding_KOI8To1251

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_OEM2ANSI
; Description ...: Converts OEM encoded string to ANSI string.
; Syntax ........: _Encoding_OEM2ANSI($sString)
; Parameters ....: $sString     - String to convert.
; Return values .: Success      - Encoded string.
;				   Failure		- Empty string and sets the @error flag:
;								|1 - DLL error.
;								|2 - OemToChar failure.
; Author ........: amel27
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_OEM2ANSI($sString)
	Local $sBuffer = DllStructCreate('char[' & StringLen($sString) + 1 & ']')
	Local $aRet = DllCall('User32.dll', 'int', 'OemToChar', 'str', $sString, 'ptr', DllStructGetPtr($sBuffer))
	
	If Not IsArray($aRet) Then
		Return SetError(1, 0, '') ; DLL error
	EndIf
	
	If $aRet[0] = 0 Then
		Return SetError(2, 0, '') ; Function error
	EndIf
	
	Return DllStructGetData($sBuffer, 1)
EndFunc ;==>_Encoding_OEM2ANSI

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_QuotedPrintableToANSI
; Description ...: Converts HEX symbols (quoted-printable) in string (Ex.: "=D2=C1=C2=CF") to ANSI symbols.
; Syntax ........: _Encoding_QuotedPrintableToANSI($sString)
; Parameters ....: $sString             - String to convert.
; Return values .: Success      - ANSI string.
;				   Failure		- Empty string and sets @error to non-zero.
; Author ........: Latoid
; Modified ......: G.Sandler
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_QuotedPrintableToANSI($sString)
	Local $iSymbolFound = 2, $sRet = ''
	
	$sString = StringRegExpReplace($sString, '=\r\n', '')
	
	For $i = 1 To StringLen($sString)
		If StringRegExp(StringMid($sString, $i, 3), '=[A-Fa-f0-9]{2}') Then
			$iSymbolFound = 0
			$sRet &= BinaryToString('0x' & StringMid($sString, $i + 1, 2))
		Else
			$iSymbolFound += 1
			
			If $iSymbolFound = 3 Then
				$iSymbolFound = 2
				$sRet &= StringMid($sString, $i, 1)
			EndIf
		EndIf
	Next
	
	If $sRet = '' Then
		Return SetError(1, 0, '')
	EndIf
	
	Return $sRet
EndFunc ;==>_Encoding_QuotedPrintableToANSI

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_StringToUTF8
; Description ...: Converts any string to UTF-8 encoding.
; Syntax ........: _Encoding_StringToUTF8($sString)
; Parameters ....: $sString     - String to convert.
; Return values .: Success      - String in UTF8 encoding.
;				   Failure		- Empty string and sets @error to non-zero.
; Author ........: LEX1
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_StringToUTF8($sString)
	Local $sRet = '', $iCode
	Local $aSplit = StringSplit($sString, '')
	
	If @error Then
		Return SetError(1, 0, '')
	EndIf
	
	For $i = 1 To $aSplit[0]
		$iCode = Asc($aSplit[$i])
		
		Switch $iCode
			Case 192 To 239
				$aSplit[$i] = Chr(208) & Chr($iCode - 48)
			Case 240 To 255
				$aSplit[$i] = Chr(209) & Chr($iCode - 112)
			Case 168
				$aSplit[$i] = Chr(208) & Chr(129)
			Case 184
				$aSplit[$i] = Chr(209) & Chr(145)
			Case Else
				$aSplit[$i] = Chr($iCode)
		EndSwitch
		
		$sRet &= $aSplit[$i]
	Next
	
	Return $sRet
EndFunc ;==>_Encoding_StringToUTF8

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_URIDecode
; Description ...: Decodes URI encoded string.
; Syntax ........: _Encoding_URIDecode($sString)
; Parameters ....: $sURLHex     - String to decode.
; Return values .: Success      - Decoded string.
;				   Failure		- Empty string and sets @error to non-zero.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_URIDecode($sString)
	Local $sRet = '', $iDec
	Local $aSplit = StringSplit($sString, '')
	
	If @error Then
		Return SetError(1, 0, '')
	EndIf
	
	For $i = 1 To $aSplit[0]
		If $aSplit[$i] = '%' And $i + 2 <= $aSplit[0] Then
			$i += 2
			$iDec = Dec($aSplit[$i - 1] & $aSplit[$i])
			
			If Not @error Then
				$sRet &= Chr($iDec)
			Else
				$sRet &= $aSplit[$i - 2]
			EndIf
		Else
			$sRet &= $aSplit[$i]
		EndIf
	Next
	
	Return $sRet
EndFunc ;==>_Encoding_URIDecode

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_URIEncode
; Description ...: Encodes string to URI qualified string.
; Syntax ........: _Encoding_URIEncode($sString)
; Parameters ....: $sString     - String to convert.
; Return values .: Success      - Encoded string.
;				   Failure		- Empty string and sets @error to non-zero.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_URIEncode($sString)
	Local $sRet = ''
	Local $aSplit = StringSplit($sString, '')
	
	If @error Then
		Return SetError(1, 0, '')
	EndIf
	
	For $i = 1 To UBound($aSplit) - 1
		If Not StringRegExp($aSplit[$i], '(?i)[a-z]|\.|-|_') Then
			$aSplit[$i] = '%' & Hex(Asc($aSplit[$i]), 2)
		EndIf
		
		$sRet &= $aSplit[$i]
	Next
	
	Return $sRet
EndFunc  ;==>_Encoding_URIEncode

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_UTF8BOMDecode
; Description ...: Converts UTF-8 (with BOM) string to ANSI encoding.
; Syntax ........: _Encoding_UTF8BOMDecode($sString)
; Parameters ....: $sString     - String to decode.
; Return values .: Decoded string.
; Author ........: amel27 (mod. by AZJIO)
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_UTF8BOMDecode($sString)
	Local $sRet, $sStr, $iPos, $sChck
	
	Local $sDecodeStr = BinaryToString('0xC3A0C3A1C3A2C3A3C3A4C3A5C3A6C3A7C3A8C3A9C3AAC3ABC3ACC3ADC3AEC3AFC3B0C3B1C3B2C3B3C3B4C3B5C3B6C3B7C3B8C3B9C3BCC3BBC3BDC3BEC3BF', 4)
	Local $sEncodeStr = ''
	
	For $i = 1 To StringLen($sString)
		$sStr = StringMid($sString, $i, 1)
		$iPos = StringInStr($sDecodeStr, $sStr)
		
		If $iPos Then
			$sChck = StringMid($sEncodeStr, $iPos, 1)
			
			If StringIsUpper($sStr) Then
				$sChck = StringUpper($sChck) ;Valid only for changed char
			EndIf
		Else
			$sChck = $sStr
		EndIf
		
		$sRet &= $sChck
	Next
	
	Return $sRet
EndFunc ;==>_Encoding_UTF8BOMDecode

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_UTF8ToANSI
; Description ...: Converts UTF-8 string to ANSI encoding.
; Syntax ........: _Encoding_UTF8ToANSI($sString)
; Parameters ....: $sString        - String to convert.
; Return values .: Encoded string.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_UTF8ToANSI($sString)
	Return BinaryToString(StringToBinary($sString), 4)
EndFunc ;==>_Encoding_UTF8ToANSI

; #FUNCTION# ====================================================================================================================
; Name ..........: _Encoding_UTF8ToUnicode
; Description ...: Converts UTF-8 string to Unicode encoding.
; Syntax ........: _Encoding_UTF8ToUnicode($sString)
; Parameters ....: $sString        - String to convert.
; Return values .: Encoded string.
; Author ........: amel27
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Encoding_UTF8ToUnicode($sString)
	Local $iBufferSize = StringLen($sString)
	Local $stBuffer = DllStructCreate('byte[' & $iBufferSize * 2 & ']')
	
	DllCall('Kernel32.dll', 'int', 'MultiByteToWideChar', 'int', 65001, 'int', 0, 'str', $sString, 'int', StringLen($sString), 'ptr', DllStructGetPtr($stBuffer), 'int', $iBufferSize)
	
	Local $bUnicodeBinary = DllStructGetData($stBuffer, 1)
	Local $sUnicodeHex = StringReplace($bUnicodeBinary, '0x', '')
	Local $iStrLen = StringLen($sUnicodeHex)
	Local $sHexTmp, $iDecTmp, $sRet
	
	For $i = 1 To $iStrLen Step 4
		$sHexTmp = StringMid($sUnicodeHex, $i, 4)
		$iDecTmp = Dec(StringMid($sHexTmp, 3, 2) & StringMid($sHexTmp, 1, 2))
		
		If $iDecTmp Then
			$sRet &= ChrW($iDecTmp)
		EndIf
	Next
	
	Return $sRet
EndFunc ;==>_Encoding_UTF8ToUnicode
