#include-once

#CS Header
	Name:               Settings.au3
	Description:        Library designed to simplify usage, reading and writing internal program settings.
	Version:            1.1
	AutoIt Version(s):  3.3.10.2 - 3.3.14.2
	Author(s):          G.Sandler (CreatoR)
	Remarks:            Uses Scripting.Dictionary object.
	
	History:
	                    v1.0
						* First public version.
						
						v1.1
						* _Settings_SetDefaults renamed to _Settings_LoadDefaults.
						* $aDefs parameter in _Settings_Startup is not optional anymore.
						+ Added optional $bApply parameter to _Settings_Shutdown function (default is True)
						+ Added _Settings_Clone function to clone settings object.
						* Changed examples.
#CE

Global Const $_sSettings_File_Key = '___FileKey___'
Global Const $_sSettings_Sect_Key = '___SectKey___'
Global Const $_sSettings_Defs_Key = '___DefsKey___'

; #FUNCTION# ====================================================================================================================
; Name ..........: _Settings_Startup
; Description ...: Initialize settings module
; Syntax ........: _Settings_Startup($sFile, $sSect[, $aDefs = 0[, $fCaseSensitive = False]])
; Parameters ....: $sFile               - Path to settings file (to store data, usualy ini-file).
;                  $sSect               - Section name for current settings object (used later to read and write data).
;                  $aDefs               - 2D array of settings defaults. Should be constructed as following (zero-based):
;                                                                                                        [n][0] - Key
;                                                                                                        [n][1] - Value
;                  $fCaseSensitive      - [optional] Determines whether to use case sensitive keys in settings object. Default is False.
; Return values .: On Success: $oSettings object.
;                  On Failure: 0 and sets @error to: 1 if $aDefs is not a 2D array, 2 if unable to create Scripting.Dictionary object.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: $oSettings object usage: $oSettings('MySettingsItem') [= 'New Value']
; Related .......: _Settings_Shutdown
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _Settings_Startup($sFile, $sSect, $aDefs, $fCaseSensitive = False)
	If UBound($aDefs, 0) <> 2 Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $oSettings = ObjCreate('Scripting.Dictionary')
	
	If Not IsObj($oSettings) Then
		Return SetError(2, 0, 0)
	EndIf
	
	ObjEvent('AutoIt.Error', '__Settings_Error')
	$oSettings.CompareMode = Int(Not $fCaseSensitive)
	
	$oSettings($_sSettings_File_Key) = $sFile
	$oSettings($_sSettings_Sect_Key) = $sSect
	$oSettings($_sSettings_Defs_Key) = $aDefs
	
	_Settings_LoadDefaults($oSettings, False)
	
	For $i = 0 To UBound($aDefs) - 1
		If Not $oSettings.Exists($aDefs[$i][0]) Then
			$oSettings($aDefs[$i][0]) = $aDefs[$i][1]
		EndIf
	Next
	
	Return $oSettings
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Settings_GetList
; Description ...: Get settings list.
; Syntax ........: _Settings_GetList($oSettings)
; Parameters ....: $oSettings           - $oSettings object as returned by _Settings_Startup.
; Return values .: On Success: 2D array of settings list, where [0][0] is the total count of keys in settings list, [n][0] is the key name, and [n][1] is the key value.
;                  On Failure: 0 and sets @error to 1 if $oSettings is not an object.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Settings_GetList($oSettings)
	If Not IsObj($oSettings) Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $aKeys = $oSettings.Keys
	Local $aRet[1][2]
	
	For $i = 0 To $oSettings.Count - 1
		If $aKeys[$i] == $_sSettings_File_Key Or $aKeys[$i] == $_sSettings_Sect_Key Or $aKeys[$i] == $_sSettings_Defs_Key Then
			ContinueLoop
		EndIf
		
		$aRet[0][0] += 1
		ReDim $aRet[$aRet[0][0] + 1][2]
		
		$aRet[$aRet[0][0]][0] = $aKeys[$i]
		$aRet[$aRet[0][0]][1] = $oSettings($aKeys[$i])
	Next
	
	Return $aRet
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Settings_Apply
; Description ...: Apply current settings data. Writes settings data to settings file (as set by _Settings_Startup function).
; Syntax ........: _Settings_Apply($oSettings)
; Parameters ....: $oSettings           - $oSettings object as returned by _Settings_Startup.
; Return values .: On Success: 1.
;                  On Failure: 0 and sets @error to 1 if $oSettings is not an object.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: 
; Related .......: _Settings_GetList
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _Settings_Apply($oSettings)
	If Not IsObj($oSettings) Then
		Return SetError(1, 0, 0)
	EndIf
	
	IniWriteSection($oSettings($_sSettings_File_Key), $oSettings($_sSettings_Sect_Key), _Settings_GetList($oSettings))
	
	Return 1
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Settings_LoadDefaults
; Description ...: Load default settings.
; Syntax ........: _Settings_LoadDefaults(Byref $oSettings[, $bFactory = False])
; Parameters ....: $oSettings           - [in/out] $oSettings object as returned by _Settings_Startup.
;                  $bFactory            - [optional] Determines whether to set factory defaults or startup defaults. Default is False, set startup defaults.
; Return values .: On Success: 1.
;                  On Failure: 0 and set @error to: 1 if $oSettings is not an object, 2 if unable to read from data file specified in _Settings_Startup function.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _Settings_LoadDefaults(ByRef $oSettings, $bFactory = False)
	If Not IsObj($oSettings) Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $sFile = $oSettings($_sSettings_File_Key)
	Local $sSect = $oSettings($_sSettings_Sect_Key)
	Local $aDefs = $oSettings($_sSettings_Defs_Key)
	
	If $bFactory Then
		For $i = 0 To UBound($aDefs) - 1
			$oSettings($aDefs[$i][0]) = $aDefs[$i][1]
		Next
		
		Return 1
	EndIf
	
	Local $aRead = IniReadSection($sFile, $sSect)
	
	If Not IsArray($aRead) Then
		Return SetError(2, 0, 0)
	EndIf
	
	For $i = 1 To UBound($aRead) - 1
		$oSettings($aRead[$i][0]) = (StringIsDigit($aRead[$i][1]) ? Int($aRead[$i][1]) : $aRead[$i][1])
	Next
	
	Return 1
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Settings_GetDefaults
; Description ...: Get default settings. Usefull if you need to retriev specific initial (default) setting value.
; Syntax ........: _Settings_GetDefaults($oSettings[, $bFactory = False])
; Parameters ....: $oSettings           - $oSettings object as returned by _Settings_Startup.
;                  $bFactory            - [optional] Determines whether to get factory defaults or startup defaults. Default is False - get startup defaults.
; Return values .: On Success: $oTemp object with default settings.
;                  On Failure: 0 and sets @error to 1 if $oSettings is not an object.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: $oTemp must be destroyed ($oTemp = 0) if not used anymore.
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _Settings_GetDefaults($oSettings, $bFactory = False)
	If Not IsObj($oSettings) Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $oTemp = _Settings_Startup($oSettings($_sSettings_File_Key), $oSettings($_sSettings_Sect_Key), $oSettings($_sSettings_Defs_Key), ($oSettings.CompareMode = 0))
	
	_Settings_LoadDefaults($oTemp, $bFactory)
	
	Return $oTemp
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Settings_Clone
; Description ...: Clone settings object. Usefull if you need to use settings temporary without affecting main settings object.
; Syntax ........: _Settings_Clone($oSettings)
; Parameters ....: $oSettings           - $oSettings object as returned by _Settings_Startup.
; Return values .: On Success: $oTemp object with cloned settings.
;                  On Failure: 0 and sets @error to 1 if $oSettings is not an object.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: $oTemp must be destroyed ($oTemp = 0) if not used anymore.
; Related .......: 
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _Settings_Clone($oSettings, $bFactory = False)
	If Not IsObj($oSettings) Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $oTemp = _Settings_Startup($oSettings($_sSettings_File_Key), $oSettings($_sSettings_Sect_Key), $oSettings($_sSettings_Defs_Key), ($oSettings.CompareMode = 0))
	
	Local $aKeys = $oTemp.Keys
	
	For $i = 0 To $oTemp.Count - 1
		If $aKeys[$i] == $_sSettings_File_Key Or $aKeys[$i] == $_sSettings_Sect_Key Or $aKeys[$i] == $_sSettings_Defs_Key Then
			ContinueLoop
		EndIf
		
		$oTemp($aKeys[$i]) = $oSettings($aKeys[$i])
	Next
	
	Return $oTemp
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Settings_Shutdown
; Description ...: Shutdown previously initialized $oSettings object and writes settings data to settings file (as set by _Settings_Startup function).
; Syntax ........: _Settings_Shutdown($oSettings[, $bApply = True])
; Parameters ....: $oSettings           - $oSettings object as returned by _Settings_Startup.
;                  $bApply              - [optional] Determines whether to apply the settings or not. Ddefault is True - apply settings.
; Return values .: On Success: 1.
;                  On Failure: 0 and sets @error to 1 if $oSettings is not an object.
; Author ........: G.Sandler
; Modified ......: 
; Remarks .......: 
; Related .......: _Settings_GetList
; Link ..........: 
; Example .......: Yes
; ===============================================================================================================================
Func _Settings_Shutdown($oSettings, $bApply = True)
	If Not IsObj($oSettings) Then
		Return SetError(1, 0, 0)
	EndIf
	
	_Settings_Apply($oSettings)
	
	$oSettings = 0
	Return 1
EndFunc
