#Region Header

#CS
	Name:				PluginInclude UDF
	Author:				Copyright  2011 CreatoR's Lab (G.Sandler), www.creator-lab.ucoz.ru, www.autoit-script.ru. All rights reserved.
	AutoIt version:		3.3.6.1
	UDF version:		1.1
	
	Remarks:			
	
	History:
						[1.1]
						* Now the UDF uses Container.au3 to communicate between the main script and the included plugins.
						 It solves an issue with passing arrays (only 2D atm) and other data types as parameters.
						
						[1.0]
						* First public version.
	
#CE

;Includes
#include-once
#include "Container.au3"

Global $oContainer = _Container_Open("PI_Storage", 1)
OnAutoItExitRegister("__PluginInclude_Exit")

#EndRegion Header

#Region Global Variables

Global $aPlgIncld_List[1]

#EndRegion Global Variables

#Region Example

#CS
	
	#include <PluginInclude.au3>
	
	_PluginInclude_AddInclude(@ScriptDir & "\External_UDFs.fnc")
	_PluginInclude_AddInclude(@ScriptDir & "\Other_UDFs.fnc")
	
	$vRet = _PluginInclude_ExecuteFunc("Some_UDF", "Are you sure?")
	
	If Not @error Then
		MsgBox(64, @ScriptName & " - ExecuteFunc", "Returned Message from Some_UDF: " & $vRet)
	EndIf
	
	$vRet = _PluginInclude_ExecuteFunc("Other_UDF", "Notepad.exe") ;Run the notepad

	If Not @error Then
		MsgBox(64, @ScriptName & " - ExecuteFunc", "Returned Message from Other_UDF.fnc (Other_UDF): " & $vRet)
	EndIf
	
	;=== Passing arrays ===
	Dim $aArr[2]
	$aRet = _PluginInclude_ExecuteFunc("SetArray", $aArr)
	
	If Not @error Then
		MsgBox(64, @ScriptName & " - ExecuteFunc", "Returned Message from Other_UDF.fnc (SetArray): " & $aRet[1])
	EndIf
	
#CE

#EndRegion Example

#Region Public Functions

; #FUNCTION# ====================================================================================================
; Name...........:	_PluginInclude_AddInclude
; Description....:	Adds external include plugin to the global includes list.
; Syntax.........:	_PluginInclude_AddInclude($sIncludeFile)
; Parameters.....:	$sIncludeFile - Plugin file to include.
;					
; Return values..:	Success - Returns 1.
;					Failure - Set @error to 1 and returns 0 if the include plugin file ($sIncludeFile) not exists.
; Author.........:	G.Sandler (CreatoR).
; Modified.......:	
; Remarks........:	
; Related........:	_PluginInclude_RemoveInclude, _PluginInclude_ExecuteFunc
; Link...........:	
; Example........:	Yes.
; ===============================================================================================================
Func _PluginInclude_AddInclude($sIncludeFile)
	If Not FileExists($sIncludeFile) Then
		Return SetError(1, 0, 0)
	EndIf
	
	$aPlgIncld_List[0] += 1
	ReDim $aPlgIncld_List[$aPlgIncld_List[0]+1]
	$aPlgIncld_List[$aPlgIncld_List[0]] = $sIncludeFile
	
	Return 1
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........:	_PluginInclude_RemoveInclude
; Description....:	Remove included plugin from the includes list.
; Syntax.........:	_PluginInclude_RemoveInclude($sIncludeFile)
; Parameters.....:	$sIncludeFile - Plugin file to remove from the list.
;					
; Return values..:	Success - Returns 1.
;					Failure - Returns 0 if the include was not found in the list.
; Author.........:	G.Sandler (CreatoR).
; Modified.......:	
; Remarks........:	
; Related........:	_PluginInclude_AddInclude, _PluginInclude_ExecuteFunc
; Link...........:	
; Example........:	Yes.
; ===============================================================================================================
Func _PluginInclude_RemoveInclude($sIncludeFile)
	Local $aTmp[1], $iRet = 0
	
	For $i = 1 To $aPlgIncld_List[0]
		If $aPlgIncld_List[$i] = $sIncludeFile Then
			$iRet = 1
		Else
			$aTmp[0] += 1
			ReDim $aTmp[$aTmp[0]+1]
			$aTmp[$aTmp[0]] = $aPlgIncld_List[$i]
		EndIf
	Next
	
	$aPlgIncld_List = $aTmp
	Return $iRet
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........:	_PluginInclude_ExecuteFunc
; Description....:	Executes function from external included plugins.
; Syntax.........:	_PluginInclude_ExecuteFunc($sFuncName [, $sParams = "" [, $iWait = 1 [, $sIncludeFile = ""]]])
; Parameters.....:	$sFuncName    - Function name to execute.
;					$sParams      - [Optional] Parameter(s) for the executed function. Arrays (only 2D) are supported.
;					$iWait        - [Optional] If this parameter = 0, then the function returns immediately without waiting for external function to return.
;					$sIncludeFile - [Optional] If this parameter <> "", then function is executed specificly from that included plugin.
;                                              Otherwise the function is been searched in all includes (and executes the first found).
;					
; Return values..:	Success - Depending on $iWait parameter. If $iWait = 1, then StdOut from executed function (the return value) returned, otherwise returns 1.
;					Failure - Set @error as following:
;                                                      1 - Included plugins was not found.
;                                                      2 - Unable to execute external function.
;                                                      3 - Unable to run sub-process (@AutoItExe failed to run).
; Author.........:	G.Sandler (CreatoR).
; Modified.......:	
; Remarks........:	
; Related........:	_PluginInclude_AddInclude, _PluginInclude_RemoveInclude
; Link...........:	
; Example........:	Yes.
; ===============================================================================================================
Func _PluginInclude_ExecuteFunc($sFuncName, $sParams = "", $iWait = 1, $sIncludeFile = "")
	Local $iFuncFound = 0, $sPluginInclude = "", $sRead_PluginInclude = "", $sData_Parser, $iPID
	Local $sRet
	
	If $sIncludeFile <> "" Then
		$sPluginInclude = $sIncludeFile
		$sRead_PluginInclude = FileRead($sPluginInclude)
	Else
		For $i = 1 To $aPlgIncld_List[0]
			If Not FileExists($aPlgIncld_List[$i]) Then
				ContinueLoop
			EndIf
			
			$sRead_PluginInclude = FileRead($aPlgIncld_List[$i])
			$sPluginInclude = $aPlgIncld_List[$i]
			
			If StringRegExp($sRead_PluginInclude, "(?im)^\s*Func\s*" & $sFuncName & "\s*\(") Then
				$iFuncFound = 1
				ExitLoop
			EndIf
		Next
	EndIf
	
	If Not FileExists($sPluginInclude) Or $sPluginInclude = "" Then
		Return SetError(1, 0, "")
	EndIf
	
	If Not $iFuncFound Then
		Return SetError(2, 0, "")
	EndIf
	
	$sData_Parser = '0x234E6F5472617949636F6E0D0A0D0A476C6F62616C20247350495F4E616D65203D20405363726970744E616D650D0A0D0A49662057696E45786973747328247350495F4E616D6529205468656E0D0A09457869740D0A456E6449660D0A0D0A4175746F497457696E5365745469746C6528247350495F4E616D65290D0A0D0A23526567696F6E20436F6E7461696E6572205544463A20687474703A2F2F7777772E6175746F69747363726970742E636F6D2F666F72756D2F746F7069632F3132363933362D636F6E7461696E65722D7564662D736372697074732D696E746572616374696F6E2D6C6962726172790D0A0D0A476C6F62616C20246F436F6E7461696E65724F4D4572720D0A0D0A46756E63205F436F6E7461696E65725F4F70656E2824734E616D652C2024694D6F6465290D0A094C6F63616C20246F5368656C6C2C20246F5368656C6C57696E646F772C20246F5368656C6C57696E646F77730D0A090D0A094966204E6F74204F626A4576656E7428224175746F49742E4572726F722229205468656E0D0A0909246F436F6E7461696E65724F4D457272203D204F626A4576656E7428224175746F49742E4572726F72222C20225F5F436F6E7461696E65725F434F4D45727248616E646C657222290D0A09456E6449660D0A090D0A09246F5368656C6C203D204F626A43726561746528225368656C6C2E4170706C69636174696F6E22290D0A090D0A094966204E6F742049734F626A28246F5368656C6C29205468656E0D0A090952657475726E205365744572726F7228322C20302C2030290D0A09456E6449660D0A090D0A09246F5368656C6C57696E646F7773203D20246F5368656C6C2E57696E646F77730D0A090D0A09466F7220246F5368656C6C57696E646F7720496E20246F5368656C6C57696E646F77730D0A0909496620246F5368656C6C57696E646F772E53746174757354657874203D2024734E616D65205468656E0D0A09090952657475726E20246F5368656C6C57696E646F770D0A0909456E6449660D0A09090D0A0909496620406572726F72205468656E0D0A090909457869744C6F6F700D0A0909456E6449660D0A094E6578740D0A090D0A0949662024694D6F6465203D2030205468656E0D0A090952657475726E205365744572726F7228312C20302C2030290D0A09456E6449660D0A090D0A093B246F436F6E7461696E6572203D204F626A47657428226E65773A7B43303841464439302D463241312D313144312D383435352D3030413043393146333838307D22290D0A09246F436F6E7461696E6572203D204F626A4372656174652822496E7465726E65744578706C6F7265722E4170706C69636174696F6E22290D'
	$sData_Parser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
	
	$sData_Parser = BinaryToString($sData_Parser)
	
	If Not StringInStr($sRead_PluginInclude, $sData_Parser) Then
		$sRead_PluginInclude = $sData_Parser & @CRLF & $sRead_PluginInclude
		$hFile = FileOpen($sPluginInclude, 2)
		FileWrite($hFile, $sRead_PluginInclude)
		FileClose($hFile)
	EndIf
	
	_Container_PutProperty($oContainer, "PI_Func_Name", $sFuncName)
	_Container_PutProperty($oContainer, "PI_Func_Params", $sParams)
	
	$iPID = Run(@AutoItExe & ' /AutoIt3ExecuteScript "' & $sPluginInclude & '"')
	
	If @error Then
		Return SetError(3, 0, "")
	EndIf
	
	If $iWait Then
		While $sRet = '' And ProcessExists($iPID)
			Sleep(100)
			$sRet = _Container_GetProperty($oContainer, "PI_Func_Return")
		WEnd
	Else
		$sRet = 1
	EndIf
	
	_Container_PutProperty($oContainer, "PI_Func_Return", "")
	
	Return $sRet
EndFunc

#EndRegion Public Functions

#Region Internal Functions

Func __PluginInclude_Exit()
	_Container_Close($oContainer)
EndFunc

#EndRegion Internal Functions
