#Region Header

#CS
	Name:				AppCheckUpdates
	Author:				Copyright  2019 CreatoR's Lab. All rights reserved.
	AutoIt version:		3.3.14.x
	UDF version:		1.1
	
	History:
						1.1
						* Fixed issue with closing program after update.
						* Fixed issue with backuping all files in current dir. Now only updated files backuped.
						* Now only zip update supported.
						* Removed limitation for non compiled script.
						* $APPCU_DAT_FILE renamed to $APPCU_DAT_URL.
						* Renamed internal functions from __AppCheckUpdates_* to __APPCU_*.
						* Removed unnecessary includes and global variables.
						* Added "Updater" utility for external programs, based on this UDF.
						* Example changed.
						
						1.0
						* First public version.
	
#CE

#include-once

#EndRegion Header

#Region User Vars

Global $APPCU_7ZIP						= @ScriptDir & '\Resources\7z.exe'
Global $APPCU_DAT_URL 					= 'http://mysite.com/mysoft/myapp.zip'
Global $APPCU_APP_NAME 					= 'My App'
Global $APPCU_APP_VER 					= '1.0'

Global Enum _
	$APPCU_MSG_ATNTN, $APPCU_MSG_ERR, $APPCU_MSG_UPDCHCK, $APPCU_MSG_CHCKUPDERR, $APPCU_MSG_NEWVER, _
	$APPCU_MSG_DNLDUPDERR, $APPCU_MSG_EXTRCTUPDERR, $APPCU_MSG_USNGNEWSTVER, $APPCU_MSG_CHCKINETCNCT, $APPCU_MSG_DNLDNG, $APPCU_MSG_CMPLTD, _
	$APPCU_MSG_TOTAL

;Translate messages
Global $APPCU_MSGS[$APPCU_MSG_TOTAL]

$APPCU_MSGS[$APPCU_MSG_ATNTN] 			= 'Attention'
$APPCU_MSGS[$APPCU_MSG_ERR]				= 'Error'
$APPCU_MSGS[$APPCU_MSG_UPDCHCK]			= 'Update check'
$APPCU_MSGS[$APPCU_MSG_CHCKUPDERR]		= 'There was an error (%i) checking the update, please contact with software developer.'
$APPCU_MSGS[$APPCU_MSG_NEWVER]			= 'There is new version available (%s).\n\nChange Log:\n\n%s\n\nWould you like to download the update?'
$APPCU_MSGS[$APPCU_MSG_DNLDUPDERR]		= 'There was an error (%i) downloading the update, please contact with software developer.'
$APPCU_MSGS[$APPCU_MSG_EXTRCTUPDERR]	= 'There was an error (%i) to extract archive, please contact with software developer.'
$APPCU_MSGS[$APPCU_MSG_USNGNEWSTVER]	= 'You are using the newest version of this software (%s).'
$APPCU_MSGS[$APPCU_MSG_CHCKINETCNCT]	= 'Please check your internet connection and try again.'
$APPCU_MSGS[$APPCU_MSG_DNLDNG]			= 'Downloading [%s]'
$APPCU_MSGS[$APPCU_MSG_CMPLTD]			= 'Completed: %s%%'

#EndRegion User Vars

#Region Options

If StringInStr($CmdLineRaw, '/AppCU') Then
	Opt('TrayIconHide', 1)
	
	$hParent = HWnd(StringRegExpReplace($CmdLineRaw, '(?s).*/hParent="([^"]*)".*', '\1'))
	$iSilentMode = Number(StringRegExpReplace($CmdLineRaw, '(?s).*/iSMode="([^"]*)".*', '\1'))
	$iMakeBackup = Number(StringRegExpReplace($CmdLineRaw, '(?s).*/iBackup="([^"]*)".*', '\1'))
	$sLanguage = StringRegExpReplace($CmdLineRaw, '(?s).*/sLang="([^"]*)".*', '\1')
	
	$APPCU_7ZIP = StringRegExpReplace($CmdLineRaw, '(?s).*/s7zip="([^"]*)".*', '\1')
	$APPCU_DAT_URL = StringRegExpReplace($CmdLineRaw, '(?s).*/sDatUrl="([^"]*)".*', '\1')
	$APPCU_APP_NAME = StringRegExpReplace($CmdLineRaw, '(?s).*/sAppName="([^"]*)".*', '\1')
	$APPCU_APP_VER = StringRegExpReplace($CmdLineRaw, '(?s).*/sAppVer="([^"]*)".*', '\1')
	
	$APPCU_MSGS = StringSplit(StringRegExpReplace($CmdLineRaw, '(?s).*/sMsgs="([^"]*)".*', '\1'), @LF, 2)
	
	_AppCheckUpdates($hParent, $iSilentMode, $iMakeBackup, $sLanguage)
	ConsoleWrite(@error)
	
	Exit
EndIf

#EndRegion Options

#Region Public Functions

; #FUNCTION# ====================================================================================================================
; Name ..........: _AppCheckUpdates
; Description ...: Check for updates for your program.
; Syntax ........: _AppCheckUpdates([$hParent = 0[, $iSilentMode = 0[, $iMakeBackup = 1[, $sLanguage = '']]]])
; Parameters ....: $hParent             - [optional] Parent window handle.
;                  $iSilentMode         - [optional] This parameter can be one of the following (Default is -1):
;																						0 - Check update and display all messages (on failure/success, ask to update, when update is not available).
;																						1 - Check update quitly (with no messages when there is no update, only when update is available).
;																						2 - Check update quitly (with no messages at all, update will be performed without prompt!).
;                  $iMakeBackup         - [optional] If 1 (default), the updater will backup all files in program dir.
;                  $sLanguage           - [optional] Language string to support multilanguage app (used in "Update ChangeLog[ Language]" in update.dat file). Default is ''.
; Return values .: PID of executed process that handles update process, StdoutRead can be used to get error code that returned as following:
;															1 - No internet connection.
;															2 - Unable to read update info.
;															3 - Unable to download update.
;															4 - Unable to extract zipped update (7zip should be located at "@ScriptDir\Resources\7z.exe" or as set in $APPCU_7ZIP variable).
; Author ........: G.Sandler
; Remarks .......: This function calls second copy of the process to check update in background.
;                  * Update.dat file supports the following structure:
;                                                           [Info]
;                                                           	;New version (compared with file version)
;                                                           Update Version=1.1.0.0
;                                                           	;Direct link to update file
;                                                           Update File=http://mysite.com/mysoft/myapp.zip
;                                                           	;Changelog delimited with \n (for new line to display). LANG is the language set by $sLanguage (can be ommited if $sLanguage = "").
;                                                           Update ChangeLog LANG=Many bug fixes.\nNew option added.
;                                                           	;File(s) to execute after the update, make sure that main program executed AFTER all other files (put it last)
;                                                           	;Each file delimited with ;, number after comma determines file deletion after it's execution
;                                                           Execute After Update=Update.exe,1;MyApp.exe,0
; Related .......: 
; Example .......: Yes
; ===============================================================================================================================
Func _AppCheckUpdates($hParent = 0, $iSilentMode = 0, $iMakeBackup = 1, $sLanguage = '')
	If Not StringInStr($CmdLineRaw, '/AppCU') Then
		Local $sApp_Exe = @AutoItExe & (@Compiled ? '' : ' "' & @ScriptFullPath & '"')
		Local $sMsgs = ''
		
		For $iMsg = 0 To $APPCU_MSG_TOTAL - 1
			$sMsgs &= ($sMsgs ? @LF : '') & $APPCU_MSGS[$iMsg]
		Next
		
		Local $iPID = Run($sApp_Exe & ' ' & _
			'/AppCU /hParent="' & $hParent & '" /iSMode="' & $iSilentMode & '" /iBackup="' & $iMakeBackup & '" /sLang="' & $sLanguage & '" ' & _
			'/sDatUrl="' & $APPCU_DAT_URL & '" /s7zip="' & $APPCU_7ZIP & '" /sAppName="' & $APPCU_APP_NAME & '" /sAppVer="' & $APPCU_APP_VER & '" /sMsgs="' & $sMsgs & '"', _
			@ScriptDir, @SW_SHOW, 8)
		
		Return $iPID
	EndIf
	
	If $hParent <> '' And IsString($hParent) Then
		$hParent = WinGetHandle($hParent)
	EndIf
	
	If Not __APPCU_InetConnected() Then
		If $iSilentMode <= 0 Then
			MsgBox(262144+48, $APPCU_APP_NAME & ' - ' & $APPCU_MSGS[$APPCU_MSG_UPDCHCK] & ' : ' & $APPCU_MSGS[$APPCU_MSG_ERR], $APPCU_MSGS[$APPCU_MSG_CHCKINETCNCT], 0, $hParent)
		EndIf
		
		Return SetError(1, 0, 0)
	EndIf
	
	Local $sUpdate_Info = BinaryToString(InetRead($APPCU_DAT_URL, 1))
	
	If Not StringInStr($sUpdate_Info, '[Info]') Then
		If $iSilentMode <= 0 Then
			MsgBox(262144+48, $APPCU_APP_NAME & " - " & $APPCU_MSGS[$APPCU_MSG_UPDCHCK] & ' : ' & $APPCU_MSGS[$APPCU_MSG_ERR], StringFormat($APPCU_MSGS[$APPCU_MSG_CHCKUPDERR], 2), 0, $hParent)
		EndIf
		
		Return SetError(2, 0, 0)
	EndIf
	
	Local $sUpdate_Version = __APPCU_GetStrInfo($sUpdate_Info, 'Update Version')
	
	If __APPCU_VersCompare($sUpdate_Version, $APPCU_APP_VER) = 1 Then
		Local $sUpdate_File = __APPCU_GetStrInfo($sUpdate_Info, 'Update File')
		Local $sUpdate_ChangeLog = StringReplace(__APPCU_GetStrInfo($sUpdate_Info, 'Update ChangeLog' & ($sLanguage ? ' ' & $sLanguage : ''), 'N/A'), '\n', @CRLF)
		Local $sExec_After_Update = __APPCU_GetStrInfo($sUpdate_Info, 'Execute After Update')
		
		;Show message when update available
		If $iSilentMode <= 1 Then
			If MsgBox(262144+36, $APPCU_APP_NAME & ' - ' & $APPCU_MSGS[$APPCU_MSG_UPDCHCK], StringFormat($APPCU_MSGS[$APPCU_MSG_NEWVER], $sUpdate_Version, $sUpdate_ChangeLog), 0, $hParent) <> 6 Then
				Return 0
			EndIf
		EndIf
		
		Local $sFile_Name = StringRegExpReplace($sUpdate_File, '^.*/', '')
		Local $sApp_Dir = @ScriptDir
		Local $sUpdate_Local_File = $sApp_Dir & '\' & $sFile_Name
		
		Local $sUpdate_Dir = $sApp_Dir & '\Update_Tmp'
		Local $sBackup_Dir = $sApp_Dir & '\Backup'
		
		Local $hInetGet = InetGet($sUpdate_File, $sUpdate_Local_File, 1, 1)
		
		If $iSilentMode <= 1 Then
			ProgressOn($APPCU_APP_NAME & ' - ' & $APPCU_MSGS[$APPCU_MSG_UPDCHCK], StringFormat($APPCU_MSGS[$APPCU_MSG_DNLDNG], $sFile_Name))
			
			Local $iPrecent
			
			Do
				Sleep(10)
				$iPrecent = 100 / (InetGetInfo($hInetGet, 1) / InetGetInfo($hInetGet, 0))
				
				If Not __APPCU_NumberIsOverflow($iPrecent) Then
					ProgressSet($iPrecent, StringFormat($APPCU_MSGS[$APPCU_MSG_CMPLTD], Round($iPrecent, 1)))
				EndIf
			Until InetGetInfo($hInetGet, 2)    ; Check if the download is complete.
		Else
			Do
				Sleep(100)
			Until InetGetInfo($hInetGet, 2)    ; Check if the download is complete.
		EndIf
		
		Local $nBytes = InetGetInfo($hInetGet, 0)
		InetClose($hInetGet)
		
		If $iSilentMode <= 1 Then
			ProgressOff()
		EndIf
		
		If $nBytes = 0 Then
			If $iSilentMode <= 0 Then
				MsgBox(48, $APPCU_APP_NAME & ' - ' & $APPCU_MSGS[$APPCU_MSG_UPDCHCK] & ' : ' & $APPCU_MSGS[$APPCU_MSG_ERR], StringFormat($APPCU_MSGS[$APPCU_MSG_DNLDUPDERR], 3), 0, $hParent)
			EndIf
			
			Return SetError(3, 0, 0)
		EndIf
		
		RunWait('"' & $APPCU_7ZIP & '" x "' & $sUpdate_Local_File & '" -y -o"' & $sUpdate_Dir & '" * -r', $sApp_Dir, @SW_HIDE)
		Local $iError = @error
		
		FileDelete($sUpdate_Local_File)
		
		If $iError Then
			If $iSilentMode <= 0 Then
				MsgBox(48, $APPCU_APP_NAME & ' - ' & $APPCU_MSGS[$APPCU_MSG_UPDCHCK] & ' : ' & $APPCU_MSGS[$APPCU_MSG_ERR], StringFormat($APPCU_MSGS[$APPCU_MSG_EXTRCTUPDERR], 4), 0, $hParent)
			EndIf
			
			Return SetError(4, 0, 0)
		EndIf
		
		If $iMakeBackup Then
			__APPCU_DirBackupAsSrc($sUpdate_Dir, $sApp_Dir, $sBackup_Dir)
		EndIf
		
		If $sExec_After_Update Then
			Local $aExec, $vDel
			
			$aExec = StringSplit($sExec_After_Update, ';')
			$sExec_After_Update = ''
			
			For $i = 1 To $aExec[0]
				If Not $aExec[$i] Then
					ContinueLoop
				EndIf
				
				$vDel = StringRegExp($aExec[$i], ',\h*(0|1)$', 3)
				
				If Not @error Then
					$vDel = Int($vDel[0])
					$aExec[$i] = StringRegExpReplace($aExec[$i], '\h*,\h*\d+$', '')
				EndIf
				
				$sExec_After_Update &= ' & Start ' & ($vDel ? '/Wait ' : '') & '"" "' & $sApp_Dir & '\' & $aExec[$i] & '"' & ($vDel ? ' & Del /Q "' & $sApp_Dir & '\' & $aExec[$i] & '"' : '')
			Next
		EndIf
		
		Local $iPID = __APPCU_GetParentProcess(@AutoItPID)
		Local $hWnd = __APPCU_WinGetByProcess($iPID)
		
		If $hWnd Then
			WinClose($hWnd)
		EndIf
		
		If Not WinWaitClose($hWnd, '', 10) Or ProcessExists($iPID) Then
			ProcessClose($iPID)
		EndIf
		
		ProcessWaitClose($iPID, 10)
		
		Run(StringFormat('%s /C Ping -n 2 localhost > nul & xcopy /y /s /c "%s\*" "%s\*"%s & RD /S /Q "%s"', @ComSpec, $sUpdate_Dir, $sApp_Dir, $sExec_After_Update, $sUpdate_Dir), '', @SW_HIDE)
		
		Exit
	EndIf
	
	If $iSilentMode <= 0 Then
		MsgBox(64, $APPCU_APP_NAME & ' - ' & $APPCU_MSGS[$APPCU_MSG_UPDCHCK], StringFormat($APPCU_MSGS[$APPCU_MSG_USNGNEWSTVER], $APPCU_APP_VER), 0, $hParent)
	EndIf
	
	Return 1
EndFunc

#EndRegion Public Functions

#Region Internal Functions

Func __APPCU_WinGetByProcess($iPID)
	Local $aWins = WinList()
	
	For $i = 1 To UBound($aWins) - 1
		If BitAND(WinGetState($aWins[$i][1]), 2) And WinGetProcess($aWins[$i][1]) = $iPID Then
			Return $aWins[$i][1]
		EndIf
	Next
	
	Return 0
EndFunc

Func __APPCU_GetParentProcess($iPID = 0)
	If Not $iPID Then $iPID = @AutoItPID
	
	Local $hSnapshot = DllCall('kernel32.dll', 'handle', 'CreateToolhelp32Snapshot', 'dword', 0x00000002, 'dword', 0)
	If @error Or Not $hSnapshot[0] Then Return SetError(@error + 10, @extended, 0)
	
	Local $tagPROCESSENTRY32_ = 'dword Size;dword Usage;dword ProcessID;ulong_ptr DefaultHeapID;dword ModuleID;dword Threads;dword ParentProcessID;long PriClassBase;dword Flags;wchar ExeFile[260]'
	
	Local $tPROCESSENTRY32 = DllStructCreate($tagPROCESSENTRY32_)
	Local $iResult = 0
	
	$hSnapshot = $hSnapshot[0]
	DllStructSetData($tPROCESSENTRY32, 'Size', DllStructGetSize($tPROCESSENTRY32))
	Local $aRet = DllCall('kernel32.dll', 'bool', 'Process32FirstW', 'handle', $hSnapshot, 'struct*', $tPROCESSENTRY32)
	Local $iError = @error
	
	While (Not @error) And ($aRet[0])
		If DllStructGetData($tPROCESSENTRY32, 'ProcessID') = $iPID Then
			$iResult = DllStructGetData($tPROCESSENTRY32, 'ParentProcessID')
			ExitLoop
		EndIf
		
		$aRet = DllCall('kernel32.dll', 'bool', 'Process32NextW', 'handle', $hSnapshot, 'struct*', $tPROCESSENTRY32)
		$iError = @error
	WEnd
	
	DllCall("kernel32.dll", "bool", "CloseHandle", "handle", $hSnapshot)
	If Not $iResult Then Return SetError($iError, 0, 0)
	
	Return $iResult
EndFunc

Func __APPCU_GetStrInfo($sStr, $sInfo, $sDefault = '')
	Local $sRet = StringRegExpReplace($sStr, '(?si).*' & $sInfo & '=([^\r\n]+).*', '\1')
	
	If @extended = 0 Or $sRet = $sStr Or StringStripWS($sRet, 8) = '' Then
		Return SetError(1, 0, $sDefault)
	EndIf
	
	Return $sRet
EndFunc

Func __APPCU_DirBackupAsSrc($sSrcDir, $sRootDir, $sBckpDir)
	Local $sRelPath, $aDirList = __APPCU_DirListToArray($sSrcDir)
	
	If Not IsArray($aDirList) Then
		Return SetError(1, 0, 0)
	EndIf
	
	For $i = 1 To UBound($aDirList) - 1
		$sRelPath = StringReplace($aDirList[$i], $sSrcDir & '\', '')
		
		If Not StringInStr(FileGetAttrib($aDirList[$i]), 'D') Then
			FileCopy($sRootDir & '\' & $sRelPath, $sBckpDir & '\' & $sRelPath, 9)
		EndIf
	Next
EndFunc

Func __APPCU_DirListToArray($sPath)
	Local $iPID = Run(@ComSpec & ' /C dir "' & $sPath & '" /s /b', '', @SW_HIDE, 6)
	ProcessWaitClose($iPID)
	
	Local $sRead = StdoutRead($iPID)
	
	If @error Or Not $sRead Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $sBuffer = DllStructCreate('char[' & StringLen($sRead) + 1 & ']')
	Local $aRet = DllCall('User32.dll', 'int', 'OemToChar', 'str', $sRead, 'ptr', DllStructGetPtr($sBuffer))
	
	If Not IsArray($aRet) Or $aRet[0] = 0 Then
		Return SetError(2, 0, 0)
	EndIf
	
	$sRead = StringStripCR(StringStripWS(DllStructGetData($sBuffer, 1), 3))
	Return StringSplit($sRead, @LF)
EndFunc

Func __APPCU_VersCompare($sVer1, $sVer2)
	Local $aVer1 = StringSplit($sVer1, '.')
	$sVer1 = ''
	
	For $i = 1 To $aVer1[0]
		$sVer1 &= ($sVer1 ? '.' : '') & StringFormat('%02s', $aVer1[$i])
	Next
	
	Local $aVer2 = StringSplit($sVer2, '.')
	$sVer2 = ''
	
	For $i = 1 To $aVer2[0]
		$sVer2 &= ($sVer2 ? '.' : '') & StringFormat('%02s', $aVer2[$i])
	Next
	
	$sVer1 = StringRegExpReplace($sVer1 & '.0.0.0', '^(\d\d?\.\d\d?\.\d\d?\.\d\d?).*', '\1')
	$sVer2 = StringRegExpReplace($sVer2 & '.0.0.0', '^(\d\d?\.\d\d?\.\d\d?\.\d\d?).*', '\1')
	
	If $sVer1 = $sVer2 Then
		Return 0
	EndIf
	
	Return ($sVer1 > $sVer2 ? 1 : -1)
EndFunc

Func __APPCU_InetConnected($sPing_Host = 'google.com', $iPing_TimeOut = 1000)
	Local Const $INTERNET_CONNECTION_MODEM = 0x1
	Local Const $INTERNET_CONNECTION_LAN = 0x2
	Local Const $INTERNET_CONNECTION_PROXY = 0x4
	Local Const $INTERNET_CONNECTION_OFFLINE = 0x20
	Local Const $INTERNET_CONNECTION_CONFIGURED = 0x40
	
	Local $stInetGetConnectedState = DllStructCreate('int')
	
	DllCall('wininet.dll', 'int', 'InternetGetConnectedState', 'ptr', DllStructGetPtr($stInetGetConnectedState), 'dword', 0)
	
	Local $s_stDataState = DllStructGetData($stInetGetConnectedState, 1)
	Local $iRetState = 0
	
	If BitAND($s_stDataState, $INTERNET_CONNECTION_MODEM) Or BitAND($s_stDataState, $INTERNET_CONNECTION_PROXY) Then
		$iRetState = 1
	EndIf
	
	If BitAND($s_stDataState, $INTERNET_CONNECTION_OFFLINE) Then
		$iRetState = 0
	EndIf
	
	If $iRetState = 0 Then
		If Ping($sPing_Host, $iPing_TimeOut) > 0 Then
			$iRetState = 1
		EndIf
	EndIf
	
	Return $iRetState
EndFunc

Func __APPCU_NumberIsOverflow($nValue)
    Return (StringInStr($nValue, '.#INF') > 0) Or (StringInStr($nValue, 'e') > 0)
EndFunc

#EndRegion Internal Functions
