#Region Header

#CS
	Name:				Marker UDF
	Description:		This small library designed to assist with markers and flags handling. Library uses Scripting.Dictionary to handle the data.
						Advantages:
									* Different types of values can be used as marker.
									* Expression like <$oMarker('Item') = 'Value'> can be used to add values, and <$oMarker('Item')> to retrieve it.
	
	Author:				Copyright  2019 CreatoR's Lab (G.Sandler), www.creator-lab.ucoz.ru, www.autoit-script.ru. All rights reserved.
	AutoIt version:		3.3.10.2
	UDF version:		1.0
	
	History:
						v1.0
						* First public version.
	
#CE

#include-once

#EndRegion Header

#Region Global Variables

Global $sMarker_MarkerName = 'MaRkEr_NaMe'

#EndRegion Global Variables

#Region Public Functions

; #FUNCTION# ====================================================================================================================
; Name ..........: _Marker_Create
; Description ...: Creates marker.
; Syntax ........: _Marker_Create($sName)
; Parameters ....: $sName               - The name of the marker.
; Return values .: Success: Marker object.
;                  Failure: 0 and sets @error to as following:
;                                                                1 - $sName is invalid marker name (empty).
;                                                                2 - Unable to create Scripting.Dictionary object.
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: All other functions.
; Example .......: Yes
; ===============================================================================================================================
Func _Marker_Create($sName)
	If StringStripWS($sName, 8) = '' Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $oMarker = ObjCreate('Scripting.Dictionary')
	
	If Not IsObj($oMarker) Then
		Return SetError(2, 0, 0)
	EndIf
	
	_Marker_Add($oMarker, $sMarker_MarkerName, $sName)
	
	Return $oMarker
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Marker_Delete
; Description ...: Deletes key from the marker.
; Syntax ........: _Marker_Delete($oMarker, $sKey)
; Parameters ....: $oMarker             - Marker object as returned from _Marker_Create.
;                  $sKey                - Key to delete.
; Return values .: Success: Returns 1.
;                  Failure: 0 and sets @error as following:
;                                                            1 - $oMarker is invalid marker object.
;                                                            2 - $sKey was not found in the marker.
;                                                            3 - $sKey is the marker NAME key.
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: _Marker_Create, _Marker_Add
; Example .......: Yes
; ===============================================================================================================================
Func _Marker_Delete(ByRef $oMarker, $sKey)
	If Not IsObj($oMarker) Then
		Return SetError(1, 0, 0)
	EndIf
	
	If Not _Marker_Exists($oMarker, $sKey) Then
		Return SetError(2, 0, 0)
	EndIf
	
	If $sKey == $sMarker_MarkerName Then
		Return SetError(3, 0, 0)
	EndIf
	
	$oMarker.Remove($sKey)
	
	Return 1
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Marker_Destroy
; Description ...: Destroys the marker.
; Syntax ........: _Marker_Destroy($oMarker)
; Parameters ....: $oMarker             - Marker object as returned from _Marker_Create.
; Return values .: Success: Returns 1.
;                  Failure: 0 and sets @error to 1 if $oMarker is invalid marker object.
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: _Marker_Create
; Example .......: Yes
; ===============================================================================================================================
Func _Marker_Destroy(ByRef $oMarker)
	If Not IsObj($oMarker) Then
		Return SetError(1, 0, 0)
	EndIf
	
	$oMarker.RemoveAll
	$oMarker = 0
	
	Return 1
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Marker_Add
; Description ...: Adds value to the marker.
; Syntax ........: _Marker_Add($oMarker, $sKey[, $sValue = ''])
; Parameters ....: $oMarker             - Marker object as returned from _Marker_Create.
;                  $sKey                - Key to add.
;                  $vValue              - [optional] Value to set (default is '').
; Return values .: Success: Returns 1.
;                  Failure: 0 and sets @error as following:
;                                                           1 - $oMarker is invalid marker object.
;                                                           2 - $sKey is invalid key (empty).
;                                                           3 - $sKey already exists in the marker.
; Author ........: G.Sandler
; Remarks .......: An expression like <$oMarker('Item') = ''> can be used instead of this function, in this case it is the user's responsibility to check for errors and data in the marker.
; Related .......: _Marker_Create, _Marker_Delete
; Example .......: Yes
; ===============================================================================================================================
Func _Marker_Add(ByRef $oMarker, $sKey, $vValue = '')
	If Not IsObj($oMarker) Then
		Return SetError(1, 0, 0)
	EndIf
	
	If StringStripWS($sKey, 8) = '' Then
		Return SetError(2, 0, 0)
	EndIf
	
	If _Marker_Exists($oMarker, $sKey) Then
		Return SetError(3, 0, 0)
	EndIf
	
	$oMarker.Add($sKey, $vValue)
	
	Return 1
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Marker_Exists
; Description ...: Checks if specified key exists in the marker.
; Syntax ........: _Marker_Exists($oMarker, $sKey)
; Parameters ....: $oMarker             - Marker object as returned from _Marker_Create.
;                  $sKey                - Key to check.
; Return values .: Success: Returns True if found, and False if not.
;                  Failure: 0 and sets @error to 1 if $oMarker is invalid marker object.
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: _Marker_Create, _Marker_Add, _Marker_SetValue
; Example .......: Yes
; ===============================================================================================================================
Func _Marker_Exists($oMarker, $sKey)
	If Not IsObj($oMarker) Then
		Return SetError(1, 0, 0)
	EndIf
	
	Return $oMarker.Exists($sKey)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Marker_SetValue
; Description ...: Sets value for specified key in the marker.
; Syntax ........: _Marker_SetValue($oMarker, $sKey, $vValue)
; Parameters ....: $oMarker             - Marker object as returned from _Marker_Create.
;                  $sKey                - Key to set the value for.
;                  $vValue              - Value to set.
; Return values .: Success: Returns 1.
;                  Failure: 0 and sets @error as following:
;                                                            1 - $oMarker is invalid marker object.
;                                                            2 - $sKey not found in the marker.
;                                                            3 - $sKey is the marker NAME key.
; Author ........: G.Sandler
; Remarks .......: An expression like <$oMarker('Item') = 'Value'> can be used instead of this function, in this case it is the user's responsibility to check for errors and data in the marker.
; Related .......: _Marker_Create, _Marker_Add
; Example .......: Yes
; ===============================================================================================================================
Func _Marker_SetValue(ByRef $oMarker, $sKey, $vValue)
	If Not IsObj($oMarker) Then
		Return SetError(1, 0, 0)
	EndIf
	
	If Not _Marker_Exists($oMarker, $sKey) Then
		Return SetError(2, 0, 0)
	EndIf
	
	If $sKey == $sMarker_MarkerName Then
		Return SetError(3, 0, 0)
	EndIf
	
	$oMarker.Item($sKey) = $vValue
	
	Return 1
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Marker_GetValue
; Description ...: Gets marker key value.
; Syntax ........: _Marker_GetValue($oMarker, $sKey)
; Parameters ....: $oMarker             - Marker object as returned from _Marker_Create.
;                  $sKey                - Key to get the value from.
; Return values .: Success: Requested value.
;                  Failure: 0 and sets @error as following:
;                                                            1 - $oMarker is invalid marker object.
;                                                            2 - Invalid $iPos.
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: 
; Example .......: Yes
; ===============================================================================================================================
Func _Marker_GetValue($oMarker, $sKey)
	If Not IsObj($oMarker) Then
		Return SetError(1, 0, 0)
	EndIf
	
	Return $oMarker.Item($sKey)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Marker_GetCount
; Description ...: Gets count of keys in the marker.
; Syntax ........: _Marker_GetCount($oMarker)
; Parameters ....: $oMarker             - Marker object as returned from _Marker_Create.
; Return values .: Success: Cont of keys in the marker.
;                  Failure: 0 and sets @error to 1 if $oMarker is invalid marker object.
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: _Marker_Create, _Marker_Add, _Marker_Delete
; Example .......: Yes
; ===============================================================================================================================
Func _Marker_GetCount($oMarker)
	If Not IsObj($oMarker) Then
		Return SetError(1, 0, 0)
	EndIf
	
	Return $oMarker.Count - 1
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _Marker_GetMarkerName
; Description ...: Gets the marker's name.
; Syntax ........: _Marker_GetMarkerName($oMarker)
; Parameters ....: $oMarker             - Marker object as returned from _Marker_Create.
; Return values .: Success: Marker name.
;                  Failure: 0 and sets @error to 1 if $oMarker is invalid marker object.
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: _Marker_Create
; Example .......: Yes
; ===============================================================================================================================
Func _Marker_GetMarkerName($oMarker)
	If Not IsObj($oMarker) Then
		Return SetError(1, 0, 0)
	EndIf
	
	Return $oMarker.Item($sMarker_MarkerName)
EndFunc

#EndRegion Public Functions
