#Region Header

#CS
	Name:				GUIDynamicMenu.au3
	Description:		UDF designed to allow loading dynamic menu from ini-file.
	Author:				Copyright  2019 CreatoR's Lab (G.Sandler), www.creator-lab.ucoz.ru, www.autoit-script.ru. All rights reserved.
	AutoIt version:		3.3.14.5
	
	Remarks:			* This library registers $WM_COMMAND, if you use it in your script, or it's used in the includes that used by your script,
						you should add the following code to begining to WM_CCOMMAND function:
						
						$vRet = __GDM_MENU_HANDLER($hWnd, $iMsg, $wParam, $lParam)
						
						If $vRet <> $GUI_RUNDEFMSG Then
							Return $vRet
						EndIf
	
						* Ini file menu structure/syntax:
													[MENU]
													Item, "ITEM_NAME" = DISABLED
													--- = 0
													Item, "ITEM_NAME" = COMMAND, "PATH/URL", "ARGUMENTS", "ICON_FILE", ICON_ID, "HOTKEY"
													Submenu, "MENU_NAME" = SECTION_NAME, "EXTERNAL_INI_PATH", "ICON_FILE", ICON_ID
													Include = SECTION_NAME, "EXTERNAL_INI_PATH"
						
							Explanations:
													[MENU] --> Section name:
														* Should use only ansi characters.
													
													Item, ... --> Disabled item:
														"ITEM_NAME" 		- Displayed text for disabled menu item.
														DISABLED			- Tells to menu to set item as disabled, only this instruction must be used.
													
													--- = ... --> Separator item:
														* Must use 3 dashes in order to specify a separator, after the "=" sign can be anything.
													
													Item, ... --> Menu item:
														"ITEM_NAME" 		- Displayed text for menu item.
														COMMAND 			- [optional] Command to execute on item trigger, supported commands:
																																		Execute 		= executes specified program or file (PATH/URL).
																																		Execute Line 	= executes AutoIt single line.
																																		Go to Page		= Opens specified URL (PATH/URL) in default browser.
																																		Send			= Sends text (can be used to trigger system hotkeys).
																																		Exit			= Programmatically exits the program.
														"PATH/URL" 			- File path or URL to use in specified command (COMMAND).
														"ARGUMENTS" 		- Arguments to pass to executed path ("PATH/URL") - only for <Execute> command.
														"ICON_FILE" 		- [optional] Icon file to use for menu item.
														ICON_ID 			- [optional] Icon id in icon file ("ICON_FILE").
														"HOTKEY" 			- [optional] Hotkey to trigger menu item (in format of "Ctrl+Shift+Q").
														"HOTKEY_IS_GLOBAL" 	- [optional] Defines hotkey scope.
																						0 to set hotkey only for current window.
																						1 to set it globaly.
																						-1 or ommit to use default behaviour (0 or as set by _GUICtrlDynamicMenu_SetHotKeysGlobal)
													
													Submenu, ... --> Submenu item:
														"MENU_NAME" 		- Displayed text for submenu item.
														SECTION_NAME		- Section name to read the submenu from.
														"EXTERNAL_INI_PATH"	- [optional] External ini-file to read the section from.
														"ICON_FILE"			- [optional] Icon file to use for submenu item.
														ICON_ID				- [optional] Icon id in icon file ("ICON_FILE").
													
													Include, ... --> Includes raw menu in current position:
														SECTION_NAME		- Section name to read the submenu from.
														"EXTERNAL_INI_PATH"	- [optional] External ini-file to read the section from.
													
	History:
						0.3
						* Fixed issue with passing argument as "" to Execute command.
						* Better error handling in _GUICtrlDynamicMenu_Destroy.
						* Example changed.
						
						0.2
						+ Added _GUICtrlDynamicMenu_SetIconSize function to set icon size for menu item (needs menu reload to apply if used after menu load).
						+ Added _GUICtrlDynamicMenu_SetHotKeysGlobal function to determine if hotkeys will be recognised globaly or only for current GUI (overided by menu item, see menu ini file structure above).
						+ Added $GDM_INFO_HK_ISGLOBAL user variable to use with _GUICtrlDynamicMenu_GetInfo.
						* Fixed issue with global hotkeys, they were blocked for rest programms even if not used by the menu.
						* Fixed issue with empty parameters (in quotes) in menu item.
						* Fixed issue with relative path (wrong working dir was set).
						* Fixed issue with "Execute Line" command in menu.
						* Example changed.
						
						0.1
						* First public version.
#CE

#include-once
#Include <Array.au3>
#Include <APIConstants.au3>
#Include <GUIMenu.au3>
#Include <WinAPIEx.au3>

#EndRegion Header

#Region Public Variables

;Constants to use with _GUICtrlDynamicMenu_GetInfo
Global Enum $GDM_INFO_HWND, $GDM_INFO_HMENU, $GDM_INFO_ITEMID, $GDM_INFO_COMMAND, $GDM_INFO_PATH, $GDM_INFO_ARGS, $GDM_INFO_ICONFILE, $GDM_INFO_ICONID, $GDM_INFO_HOTKEY, $GDM_INFO_HK_ISGLOBAL

#EndRegion

#Region Global Variables

Global $iGDM_MenuIcon_Size = 16
Global $iGDM_HotKeys_Global = False

;THE ORDER IS IMPORTANT
Global Enum $iGDM_hWnd, $iGDM_hMenu, $iGDM_ItemID, $iGDM_Command, $iGDM_Path, $iGDM_Args, $iGDM_IconFile, $iGDM_IconID, $iGDM_Hotkey, $iGDM_HKIsGlobal, _
	$iGDM_Total

Global Enum $iGDM_SprtdCmd_Command, $iGDM_SprtdCmd_Func, $iGDM_SprtdCmd_UseArgs, _
	$iGDM_SprtdCmd_Total

Global Enum $iGDM_SprtdItem_Sprtr, $iGDM_SprtdItem_Incld, $iGDM_SprtdItem_SubMenu, $iGDM_SprtdItem_Item, _
	$iGDM_SprtdItem_Total

Global $aGDM_Items[1][$iGDM_Total]

Global $aGDM_SprtdCmds[][$iGDM_SprtdCmd_Total] = _
	[ _
		['Execute', __GDM_ShellExecuteEx, True], _
		['Execute Line', __GDM_ExecuteLine, False], _
		['Go to Page', __GDM_ShellExecuteEx, False], _
		['Send', Send, False], _
		['Exit', __GDM_Exit, False] _
	]

Global $iGDM_SprtdCmd_Count = UBound($aGDM_SprtdCmds)
Global $aGDM_SprtdItems[$iGDM_SprtdItem_Total] = ['---', 'Include', 'Submenu', 'Item']

#EndRegion Global Variables

#Region Options

GUIRegisterMsg($WM_COMMAND, '__GDM_MENU_HANDLER')

#EndRegion Options

#Region Public Functions

; #FUNCTION# ====================================================================================================================
; Name ..........: _GUICtrlDynamicMenu_Create
; Description ...: Creates (or gets handle for) main menu.
; Syntax ........: _GUICtrlDynamicMenu_Create([$iID = 0])
; Parameters ....: $iID                 - [optional] GUI menu identifier, if not set - Popup menu created. Default is 0.
; Return values .: Success: Handle to popup menu or specified menu identifier.
;                  Failure: 0 and sets @error to 1 if $iID set and it's not menu id.
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: _GUICtrlDynamicMenu_Destroy, _GUICtrlDynamicMenu_Load
; Example .......: Yes
; ===============================================================================================================================
Func _GUICtrlDynamicMenu_Create($iID = 0)
	If $iID Then
		$iID = GUICtrlGetHandle($iID)
		
		If Not _GUICtrlMenu_IsMenu($iID) Then
			Return SetError(1, 0, 0)
		EndIf
		
		Return $iID
	EndIf
	
	Return _GUICtrlMenu_CreatePopup()
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GUICtrlDynamicMenu_Destroy
; Description ...: Destroys menu created with _GUICtrlDynamicMenu_Create (when $iID = 0), should be called before script exists to release resources.
; Syntax ........: _GUICtrlDynamicMenu_Destroy($hMenu)
; Parameters ....: $hMenu               - Handle to menu as returned by _GUICtrlDynamicMenu_Create, or _GUICtrlDynamicMenu_Load function, in this case menu is unloaded (using _GUICtrlDynamicMenu_UnLoad) and destroyed.
; Return values .: Success: True
;                  Failure: False
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: _GUICtrlDynamicMenu_Create, _GUICtrlDynamicMenu_UnLoad
; Example .......: Yes
; ===============================================================================================================================
Func _GUICtrlDynamicMenu_Destroy($hMenu)
	If UBound($hMenu, 2) = $iGDM_Total Then
		Local $iRet = Int(_GUICtrlDynamicMenu_UnLoad($hMenu)) + Int(_GUICtrlMenu_DestroyMenu($hMenu[1][$iGDM_hMenu]))
		Return ($iRet = 2)
	EndIf
	
	Return _GUICtrlMenu_DestroyMenu($hMenu)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GUICtrlDynamicMenu_Load
; Description ...: Loads Dynamic Menu from file.
; Syntax ........: _GUICtrlDynamicMenu_Load($hGUI, $hMenu, $sIniFile[, $sSectName = ''])
; Parameters ....: $hGUI                - Handle to main gui.
;                  $hMenu               - Handle to menu as returned by _GUICtrlDynamicMenu_Create.
;                  $sIniFile            - File to load menu from. Read UDF header to understand the hierarchy of such menu file.
;                  $sSectName           - [optional] Section name in $sIniFile to read from, if not set (default), reads all sections in menu.
; Return values .: Success: Handle (array actualy) to loaded menu. Can be used to get menu data or by _GUICtrlDynamicMenu_UnLoad.
;                  Failure: 0 and sets @error as following:
;                                                              1 - $sIniFile not exists.
;                                                              2 - $sSectName is not set (empty string) and unable to read sections from $sIniFile file.
;                                                              3 - $hMenu is not a menu handler.
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: _GUICtrlDynamicMenu_UnLoad, _GUICtrlDynamicMenu_Create
; Example .......: Yes
; ===============================================================================================================================
Func _GUICtrlDynamicMenu_Load($hGUI, $hMenu, $sIniFile, $sSectName = '')
	Local $aSect_Names, $aMenu_Sect, $sIconFile, $iIconID, $sSubMenuName, $sSubMenuSect
	Local $aSplit, $hSubMenuID, $sCommand, $sItemName, $sItemPath, $sItemArgs, $sItemHotkey
	Local $aRet[1][$iGDM_Total], $aSubRet, $iItemID
	
	If Not FileExists($sIniFile) Then
		Return SetError(1, 0, 0)
	EndIf
	
	If $sSectName = '' Then
		$aSect_Names = IniReadSectionNames($sIniFile)
		
		If Not IsArray($aSect_Names) Then
			Return SetError(2, 0, 0)
		EndIf
	Else
		Dim $aSect_Names[2] = [1, $sSectName]
	EndIf
	
	If Not _GUICtrlMenu_IsMenu($hMenu) Then
		Return SetError(3, 0, 0)
	EndIf
	
	For $iSect = 1 To $aSect_Names[0]
		$aMenu_Sect = IniReadSection($sIniFile, $aSect_Names[$iSect])
		
		If Not IsArray($aMenu_Sect) Then
			ContinueLoop
		EndIf
		
		For $i = 1 To $aMenu_Sect[0][0]
			Dim $sItemPath = '', $sItemArgs = '', $sIconFile = '', $iIconID = 0, $sItemHotkey = '', $sCommand = ''
			
			$aSplit = __GDM_StringSplitIgnoreQuoted($aMenu_Sect[$i][0], ',', True)
			
			Switch $aSplit[1]
				Case $aGDM_SprtdItems[$iGDM_SprtdItem_Sprtr]
					__GDM_ArrayAddElmnts($aRet, $hGUI, $hMenu, __GDM_AddMenu('', $hMenu, 0, 0, True, $MF_SEPARATOR))
				Case $aGDM_SprtdItems[$iGDM_SprtdItem_Incld]
					$aSplit = __GDM_StringSplitIgnoreQuoted($aMenu_Sect[$i][1], ',', True)
					$sSubMenuSect = $aSplit[1]
					$sItemPath = $aSplit[2]
					
					If Not FileExists($sItemPath) Then
						$sItemPath = $sIniFile
					EndIf
					
					$aSubRet = _GUICtrlDynamicMenu_Load($hGUI, $hMenu, $sItemPath, $sSubMenuSect)
					
					If Not @error Then
						For $j = 1 To $aSubRet[0][0]
							__GDM_ArrayAddElmnts($aRet, _
								$aSubRet[$j][$iGDM_hWnd], $aSubRet[$j][$iGDM_hMenu], $aSubRet[$j][$iGDM_ItemID], $aSubRet[$j][$iGDM_Command], $aSubRet[$j][$iGDM_Path], _
								$aSubRet[$j][$iGDM_Args], $aSubRet[$j][$iGDM_IconFile], $aSubRet[$j][$iGDM_IconID], $aSubRet[$j][$iGDM_Hotkey], $aSubRet[$j][$iGDM_HKIsGlobal])
						Next
					EndIf
				Case $aGDM_SprtdItems[$iGDM_SprtdItem_SubMenu]
					$sSubMenuName = $aSplit[2]
					
					If $sSubMenuName = '' Then
						ContinueLoop
					EndIf
					
					$aSplit = __GDM_StringSplitIgnoreQuoted($aMenu_Sect[$i][1], ',', True)
					
					$sSubMenuSect = $aSplit[1]
					$sIconFile = ($aSplit[2] = '' ? @AutoItExe : $aSplit[2])
					$iIconID = Number($aSplit[3])
					$sItemPath = $aSplit[4]
					
					$hSubMenuID = __GDM_AddMenu($sSubMenuName, $hMenu, $sIconFile, $iIconID, False)
					__GDM_ArrayAddElmnts($aRet, $hGUI, $hMenu, @extended, '', $sIconFile, $iIconID) 
					
					If Not FileExists($sItemPath) Then
						$sItemPath = $sIniFile
					EndIf
					
					$aSubRet = _GUICtrlDynamicMenu_Load($hGUI, $hSubMenuID, $sItemPath, $sSubMenuSect)
					
					If Not @error Then
						For $j = 1 To $aSubRet[0][0]
							__GDM_ArrayAddElmnts($aRet, _
								$aSubRet[$j][$iGDM_hWnd], $aSubRet[$j][$iGDM_hMenu], $aSubRet[$j][$iGDM_ItemID], $aSubRet[$j][$iGDM_Command], $aSubRet[$j][$iGDM_Path], _
								$aSubRet[$j][$iGDM_Args], $aSubRet[$j][$iGDM_IconFile], $aSubRet[$j][$iGDM_IconID], $aSubRet[$j][$iGDM_Hotkey], $aSubRet[$j][$iGDM_HKIsGlobal])
						Next
					EndIf
				Case $aGDM_SprtdItems[$iGDM_SprtdItem_Item]
					$sItemName = $aSplit[2]
					
					If $sItemName = '' Then
						ContinueLoop
					EndIf
					
					$aSplit = __GDM_StringSplitIgnoreQuoted($aMenu_Sect[$i][1], ',', True)
					
					If $aSplit[1] = 'DISABLED' Then
						__GDM_ArrayAddElmnts($aRet, $hGUI, $hMenu, __GDM_AddMenu($sItemName, $hMenu, 0, 0, True, BitOR($MF_STRING, $MF_GRAYED)))
						ContinueLoop
					EndIf
					
					$sCommand = $aSplit[1]
					$sItemPath = __GDM_FileSearchPath($aSplit[2])
					$sItemArgs = $aSplit[3]
					$sIconFile = ($aSplit[4] = '' ? ((Not $sItemPath Or Not FileExists($sItemPath)) ? @AutoItExe : $sItemPath) : $aSplit[4])
					$iIconID = Number($aSplit[5])
					$sItemHotkey = StringStripWS($aSplit[6], 3)
					$fHotkeyIsGlobal = ($aSplit[7] ? Int($aSplit[7]) : $aSplit[7])
					
					If $sItemHotkey <> '' Then
						$sItemName &= @TAB & '(' & StringUpper($sItemHotkey) & ')'
						$sItemHotkey = __GDM_GetHotKeyCompatibleString(StringLower($sItemHotkey))
						
						HotKeySet($sItemHotkey, '__GDM_HK_Event_Handler')
					EndIf
					
					$iItemID = __GDM_AddMenu($sItemName, $hMenu, $sIconFile, $iIconID, True)
					
					__GDM_ArrayAddElmnts($aGDM_Items, $hGUI, $hMenu, $iItemID, $sCommand, $sItemPath, $sItemArgs, $sIconFile, $iIconID, $sItemHotkey, $fHotkeyIsGlobal)
					__GDM_ArrayAddElmnts($aRet, $hGUI, $hMenu, $iItemID, $sCommand, $sItemPath, $sItemArgs, $sIconFile, $iIconID, $sItemHotkey, $fHotkeyIsGlobal)
			EndSwitch
		Next
	Next
	
	Return $aRet
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GUICtrlDynamicMenu_UnLoad
; Description ...: Unloads menu (deletes menu items and release data) loaded with _GUICtrlDynamicMenu_Load.
; Syntax ........: _GUICtrlDynamicMenu_UnLoad(Byref $hDynamicMenu)
; Parameters ....: $hDynamicMenu        - DynamicMenu handle as returned by _GUICtrlDynamicMenu_Load.
; Return values .: Success: 1
;                  Failure: 0 and sets @error to 1.
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: _GUICtrlDynamicMenu_Load
; Example .......: Yes.
; ===============================================================================================================================
Func _GUICtrlDynamicMenu_UnLoad($hDynamicMenu)
	If UBound($hDynamicMenu, 2) < 2 Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $iFind
	
	For $i = 1 To $hDynamicMenu[0][0]
		_GUICtrlMenu_DeleteMenu($hDynamicMenu[$i][$iGDM_hMenu], $hDynamicMenu[$i][$iGDM_ItemID], False)
		
		$iFind = _ArraySearch($aGDM_Items, $hDynamicMenu[$i][$iGDM_ItemID], 1, $aGDM_Items[0][0], 0, 2, 1, $iGDM_ItemID)
		
		If $iFind <> -1 Then
			_ArrayDelete($aGDM_Items, $iFind)
			$aGDM_Items[0][0] -= 1
		EndIf
	Next
	
	If UBound($aGDM_Items) = 0 Then
		Dim $aGDM_Items[1][$iGDM_Total]
	EndIf
	
	Return 1
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GUICtrlDynamicMenu_Show
; Description ...: Shows loaded menu as created with _GUICtrlDynamicMenu_Create.
; Syntax ........: _GUICtrlDynamicMenu_Show($hGUI, $hMenu, $iControlID[, $bUnderMouse = False[, $iNotify = 0[, $iButtons = 0[, $bAboveCtrl = False]]]])
; Parameters ....: $hGUI                - Handle to main gui.
;                  $hMenu               - Handle to menu as returned by _GUICtrlDynamicMenu_Create.
;                  $iControlID          - [optional] Control id to show the menu under. -1 (default) can be used as last control id in GUI. If not set or id not found menu shown in mouse position.
;                  $bUnderMouse         - [optional] If True, then menu shown in mouse position ($iControlID ignored). Default is False.
;                  $iNotify             - [optional] Use to determine the selection without parent window:
;                                                                                            0 - Use built-in events handler (default).
;                                                                                            1 - Do not send notification messages.
;                                                                                            2 - Return the menu item identifier of the user's selection. In this case it is user's responsibility to handle item event (use _GUICtrlDynamicMenu_GetInfo).
;                  $iButtons            - [optional] Mouse button the shortcut menu tracks:
;                                                                                            0 - The user can select items with only the left mouse button (default).
;                                                                                            1 - The user can select items with both left and right buttons .
;                  $bAboveCtrl          - [optional] If True, the menu shown above the control (if $iControlID <> -1), default is False - show under control.
; Return values .: Success: If $iNotify is set to 2, the return value is the menu item identifier of the item that the user selected. If the user cancels the menu without making a selection or if an error occurs, then the return value is 0.
;                  Failure: 0 and sets @error to 1 in case $iControlID is set and there was a problem to get it's position.
; Author ........: G.Sandler
; Remarks .......: The function will return only after menu is closed.
; Related .......: _GUICtrlDynamicMenu_Create, _GUICtrlDynamicMenu_GetInfo
; Example .......: Yes
; ===============================================================================================================================
Func _GUICtrlDynamicMenu_Show($hGUI, $hMenu, $iControlID = -1, $bUnderMouse = False, $iNotify = 0, $iButtons = 0, $bAboveCtrl = False)
	If Not _GUICtrlMenu_IsMenu($hMenu) Then
		$hMenu = GUICtrlGetHandle($hMenu)
	EndIf
	
	If $iControlID = -1 Then
		$iControlID = _WinAPI_GetDlgCtrlID(GUICtrlGetHandle(-1))
	EndIf
	
	If $bUnderMouse Or Not $iControlID Then
		Return _GUICtrlMenu_TrackPopupMenu($hMenu, $hGUI, -1, -1, 1, 1, $iNotify, $iButtons)
	EndIf
	
	Local $aCtrlPos = ControlGetPos($hGUI, '', $iControlID)
	
	If @error Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $stPoint = DllStructCreate('int;int')
	
    DllStructSetData($stPoint, 1, $aCtrlPos[0])
    DllStructSetData($stPoint, 2, $aCtrlPos[1] + ($bAboveCtrl ? 0 : $aCtrlPos[3]))
	
	_WinAPI_ClientToScreen($hGUI, $stPoint)
	
	Return _GUICtrlMenu_TrackPopupMenu($hMenu, $hGUI, DllStructGetData($stPoint, 1), DllStructGetData($stPoint, 2), 1, 1, $iNotify, $iButtons)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GUICtrlDynamicMenu_GetInfo
; Description ...: Gets info from DynamicMenu.
; Syntax ........: _GUICtrlDynamicMenu_GetInfo($hDynamicMenu[, $iItemID = -1])
; Parameters ....: $hDynamicMenu        - DynamicMenu handle as returned by _GUICtrlDynamicMenu_Load.
;                  $iItemID             - [optional] Menu Item identifier, can be used a return from _GUICtrlDynamicMenu_Show with $iNotify = 2. Default is -1 - get info for all menu items.
; Return values .: Success: If $iItemID is set (<> -1), returns 1D array of item info as following:
;                                                     [$GDM_INFO_HWND]          - Handle to main gui window.
;                                                     [$GDM_INFO_HMENU]         - Handle to main menu.
;                                                     [$GDM_INFO_ITEMID]        - Item ID.
;                                                     [$GDM_INFO_COMMAND]       - Command.
;                                                     [$GDM_INFO_PATH]          - Path to use in command.
;                                                     [$GDM_INFO_ARGS]          - Arguments for Path ([4]).
;                                                     [$GDM_INFO_ICONFILE]      - Icon file.
;                                                     [$GDM_INFO_ICONID]        - Icon ID.
;                                                     [$GDM_INFO_HOTKEY]        - Hotkey.
;                                                     [$GDM_INFO_HK_ISGLOBAL]   - Hotkey is Global.
;                           If $iItemID = -1 then return is 2D array where each row have item info as shown above (the same as using the $hDynamicMenu itself).
;                  Failure: 0 and sets @error to 1.
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: _GUICtrlDynamicMenu_Show, _GUICtrlDynamicMenu_Load
; Example .......: Yes
; ===============================================================================================================================
Func _GUICtrlDynamicMenu_GetInfo($hDynamicMenu, $iItemID = -1)
	If $iItemID = -1 Then
		Return $hDynamicMenu
	EndIf
	
	Local $aRet[$iGDM_Total + 1]
	
	For $i = 1 To $hDynamicMenu[0][0]
		If $hDynamicMenu[$i][$iGDM_ItemID] = $iItemID Then
			For $j = 0 To $iGDM_Total - 1
				$aRet[$j] = $hDynamicMenu[$i][$j]
			Next
			
			Return $aRet
		EndIf
	Next
	
	Return SetError(1, 0, 0)
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GUICtrlDynamicMenu_SetHotKeysGlobal
; Description ...: Sets HotKey global state (scope).
; Syntax ........: _GUICtrlDynamicMenu_SetHotKeysGlobal([$fGlobal = False])
; Parameters ....: $fGlobal             - [optional] If True, the hotkeys will be recognised globaly (overided by menu item, see menu ini file structure description in udf header). Default is False.
; Return values .: Old state.
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: 
; Example .......: Yes
; ===============================================================================================================================
Func _GUICtrlDynamicMenu_SetHotKeysGlobal($fGlobal = False)
	Local $iOld = $iGDM_HotKeys_Global
	$iGDM_HotKeys_Global = ($fGlobal == True)
	Return $iOld
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GUICtrlDynamicMenu_SetIconSize
; Description ...: Sets menu item icon size.
; Syntax ........: _GUICtrlDynamicMenu_SetIconSize($iSize)
; Parameters ....: $iSize               - Icon size (default is 16). Minimum icon size is 16, maximum is 48.
; Return values .: None
; Author ........: G.Sandler
; Remarks .......: IMPORTANT: If this function called after the menu was loaded, the menu should be reloaded to apply changes.
; Related .......: 
; Example .......: Yes
; ===============================================================================================================================
Func _GUICtrlDynamicMenu_SetIconSize($iSize = 16)
	$iGDM_MenuIcon_Size = (($iSize < 16 Or $iSize > 48) ? 16 : $iSize)
EndFunc

#EndRegion Public Functions

#Region Internal Functions

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Description ...: Adds elements to array.
; Syntax ........: __GDM_ArrayAddElmnts(Byref $aArr, $vElmnt1, $vElmnt2 ... $vElmnt10)
; Author ........: G.Sandler
; ===============================================================================================================================
Func __GDM_ArrayAddElmnts(ByRef $aArr, $vElmnt1, $vElmnt2, $vElmnt3 = '', $vElmnt4 = '', $vElmnt5 = '', $vElmnt6 = '', $vElmnt7 = '', $vElmnt8 = '', $vElmnt9 = '', $vElmnt10 = '')
	If Not IsArray($aArr) Then
		Return SetError(1, 0, 0)
	EndIf
	
	$aArr[0][0] += 1
	ReDim $aArr[$aArr[0][0] + 1][$iGDM_Total]
	
	For $i = 0 To $iGDM_Total - 1
		$aArr[$aArr[0][0]][$i] = Eval('vElmnt' & $i + 1)
	Next
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Description ...: Add items to menu wrapper.
; Syntax ........: __GDM_AddMenu($sText, $hMenuID, $sIcon, $iIconID[, $bItem = True[, $iFlags = 0]])
; Author ........: G.Sandler
; ===============================================================================================================================
Func __GDM_AddMenu($sText, $hMenuID, $sIcon, $iIconID, $bItem = True, $iFlags = 0)
	Local $iItem, $hSubMenu
	Local Static $iIDCount = 999
	
	$iIDCount += 1
	
	If $bItem Then
		If $iFlags Then
			$iItem = _GUICtrlMenu_AppendMenu($hMenuID, $iFlags, $iIDCount, $sText)
		Else
			$iItem = _GUICtrlMenu_AddMenuItem($hMenuID, $sText, $iIDCount)
		EndIf
	Else
		$hSubMenu = _GUICtrlMenu_CreateMenu()
		$iItem = _GUICtrlMenu_AddMenuItem($hMenuID, $sText, $iIDCount, $hSubMenu)
	EndIf
	
	If $sIcon Then
		_GUICtrlMenu_SetItemBmp($hMenuID, $iItem, _WinAPI_Create32BitHBITMAP(_WinAPI_ShellExtractIcon($sIcon, $iIconID, $iGDM_MenuIcon_Size, $iGDM_MenuIcon_Size), True, True))
	EndIf
	
	Return ($bItem ? $iIDCount : SetExtended($iIDCount, $hSubMenu))
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Description ...: Splits string while ignoring quoted subtext.
; Syntax ........: __GDM_StringSplitIgnoreQuoted($sStr, $sDelim[, $bStripQuotes = True[, $iRetDim = 10]])
; Author ........: G.Sandler
; ===============================================================================================================================
Func __GDM_StringSplitIgnoreQuoted($sStr, $sDelim, $bStripQuotes = True, $iRetDim = 10)
	Local $aQuote1, $aQuote2, $aQuotes, $aQMarkers[1], $aSplit, $aRet[$iRetDim], $sRet
	
	$aQuote1 = StringRegExp('0' & $sStr, '(^0|"[^"]*")', 3)
	$aQuote2 = StringRegExp('0' & $sStr, "(^0|'[^']*')", 3)
	
	_ArrayConcatenate($aQuote1, $aQuote2, 1)
	
	$aQuotes = $aQuote1
	$aQuotes[0] = UBound($aQuotes) - 1
	
	For $i = 1 To $aQuotes[0]
		$aQMarkers[0] += 1
		ReDim $aQMarkers[$aQMarkers[0] + 1]
		$aQMarkers[$aQMarkers[0]] = 'QUOTES'
		
		While StringInStr($sStr, $aQMarkers[$aQMarkers[0]], 2)
			$aQMarkers[$aQMarkers[0]] &= Random(10000, 99999)
		WEnd
		
		$sStr = StringReplace($sStr, $aQuotes[$i], $aQMarkers[$aQMarkers[0]], 1, 1)
	Next
	
	$aSplit = StringSplit($sStr, $sDelim)
	
	For $i = 1 To $aSplit[0]
		$aSplit[$i] = StringStripWS($aSplit[$i], 3)
		$sRet = $aSplit[$i]
		
		For $j = 1 To $aQMarkers[0]
			If $aSplit[$i] == $aQMarkers[$j] Then
				$sRet = ($bStripQuotes ? StringRegExpReplace($aQuotes[$j], '^["'']+|["'']+$', '') : $aQuotes[$j])
				ExitLoop
			EndIf
		Next
		
		$aRet[0] += 1
		$aRet[$aRet[0]] = $sRet
	Next
	
	Return $aRet
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Description ...: Checks if passed item text match predefined supported command.
; Syntax ........: __GDM_IsSupportedItemCommand($sCommand)
; Author ........: G.Sandler
; ===============================================================================================================================
Func __GDM_IsSupportedItemCommand($sCommand)
	For $i = 0 To $iGDM_SprtdCmd_Count - 1
		If $aGDM_SprtdCmds[$i][$iGDM_SprtdCmd_Command] = $sCommand Then
			Return 1
		EndIf
	Next
	
	Return 0
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Description ...: Gets compatible hotkey string (converts "ctrl+shift+a" into "^+a").
; Syntax ........: __GDM_GetHotKeyCompatibleString($sHotkey)
; Author ........: G.Sandler
; ===============================================================================================================================
Func __GDM_GetHotKeyCompatibleString($sHotkey)
	Local $sModifiers, $sKeys
	
	$sHotkey = StringRegExpReplace($sHotkey, '[\+\s]', '')
	$sHotkey = StringReplace($sHotkey, 'ctrl', '^', 0, 2)
	$sHotkey = StringReplace($sHotkey, 'shift', '+', 0, 2)
	$sHotkey = StringReplace($sHotkey, 'alt', '!', 0, 2)
	$sHotkey = StringReplace($sHotkey, 'win', '#', 0, 2)
	
	$sModifiers = StringRegExpReplace($sHotkey, '[^\^\+!#]', '')
	$sKeys = StringRegExpReplace($sHotkey, '[\^\+!#]*', '')
	
	If $sKeys <> '' And StringLen($sKeys) > 1 Then
		$sKeys = '{' & $sKeys & '}'
	EndIf
	
	Return $sModifiers & $sKeys
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Description ...: Searches for a matching file using a registry-dependent system search path.
; Syntax ........: __GDM_FileSearchPath($sFile)
; Author ........: G.Sandler
; ===============================================================================================================================
Func __GDM_FileSearchPath($sFile)
	Local $sRet = _WinAPI_SearchPath($sFile)
	
	If Not @error Then
		Return $sRet
	EndIf
	
	Return SetError(1, 0, $sFile)
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Description ...: Performs a default system operation on a specified file.
; Syntax ........: __GDM_ShellExecuteEx($sCmd[, $sArgs = ''[, $sFolder = ''[, $sVerb = ''[, $iState = @SW_SHOWNORMAL[, $hWnd = 0]]]]])
; Author ........: G.Sandler
; ===============================================================================================================================
Func __GDM_ShellExecuteEx($sCmd, $sArgs = '', $sFolder = '', $sVerb = '', $iState = @SW_SHOWNORMAL, $hWnd = 0)
	Local $tINFO = DllStructCreate('long;long;long;ptr;ptr;ptr;ptr;long;long;long;ptr;long;long;long;long')
	Local $tVerb = DllStructCreate('char[15];char')
	Local $tPath = DllStructCreate('char[255];char')
	Local $tArgs = DllStructCreate('char[255];char')
	Local $tWDir = DllStructCreate('char[255];char')
	
	DllStructSetData($tVerb, 1, $sVerb)
	
	If StringRight($sCmd, 3) = 'lnk' Then
		Local $aShortcutInfo = FileGetShortcut($sCmd)
		
		If IsArray($aShortcutInfo) Then
			DllStructSetData($tPath, 1, $aShortcutInfo[0])
			DllStructSetData($tWDir, 1, $aShortcutInfo[1])
			DllStructSetData($tArgs, 1, $aShortcutInfo[2])
			$iState = $aShortcutInfo[6]
		Else
			Return SetError(1, 0, 0)
		EndIf
	Else
		DllStructSetData($tPath, 1, $sCmd)
		DllStructSetData($tWDir, 1, $sFolder)
		DllStructSetData($tArgs, 1, $sArgs)
	EndIf

	DllStructSetData($tINFO, 1, DllStructGetSize($tINFO))
	DllStructSetData($tINFO, 2, BitOR(0xC, 0x40, 0x400))
	DllStructSetData($tINFO, 3, $hWnd)
	DllStructSetData($tINFO, 4, DllStructGetPtr($tVerb))
	DllStructSetData($tINFO, 5, DllStructGetPtr($tPath))
	DllStructSetData($tINFO, 6, DllStructGetPtr($tArgs))
	DllStructSetData($tINFO, 7, DllStructGetPtr($tWDir))
	DllStructSetData($tINFO, 8, $iState)
	
	Local $aRet = DllCall('shell32.dll', 'int', 'ShellExecuteEx', 'ptr', DllStructGetPtr($tINFO))
	If @error Then Return SetError(2, 0, 0)
	
	Return $aRet[0]
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Description ...: Execute AutoIt script single line.
; Syntax ........: __GDM_ExecuteLine($sLine)
; Author ........: G.Sandler
; ===============================================================================================================================
Func __GDM_ExecuteLine($sLine)
	Run(@AutoItExe & ' /AutoIt3ExecuteLine "' & StringReplace($sLine, '"', "'") & '"')
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Description ...: Event handler wrapper when menu item was clicked.
; Syntax ........: __GDM_Item_Event_Handler($iGDM)
; Author ........: G.Sandler
; ===============================================================================================================================
Func __GDM_Item_Event_Handler($iGDM)
	If Not __GDM_IsSupportedItemCommand($aGDM_Items[$iGDM][$iGDM_Command]) Then
		Return
	EndIf
	
	Local $sWorkDir = StringRegExpReplace($aGDM_Items[$iGDM][$iGDM_Path], '\\[^\\]*$', '')
	Local $aArgs = StringSplit($aGDM_Items[$iGDM][$iGDM_Args], '|')
	Local $sArguments = '', $hFunc
	
	If _WinAPI_PathIsRelative($sWorkDir) Then
		$sWorkDir = @WorkingDir
	EndIf
	
	For $iArg = 1 To $aArgs[0]
		If $aArgs[$iArg] Then
			$sArguments &= ($sArguments ? ' ' : '') & '"' & $aArgs[$iArg] & '"'
		EndIf
	Next
	
	For $i = 0 To $iGDM_SprtdCmd_Count - 1
		If $aGDM_Items[$iGDM][$iGDM_Command] = $aGDM_SprtdCmds[$i][$iGDM_SprtdCmd_Command] Then
			$hFunc = $aGDM_SprtdCmds[$i][$iGDM_SprtdCmd_Func]
			
			If $aGDM_SprtdCmds[$i][$iGDM_SprtdCmd_UseArgs] Then
				$hFunc($aGDM_Items[$iGDM][$iGDM_Path], $sArguments, $sWorkDir)
			Else
				$hFunc($aGDM_Items[$iGDM][$iGDM_Path])
			EndIf
			
			ExitLoop
		EndIf
	Next
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Description ...: Event handler wrapper when hotkey is trigered.
; Syntax ........: __GDM_HK_Event_Handler()
; Author ........: G.Sandler
; ===============================================================================================================================
Func __GDM_HK_Event_Handler()
	For $i = 1 To $aGDM_Items[0][0]
		If @HotKeyPressed = $aGDM_Items[$i][$iGDM_Hotkey] Then
			Local $fHK_Global = ($aGDM_Items[$i][$iGDM_HKIsGlobal] == 0 ? False : ($aGDM_Items[$i][$iGDM_HKIsGlobal] == 1 ? True : $iGDM_HotKeys_Global))
			
			If $fHK_Global Or (Not $fHK_Global And WinActive($aGDM_Items[$i][$iGDM_hWnd])) Then
				_SendMessage($aGDM_Items[$i][$iGDM_hWnd], $WM_COMMAND, $aGDM_Items[$i][$iGDM_ItemID], 0)
			Else ;If not handled, send the hotkey to it's target
				HotKeySet(@HotKeyPressed)
				Send(@HotKeyPressed)
				HotKeySet(@HotKeyPressed, '__GDM_HK_Event_Handler')
			EndIf
			
			Return
		EndIf
	Next
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Description ...: Exit function for "Exit" command.
; Syntax ........: __GDM_Exit()
; Author ........: G.Sandler
; ===============================================================================================================================
Func __GDM_Exit($iParam)
	Exit Int($iParam)
EndFunc

; #INTERNAL_USE_ONLY# ===========================================================================================================
; Description ...: Main handler for menu events.
; Syntax ........: __GDM_MENU_HANDLER($hWndGUI, $iMsg, $wParam, $lParam)
; Author ........: G.Sandler
; ===============================================================================================================================
Func __GDM_MENU_HANDLER($hWnd, $iMsg, $wParam, $lParam)
	For $i = 1 To $aGDM_Items[0][0]
		If $aGDM_Items[$i][$iGDM_ItemID] = BitAND($wParam, 0xFFFF) Then
			__GDM_Item_Event_Handler($i)
			ExitLoop
		EndIf
	Next
	
	Return $GUI_RUNDEFMSG
EndFunc

#EndRegion Internal Functions
