#Region Header

#CS
	Name:				SciTE.au3
	Description:		Functions to handle SciTE Editor
	Author:				Copyright  2012 - 2015 CreatoR's Lab (G.Sandler), www.creator-lab.ucoz.ru. All rights reserved.
	AutoIt version:		3.3.10.2 - 3.3.12.0
	UDF version:		1.2
	
	Remarks:			This UDF uses WM_COPYDATA windows message, any other UDF that might use it and defined *after* this UDF, should call internal function __SciTE_WM_COPYDATA, e.g:
						
						Func WM_COPYDATA($hWnd, $nMsg, $wParam, $lParam)
							__SciTE_WM_COPYDATA($hWnd, $nMsg, $wParam, $lParam)
							
							;...
						EndFunc
	
	Functions List:		
						[PUBLIC]
						_SciTE_ChangeWorkingDir
						_SciTE_CloseCurrentFile
						_SciTE_ConsoleWrite
						_SciTE_EnumProperties
						_SciTE_ExportAs
						_SciTE_ExpandProperties
						_SciTE_Extender
						_SciTE_FindText
						_SciTE_GetAu3Extensions
						_SciTE_GetCurrentFile
						_SciTE_GetCurrentWord
						_SciTE_WinGetHandleByPID
						_SciTE_GetHomePath
						_SciTE_GetSelection
						_SciTE_GetProperty
						_SciTE_GoToLine
						_SciTE_InsertText
						_SciTE_LoadSession
						_SciTE_MacroCommand
						_SciTE_MenuCommand
						_SciTE_Open
						_SciTE_OpenFile
						_SciTE_Quit
						_SciTE_ReloadProperties
						_SciTE_ReplaceAll
						_SciTE_ReplaceSelection
						_SciTE_SaveAs
						_SciTE_SaveSession
						_SciTE_SetSelection
						_SciTE_SetProperty
						_SciTE_SetMainPID
						_SciTE_SendCommand
						
						[INTERNAL]
						__SciTE_SendCommand
						__SciTE_WM_COPYDATA
	
	History:
						v1.2
						* Fixed issue with initial $iSCITE_MAIN_PID set.
						* Fixed issue with _SciTE_ReplaceAll function.
						* Changed return value from _SciTE_GetSelection (for details see functions description).
						* Now _SciTE_MenuCommand can accept 'IDM_CONSTANT' string as $vCommand parameter.
						+ Added _SciTE_SetSelection function to set current selection.
						+ Added _SciTE_ExpandProperties function to expand properties in string to it's true value.
						+ Added _SciTE_ConsoleRead function to read from SciTE's output pane.
						+ Added $fOverwrite optional parameter to _SciTE_ConsoleWrite function to clear the output before sending the text.
						+ Added $iFlags optional parameter to _SciTE_ReplaceAll function (Search flags).
							(for details see functions description)
						+ Added following optional parameters to _SciTE_FindText function:
							$iFlags - Search flags
							$iStart - Search start position in the document
							$fWrapAround - Start the search from the beginning if not found from current position
							(for parameters details see functions description)
						
						v1.1
						* Better handling SciTE instances.
						+ Added following functions:
							_SciTE_Extender, _SciTE_WinGetHandleByPID, _SciTE_SetMainPID, _SciTE_SendCommand
							(for details see functions description)
						+ Added $sAutoIt_Dir optional parameter to _SciTE_GetHomePath function (AutoIt path to search the SciTE path).
						+ Added $iWaitDE optional parameter to _SciTE_Open function (to wait for DirectorExtension handle).
						
						v1.0
						* First public version.
	
#CE

;Includes
#include-once

#EndRegion

#Region Global Variables

Global Const $SCFIND_WHOLEWORD 			= 2
Global Const $SCFIND_MATCHCASE 			= 4
Global Const $SCFIND_WORDSTART			= 0x00100000
Global Const $SCFIND_REGEXP 			= 0x00200000
Global Const $SCFIND_POSIX 				= 0x00400000

Global Const $sSCITE_WND_CLASS			= '[CLASS:SciTEWindow]'
Global Const $sSCITE_DIREXTWND_CLASS	= '[CLASS:DirectorExtension;TITLE:DirectorExtension]'
Global Const $sSCITE_EXE_FILENAME		= 'SciTE.exe'
Global Const $iSCITE_WM_COPYDATA 		= 74

Global $sSCITE_CMD_BUFFER 				= '' ;SciTE command return data
Global $hSCITE_WND_HANDLE				= 0
Global $iSCITE_MAIN_PID					= WinGetProcess($sSCITE_WND_CLASS)

#EndRegion

#Region Public Functions

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_ChangeWorkingDir
; Description....: Change the working directory
; Syntax.........: _SciTE_ChangeWorkingDir($sWorkDir)
; Parameters.....: $sWorkDir - Path to new working dir.
;                  
;                 
; Return values..: None.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_ChangeWorkingDir($sWorkDir)
	__SciTE_SendCommand('cwd:' & $sWorkDir)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_Close
; Description....: Close the SciTE editor
; Syntax.........: _SciTE_Close($iPID)
; Parameters.....: $iPID [Optional] - Process ID of SciTE instance (default is 0, wich means that all SciTE instances will be closed by SciTEWindow classname).
;                 
; Return values..: Success - Returns 1, and closes specified SciTE instance.
;                  Failure - Returns 0 and set @error to non-zero.
;                
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_Close($iPID = 0)
	Local $aWinList = WinList($sSCITE_WND_CLASS)
	
	$iPID = ProcessExists($iPID)
	
	For $i = 1 To UBound($aWinList) - 1
		If Not $iPID Then
			WinClose($aWinList[$i][1])
		Else
			If WinGetProcess($aWinList[$i][1]) = $iPID Then
				Return ProcessClose($iPID)
			EndIf
		EndIf
	Next
	
	Return SetError(1, 0, 0)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_CloseCurrentFile
; Description....: Close the current file
; Syntax.........: _SciTE_CloseCurrentFile()
; Parameters.....: 
;                 
; Return values..: None. On success - Closes current document file.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_CloseCurrentFile()
	__SciTE_SendCommand('close:')
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_ConsoleWrite
; Description....: Writes to SciTE's output pane.
; Syntax.........: _SciTE_ConsoleWrite($sText)
; Parameters.....: $sText - Text to write to the SciTE console (output pane).
;                  $fOverwrite [Optional] - If True (default is False), the output will be cleared before sending the text.
;                 
; Return values..: None. On success - Writes to SciTE's output pane.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_ConsoleWrite($sText, $fOverwrite = False)
	If $fOverwrite Then
		_SciTE_MenuCommand('IDM_CLEAROUTPUT')
	EndIf
	
	__SciTE_SendCommand('output:' & StringReplace(StringStripCR($sText), @LF, @CRLF, 0, 2))
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_ConsoleRead
; Description....: Reads from SciTE's output pane.
; Syntax.........: _SciTE_ConsoleRead()
; Parameters.....: None.
;                 
; Return values..: SciTE's output pane text.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_ConsoleRead()
	_SciTE_Extender('dostring props["_scite_consoleread.prop.gettext"]=output:GetText()')
	Return StringReplace(_SciTE_GetProperty('_scite_consoleread.prop.gettext'), '\n', @CRLF, 0, 2)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_EnumProperties
; Description....: Enumerate all the properties in the argument set
;                 
; Syntax.........: _SciTE_EnumProperties($sType = Default)
; Parameters.....: $sType [Optional] - Properties type (set), supported: dyn|local|user|base|embed.
;                 
;                 
; Return values..: Success - List of properties, delimited with @CRLF.
;                  Failure - Empty string ("").
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_EnumProperties($sType = Default)
	If (IsKeyword($sType) And $sType = Default) Or $sType = -1 Then
		$sType = 'dyn'
	EndIf
	
	Return __SciTE_SendCommand('enumproperties:' & $sType)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_ExportAs
; Description....: Save the document in HTML/RTF/PDF/LaTeX/XML format as the indicated file
; Syntax.........: _SciTE_ExportAs($sFile, $sType)
; Parameters.....: $sFile - File path to export.
;                  $sType - Type of export format, supported: HTML|RTF|PDF|LaTeX|XML
;                  
;                 
; Return values..: None. On success - Exports current document to specified file.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_ExportAs($sFile, $sType)
	If Not StringRegExp($sType, '(?i)^(html|rtf|pdf|latex|xml)$') Then
		Return SetError(1, 0, 0)
	EndIf
	
	__SciTE_SendCommand('exportas' & StringLower($sType) & ':' & $sFile)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_ExpandProperties
; Description....: Expands properties in string to it's true value (using _SciTE_GetProperty).
; Syntax.........: _SciTE_ExpandProperties($sText)
; Parameters.....: $sText - String to expand.
;                  
;                 
; Return values..: Success - Expanded string.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_ExpandProperties($sText)
	Local $aProps = StringRegExp($sText, '\$\(([^)]+)\)', 3)
	
	For $i = 0 To UBound($aProps) - 1
		$sText = StringReplace($sText, '$(' & $aProps[$i] & ')', _SciTE_GetProperty($aProps[$i]), 1, 2)
	Next
	
	Return $sText
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_Extender
; Description....: Call the extension interface with the given command.
; Syntax.........: _SciTE_Extender($sCommand)
; Parameters.....: $sCommand - Command to execute. See the SciTE source code for the syntax of the command argument.
;                  
;                 
; Return values..: None. On success - Executes extender command.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: To get data from extender, assign unique property (props['unique_prop']=...) and use _SciTE_GetProperty (See example in _SciTE_FindText function).
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_Extender($sCommand)
	Return __SciTE_SendCommand('extender:' & $sCommand)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_FindText
; Description....: Search for a string, select and show it
; Syntax.........: _SciTE_FindText($sText)
; Parameters.....: $sText - Text to find.
;                  $iFlags [Optional] - Search flags. Supported flags (can be combined with BitOR):
;																		$SCFIND_WHOLEWORD - Search only whole word
;																		$SCFIND_MATCHCASE - Case sensitive search
;																		$SCFIND_WORDSTART - Search only at word start
;																		$SCFIND_REGEXP - Search using regular expression
;																		$SCFIND_POSIX - Posix Search
;                  $iStart [Optional] - Search start position in the document
;										(default is -1, search from current selection end position, use -2 to search from current selection start position).
;                  $fWrapAround [Optional] - If True (default), the search will start from the begining if not found from current position.
;                  
; Return values..: Success - Selects found text in current document.
;                  Failure - Sets @error to non-zero.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_FindText($sText, $iFlags = 0, $iStart = -1, $fWrapAround = True)
	;__SciTE_SendCommand('find:' & $sText)
	
	If $iStart = -1 Then
		$iStart = 'editor.SelectionEnd'
	ElseIf $iStart = -2 Then
		$iStart = 'editor.SelectionStart'
	EndIf
	
	Local $sPropStart = '_scite_findtext.prop.start.pos'
	Local $sPropEnd = '_scite_findtext.prop.end.pos'
	
	Local $iPropStart, $iPropEnd
	
	$sText = StringReplace($sText, '\', '\\', 0, 2)
	$sText = StringReplace($sText, @CR, '\r', 0, 2)
	$sText = StringReplace($sText, @LF, '\n', 0, 2)
	$sText = StringReplace($sText, @TAB, '\t', 0, 2)
	
	For $i = 0 To 1
		_SciTE_Extender('dostring props["' & $sPropStart & '"], props["' & $sPropEnd & '"] = editor:findtext("' & _
			$sText & '", ' & $iFlags & ', ' & $iStart & ', editor.Length)')
		
		$iPropStart = Int(_SciTE_GetProperty($sPropStart))
		$iPropEnd = Int(_SciTE_GetProperty($sPropEnd))
		
		If $iPropEnd > $iPropStart Or Not $fWrapAround Then
			ExitLoop
		Else
			$iStart = 0
		EndIf
	Next
	
	If $iPropEnd <= $iPropStart Then
		Return SetError(1, 0, 0)
	EndIf
	
	_SciTE_SetSelection($iPropStart, $iPropEnd)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_GetAu3Extensions
; Description....: Return AutoIt extensions (files that recognised by the SciTE as AutoIt scripts)
; Syntax.........: _SciTE_GetAu3Extensions()
; Parameters.....: 
;                 
; Return values..: List of AutoIt extensions that recognised by the SciTE as AutoIt scripts
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_GetAu3Extensions()
	Return _SciTE_GetProperty('au3')
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_GetCurrentFile
; Description....: Return the path of the file being edited
; Syntax.........: _SciTE_GetCurrentFile()
; Parameters.....: 
;                 
; Return values..: Current file path open in SciTE.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_GetCurrentFile()
	Return __SciTE_SendCommand('askfilename:')
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_GetCurrentWord
; Description....: Return currently selected/focused word
; Syntax.........: _SciTE_GetCurrentWord()
; Parameters.....: 
;                 
; Return values..: Currently selected/focused word.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_GetCurrentWord()
	Return _SciTE_GetProperty('CurrentWord')
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_GetHomePath
; Description....: Get SciTE home directory
; Syntax.........: _SciTE_GetHomePath()
; Parameters.....: $sAutoIt_Dir [Optional] - AutoIt path to search the SciTE path (default is '', will search for SciteDefaultHome property or in the registry).
;                 
; Return values..: Success - SciTE directory path.
;                  Failure - Empty string ("") and sets @error to 1 if SciTE path not found.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_GetHomePath($sAutoIt_Dir = '')
	If $sAutoIt_Dir And FileExists($sAutoIt_Dir & '\SciTE\' & $sSCITE_EXE_FILENAME) Then
		Return $sAutoIt_Dir & '\SciTE'
	EndIf
	
	Local $sSciTE_Dir = _SciTE_GetProperty('SciteDefaultHome')
	
	If FileExists($sSciTE_Dir & '\' & $sSCITE_EXE_FILENAME) Then
		Return $sSciTE_Dir
	EndIf
	
	$sAutoIt_Dir = RegRead('HKEY_LOCAL_MACHINE\SOFTWARE\AutoIt v3\AutoIt', 'InstallDir')
	
	If Not @Compiled And Not FileExists($sAutoIt_Dir & '\SciTE\' & $sSCITE_EXE_FILENAME) Then
		$sAutoIt_Dir = StringRegExpReplace(@AutoItExe, '\\[^\\]*$', '')
	EndIf
	
	If FileExists($sAutoIt_Dir & '\SciTE\' & $sSCITE_EXE_FILENAME) Then
		Return $sAutoIt_Dir & '\SciTE'
	EndIf
	
	Return SetError(1, 0, '')
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_GetSelection
; Description....: Return current selection data
; Syntax.........: _SciTE_GetSelection()
; Parameters.....: 
;                 
; Return values..: Array of the selection info with 7 elements:
;                                                    [0] - Selection start column
;                                                    [1] - Selection start line
;                                                    [2] - Selection end column
;                                                    [3] - Selection end line
;                                                    [4] - Selected text
;                                                    [5] - Selection start position
;                                                    [6] - Selection end position
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_GetSelection()
	Local $sProp = '_scite_getselection.prop.sel'
	Local $iStart, $iEnd
	
	_SciTE_Extender('dostring props["' & $sProp & '"] = editor.SelectionStart')
	$iStart = Int(_SciTE_GetProperty($sProp))
	
	_SciTE_Extender('dostring props["' & $sProp & '"] = editor.SelectionEnd')
	$iEnd = Int(_SciTE_GetProperty($sProp))
	
	Local $aRet[7] = _ 
		[ _
			_SciTE_GetProperty('SelectionStartColumn'), _SciTE_GetProperty('SelectionStartLine'), _SciTE_GetProperty('SelectionEndColumn'), _
			_SciTE_GetProperty('SelectionEndLine'), _SciTE_GetProperty('CurrentSelection'), $iStart, $iEnd _
		]
	
	Return $aRet
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_GetProperty
; Description....: Return the value of a property
; Syntax.........: _SciTE_GetProperty($sKey)
; Parameters.....: $sKey - Property key.
;                  
;                 
; Return values..: Success - Value of the key property.
;                  Failure - Empty string ("").
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_GetProperty($sKey)
	Local $sRet = __SciTE_SendCommand('askproperty:' & $sKey)
	
	$sRet = StringReplace($sRet, '\\', '\')
	$sRet = StringReplace($sRet, '\t', @TAB)
	$sRet = StringReplace($sRet, '\r\n', @CRLF)
	
	Return $sRet
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_WinGetHandleByPID
; Description....: Get SciTE main window handle of specified Process ID.
; Syntax.........: _SciTE_WinGetHandleByPID()
; Parameters.....: 	$iPID - Process ID.
;					$fVisible [Optional] - If True (default), will return only visible window.
;                 
; Return values..: Success - SciTE main window handle that associated with specified Process ID ($iPID).
;                  Failure - 0 and sets @error as following:
;															1 - Specified PID not exists.
;															2 - SciTE main window handle associated with specified PID not found.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_WinGetHandleByPID($iPID, $fVisible = True)
	If Not ProcessExists($iPID) Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $aWinList = WinList($sSCITE_WND_CLASS)
	Local $f_Visible
	
	For $i = 1 To UBound($aWinList) - 1
		$f_Visible = (BitAND(WinGetState($aWinList[$i][1]), 2) = 2)
		
		If (($fVisible And $f_Visible) Or (Not $fVisible And Not $f_Visible)) And WinGetProcess($aWinList[$i][1]) = $iPID Then
			Return $aWinList[$i][1]
		EndIf
	Next
	
	Return SetError(2, 0, 0)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_GoToLine
; Description....: Move caret to a particular line and make it visible
; Syntax.........: _SciTE_GoToLine($iLine, $iColumn = -1)
; Parameters.....: $iLine - Line number.
;                  $iColumn [Optional] - Column number. If this parameter > 0, then function will select the word at that column number or move the caret there if no word is present.
;                  
;                 
; Return values..: None. On success - Jumps to specific line and column in that line.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_GoToLine($iLine, $iColumn = -1)
	Local $sCmd = 'goto:' & $iLine
	
	If $iColumn > 0 Then
		$sCmd &= ',' & $iColumn
	EndIf
	
	__SciTE_SendCommand($sCmd)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_InsertText
; Description....: Display the value in the editor pane replacing the selection
; Syntax.........: _SciTE_InsertText($sText)
; Parameters.....: $sText - Text to insert.
;                  
;                 
; Return values..: None. On success - Inserts specified text to SciTE editor.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_InsertText($sText)
	__SciTE_SendCommand('insert:' & $sText)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_LoadSession
; Description....: Load a session as given by the indicated file
; Syntax.........: _SciTE_LoadSession($sFile)
; Parameters.....: $sFile - Session file path.
;                  
;                 
; Return values..: None. On success - Loads session file.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_LoadSession($sFile)
	__SciTE_SendCommand('loadsession:' & $sFile)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_MacroCommand
; Description....: Execute a macro command.
; Syntax.........: _SciTE_MacroCommand($sCommand)
; Parameters.....: $sCommand - Command to execute. See the SciTE source code for the syntax of the command argument.
;                  
;                 
; Return values..: None. On success - Executes macro command.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_MacroCommand($sCommand)
	__SciTE_SendCommand('macrocommand:' & $sCommand)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_MenuCommand
; Description....: Execute a menu command based on numeric ID
; Syntax.........: _SciTE_MenuCommand($vCommand)
; Parameters.....: $vCommand - Menu Command (Menu Item ID, or 'IDM_CONSTANT' string).
;                  
;                 
; Return values..: None. On success - Call specified menu command.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: List of Menu Item IDs or IDM_* constants can be found in SciTE help file "SciTE4AutoIt3.chm::/Scitedoc/CommandValues.html" (CTRL + F1).
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_MenuCommand($vCommand)
	If IsString($vCommand) Then
		_SciTE_Extender('dostring scite.MenuCommand(' & StringUpper($vCommand) & ')')
	Else
		__SciTE_SendCommand('menucommand:' & $vCommand)
	EndIf
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_Open
; Description....: Open SciTE editor
; Syntax.........: _SciTE_Open($sParams = '')
; Parameters.....: 	$sParams [Optional] - Command line parameters to pass to the SciTE when opening.
;					$iWaitDE [Optional] - If this parameter is <> 0 (default is 1), then function will wait $iWaitDE seconds for DirectorExtension handle.
;                 
; Return values..: Success - Returns SciTE's pid.
;                  Failure - Returns 0 and sets @error to non-zero.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_Open($sParams = '', $iWaitDE = 1)
	Local $sRun = _SciTE_GetHomePath() & '\' & $sSCITE_EXE_FILENAME
	
	If @error Then
		Return SetError(1, 0, 0)
	EndIf
	
	If $sParams <> '' Then
		$sRun &= ' ' & $sParams
	EndIf
	
	Local $iPID = Run($sRun)
	
	If $iWaitDE Then
		Local $iTimer = TimerInit()
		
		While WinGetProcess($sSCITE_DIREXTWND_CLASS) <> $iPID
			If TimerDiff($iTimer) / 1000 >= $iWaitDE Then
				ExitLoop
			EndIf
		WEnd
	EndIf
	
	Return $iPID
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_OpenFile
; Description....: Open the indicated file
; Syntax.........: _SciTE_OpenFile($sFile)
; Parameters.....: $sFile - File to open.
;                  
;                 
; Return values..: None. On success - Opens the specified file in SciTE.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_OpenFile($sFile)
	__SciTE_SendCommand('open:' & $sFile)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_Quit
; Description....: Shut down SciTE
; Syntax.........: _SciTE_Quit()
; Parameters.....: 
;                 
; Return values..: None. On success - Closes SciTE Editor.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_Quit()
	__SciTE_SendCommand('quit:')
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_ReloadProperties
; Description....: Reload properties from files (usefull to apply changes made to the SciTE)
; Syntax.........: _SciTE_ReloadProperties()
; Parameters.....: 
;                 
; Return values..: None. On success - Reloads SciTE properties.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_ReloadProperties()
	__SciTE_SendCommand('reloadproperties:')
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_ReplaceAll
; Description....: Replace all instances of the search string in the document with the replace string
; Syntax.........: _SciTE_ReplaceAll($sSearch, $sReplace)
; Parameters.....: $sSearch - Text to search.
;                  $sReplace - Text to replace.
;                  $iFlags [Optional] - Search flags. Supported flags (can be combined with BitOR):
;																		$SCFIND_WHOLEWORD - Search only whole word
;																		$SCFIND_MATCHCASE - Case sensitive search
;																		$SCFIND_WORDSTART - Search only at word start
;																		$SCFIND_REGEXP - Search using regular expression
;																		$SCFIND_POSIX - Posix Search
;                  
;                 
; Return values..: Success - Replaces all instances of the found text and returns number of replacements.
;				   Failure - Returns 0 if no replacements performed.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_ReplaceAll($sSearch, $sReplace, $iFlags = 0)
	;__SciTE_SendCommand('replaceall:' & $sSearch & '\000' & $sReplace)
	
	Local $iRet = 0
	
	While 1
		_SciTE_FindText($sSearch, $iFlags, 0)
		
		If @error Then
			ExitLoop
		EndIf
		
		_SciTE_ReplaceSelection($sReplace)
		$iRet += 1
	WEnd
	
	Return $iRet
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_ReplaceSelection
; Description....: Replace currently selected text with other text
; Syntax.........: _SciTE_ReplaceSelection($sText)
; Parameters.....: $sText - Text to replace with.
;                  
;                 
; Return values..: None. On success - Replaces selected text with specified in $sText.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_ReplaceSelection($sText)
	_SciTE_InsertText($sText)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_SaveAs
; Description....: Save the document as the indicated file
; Syntax.........: _SciTE_SaveAs($sFile)
; Parameters.....: $sFile - File path to save.
;                  
;                 
; Return values..: None. On success - Saves current document as $sFile.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_SaveAs($sFile)
	__SciTE_SendCommand('saveas:' & $sFile)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_SaveSession
; Description....: Save a session as given by the indicated file
; Syntax.........: _SciTE_SaveSession($sFile)
; Parameters.....: $sFile - File path to save.
;                  
;                 
; Return values..: None. On success - Saves current session as $sFile.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_SaveSession($sFile)
	__SciTE_SendCommand('savesession:' & $sFile)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_SetSelection
; Description....: Set current selection.
; Syntax.........: _SciTE_SetSelection($iStart, $iEnd)
; Parameters.....: $iStart - Start selection position in the document (set to -1 to select end of buffer text).
;                  $iEnd - End selection position in the document (set to -1 to select end of buffer text).
;                  $fFocus [Optional] - If True (default), the line with selection will be focused.
;                  
; Return values..: None. On success - Sets selection from $iStart to $iEnd.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_SetSelection($iStart, $iEnd, $fFocus = True)
	If $iStart = -1 Then
		$iStart = 'editor.Length'
	EndIf
	
	If $iEnd = -1 Then
		$iEnd = 'editor.Length'
	EndIf
	
	_SciTE_Extender('dostring editor:SetSel(' & $iStart & ',' & $iEnd & ')')
	
	If $fFocus Then
		_SciTE_GoToLine(_SciTE_GetProperty('SelectionStartLine'))
		_SciTE_Extender('dostring editor:SetSel(' & $iStart & ',' & $iEnd & ')')
	EndIf
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_SetProperty
; Description....: Set a property to a value
; Syntax.........: _SciTE_SetProperty($sKey, $sValue)
; Parameters.....: $sKey - Property Key.
;                  $sValue - Property value.
;                  
;                 
; Return values..: None. On success - Sets property as key=value.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_SetProperty($sKey, $sValue)
	__SciTE_SendCommand('property:' & $sKey & '=' & $sValue)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_SetMainPID
; Description....: Set SciTE main Process ID to work with.
; Syntax.........: _SciTE_SetMainPID($iPID)
; Parameters.....: $iPID - SciTE process ID.
;                  
; Return values..: None.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_SetMainPID($iPID)
	If ProcessExists($iPID) Then
		$iSCITE_MAIN_PID = $iPID
	EndIf
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _SciTE_SendCommand
; Description....: Sends command to SciTE's DirectorExtension.
; Syntax.........: _SciTE_SendCommand($sCmd)
; Parameters.....: $sCmd - Command to send (command list can be found in SciTE documentation: http://www.scintilla.org/SciTEDirector.html).
;                  
; Return values..: Requested value as returned by DirectorExtension.
;                 
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: 
; Link...........: 
; Example........: 
; ===============================================================================================================
Func _SciTE_SendCommand($sCmd)
	Return __SciTE_SendCommand($sCmd)
EndFunc

#EndRegion

#Region Internal Functions

Func __SciTE_SendCommand($sCmd)
	Local $iCmd_Len, $stCmd, $stCOPYDATA
	Local $hSciTE_DE = 0 ;WinGetHandle($sSCITE_DIREXTWND_CLASS)
	Local $aList = WinList($sSCITE_DIREXTWND_CLASS)
	Local Static $hSCITE_AU3_WND = GUICreate('AutoIt3-SciTE interface')
	Local Static $iWM_CD = GUIRegisterMsg($iSCITE_WM_COPYDATA, '__SciTE_WM_COPYDATA')
	
	For $i = 1 To UBound($aList) - 1
		If WinGetProcess($aList[$i][1]) = $iSCITE_MAIN_PID Then
			$hSciTE_DE = $aList[$i][1]
			ExitLoop
		EndIf
	Next
	
	If Not IsHWnd($hSciTE_DE) Then
		Return SetError(1, 0, 0)
	EndIf
	
	$sCmd = StringReplace($sCmd, '\', '\\')
	$sCmd = StringReplace($sCmd, @TAB, '\t')
	$sCmd = StringReplace($sCmd, @CRLF, '\r\n')
	
	;Convert command to UTF-8 string
	$sCmd = BinaryToString(StringToBinary($sCmd, 4))
	$sCmd = ":" & Dec(StringRight($hSCITE_AU3_WND, 8)) & ":" & $sCmd
	
	$iCmd_Len = StringLen($sCmd)
	$stCmd = DllStructCreate('char[' & $iCmd_Len + 1 & ']')
	DllStructSetData($stCmd, 1, $sCmd)
	
	$stCOPYDATA = DllStructCreate('ptr;dword;ptr')
	DllStructSetData($stCOPYDATA, 1, 1)
	DllStructSetData($stCOPYDATA, 2, $iCmd_Len + 1)
	DllStructSetData($stCOPYDATA, 3, DllStructGetPtr($stCmd))
	
	$sSCITE_CMD_BUFFER = ''
	
	DllCall('User32.dll', 'None', 'SendMessage', 'HWnd', $hSciTE_DE, 'Int', $iSCITE_WM_COPYDATA, 'HWnd', $hSCITE_AU3_WND, 'Ptr', DllStructGetPtr($stCOPYDATA))
	
	If $sSCITE_CMD_BUFFER <> '' Then
		Return StringStripWS($sSCITE_CMD_BUFFER, 2)
	EndIf
EndFunc

Func __SciTE_WM_COPYDATA($hWnd, $iMsg, $wParam, $lParam)
	Local $stCOPYDATA = DllStructCreate('ptr;dword;ptr', $lParam)
	Local $iSciTECmdLen = DllStructGetData($stCOPYDATA, 2)
	Local $stCmd = DllStructCreate('char[' & $iSciTECmdLen & ']', DllStructGetData($stCOPYDATA, 3))
	Local $sRet = StringLeft(DllStructGetData($stCmd, 1), $iSciTECmdLen)
	
	$sRet = StringRegExpReplace($sRet, '(?i)^[^:]*:\d+:[^:]+:(?:[^:]{2,}:)?', '')
	$sRet = StringReplace($sRet, '\\', '\')
	
	$sSCITE_CMD_BUFFER &= $sRet & @CRLF
EndFunc

#EndRegion
