#Region Header

#CS

    Title:          Syntax Highlighter for AutoIt v3 Code
    Filename:       Au3SyntaxHighlight.au3
    Description:    Allows to highlight AutoIt v3 syntax to html/bbcode format.
    Author:         G.Sandler (a.k.a (Mr)CreatoR), www.creator-lab.ucoz.ru, www.autoit-script.ru
    Version:        0.9
    Requirements:   AutoIt v3.3 +, Developed/Tested on WindowsXP (Service Pack 3)/Win 7.
    Uses:           
    Notes:          *Idea* based on similar old project: http://www.autoitscript.com/forum/index.php?showtopic=34236
	
	
	History:
					[1.0]
					* Speed optimization.
					
					[0.9]
					+ Compatibility with AutoIt 3.3.12.0.
					+ Added "Add missing keywords.au3" script to add missing keywords from new AutoIt version.
					* $iDebug_RegExp_Patterns user global variable renamed to $fAu3SH_Debug_RE_Patterns.
					* $iDebug_RegExp_WriteLog user global variable renamed to $fAu3SH_Debug_RE_WriteLog.
					* Examples changed.
					
					[0.8]
					* Fixed bug with highlighting variables inside special keywords (#region, #forceref etc.).
					
					[0.7]
					* Fixed issue with wrong parsing when Add Urls option is enabled.
					
					[0.6]
					+ Compatibility with AutoIt 3.3.8.1.
					* Fixed issue with wrong restored strings.
					* Fixed bug with not proper UDF naming for url (UDFs links does not recognized due to not proper case, i.e: _stringrepeat instead of _StringRepeat).
					
					[0.5]
					* Fixed bug with highlighting literal strings inside comment blocks.
					* Fixed few issues with highlighting special keywords.
					
					[0.4]
					* Fixed bug with (re)highlighting keywords and functions inside COM object methods.
					* Docs updated.
					
					[0.3]
					* Fixed bug when strings and send keys was highlighted inside commented lines.
					* Few optimizations to the code.
					
					[0.2]
					+ Added Global parameter ($iAu3SH_AbortHighlight) to abort the highlighting process (break the function execution).
					   - If this variable set to 1, the function will return the original Au3 Code and set @error to -1.
					+ Added "#White space" style support (now you can set it in the au3.styles.properties file).
					+ Added "#Background" style support, used only for highlighting with Html tags mode (now you can set it in the au3.styles.properties file).
					* Styles header classes renamed with "au3_" prefix.
					* Fixed bug with highlighting COM objects.
					* Fixed bug with highlighting keywords inside variables.
					* Fixed bug with highlighting more than one preprocessor when instead of <Include.au3> used double/single qoutes ("Include.au3").
					* Fixed bug with highlighting commented lines/comment blocks inside literal strings.
					* Fixed bug with not (properly) highlighting commented lines.
					* Fixed issue with converting to BBCode format.
					* Now the PreProcessor and Special keywords are correctly highlighted (all included tags removed).
					
					[0.1]
					* First public release.

================ ToDo: ================

#CE

If Not @Compiled Then
	ConsoleWrite('+ =============================================================' & @LF)
	ConsoleWrite('(' & 5 & ') : Syntax Highlighter for AutoIt (double-click on this line for details)' & @LF)
	ConsoleWrite('(' & @ScriptLineNumber + 5 & ') : Example included (double-click on this line to see the example)' & @LF)
	ConsoleWrite('+ =============================================================' & @LF & @LF)
EndIf

#CS Example script

#include <Au3SyntaxHighlight.au3>
#include <GUIConstantsEx.au3>
#include <WindowsConstants.au3>
#include <IE.au3>

$iAu3SH_AddURLs = 1

$sInitDir = ''
$sFilter = '*.au3'

$sMRUList = RegRead('HKEY_CURRENT_USER\Software\Microsoft\Windows\CurrentVersion\Explorer\ComDlg32\OpenSaveMRU\au3', 'MRUList')
$sLastFileOpenPath = RegRead('HKEY_CURRENT_USER\Software\Microsoft\Windows\CurrentVersion\Explorer\ComDlg32\OpenSaveMRU\au3', StringLeft($sMRUList, 1))

If Not FileExists($sLastFileOpenPath) Then
	$sInitDir = @ScriptDir
	$sFilter = 'Test Script.au3'
EndIf

$sFile = FileOpenDialog('Open Au3 Script File', $sInitDir, 'AutoIt Script (' & $sFilter & ')')
If @error Then Exit

$sAu3Code = FileRead($sFile)
$sAu3Syntax_HighlightedCode = _Au3_SyntaxHighlight_Proc($sAu3Code, 0)

$sTmp_File = @TempDir & '\Au3_HighlightSyntax.htm'

$hFile = FileOpen($sTmp_File, 2)
FileWrite($hFile, $sAu3Syntax_HighlightedCode)
FileClose($hFile)

Execute('_IEErrorHandlerRegister()')

$oIE = _IECreateEmbedded()

$iGUI_Width = @DesktopWidth - 40
$iGUI_Height = @DesktopHeight - 150

$hGUI = GUICreate('AutoIt Syntax Highlighter [' & $sFile & ']', $iGUI_Width, $iGUI_Height, -1, -1, BitOR($GUI_SS_DEFAULT_GUI, $WS_CLIPSIBLINGS, $WS_CLIPCHILDREN))
GUICtrlCreateObj($oIE, 10, 20, $iGUI_Width - 20, $iGUI_Height - 70)
$iBack_Bttn = GUICtrlCreateButton('Back', 10, $iGUI_Height - 40, 100, 30)
$iForward_Bttn = GUICtrlCreateButton('Forward', 120, $iGUI_Height - 40, 100, 30)
$iHome_Bttn = GUICtrlCreateButton('Home', 230, $iGUI_Height - 40, 100, 30)
$iStop_Bttn = GUICtrlCreateButton('Stop', 340, $iGUI_Height - 40, 100, 30)

GUISetState(@SW_SHOW, $hGUI)

_IENavigate($oIE, $sTmp_File)

While 1
    Switch GUIGetMsg()
		Case $GUI_EVENT_CLOSE
			GUIDelete($hGUI)
            Exit
		Case $iHome_Bttn
            _IENavigate($oIE, $sTmp_File)
        Case $iBack_Bttn
            _IEAction($oIE, 'back')
        Case $iForward_Bttn
            _IEAction($oIE, 'forward')
        Case $iStop_Bttn
            _IEAction($oIE, 'stop')
    EndSwitch
WEnd

#CE Example script

#include-once

Global $iAu3SH_MDV_Opt = Opt("MustDeclareVars", 1)

#EndRegion Header

#Region Global Variables

Global $iDebug_RegExp_Step 			= 1
Global $sDebug_RegExp_LogContent 	= ""
Global $sDebug_RegExp_LogFile 		= @ScriptDir & "\Debug_RegExp.log"

Global $sAu3_Keywords_File 			= @ScriptDir & "\Resources\au3.keywords.properties"
Global $sAu3_Styles_File 			= @ScriptDir & "\Resources\au3.styles.properties"

;Abort _Au3_SyntaxHighlight_Proc
;ATTENTION: If this is set to 1, then it's should be reset to 0 after the "_Au3_SyntaxHighlight_Proc" function returns.
Global $iAu3SH_AbortHighlight 		= 0

Global $sAu3SH_Funcs_URL 			=  __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, _
	"au3.keywords.functions.url", "http://www.autoitscript.com/autoit3/docs/functions/%s.htm", 0)
Global $sAu3SH_UDFs_URL 			= __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, _
	"au3.keywords.udfs.url", "http://www.autoitscript.com/autoit3/docs/libfunctions/%s.htm", 0)
Global $sAu3SH_Keywords_URL 		= __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, _
	"au3.keywords.keywotds.url", "http://www.autoitscript.com/autoit3/docs/keywords.htm#%s", 0)
Global $sAu3SH_Macros_URL 			= __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, _
	"au3.keywords.macros.url", "http://www.autoitscript.com/autoit3/docs/macros.htm#%s", 0)
Global $sAu3SH_PreProcessor_URL 	= __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, _
	"au3.keywords.preprocessor.url", "http://www.autoitscript.com/autoit3/docs/keywords.htm#%s", 0)

Global Enum _
	$iAu3SH_WSpace_Style = 0, $iAu3SH_Cmnt_Style, $iAu3SH_CmntBlck_Style, $iAu3SH_Nmbr_Style, $iAu3SH_Fnc_Style, $iAu3SH_Kwd_Style, $iAu3SH_Mcro_Style, $iAu3SH_Str_Style, _
	$iAu3SH_Oprtr_Style, $iAu3SH_Var_Style, $iAu3SH_SntKys_Style, $iAu3SH_PrPrc_Style, $iAu3SH_Spec_Style, $iAu3SH_Abrv_Style, $iAu3SH_COM_Style, $iAu3SH_Udf_Style

Global $aAu3SH_Styles[$iAu3SH_Udf_Style + 1]

For $i = $iAu3SH_WSpace_Style To $iAu3SH_Udf_Style
	$aAu3SH_Styles[$i] 			= '<span class="au3_S' & $i & '">'
Next

Global $sAu3SH_CloseTag 		= '</span>'

#EndRegion Global Variables

#Region User Variables

Global $fAu3SH_Debug_RE_Patterns 	= False
Global $fAu3SH_Debug_RE_WriteLog 	= False
Global $iAu3SH_AddURLs 				= True

#EndRegion User Variables

#Region Public Functions

; #FUNCTION# ====================================================================================================
; Name...........:	_Au3_SyntaxHighlight_Proc
; Description....:	Allows to highlight AutoIt v3 syntax to html/bbcode format.
; Syntax.........:	_Au3_SyntaxHighlight_Proc($sAu3Code, $iOutput = -1)
; Parameters.....:	$sAu3Code - AutoIt v3 plain code.
;					$iOutput  - [Optional] Sets the output format:
;					                                      -1 - (Default) Return CSS classes header and the AutoIt Syntax Highlighted code as string (wrapped with code tags)
;					                                       1 - Return CSS classes header and AutoIt Syntax Highlighted code as array, where...
;					                                            [0] = CSS styles.
;					                                            [1] = AutoIt Syntax Highlighted code.
;					                                       2 - Return the result as Html formatted string.
;					                                       3 - Return the result as BBCode formatted string (html tags replaced with bbcode tags).
;					
; Return values..:	Success - Returns AutoIt Syntax Highlighted code (see $iOutput parameter above for more details).
;					Failure - Sets @error to:
;												1 	- The $sAu3Code is empty string.
;											   -1 	- The process is aborted by global parameter $iAu3SH_AbortHighlight, and the return value is the original $sAu3Code.
;												2 	- Keywords or styles file not found (check the @ScriptDir & "\Resources").
;
; Author.........:	G.Sandler (a.k.a (Mr)CreatoR), www.creator-lab.ucoz.ru, www.autoit-script.ru
; Modified.......:	
; Remarks........:	*Idea* based on similar old project: http://www.autoitscript.com/forum/index.php?showtopic=34236
; Related........:	
; Link...........:	
; Example........:	Yes (see #Region Header).
; ===============================================================================================================
Func _Au3_SyntaxHighlight_Proc($sAu3Code, $iOutput = -1)
	;************************** I M P O R T A N T *************************
	;   Highlighting elements order must be preserved (DO NOT CHANGE IT)   
	;************************** I M P O R T A N T *************************
	
	Local $sOrigin_Au3Code = $sAu3Code
	
	If StringStripWS($sAu3Code, 8) = '' Then
		Return SetError(1, 0, 0)
	EndIf
	
	If Not FileExists($sAu3_Keywords_File) Or Not FileExists($sAu3_Styles_File) Then
		Return SetError(2, 0, 0)
	EndIf
	
	Local $sPattern1, $sPattern2, $sPattern3
	Local $sReplace1, $sReplace2, $sReplace3
	
	Local $sUnique_Str_Quote = __Au3_SyntaxHighlight_GetUniqueStr($sAu3Code, '%@~@%')
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	Local $sUnique_Str_Include = __Au3_SyntaxHighlight_GetUniqueStr($sAu3Code, '!~@%@~!') ; "!" must be the first character
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	Local $sUnique_Str_CmntBlk = __Au3_SyntaxHighlight_GetUniqueStr($sAu3Code, '%~@~%~@~%')
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	Local $sUnique_Str_Cmnt = __Au3_SyntaxHighlight_GetUniqueStr($sAu3Code, '%~@%@~%')
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	; Get the comment blocks to array
	Local $aCmntBlk_Strings = __Au3_SyntaxHighlight_GetCommentBlocks($sAu3Code)
	
	; Replace all the comment blocks with unique marks (must be done before we get the literal strings)
	For $i = 1 To $aCmntBlk_Strings[0]
		$sAu3Code = StringReplace($sAu3Code, $aCmntBlk_Strings[$i], $sUnique_Str_CmntBlk, 0, 2)
	Next
	
	; Get all strings to array (all between "", '' and <> chars), so we can replace later with unique marks
	$sPattern1 = '("[^"\r\n]*?"|''[^''\r\n]*?'')'
	$sPattern2 = "(?i)#include\s+?(<[^\>]*>).*?"
	
	Local $aQuote_Strings = StringRegExp($sAu3Code, $sPattern1, 3)
	Local $aInclude_Strings = StringRegExp($sAu3Code, $sPattern2, 3)
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("a", $sPattern1, 3, "", $sAu3Code, @ScriptLineNumber - 4)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("b", $sPattern2, 3, "Get all strings to array (all between """", '' and <> chars)", $sAu3Code, @ScriptLineNumber - 4)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
	
	; Replace all the strings with unique marks
	$sPattern1 = '(?s)("|'')([^\1\r\n])*?(\1)'
	$sPattern2 = "(?si)(#include\s+?)<[^\>]*>(.*?)"
	
	$sReplace1 = $sUnique_Str_Quote
	$sReplace2 = '\1' & $sUnique_Str_Include & '\2'
	
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern2, $sReplace2)
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("a", $sPattern1, $sReplace1, "", $sAu3Code, @ScriptLineNumber - 4)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("b", $sPattern2, $sReplace2, "Replace all the strings with unique marks", $sAu3Code, @ScriptLineNumber - 4)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
	
	; Get all commented lines to array (must be done AFTER "Replace all the strings unique marks")
	$sPattern1 = '(;.*)'
	
	Local $aCmnt_Strings = StringRegExp($sAu3Code, $sPattern1, 3)
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("", $sPattern1, 3, "Get all commented lines to array", $sAu3Code, @ScriptLineNumber - 3)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
	
	; Replace all the commented lines with unique marks
	$sPattern1 = '(;.*)'
	$sReplace1 = $sUnique_Str_Cmnt
	
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("", $sPattern1, $sReplace1, "Replace all the commented lines with unique marks", $sAu3Code, @ScriptLineNumber - 3)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
	
	$sPattern1 = '([\(\)\[\]\<\>\.\*\+\-\=\&\^\,\/])'
	$sReplace1 = $aAu3SH_Styles[$iAu3SH_Oprtr_Style] & '\1' & $sAu3SH_CloseTag
	
	; Highlight the operators, brakets, commas (must be done before all other parsers)
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("", $sPattern1, $sReplace1, "Highlight the operators, brakets, commas", $sAu3Code, @ScriptLineNumber - 3)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
	
	$sPattern1 = '(\W+)(_)(\W+)'
	$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Oprtr_Style] & '\2' & $sAu3SH_CloseTag & '\3'
	
	; Highlight the line braking character, wich is the underscore (must be done before all other parsers)
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("", $sPattern1, $sReplace1, "Highlight the line braking character ( _ )", $sAu3Code, @ScriptLineNumber - 3)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
	
	$sPattern1 = '((?:\s+)?' & $aAu3SH_Styles[$iAu3SH_Oprtr_Style] & '\.' & $sAu3SH_CloseTag & '|(?:\s+)?\.)([^\d\$<]\w+)'
	$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_COM_Style] & '\2' & $sAu3SH_CloseTag
	
	; Highlight the COM Objects (must be done before all other parsers)
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("", $sPattern1, $sReplace1, _
			"Highlight the COM Objects (""S8"" is the highlight for operators, here it is a dot (.) char)", $sAu3Code, @ScriptLineNumber - 3)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
	
	$sPattern1 = '([^\w#@])(\d+)([^\w]|\R|$)'
	$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Nmbr_Style] & '\2' & $sAu3SH_CloseTag & '\3'
	
	; Highlight the number
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("", $sPattern1, $sReplace1, "Highlight the numbers", $sAu3Code, @ScriptLineNumber - 3)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
	
	; Highlight the function
	__Au3_SyntaxHighlight_ParseFunctions($sAu3Code)
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	; Highlight the UDFs
	__Au3_SyntaxHighlight_ParseUDFs($sAu3Code)
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	; Highlight the keyword
	__Au3_SyntaxHighlight_ParseKeywords($sAu3Code)
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	; Highlight the macros
	__Au3_SyntaxHighlight_ParseMacros($sAu3Code)
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	; Highlight the PreProcessor
	__Au3_SyntaxHighlight_ParsePreProcessor($sAu3Code)
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	; Highlight special keywords
	__Au3_SyntaxHighlight_ParseSpecial($sAu3Code)
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	$sPattern1 = '([^\w#@]|\R)((?i)0x[abcdef\d]+)([^abcdef\d])'
	$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Nmbr_Style] & '\2' & $sAu3SH_CloseTag & '\3'
	
	; Highlight the hexadecimal number
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("", $sPattern1, $sReplace1, "Highlight the hexadecimal number", $sAu3Code, @ScriptLineNumber - 3)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
	
	$sPattern1 = '\$(\w+)?'
	$sReplace1 = $aAu3SH_Styles[$iAu3SH_Var_Style] & '$\1' & $sAu3SH_CloseTag
	
	; Highlight variables (also can be just the dollar sign)
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("", $sPattern1, $sReplace1, "Highlight variables", $sAu3Code, @ScriptLineNumber - 3)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
	
	; Highlight send keys (must be done before the strings been restored, because send keys should be highlighted inside the strings)
	__Au3_SyntaxHighlight_ParseSendKeys($aQuote_Strings)
	If @error = -1 Then Return SetError(-1, 0, $sOrigin_Au3Code)
	
	$sPattern1 = '(\w+)(\h*' & $aAu3SH_Styles[$iAu3SH_Oprtr_Style] & '\()'
	$sReplace1 = $aAu3SH_Styles[$iAu3SH_WSpace_Style] & '\1' & $sAu3SH_CloseTag & '\2'
	
	; Highlight finaly the '#White space' (only user defined functions)
	$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("", $sPattern1, $sReplace1, "Highlight '#White space' (only user defined functions)", $sAu3Code, @ScriptLineNumber - 3)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
	
	; Replace back the unique marks with the original one and wrap them with "string" tags
	For $i = 1 To UBound($aCmntBlk_Strings) - 1
		$sAu3Code = StringReplace($sAu3Code, $sUnique_Str_CmntBlk, $aAu3SH_Styles[$iAu3SH_CmntBlck_Style] & $aCmntBlk_Strings[$i] & $sAu3SH_CloseTag, 1, 2)
		
		If $iAu3SH_AbortHighlight = 1 Then
			Return SetError(-1, 0, $sOrigin_Au3Code)
		EndIf
	Next
	
	For $i = 0 To UBound($aCmnt_Strings) - 1
		$sAu3Code = StringReplace($sAu3Code, $sUnique_Str_Cmnt, $aAu3SH_Styles[$iAu3SH_Cmnt_Style] & $aCmnt_Strings[$i] & $sAu3SH_CloseTag, 1, 2)
		
		If $iAu3SH_AbortHighlight = 1 Then
			Return SetError(-1, 0, $sOrigin_Au3Code)
		EndIf
	Next
	
	For $i = 0 To UBound($aQuote_Strings) - 1
		$sAu3Code = StringReplace($sAu3Code, $sUnique_Str_Quote, $aAu3SH_Styles[$iAu3SH_Str_Style] & $aQuote_Strings[$i] & $sAu3SH_CloseTag, 1, 2)
		
		If $iAu3SH_AbortHighlight = 1 Then
			Return SetError(-1, 0, $sOrigin_Au3Code)
		EndIf
	Next
	
	For $i = 0 To UBound($aInclude_Strings) - 1
		$aInclude_Strings[$i] = StringReplace($aInclude_Strings[$i], '<', '&lt;', 0, 2)
		$aInclude_Strings[$i] = StringReplace($aInclude_Strings[$i], '>', '&gt;', 0, 2)
		
		$sAu3Code = StringReplace($sAu3Code, $sUnique_Str_Include, $aAu3SH_Styles[$iAu3SH_Str_Style] & $aInclude_Strings[$i] & $sAu3SH_CloseTag, 1, 2)
		
		If $iAu3SH_AbortHighlight = 1 Then
			Return SetError(-1, 0, $sOrigin_Au3Code)
		EndIf
	Next
	
	; Strip tags from "string" inside commented lines and regions (special keywords)
	Do
		$sAu3Code = StringRegExpReplace($sAu3Code, _
			'(.*?(?:' & $aAu3SH_Styles[$iAu3SH_Cmnt_Style] & ';|' & $aAu3SH_Styles[$iAu3SH_Spec_Style] & '#).*?)' & _
			'(?:' & $aAu3SH_Styles[$iAu3SH_Str_Style] & '|' & $aAu3SH_Styles[$iAu3SH_SntKys_Style] & '|' & $aAu3SH_Styles[$iAu3SH_Var_Style] & ')(.*?)' & $sAu3SH_CloseTag & '(.*?)', _
			'\1\2\3')
	Until @extended = 0
	
	; Replace tabs with 4 spaces, in IE the tabs not looking good :(.
	$sAu3Code = StringReplace($sAu3Code, @TAB, '    ', 0, 2) ;'&nbsp;&nbsp;&nbsp;&nbsp;'
	
	; Output debug data to log file
	If $fAu3SH_Debug_RE_WriteLog Then
		Local $hFile = FileOpen($sDebug_RegExp_LogFile, 1)
		
		FileWrite($hFile, StringFormat('_Au3_SyntaxHighlight Debug RegExp Log started at: %.2i/%.2i/%i, %.2i:%.2i:%.2i\r\n' & _
			'~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~\r\n%s\r\n', _
			@MDAY, @MON, @YEAR, @HOUR, @MIN, @SEC, $sDebug_RegExp_LogContent) & _
			'~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~=~' & @CRLF)
		
		FileClose($hFile)
	EndIf
	
	; Get the CSS styles header (to use on return)
	Local $sStyles_Header = __Au3_SyntaxHighlight_GetKeywordStylesHeader()
	
	; Check the $iOutput flag
	Switch $iOutput
		; Return CSS classes header and AutoIt Syntax Highlighted code as array ([0] = CSS styles, [1] = Au3 Code)
		Case 1
			Local $aRet[2] = [$sStyles_Header, $sAu3Code]
			Return $aRet
		; Return AutoIt Syntax Highlighted code as string
		Case 2
			Return $sAu3Code
		; Return the result as BBCode formatted string
		Case 3
			__Au3_SyntaxHighlight_ConvertHtmlToBBCodeTags($sAu3Code)
			
			If @error = -1 Then
				Return SetError(-1, 0, $sOrigin_Au3Code)
			EndIf
			
			Return $sAu3Code
		; Return CSS classes header and the AutoIt Syntax Highlighted code as Html formatted string
		Case Else
			Return $sStyles_Header & @CRLF & '<pre class="au3_codebox"><span>' & $sAu3Code & '</span></pre>'
	EndSwitch
EndFunc

#EndRegion Public Functions

#Region Internal Functions

Func __Au3_SyntaxHighlight_ParseFunctions(ByRef $sAu3Code)
	Local $aFuncs = __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, "au3.keywords.functions")
	Local $sPattern1, $sReplace1, $iLineNum
	
	If $iAu3SH_AddURLs Then
		For $i = 1 To $aFuncs[0]
			If $iAu3SH_AbortHighlight = 1 Then
				Return SetError(-1)
			EndIf
			
			$sPattern1 = '([^\w\$]+|\A)(?<!' & $aAu3SH_Styles[$iAu3SH_COM_Style] & ')((?i)' & $aFuncs[$i] & ')(\W+|$)'
			$sReplace1 = '\1<a href="' & StringFormat($sAu3SH_Funcs_URL, $aFuncs[$i]) & '">' & $aAu3SH_Styles[$iAu3SH_Fnc_Style] & '\2' & $sAu3SH_CloseTag & '</a>\3'
			
			If StringInStr($sAu3Code, $aFuncs[$i], 2) Then
				$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
			EndIf
		Next
		
		$iLineNum = @ScriptLineNumber - 4
	Else
		For $i = 1 To $aFuncs[0]
			If $iAu3SH_AbortHighlight = 1 Then
				Return SetError(-1)
			EndIf
			
			$sPattern1 = '([^\w\$]+|\A)(?<!' & $aAu3SH_Styles[$iAu3SH_COM_Style] & ')((?i)' & $aFuncs[$i] & ')(\W+|$)'
			$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Fnc_Style] & '\2' & $sAu3SH_CloseTag & '\3'
			
			If StringInStr($sAu3Code, $aFuncs[$i], 2) Then
				$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
			EndIf
		Next
		
		$iLineNum = @ScriptLineNumber - 4
	EndIf
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("", _
			$sPattern1, $sReplace1, _
			"Highlight built-in functions - The function name/url in the <Search/Replace Pattern> is the last found name/url in the functions list (after the loop ends)", _
			$sAu3Code, $iLineNum)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
EndFunc

Func __Au3_SyntaxHighlight_ParseUDFs(ByRef $sAu3Code)
	Local $aUDFs = __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, "au3.keywords.udfs")
	Local $sPattern1, $sReplace1, $iLineNum
	
	If $iAu3SH_AddURLs Then
		For $i = 1 To $aUDFs[0]
			If $iAu3SH_AbortHighlight = 1 Then
				Return SetError(-1)
			EndIf
			
			$sPattern1 = '([^\w\$]+|\A)(?<!' & $aAu3SH_Styles[$iAu3SH_COM_Style] & ')((?i)' & $aUDFs[$i] & ')(\W+|$)'
			$sReplace1 = '\1<a href="' & StringFormat($sAu3SH_UDFs_URL, $aUDFs[$i]) & '">' & $aAu3SH_Styles[$iAu3SH_Udf_Style] & '\2' & $sAu3SH_CloseTag & '</a>\3'
			
			If StringInStr($sAu3Code, $aUDFs[$i], 2) Then
				$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
			EndIf
		Next
		
		$iLineNum = @ScriptLineNumber - 4
	Else
		For $i = 1 To $aUDFs[0]
			If $iAu3SH_AbortHighlight = 1 Then
				Return SetError(-1)
			EndIf
			
			$sPattern1 = '([^\w\$]+|\A)(?<!' & $aAu3SH_Styles[$iAu3SH_COM_Style] & ')((?i)' & $aUDFs[$i] & ')(\W+|$)'
			$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Udf_Style] & '\2' & $sAu3SH_CloseTag & '\3'
			
			If StringInStr($sAu3Code, $aUDFs[$i], 2) Then
				$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
			EndIf
		Next
		
		$iLineNum = @ScriptLineNumber - 4
	EndIf
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("", $sPattern1, $sReplace1, _
			"Highlight the UDFs - The UDF name/url in the <Search/Replace Pattern> is the last found name/url in the UDFs list (after the loop ends)", $sAu3Code, $iLineNum)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
EndFunc

Func __Au3_SyntaxHighlight_ParseKeywords(ByRef $sAu3Code)
	Local $aKeywords = __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, "au3.keywords.keywords")
	Local $sPattern1, $sReplace1, $iLineNum
	
	If $iAu3SH_AddURLs Then
		For $i = 1 To $aKeywords[0]
			If $iAu3SH_AbortHighlight = 1 Then
				Return SetError(-1)
			EndIf
			
			$sPattern1 = '([^\w\$@]|\R|\A)(?<!' & $aAu3SH_Styles[$iAu3SH_COM_Style] & ')((?i)' & $aKeywords[$i] & ')(\W|$)'
			$sReplace1 = '\1<a href="' & StringFormat($sAu3SH_Keywords_URL, $aKeywords[$i]) & '">' & $aAu3SH_Styles[$iAu3SH_Kwd_Style] & '\2' & $sAu3SH_CloseTag & '</a>\3'
			
			If StringInStr($sAu3Code, $aKeywords[$i], 2) Then
				$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
			EndIf
		Next
		
		$iLineNum = @ScriptLineNumber - 4
	Else
		For $i = 1 To $aKeywords[0]
			If $iAu3SH_AbortHighlight = 1 Then
				Return SetError(-1)
			EndIf
			
			$sPattern1 = '([^\w\$@]|\R|\A)(?<!' & $aAu3SH_Styles[$iAu3SH_COM_Style] & ')((?i)' & $aKeywords[$i] & ')(\W|$)'
			$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Kwd_Style] & '\2' & $sAu3SH_CloseTag & '\3'
			
			If StringInStr($sAu3Code, $aKeywords[$i], 2) Then
				$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
			EndIf
		Next
		
		$iLineNum = @ScriptLineNumber - 4
	EndIf
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("", _
			$sPattern1, $sReplace1, _
			"Highlight the Keywords - The Keyword name/url in the <Search/Replace Pattern> is the last found name/url in the Keywords list (after the loop ends)", _
			$sAu3Code, $iLineNum)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
EndFunc

Func __Au3_SyntaxHighlight_ParseMacros(ByRef $sAu3Code)
	Local $aMacros = __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, "au3.keywords.macros")
	Local $sPattern1, $sReplace1, $iLineNum
	
	If $iAu3SH_AddURLs Then
		For $i = 1 To $aMacros[0]
			If $iAu3SH_AbortHighlight = 1 Then
				Return SetError(-1)
			EndIf
			
			$sPattern1 = '(\W+|\R|\A)((?i)' & $aMacros[$i] & ')(\W+|$)'
			$sReplace1 = '\1<a href="' & StringFormat($sAu3SH_Macros_URL, $aMacros[$i]) & '">' & $aAu3SH_Styles[$iAu3SH_Mcro_Style] & '\2' & $sAu3SH_CloseTag & '</a>\3'
			
			If StringInStr($sAu3Code, $aMacros[$i], 2) Then
				$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
			EndIf
		Next
		
		$iLineNum = @ScriptLineNumber - 4
	Else
		For $i = 1 To $aMacros[0]
			If $iAu3SH_AbortHighlight = 1 Then
				Return SetError(-1)
			EndIf
			
			$sPattern1 = '(\W+|\R|\A)((?i)' & $aMacros[$i] & ')(\W+|$)'
			$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_Mcro_Style] & '\2' & $sAu3SH_CloseTag & '\3'
			
			If StringInStr($sAu3Code, $aMacros[$i], 2) Then
				$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
			EndIf
		Next
		
		$iLineNum = @ScriptLineNumber - 4
	EndIf
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("", $sPattern1, $sReplace1, _
			"Highlight the Macros - The Macro name/url in the <Search/Replace Pattern> is the last found name/url in the Macros list (after the loop ends)", $sAu3Code, $iLineNum)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
EndFunc

Func __Au3_SyntaxHighlight_ParsePreProcessor(ByRef $sAu3Code)
	Local $aPreProcessor = __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, "au3.keywords.preprocessor")
	Local $sPattern1, $sPattern2, $sReplace1, $sReplace2, $iLineNum
	
	If $iAu3SH_AddURLs Then
		For $i = 1 To $aPreProcessor[0]
			If $iAu3SH_AbortHighlight = 1 Then
				Return SetError(-1)
			EndIf
			
			$sPattern1 = '(\W+|\R|\A)((?i)' & $aPreProcessor[$i] & '.*)<(?:span|a href=).*?>(.*)</(?:span|a)>'
			$sPattern2 = '(\W+|\R|\A)((?i)' & $aPreProcessor[$i] & ')([^\w!%-]+?|\Z)'
			
			$sReplace1 = '\1\2\3'
			$sReplace2 = '\1<a href="' & StringFormat($sAu3SH_PreProcessor_URL, $aPreProcessor[$i]) & '">' & _
				$aAu3SH_Styles[$iAu3SH_PrPrc_Style] & '\2' & $sAu3SH_CloseTag & '</a>\3'
			
			If StringInStr($sAu3Code, $aPreProcessor[$i], 2) Then
				Do
					$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
				Until @extended = 0
				
				$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern2, $sReplace2)
			EndIf
		Next
		
		$iLineNum = @ScriptLineNumber - 4
	Else
		For $i = 1 To $aPreProcessor[0]
			If $iAu3SH_AbortHighlight = 1 Then
				Return SetError(-1)
			EndIf
			
			$sPattern1 = '(\W+|\R|\A)((?i)' & $aPreProcessor[$i] & '.*)<(?:span|a href=).*?>(.*)</(?:span|a)>'
			$sPattern2 = '(\W+|\R|\A)((?i)' & $aPreProcessor[$i] & ')([^\w!%-]+?|\Z)'
			
			$sReplace1 = '\1\2\3'
			$sReplace2 = '\1' & $aAu3SH_Styles[$iAu3SH_PrPrc_Style] & '\2' & $sAu3SH_CloseTag & '\3'
			
			If StringInStr($sAu3Code, $aPreProcessor[$i], 2) Then
				Do
					$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
				Until @extended = 0
				
				$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern2, $sReplace2)
			EndIf
		Next
		
		$iLineNum = @ScriptLineNumber - 4
	EndIf
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("", _
			$sPattern1, $sReplace1, _
			"Highlight the PreProcessor - The PreProc. name/url in the <Search/Replace Pattern> is the last found name/url in the PreProc. list (after the loop ends)", _
			$sAu3Code, $iLineNum)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
EndFunc

Func __Au3_SyntaxHighlight_ParseSpecial(ByRef $sAu3Code)
	Local $aSpecial = __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, "au3.keywords.special")
	Local $sPattern1, $sPattern2, $sReplace1, $sReplace2
	
	For $i = 1 To $aSpecial[0]
		If $iAu3SH_AbortHighlight = 1 Then
			Return SetError(-1)
		EndIf
		
		$sPattern1 = '(\W+|\R|\A)((?i)' & $aSpecial[$i] & '.*)<(?:span|a href=).*?>(.*)</(?:span|a)>'
		$sPattern2 = '(\W+|\R|\A)((?i)' & $aSpecial[$i] & '.*)'
		$sReplace1 = '\1\2\3'
		$sReplace2 = '\1' & $aAu3SH_Styles[$iAu3SH_Spec_Style] & '\2' & $sAu3SH_CloseTag
		
		If StringInStr($sAu3Code, $aSpecial[$i], 2) Then
			Do
				$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern1, $sReplace1)
			Until @extended = 0
			
			$sAu3Code = StringRegExpReplace($sAu3Code, $sPattern2, $sReplace2)
		EndIf
	Next
	
	$sAu3Code = StringReplace($sAu3Code, @CR & '</span>', '</span>', 0, 2)
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("a", $sPattern1, $sReplace1, "", $sAu3Code, @ScriptLineNumber - 10)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("b", _
			$sPattern2, $sReplace2, _
			"Highlight Special Keywords - The Spec. keywords name in the <Search Pattern> is the last found name in the Spec. keywords list (after the loop ends)", _
			$sAu3Code, @ScriptLineNumber - 8)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
EndFunc

Func __Au3_SyntaxHighlight_ParseSendKeys(ByRef $aStrings)
	Local $aSendKeys = __Au3_SyntaxHighlight_PropertyRead($sAu3_Keywords_File, "au3.keywords.sendkeys")
	Local $sPattern1, $sPattern2, $sReplace1, $sReplace2
	Local $sStrings = ''
	
	;$sReplace1 = '\1' & $aAu3SH_Styles[$iAu3SH_SntKys_Style] & '\3' & $sAu3SH_CloseTag & '\4'
	;$sPattern2 = '((''|")[^\2\>]*?)((?i)[\^+!#]*?{\S(?:[\h]+?[\h\d]*?)?})([^\2</]*?\2)'
	$sReplace1 = $aAu3SH_Styles[$iAu3SH_SntKys_Style] & '\1' & $sAu3SH_CloseTag
	$sPattern2 = '(?i)([\^+!#]*?{\S(?:[\h]+?[\h\d]*?)?})'
	$sReplace2 = $sReplace1
	
	For $i = 0 To UBound($aStrings) - 1
		If $iAu3SH_AbortHighlight = 1 Then
			Return SetError(-1)
		EndIf
		
		If $sStrings Then
			$sStrings &= @LF
		EndIf
		
		$sStrings &= $aStrings[$i]
	Next
	
	For $i = 1 To $aSendKeys[0]
		If $iAu3SH_AbortHighlight = 1 Then
			Return SetError(-1)
		EndIf
		
		;$sPattern1 = '((''|")[^\2]*?)((?i)[\^+!#]*?{' & $aSendKeys[$i] & '(?:[\h]+?[\h\d]*?)?})([^\2]*?\2)'
		$sPattern1 = '(?i)([\^+!#]*?{' & $aSendKeys[$i] & '(?:[\h]+?[\h\d]*?)?})'
		
		If StringInStr($sStrings, $aSendKeys[$i], 2) Then
			$sStrings = StringRegExpReplace($sStrings, $sPattern1, $sReplace1)
		EndIf
	Next
	
	$sStrings = StringRegExpReplace($sStrings, $sPattern2, $sReplace2)
	
	If $fAu3SH_Debug_RE_Patterns Then
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("a", $sPattern1, $sReplace1, "", $sStrings, @ScriptLineNumber - 7)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("b", $sPattern2, $sReplace2, _
			"Highlight send keys - The send key name in the <Search Pattern> is the last found name in the send keys list (after the loop ends)", $sStrings, @ScriptLineNumber - 4)
		__Au3_SyntaxHighlight_Debug_RegExp_Patterns("===")
	EndIf
	
	$aStrings = StringSplit($sStrings, @LF, 2)
EndFunc

Func __Au3_SyntaxHighlight_GetUniqueStr(ByRef $sAu3Code, $sUnique_Str)
	While StringInStr($sAu3Code, $sUnique_Str, 2)
		$sUnique_Str &= Random(10000, 99999)
		
		If $iAu3SH_AbortHighlight = 1 Then
			Return SetError(-1)
		EndIf
	WEnd
	
	Return $sUnique_Str
EndFunc

Func __Au3_SyntaxHighlight_GetCommentBlocks($sAu3Code)
	Local $aSplit_Code = StringSplit(StringStripCR($sAu3Code), @LF)
	Local $sCmmntsEnd, $sCmntBlk, $iSubCmntOpen, $sSubCmntEnd, $aCmmnt_Blocks[1] = [0]
	
	For $i = 1 To $aSplit_Code[0]
		$sCmntBlk = ''
		
		If StringRegExp($aSplit_Code[$i], '(?i)^\h*#(cs|comments-start)') Then
			$sCmmntsEnd = 'comments-end'
			
			If StringRegExp($aSplit_Code[$i], '(?i)^\h*#cs') Then
				$sCmmntsEnd = 'ce'
			EndIf
			
			$sCmntBlk &= $aSplit_Code[$i] & @CRLF
			
			For $j = $i + 1 To $aSplit_Code[0]
				If StringRegExp($aSplit_Code[$j], '(?i)^\h*#(cs|comments-start)') Then
					$iSubCmntOpen = 1
					$sSubCmntEnd = 'comments-end'
					
					If StringRegExp($aSplit_Code[$j], '(?i)^\h*#cs') Then
						$sSubCmntEnd = 'ce'
					EndIf
				EndIf
				
				$sCmntBlk &= $aSplit_Code[$j] & @CRLF
				
				If Not $iSubCmntOpen And StringRegExp($aSplit_Code[$j], '(?i)^\h*#' & $sCmmntsEnd) Then
					ExitLoop
				EndIf
				
				If $iSubCmntOpen And StringRegExp($aSplit_Code[$j], '(?i)^\h*#' & $sSubCmntEnd) Then
					$iSubCmntOpen = 0
				EndIf
			Next
			
			$i = $j + 1
		EndIf
		
		If $sCmntBlk <> '' Then
			$aCmmnt_Blocks[0] += 1
			ReDim $aCmmnt_Blocks[$aCmmnt_Blocks[0] + 1]
			$aCmmnt_Blocks[$aCmmnt_Blocks[0]] = $sCmntBlk
		EndIf
	Next
	
	Return $aCmmnt_Blocks
EndFunc

Func __Au3_SyntaxHighlight_GetKeywordStylesHeader()
	Local $sStyle
	
	Local $sFontBkColor = __Au3_SyntaxHighlight_PropertyRead($sAu3_Styles_File, 'style.au3.32', '#f0f5fa', 0)
	Local $aColor = StringSplit(__Au3_SyntaxHighlight_PropertyRead($sAu3_Styles_File, 'style.au3.0', '#000000', 0), ',')
	
	Local $sFontColor = $aColor[1]
	Local $sFontWeight = "normal"
	Local $sFontFamily = "Courier New"
	
	If $aColor[0] > 1 Then
		$sFontWeight = $aColor[2]
	EndIf
	
	If $aColor[0] > 2 Then
		$sFontFamily = $aColor[3]
	EndIf
	
	Local $sRet = _
		'<head>' & @CRLF & _
		'<style type="text/css">' & @CRLF & @CRLF & _
		'.au3_codebox' & @CRLF & _
		'{' & @CRLF & _
		'	BORDER-BOTTOM: #AAAAAA 1px solid;' & @CRLF & _
		'	BORDER-LEFT: #AAAAAA 1px solid;' & @CRLF & _
		'	BORDER-RIGHT: #AAAAAA 1px solid;' & @CRLF & _
		'	BORDER-TOP: #AAAAAA 1px solid;' & @CRLF & _
		'	PADDING-RIGHT: 8px;' & @CRLF & _
		'	PADDING-LEFT: 8px;' & @CRLF & _
		'	PADDING-BOTTOM: 8px;' & @CRLF & _
		'	PADDING-TOP: 8px;' & @CRLF & _
		'	FONT-SIZE: 12px;' & @CRLF & _
		'	FONT-FAMILY: Courier New, Verdana, Courier, Arial;' & @CRLF & _
		'	FONT-WEIGHT: ' & $sFontWeight & ';' & @CRLF & _
		'	BACKGROUND-COLOR: ' & $sFontBkColor & ';' & @CRLF & _
		'	COLOR: ' & $sFontColor & ';' & @CRLF & _
		'	WHITE-SPACE: pre;' & @CRLF & _
		'}' & @CRLF & @CRLF & _
		'a' & @CRLF & _
		'{' & @CRLF & _
		'	text-decoration:none;' & @CRLF & _
		'}' & @CRLF & @CRLF & _
		'a:hover' & @CRLF & _
		'{' & @CRLF & _
		'	text-decoration:underline;' & @CRLF & _
		'}' & @CRLF & @CRLF & _
		'pre' & @CRLF & _
		'{' & @CRLF & _
		'	font-family: Verdana, Arial, Helvetica, sans-serif, "MS sans serif";' & @CRLF & _
		'	line-height: normal;' & @CRLF & _
		'	margin-top: 0.5em;' & @CRLF & _
		'	margin-bottom: 0.5em;' & @CRLF & _
		'}' & @CRLF & @CRLF & _
		'span' & @CRLF & _
		'{' & @CRLF & _
		'	font-family: ' & $sFontFamily & ';' & @CRLF & _
		'	font-weight: ' & $sFontWeight & ';' & @CRLF & _
		'	color: ' & $sFontColor & ';' & @CRLF & _
		'}' & @CRLF & @CRLF
	
	For $i = 0 To 15
		If $iAu3SH_AbortHighlight = 1 Then
			Return SetError(-1)
		EndIf
		
		$sStyle = __Au3_SyntaxHighlight_PropertyRead($sAu3_Styles_File, "style.au3." & $i, "", 0)
		$sRet &= ".au3_S" & $i
		
		; Check if it's Special keywords, the we need to override the other style
		If $i = 11 Or $i = 12 Then
			$sRet &= ", .au3_S" & $i & " span"
		EndIf
		
		$sRet &= @CRLF & "{" & @CRLF
		
		If StringInStr($sStyle, "bold", 2) Then
			$sRet &= "	font-weight: bold;" & @CRLF
		EndIf
		
		If StringInStr($sStyle, "normal", 2) Then
			$sRet &= "	font-weight: normal;" & @CRLF
		EndIf
		
		If StringInStr($sStyle, "italics", 2) Then
			$sRet &= "	font-style: italic;" & @CRLF
		EndIf
		
		If StringInStr($sStyle, "#", 2) Then
			$sRet &= "	color: " & StringRegExpReplace($sStyle, ".*((?i)#[a-z0-9]+).*", "\1") & ";" & @CRLF
		EndIf
		
		$sRet &= "}" & @CRLF & @CRLF
	Next
	
	Return $sRet & '</style></head>' & @CRLF
EndFunc

Func __Au3_SyntaxHighlight_ConvertHtmlToBBCodeTags(ByRef $sAu3HtmlCode)
	Local $sStyle, $sTags_Start, $sTags_End, $sRet
	
	For $i = 0 To 15
		If $iAu3SH_AbortHighlight = 1 Then
			Return SetError(-1)
		EndIf
		
		$sStyle = __Au3_SyntaxHighlight_PropertyRead($sAu3_Styles_File, "style.au3." & $i, "", 0)
		
		$sTags_Start = ''
		$sTags_End = ''
		
		If StringInStr($sStyle, "bold", 2) Then
			$sTags_Start = '[b]'
			$sTags_End = '[/b]'
		EndIf
		
		If StringInStr($sStyle, "italics", 2) Then
			$sTags_Start = '[i]' & $sTags_Start
			$sTags_End &= '[/i]'
		EndIf
		
		If StringInStr($sStyle, "#", 2) Then
			$sTags_Start = '[color=' & StringRegExpReplace($sStyle, ".*((?i)#[a-z0-9]+).*", "\1") & ']' & $sTags_Start
			$sTags_End &= '[/color]'
		EndIf
		
		; Replace <a href=> tag with bbcode [url=] tag...
		$sAu3HtmlCode = StringRegExpReplace($sAu3HtmlCode, '(?si)(.*?)<a href="([^"]*)">(.*?)</a>(.*?)', '\1[url=\2]\3[/url]\4')
		
		; Replace all the styles with bbcode
		$sAu3HtmlCode = StringRegExpReplace($sAu3HtmlCode, _
			'(?si)(.*?)' & $aAu3SH_Styles[$i] & '(.*?)' & $sAu3SH_CloseTag & '(.*?)', '\1' & $sTags_Start & '\2' & $sTags_End & '\3')
		
		While StringRegExp($sAu3HtmlCode, $aAu3SH_Styles[$i] & '(.*?)' & $sAu3SH_CloseTag)
			If $iAu3SH_AbortHighlight = 1 Then
				Return SetError(-1)
			EndIf
			
			$sAu3HtmlCode = StringRegExpReplace($sAu3HtmlCode, _
				'(?si)(.*?)' & $aAu3SH_Styles[$i] & '(.*?)' & $sAu3SH_CloseTag & '(.*?)', '\1' & $sTags_Start & '\2' & $sTags_End & '\3')
		WEnd
	Next
	
	; Replace all the html entities and <br>s
	$sAu3HtmlCode = StringReplace($sAu3HtmlCode, '<br>', @CRLF, 0, 2)
	$sAu3HtmlCode = StringReplace($sAu3HtmlCode, '&lt;', '<', 0, 2)
	$sAu3HtmlCode = StringReplace($sAu3HtmlCode, '&gt;', '>', 0, 2)
	$sAu3HtmlCode = StringReplace($sAu3HtmlCode, '&nbsp;', ' ', 0, 2)
EndFunc

Func __Au3_SyntaxHighlight_PropertyRead($sFile, $sProperty, $sDefault = "", $iRetArr = 1)
	Local $aFileRead = StringSplit(StringStripCR(FileRead($sFile)), @LF)
	Local $sRet
	
	For $i = 1 To $aFileRead[0]
		If StringInStr($aFileRead[$i], $sProperty & "=", 2) Then
			$aFileRead[$i] = StringTrimLeft($aFileRead[$i], StringLen($sProperty & "="))
			
			If StringRight($aFileRead[$i], 1) = "\" Then
				$aFileRead[$i] = StringTrimRight($aFileRead[$i], 1)
			EndIf
			
			$sRet &= StringStripWS($aFileRead[$i], 3)
			
			For $j = $i+1 To $aFileRead[0]
				If StringInStr($aFileRead[$j], "=", 2) Then
					ExitLoop 2
				ElseIf StringLeft($aFileRead[$j], 1) <> "#" Then
					If StringRight($aFileRead[$j], 1) = "\" Then
						$aFileRead[$j] = StringTrimRight($aFileRead[$j], 1)
					EndIf
					
					$sRet &= " " & StringStripWS($aFileRead[$j], 3)
				EndIf
			Next
			
			ExitLoop
		EndIf
	Next
	
	If $sRet = "" Then
		$sRet = $sDefault
	EndIf
	
	$sRet = StringStripWS($sRet, 3)
	
	If $iRetArr Then
		Return __Au3_SyntaxHighlight_DeleteDuplicates(StringSplit($sRet, " 	"))
	EndIf
	
	Return $sRet
EndFunc

Func __Au3_SyntaxHighlight_DeleteDuplicates($aArray)
	Local $oDict, $aRet[UBound($aArray)]
	
	$oDict = ObjCreate('Scripting.Dictionary')
	$oDict.CompareMode = 1 ;Text mode (not case sensitive)
	
	For $i = 1 To UBound($aArray)-1
		If Not $oDict.Exists($aArray[$i]) Then
			$oDict.Add($aArray[$i], $aArray[$i])
			$aRet[0] += 1
			$aRet[$aRet[0]] = $aArray[$i]
		EndIf
	Next
	
	ReDim $aRet[$aRet[0]+1]
	Return $aRet
EndFunc

Func __Au3_SyntaxHighlight_Debug_RegExp_Patterns($iStep, $sPattern = "", $sReplace = "", $sNotes = "", $sTestStr = "", $iScriptLineNum = @ScriptLineNumber)
	If Not $fAu3SH_Debug_RE_Patterns And Not $fAu3SH_Debug_RE_WriteLog Then
		Return 0
	EndIf
	
	If $iStep = '===' Then
		$iDebug_RegExp_Step += 1
	EndIf
	
	If $iStep <> '===' Then
		$iStep = $iDebug_RegExp_Step & $iStep
	EndIf
	
	Local $sDebug_Output = '', $aLines
	
	If Not StringRegExp($iScriptLineNum, '^\d+$') Then
		$aLines = StringSplit($iScriptLineNum, '/')
		
		$sDebug_Output &= @CRLF & '   Step #' & $iStep & '...' & @CRLF
		
		For $i = 1 To $aLines[0]
			$sDebug_Output &= '    Line (' & $aLines[$i] & ') :'
			If $i < $aLines[0] Then $sDebug_Output &= @CRLF
		Next
	Else
		$sDebug_Output = @CRLF & '   Step #' & $iStep & ', Line (' & $iScriptLineNum & ') :'
	EndIf
	
	If $sPattern <> '' Then
		$sDebug_Output &= @CRLF & '- Search Pattern:' & @TAB & $sPattern
	ElseIf $iStep = '===' Then
		$sDebug_Output = '> ===================================================================================='
	ElseIf $iStep = '' Then
		$sDebug_Output = ''
	EndIf
	
	If $sPattern <> '' Then
		If IsString($sReplace) Then
			If $sReplace = '' Then
				$sDebug_Output &= @CRLF & '+ Replace Pattern:' & @TAB & ''''' (Empty string)'
			Else
				$sDebug_Output &= @CRLF & '+ Replace Pattern:' & @TAB & $sReplace
			EndIf
		Else
			If $sReplace = 0 Then
				$sDebug_Output &= @CRLF & '- Replace Pattern:' & @TAB & 'Check mathes (flag [0] in StringRegExp)'
			Else
				$sDebug_Output &= @CRLF & '+ Replace Pattern:' & @TAB & 'Array of mathes (flag [' & $sReplace & '] in StringRegExp)'
			EndIf
		EndIf
	EndIf
	
	If $sTestStr <> '' Then
		Local $iTimerInit = TimerInit()
		
		If $sReplace = '' Then
			StringRegExp($sTestStr, $sPattern, 3)
		Else
			StringRegExpReplace($sTestStr, $sPattern, $sReplace)
		EndIf
		
		$sDebug_Output &= @CRLF & ' RegExp Execution Time (ms): ' & @TAB & Round(TimerDiff($iTimerInit), 3)
	EndIf
	
	If $sNotes <> '' Then
		$sDebug_Output &= @CRLF & '! Note:' & @TAB & @TAB & @TAB & $sNotes
	EndIf
	
	If $fAu3SH_Debug_RE_WriteLog Then
		$sDebug_RegExp_LogContent &= $sDebug_Output & @CRLF
	EndIf
	
	If $fAu3SH_Debug_RE_Patterns Then
		ConsoleWrite($sDebug_Output & @CRLF)
	EndIf
EndFunc

Opt("MustDeclareVars", 0)
Opt("MustDeclareVars", $iAu3SH_MDV_Opt)

#EndRegion Internal Functions
