#pragma compile(Icon, Resources\Icon.ico)
#pragma compile(Console, True)
#pragma compile(UPX, True)
#pragma compile(Stripper, True)
#pragma compile(FileVersion, 1.0.0.0)
#pragma compile(ProductVersion, 1.0.0.0)
#pragma compile(ProductName, A3S)
#pragma compile(FileDescription, AutoIt3 Scripts Stripper)
#pragma compile(LegalCopyright,  G.Sandler)
#pragma compile(CompanyName, CreatoR's Lab)
#pragma compile(Comments, Program made by G.Sandler)
#pragma compile(Sign, G.Sandler)

#Au3Stripper_Parameters=/pe /so /rm

;http://autoit-script.ru/index.php?topic=20895.0
;https://www.autoitscript.com/autoit3/scite/docs/SciTE4AutoIt3/Au3Stripper.html

#NoTrayIcon
#include <WinAPIFiles.au3>
#include 'Includes\Dict.au3'

Global $sSrc_File = ($CmdLine[0] = 0 ? '' : $CmdLine[1])

If $CmdLine[0] = 0 Or Not FileExists($CmdLine[1]) Then
	$sSrc_File = FileOpenDialog('A3S - Select AutoIt3 Script File', '', 'AutoIt Script (*.au3)', 3, '')
	
	If @error Then
		Exit
	EndIf
EndIf

Global $sSrc_Raw 				= ''

Global $oIgnrFncs 				= _Dict_Init()
Global $oIgnrVars 				= _Dict_Init()
Global $oDfndFncs 				= _Dict_Init()
Global $oDclrdFncs 				= _Dict_Init()
Global $oDclrdVars 				= _Dict_Init()

Global Const $sConfig_File 		= @ScriptDir & '\A3S.ini' 				;Config file
Global Const $sAutoIt_Dir 		= _A3S_GetAutoItPath()					;AutoIt dir
Global Const $sIncludes_Dir 	= $sAutoIt_Dir & '\Include'				;AutoIt Includes dir
Global Const $sAu3Check_Exe 	= $sAutoIt_Dir & '\Au3Check.exe'		;AutoIt syntax check tool
Global Const $sMain_Drctv		= '#Au3Stripper' 						;Main directive prefix

;Excludes for vars strip (vars will be candidated to removal even if they have theese funcs in their value)
Global Const $sStrpVrsExclds	= 'bitor|bitxor|bitnot|bitshift|bitand|bitrotate|number|int|ptr|hex|dec|string|ubound|binary|binarylen|binarymid|hwnd|eval|asc|ascw|chr|chrw'

Global $bPreExpand 				= _A3S_GetParam('PE|PreExpand', False) 							;Replace any reference to a Global Const variable with its actual value
Global $bStrpFncs 				= _A3S_GetParam('SF|StripUnusedFunc', True) 					;Strip unused funcs
Global $bStrpVrs 				= _A3S_GetParam('SV|StripUnusedVars', True)						;Strip unused vars
Global $bStrpOnly 				= _A3S_GetParam('SO|StripOnly', False)							;Set the options to:/SF=1 /SV=1 which is now also the default when no parameters are specified
Global $bMergeOnly 				= _A3S_GetParam('MO|MergeOnly', False)							;Will produce a scriptfile as AUT2EXE includes in the Compiled EXE. This allows you to find the proper linenumber when errors are reported
Global $iStrpItrs 				= _A3S_GetParam('MI|MaxIterations', 5) 							;Number of iteration to strip
Global $bRenMin 				= _A3S_GetParam('RM|RenameMinimum', False) 						;Generates a much smaller file by substituting function and variable names with unique 2+-character names
Global $bRsln 					= _A3S_GetParam('rsln', False) 									;Replace @ScriptLineNumber with the actual line number for source debug purposes with compiled scripts

If Not _A3S_GetParam('(PE|PreExpand)|(S[FV]|StripUnused(Func|Vars))|(SO|StripOnly)|(MO|MergeOnly)|(MI|MaxIterations)|(RM|RenameMinimum)|rsln', 0, True) Then
	$bStrpOnly = True
EndIf

If $bStrpOnly Then
	$bStrpFncs = True
	$bStrpVrs = True
EndIf

If $bMergeOnly Then
	$bStrpFncs = False
	$bStrpVrs = False
EndIf

_A3S_Strip()

Func _A3S_Strip()
	Local $sAu3Check, $iTimer
	Local $aOffCode, $sOCUnqStr, $sDst_File, $hFile
	
	 $sAu3Check = _A3S_SyntaxCheck($sSrc_File)
	
	If @error Then
		ConsoleWrite($sAu3Check)
		Exit 1
	EndIf
	
	$iTimer = TimerInit()
	
	$sSrc_Raw = _A3S_MrgInclds($sSrc_File, $sSrc_Raw)
	
	$aOffCode = StringRegExp($sSrc_Raw, '(?sm)(^' & $sMain_Drctv & '_Off.*?(?:^' & $sMain_Drctv & '_On|\z))', 3)
	$sOCUnqStr = _A3S_GetUnqStr($sSrc_Raw)
	
	$sSrc_Raw = StringRegExpReplace($sSrc_Raw, '(?sm)(^' & $sMain_Drctv & '_Off.*?(?:^' & $sMain_Drctv & '_On|\z))', $sOCUnqStr)
	
	If Not $bMergeOnly Then
		_A3S_GetDefines()
		
		If $bStrpFncs Then
			_A3S_GetIgnores($sSrc_Raw, 'Funcs', $oIgnrFncs)
		EndIf
		
		If $bStrpVrs Then
			_A3S_GetIgnores($sSrc_Raw, 'Vars', $oIgnrVars)
		EndIf
		
		If $bStrpFncs Or $bStrpVrs Then
			_A3S_StripUnUsed($sSrc_Raw)
		EndIf
		
		If $bPreExpand Then
			_A3S_PreExpand($sSrc_Raw)
		EndIf
		
		If $bRenMin Then
			_A3S_RenMin($sSrc_Raw)
		EndIf
		
		If $bRsln Then
			_A3S_RpScrptLnNmbr($sSrc_Raw)
		EndIf
	EndIf
	
	;Restore string
	For $i = 0 To UBound($aOffCode) - 1
		$sSrc_Raw = StringReplace($sSrc_Raw, $sOCUnqStr, $aOffCode[$i], 1)
	Next
	
	$sDst_File = StringRegExpReplace($sSrc_File, '(\.[^\.]+)$', '_stripped$1')
	
	$hFile = FileOpen($sDst_File, 2)
	FileWrite($hFile, StringStripWS($sSrc_Raw, 3))
	FileClose($hFile)
	
	$sAu3Check = _A3S_SyntaxCheck($sDst_File)
	
	If @error Then
		ConsoleWrite($sAu3Check)
		Exit 2
	EndIf
	
	ConsoleWrite('Stripped successfully and saved as "' & $sDst_File & '"' & @CRLF & 'Time: ' & Round(TimerDiff($iTimer) / 1000, 1) & ' sec.' & @CRLF)
	Exit 0
EndFunc

Func _A3S_SyntaxCheck($sFile)
	Local $iPID, $sRead = ''
	
	If Not FileExists($sAu3Check_Exe) Then
		Return SetError(1, 0, 'Unable to find Au3Check.exe')
	EndIf
	
	$iPID = Run('"' & $sAu3Check_Exe & '" -q "' & $sFile & '"', '', @SW_HIDE, 6)
	ProcessWaitClose($iPID)
	
	$sRead = StdoutRead($iPID)
	
	If StringRegExp($sRead, '(?i)- [1-9]+ error\(s\)') Then
		Return SetError(2, 0, $sRead)
	EndIf
	
	Return ''
EndFunc

Func _A3S_GetParam($sParam, $vDefault, $bCheck = False)
	Local Static $sTmpSrcRaw
	
	If $sSrc_Raw = '' Then
		$sSrc_Raw = FileRead($sSrc_File)
		
		$sTmpSrcRaw = $sSrc_Raw
		_A3S_StrpCmntBlcks($sTmpSrcRaw)
	EndIf
	
	Local $aPttrns[2][2] = [[$sTmpSrcRaw, '(?mi)^\h*' & $sMain_Drctv & '_Parameters=.*/(?:' & $sParam & ')(?:\h*=\h*(\d))?(?:\h|\h*$)'], [$CmdLineRaw, '(?i).*/(?:' & $sParam & ')(?:\h*=\h*(\d))?(?:\h|\h*$)']]
	Local $sRet
	
	If Not StringRegExp($aPttrns[0][0], $aPttrns[0][1]) And Not StringRegExp($aPttrns[1][0], $aPttrns[1][1]) Then
		If $bCheck Then
			Return False
		EndIf
		
		Return $vDefault
	EndIf
	
	If $bCheck Then
		Return True
	EndIf
	
	For $i = 0 To UBound($aPttrns) - 1
		$sRet = StringRegExpReplace($aPttrns[$i][0], $aPttrns[$i][1], '\1')
		
		If @extended Then
			If StringIsDigit($sRet) Then
				Return Number($sRet)
			EndIf
			
			Return True
		EndIf
	Next
	
	Return False
EndFunc

Func _A3S_GetAutoItPath()
	Local $sAutoItDir = IniRead($sConfig_File, 'Main', 'AutoItDir', 'Error')
	
	If $sAutoItDir <> 'Error' And FileExists($sAutoItDir & '\AutoIt3.exe') Then
		Return _WinAPI_GetFullPathName($sAutoItDir)
	EndIf
	
	$sAutoItDir = @ScriptDir
	
	For $i = 1 To 4
		If FileExists($sAutoItDir & '\AutoIt3.exe') Then
			Return _WinAPI_GetFullPathName($sAutoItDir)
		EndIf
		
		$sAutoItDir &= '\..'
	Next
	
	$sAutoItDir = RegRead('HKLM\SOFTWARE\AutoIt v3\AutoIt', 'InstallDir')
	Return _WinAPI_GetFullPathName($sAutoItDir)
EndFunc

Func _A3S_GetDefines()
	;Functionname,Parameter to check,0=check for function/1=check for Variable
	
	;Defaults
	Local $aSect = IniReadSection($sConfig_File, 'Defined')
	
	If @error Or Not IsArray($aSect) Then
		Dim $aSect[][2] = _
			[ _
				['adlibenable', '1,0'], _
				['adlibregister', '1,0'], _
				['adlibunregister', '1,0'], _
				['call', '1,0'], _
				['dllcallbackregister', '1,0'], _
				['eval', '1,1'], _
				['guictrlregisterlistviewsort', '2,0'], _
				['guictrlsetonevent', '2,0'], _
				['guiregistermsg', '2,0'], _
				['guisetonevent', '2,0'], _
				['hotkeyset', '2,0'], _
				['isdeclared', '1,1'], _
				['objevent', '2,0'], _
				['onautoitexitregister', '1,0'], _
				['onautoitexitunregister', '1,0'], _
				['opt', '2,0'], _
				['traysetonevent', '2,0'], _
				['trayitemsetonevent', '2,0'] _
			]
	EndIf
	
	For $i = 1 To $aSect[0][0]
		_Dict_KeyAdd($aSect[$i][0], $aSect[$i][1], $oDfndFncs)
	Next
EndFunc

Func _A3S_GetIgnores($sSrcRaw, $sDrctvSufix, ByRef $oIgnrs, $bDfndHdr = False)
	Local $aIgnores, $aSplit, $aDefines, $aParams, $iParam, $iCheck, $aFuncs
	
	;Get ignores from directive #Au3Stripper_Ignore_Funcs/Vars
	$aIgnores = StringRegExp($sSrcRaw, '(?mi)^\h*' & $sMain_Drctv & '_Ignore_' & $sDrctvSufix & '\h*=\h*(.*)', 3)
	
	For $i = 0 To UBound($aIgnores) - 1
		$aSplit = StringRegExp($aIgnores[$i], '([^,]+)', 3)
		
		For $j = 0 To UBound($aSplit) - 1
			_Dict_KeyAdd($aSplit[$j], '', $oIgnrs)
		Next
	Next
	
	;Get ignores from defined funcs
	$aDefines = _Dict_GetKeys($oDfndFncs)
	
	For $i = 0 To UBound($aDefines) - 1
		$aParams = StringRegExp(_Dict_GetItem($aDefines[$i], $oDfndFncs), '(\d+)', 3)
		
		If UBound($aParams) < 2 Then
			ContinueLoop
		EndIf
		
		$iParam = $aParams[0]
		$iCheck = $aParams[1]
		
		If ($sDrctvSufix = 'Funcs' And $iCheck = 0) Or ($sDrctvSufix = 'Vars' And $iCheck = 1) Then
			If $iParam <= 0 Then
				$iParam = 1
			EndIf
			
			$aFuncs = StringRegExp($sSrcRaw, '(?mi)^.*(?:\W|\b)' & $aDefines[$i] & '\h*\((?:.+,\h*){' & $iParam - 1 & '}["''](.+?)["''].*$', 3)
			
			For $j = 0 To UBound($aFuncs) - 1
				If $bDfndHdr Then
					$aFuncs[$j] = $aDefines[$i] & '|' & $aFuncs[$j]
				EndIf
				
				_Dict_KeyAdd($aFuncs[$j], '', $oIgnrs)
			Next
		EndIf
	Next
	
	;Get ignores from #OnAutoItStartRegister directive (only for functions ignores)
	If $sDrctvSufix = 'Funcs' Then
		$aFuncs = StringRegExp($sSrcRaw, '(?mi)^#OnAutoItStartRegister\h*(?:(["'']?)([a-zA-Z_]\w*)\1|<([a-zA-Z_]\w*)>)', 3)
		
		For $i = 0 To UBound($aFuncs) - 1
			If $aFuncs[$i] <> '"' And $aFuncs[$i] <> "'" Then
				If $bDfndHdr Then
					$aFuncs[$i] = '#OnAutoItStartRegister|' & $aFuncs[$i]
				EndIf
				
				_Dict_KeyAdd($aFuncs[$i], '', $oIgnrs)
			EndIf
		Next
	EndIf
	
	;Add $CmdLine[Raw] ignore (only for variables ignores)
	If $sDrctvSufix = 'Vars' Then
		_Dict_KeyAdd('$CmdLine', '', $oIgnrs)
		_Dict_KeyAdd('$CmdLineRaw', '', $oIgnrs)
	EndIf
EndFunc

Func _A3S_MrgInclds($sSrcFile, $sRead = '')
	If $sSrcFile = '' Or Not FileExists($sSrcFile) Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local Static $sSrcRaw = ''
	Local Static $iLine = 0
	Local Static $oInclds = _Dict_Init()
	
	Local $sInclude = '', $sLine = '', $sTmp = '', $sFName = '', $sFnc = '', $aFnc
	Local $sScriptDir = StringRegExpReplace($sSrcFile, '\\[^\\]+$', '')
	
	If $sRead = '' Then
		$sRead = FileRead($sSrcFile)
	EndIf
	
	;!!! Strip comment block here (before checking #include-once)...
	_A3S_StrpCmntBlcks($sRead)
	
	If StringRegExp($sRead, '(?mi)^\h*#include-once') Then
		If _Dict_KeyExists($sSrcFile, $oInclds) Then
			$sRead = StringRegExpReplace($sRead, '(?msi)\R?^\h*#include-once.*', '')
		Else
			_Dict_KeyAdd($sSrcFile, '', $oInclds)
		EndIf
	EndIf
	
	$sRead = StringStripCR($sRead)
	Local $aRead = StringSplit($sRead, @LF)
	
	For $i = 1 To $aRead[0]
		If StringStripWS($aRead[$i], 8) = '' Then
			ContinueLoop
		EndIf
		
		If StringRegExp($aRead[$i], '(?i)^\h*(;|#include-once|#pragma\h+compile\()') Then
			ContinueLoop
		EndIf
		
		_A3S_StrpCmnts($aRead[$i])
		
		If Not StringRegExp($aRead[$i], '(?i)^\h*#include\h*[<"'']') Then
			$sLine = $aRead[$i]
			
			;Merge broken (with _) lines
			If StringRegExp($aRead[$i], '_\h*$') Then
				$sLine = StringRegExpReplace($aRead[$i], '^\h+|[_\h]+$', '')
				
				While 1
					$i += 1
					
					If $i >= $aRead[0] Then
						ExitLoop
					EndIf
					
					_A3S_StrpCmnts($aRead[$i])
					$aRead[$i] = StringRegExpReplace($aRead[$i], '[_\h]+$', '')
					
					If @extended = 0 Then
						$sLine &= ' ' & StringStripWS($aRead[$i], 1)
						ExitLoop
					EndIf
					
					$sLine &= ' ' & StringStripWS($aRead[$i], 1)
				WEnd
			EndIf
			
			$iLine += 1
			$sLine = StringStripWS($sLine, 1)
			$sSrcRaw &= $sLine & @CRLF
			
			If $bStrpFncs Or $bStrpVrs Then
				;It's a funcion start (get func name)
				If $sFName = '' Then
					$aFnc = StringRegExp($sLine, '(?i)^(?:Volatile\h+)?Func\h+([a-zA-Z_]\w*)\h*\(', 3)
					
					If IsArray($aFnc) Then
						$sFName = $aFnc[0]
					EndIf
				EndIf
				
				;Func already started, so collect here func content
				If $sFName Then
					$sFnc &= $sLine & @CRLF
					
					;It's a funcion end (add collected func to $oDclrdFncs dict)
					If StringRegExp($sLine, '(?i)^EndFunc(\h*;.*)?') Then
						_Dict_KeyAdd($sFName, $sFnc, $oDclrdFncs)
						
						$sFName = ''
						$sFnc = ''
					EndIf
				EndIf
			EndIf
			
			If $bStrpVrs Then
				;Func already started, so the current var is in the function scope/range
				If $sFName Then
					If StringRegExp($sLine, '(?i)^(?:Global|Dim)\h+(?:(?:Static\h+|Const\h+|Enum\h+(?:Step\h+[\*\+-]?\h*\d+\h*)?))?\$\w') Then
						;Current var is in the Global scope (even if it's inside the func)
						;Remove all strings...
						$sTmp = StringRegExpReplace($sLine, '"[^"]*"|''[^'']*''', '')
						
						;Check if there is no function call (except excluded funcs)...
						If Not StringRegExp($sTmp, '(?i)=\h*[a-zA-Z_]\w*(?<!' & $sStrpVrsExclds & ')\h*\(') Then
							;If not, add all vars in line to the $oDclrdVars as Global [Global,Line Pos=aVars]
							_Dict_KeyAdd('Global,' & $iLine, $sLine, $oDclrdVars)
						EndIf
					ElseIf StringRegExp($sLine, '(?i)^Local\h+(?:(?:Static\h+|Const\h+|Enum\h+(?:Step\h+[\*\+-]?\h*\d+\h*)?))?\$\w') Then
						;Current var is in the Local scope, so remember the func name to check later
						
						;Remove all strings...
						$sTmp = StringRegExpReplace($sLine, '"[^"]*"|''[^'']*''', '')
						
						;Check if there is no function call (except Bit* funcs)...
						If Not StringRegExp($sTmp, '(?i)=\h*[a-zA-Z_]\w*(?<!' & $sStrpVrsExclds & ')\h*\(') Then
							;If not, add all vars in line to the $oDclrdVars as Local [Func Name,Line Pos=aVars]
							_Dict_KeyAdd('Func_' & $sFName & ',' & $iLine, $sLine, $oDclrdVars)
						EndIf
					EndIf
				ElseIf StringRegExp($sLine, '(?i)^(?:(?:Global|Local|Dim)\h+)?(?:(?:Static\h+|Const\h+|Enum\h+(?:Step\h+[\*\+-]?\h*\d+\h*)?))?\$\w') Then
					;We are not in func, and the current var is in the global scope (always, even when Local used)
					
					;Remove all strings...
					$sTmp = StringRegExpReplace($sLine, '"[^"]*"|''[^'']*''', '')
					
					;Check if there is no function call (except Bit* funcs)...
					;If Not StringRegExp($sTmp, '\w+\(|=\h*\$\w') Then
					If Not StringRegExp($sTmp, '(?i)=\h*[a-zA-Z_]\w*(?<!' & $sStrpVrsExclds & ')\h*\(') Then
						;If not, add all vars in line to the $oDclrdVars as Global [Global,Line Pos=aVars]
						_Dict_KeyAdd('Global,' & $iLine, $sLine, $oDclrdVars)
					EndIf
				EndIf
			EndIf
			
			ContinueLoop
		EndIf
		
		;Get include path
		$sInclude = _A3S_Incld2Pth($aRead[$i], $sScriptDir)
		
		If Not @error Then
			;Recursive call for include
			$sSrcRaw = _A3S_MrgInclds($sInclude)
		EndIf
	Next
	
	Return $sSrcRaw
EndFunc

Func _A3S_StripUnUsed(ByRef $sSrcRaw)
	Local $aFuncs, $sFuncs, $sOrigFuncs, $sTmpFuncs, $sFunc, $sTmpSrcRaw, $sUnUsedFuncs
	Local $aDclrdVrs, $sScope, $sLine, $sLine, $sTmpLine, $aVars
	
	If $bStrpFncs Then
		For $xItr = 1 To $iStrpItrs
			$sUnUsedFuncs = '|'
			
			;Remove preprocessors (only when stripping unused stuff)
			$sSrcRaw = StringRegExpReplace($sSrcRaw, '(?mi)^#(?!pragma|NoTrayIcon).*$\R', '')
			
			;Src without funcs
			$sTmpSrcRaw = StringRegExpReplace($sSrcRaw, '(?smi)(^(?:Volatile\h+)?Func\h+[a-zA-Z_]\w*\h*\(.+?\r?\nEndFunc$(?:\r?\n)?)', '')
			
			;Get array of func names
			$aFuncs = _Dict_GetKeys($oDclrdFncs)
			$sFuncs = ''
			
			;Get funcs content
			For $i = 0 To UBound($aFuncs) - 1
				$sFuncs &= _Dict_GetItem($aFuncs[$i], $oDclrdFncs)
			Next
			
			;Original funcs content
			$sOrigFuncs = $sFuncs
			
			;Strip the "[Volatile ]Func funcname(" part
			$sFuncs = StringRegExpReplace($sFuncs, '(?mi)^(?:Volatile\h+)?Func\h+[a-zA-Z_]\w*\h*\(', '')
			
			For $i = 0 To UBound($aFuncs) - 1
				;If it's in the ignore list, skip to next func
				If _Dict_KeyExists($aFuncs[$i], $oIgnrFncs) Then
					ContinueLoop
				EndIf
				
				;If current function not used in src content, check funcs content...
				If Not _A3S_FncIsUsd($sTmpSrcRaw, $aFuncs[$i]) Then
					$sTmpFuncs = ''
					
					;If it's before last iteration, check if the function used only in itself
					If $xItr = $iStrpItrs - 1 Then
						;Strip this function from funcs content
						$sTmpFuncs = StringRegExpReplace($sOrigFuncs, '(?smi)(^(?:Volatile\h+)?Func\h+' & $aFuncs[$i] & '\h*\(.+?\r?\nEndFunc$(?:\r?\n)?)', '')
					EndIf
					
					;If current function not used in funcs content, or not used except in itself, remove it from the $oDclrdFncs list and add it to the $sUnUsedFuncs list (to strip later)
					If Not _A3S_FncIsUsd($sFuncs, $aFuncs[$i]) Or ($sTmpFuncs <> '' And Not _A3S_FncIsUsd($sTmpFuncs, $aFuncs[$i])) Then
						_Dict_KeyRemove($aFuncs[$i], $oDclrdFncs)
						$sUnUsedFuncs &= $aFuncs[$i] & '|'
					EndIf
				EndIf
			Next
			
			;Strip unused funcs
			If $sUnUsedFuncs <> '|' Then
				$sUnUsedFuncs = StringRegExpReplace($sUnUsedFuncs, '^\|+|\|+$', '')
				$sSrcRaw = StringRegExpReplace($sSrcRaw, '(?smi)^(?:Volatile\h+)?Func\h+(?:' & $sUnUsedFuncs & ')\h*\(.+?^EndFunc$\R*', '')
			EndIf
		Next
	EndIf
	
	If $bStrpVrs Then
		If Not $bStrpFncs Then
			;Remove preprocessors (only when stripping unused stuff)
			$sSrcRaw = StringRegExpReplace($sSrcRaw, '(?mi)^#(?!pragma|NoTrayIcon).*$\R', '')
			
			;Src without funcs
			$sTmpSrcRaw = StringRegExpReplace($sSrcRaw, '(?smi)(^(?:Volatile\h+)?Func\h+[a-zA-Z_]\w*\h*\(.+?\r?\nEndFunc$(?:\r?\n)?)', '')
		EndIf
		
		;Get array of func names (it's only used funcs)
		$aFuncs = _Dict_GetKeys($oDclrdFncs)
		$sFuncs = ''
		
		;Get funcs content
		For $i = 0 To UBound($aFuncs) - 1
			$sFuncs &= _Dict_GetItem($aFuncs[$i], $oDclrdFncs)
		Next
		
		For $x = 1 To $iStrpItrs
			;Get array of declared vars
			$aDclrdVrs = _Dict_GetKeys($oDclrdVars)
			
			For $i = 0 To UBound($aDclrdVrs) - 1
				;Get scope, line, vars
				$sScope = StringLeft($aDclrdVrs[$i], StringInStr($aDclrdVrs[$i], ',', 2) - 1)
				$sLine = _Dict_GetItem($aDclrdVrs[$i], $oDclrdVars)
				
				If $sLine = '' Then
					ContinueLoop
				EndIf
				
				$sTmpLine = StringRegExpReplace($sLine, '(?i)(?:' & $sStrpVrsExclds & ')\h*\([^()]+\)', '')
				$sTmpLine = StringRegExpReplace($sTmpLine, '\h*=\h*[^,]+', '')
				
				$aVars = StringRegExp($sTmpLine, '(\$\w+)', 3)
				
				For $j = 0 To UBound($aVars) - 1
					;If it's in the ignore list, skip to next var
					If _Dict_KeyExists(StringTrimLeft($aVars[$j], 1), $oIgnrVars) Then
						ContinueLoop 2
					EndIf
					
					;Check if current var is used...
					
					;If it's in Global scope, check in raw src without funcs ($sTmpSrcRaw), and then check in funcs src
					If $sScope = 'Global' Then
						If _A3S_VarIsUsd($sTmpSrcRaw, $aVars[$j], 'Global') Or _A3S_VarIsUsd($sFuncs, $aVars[$j], 'Global_F') Then
							;One of the vars in $aVars is used, so skip the strip
							ContinueLoop 2
						EndIf
					Else
						;It's in Local scope ($sScope is Func Name where it was declared, "Func_" at the begining should be removed)
						;So check in the func content
						$sFunc = StringTrimLeft($sScope, 5)
						
						;It's "dead" function (stripped), so skip the strip
						If Not _Dict_KeyExists($sFunc, $oDclrdFncs) Then
							ContinueLoop 2
						EndIf
						
						If _A3S_VarIsUsd(_Dict_GetItem($sFunc, $oDclrdFncs), $aVars[$j], 'Local') Then
							;One of the vars in $aVars is used, so skip the strip
							ContinueLoop 2
						EndIf
					EndIf
				Next
				
				;If all of the vars in $aVars not used, remove the line with the var (find it by line position from where it's declared)
				$sSrcRaw = StringReplace($sSrcRaw, $sLine & @CRLF, '', 1, 1)
				$sTmpSrcRaw = StringReplace($sTmpSrcRaw, $sLine & @CRLF, '', 1, 1)
				
				If $iStrpItrs > 1 Then
					_Dict_KeyRemove($aDclrdVrs[$i], $oDclrdVars)
				EndIf
			Next
		Next
	EndIf
EndFunc

Func _A3S_PreExpand(ByRef $sSrcRaw)
	Local $aDclrdVrs, $sScope, $sLine, $aVarVal
	
	;Get array of declared vars
	$aDclrdVrs = _Dict_GetKeys($oDclrdVars)
	
	For $i = 0 To UBound($aDclrdVrs) - 1
		;Get scope, line, vars
		$sScope = StringLeft($aDclrdVrs[$i], StringInStr($aDclrdVrs[$i], ',', 2) - 1)
		$sLine = _Dict_GetItem($aDclrdVrs[$i], $oDclrdVars)
		
		;If it's global scope and it's a constant...
		If $sScope = 'Global' And StringRegExp($sLine, '(?i)^Global\h+Const\h+\$') Then
			;Get var = val pair
			$aVarVal = StringRegExp($sLine, '(?i)^Global\h+Const\h+(\$\w+)\h*=\h*([^,]+)$', 3)
			
			;If found...
			If UBound($aVarVal) = 2 Then
				$iExt = 0
				
				;Replace var with actual value (if it's not declared by other call)
				While 1
					$sSrcRaw = StringRegExpReplace($sSrcRaw, _
						'(?im)^\h*(?!(?:\h*|(?:Global|Local|Dim|Const|Static|Enum(?:\h+Step\h+[\*\+-]?\h*\d))\h+)*(?:.*,\h*)?\' & $aVarVal[0] & '\b\h*(?:,|=|$)|Func\h+.*[,(]\h*\' & $aVarVal[0] & '\b\h*(?:,|=|\)))(.*)\' & $aVarVal[0] & '\b', _
						'${1}' & $aVarVal[1])
					
					If @extended = 0 Then
						ExitLoop
					EndIf
					
					$iExt += @extended
				WEnd
				
				If $iExt Then
					$sSrcRaw = StringReplace($sSrcRaw, $sLine & @CRLF, '', 1, 1)
					;_Dict_KeyRemove($aDclrdVrs[$i], $oDclrdVars)
				EndIf
			EndIf
		EndIf
	Next
EndFunc

Func _A3S_RenMin(ByRef $sSrcRaw)
	Local $aStrs, $sUnqStr, $sStr, $sUnqName, $aFuncs, $aVars
	
	;Get strings
	$aStrs = StringRegExp($sSrcRaw, '("[^"]*"|''[^'']*'')', 3)
	$sUnqStr = _A3S_GetUnqStr($sSrcRaw)
	
	;Strip strings (but not from ignores)
	For $i = 0 To UBound($aStrs) - 1
		$sStr = StringTrimRight(StringTrimLeft($aStrs[$i], 1), 1)
		
		If Not _Dict_KeyExists($sStr, $oIgnrFncs) And Not _Dict_KeyExists($sStr, $oIgnrVars) Then
			;Replace strings with unique string
			$sSrcRaw = StringReplace($sSrcRaw, $aStrs[$i], $sUnqStr & $i, 1, 2)
		EndIf
	Next
	
	;Get ignores - to get defined with header (defined func name)
	_Dict_RemoveAll($oIgnrFncs)
	_Dict_RemoveAll($oIgnrVars)
	
	_A3S_GetIgnores($sSrcRaw, 'Funcs', $oIgnrFncs, True)
	_A3S_GetIgnores($sSrcRaw, 'Vars', $oIgnrVars, True)
	
	;Get funcs
	$aFuncs = _Dict_GetKeys($oDclrdFncs)
	
	For $i = 0 To UBound($aFuncs) - 1
		If _Dict_KeyExists($aFuncs[$i], $oIgnrFncs) Or Not StringRegExp($sSrcRaw, '\b' & $aFuncs[$i] & '\b') Then
			ContinueLoop
		EndIf
		
		$sUnqName = _A3S_GetUnqName($sSrcRaw, 'Func')
		$sSrcRaw = StringRegExpReplace($sSrcRaw, '\b' & $aFuncs[$i] & '\b', $sUnqName)
	Next
	
	;Get vars
	$aVars = StringRegExp($sSrcRaw, '(\$\w+)\b', 3)
	
	For $i = 0 To UBound($aVars) - 1
		If _Dict_KeyExists($aVars[$i], $oIgnrVars) Or Not StringRegExp($sSrcRaw, '\' & $aVars[$i] & '\b') Then
			ContinueLoop
		EndIf
		
		$sUnqName = _A3S_GetUnqName($sSrcRaw, 'Var')
		$sSrcRaw = StringRegExpReplace($sSrcRaw, '\' & $aVars[$i] & '\b', '$' & $sUnqName)
	Next
	
	;Restore strings
	For $i = 0 To UBound($aStrs) - 1
		$sSrcRaw = StringReplace($sSrcRaw, $sUnqStr & $i, $aStrs[$i], 1, 2)
	Next
EndFunc

Func _A3S_RpScrptLnNmbr(ByRef $sSrcRaw)
	Local $iPos
	
	While 1
		$iPos = StringInStr($sSrcRaw, '@ScriptLineNumber', 2)
		
		If Not $iPos Then
			ExitLoop
		EndIf
		
		StringReplace(StringMid($sSrcRaw, 1, StringInStr($sSrcRaw, @LF, 2, -1, $iPos)), @LF, '')
		$sSrc_Raw = StringReplace($sSrcRaw, '@ScriptLineNumber', @extended + 1, 1, 2)
	WEnd
EndFunc

Func _A3S_FncIsUsd($sSrc, $sFName)
	Local $sLines
	
	;Strip strings & directives
	$sLines = StringRegExpReplace($sSrc, '(?m)"[^"]*"|''[^'']*''|^\h*#.*$', '')
	
	;Check if there is function in format funcname(...
	If StringRegExp($sLines, '(?im)^.*?(?:\W|\b)' & $sFName & '\h*\(') Then
		Return 1
	EndIf
	
	Return 0
EndFunc

Func _A3S_VarIsUsd($sSrc, $sVName, $sScope)
	Local $aLine
	
	;Explanation:
	;* If $sScope = 'Global', the $sSrc is $sTmpSrcRaw (src without funcs)
	;* If $sScope = 'Global_F', the $sSrc is $sFuncs (funcs content src)
	;* If $sScope = 'Local', the $sSrc is $sFunc (func content where the var was declared)
	
	If $sScope = 'Local' Then
		;Remove all lines until "Local .* $sVName" (to detect correctly local var usage)
		$sSrc = StringRegExpReplace($sSrc, '(?mi)[\s\S]+?(?=^Local\h+(?:\' & $sVName & '|.+,\h*\' & $sVName & ')\h*(?:,|=|$))', '', 1)
	EndIf
	
	;Get all occurrences of the var
	$aLine = StringRegExp($sSrc, '(?mi)(^.*\' & $sVName & '(?:\b|\W).*$)', 3)
	
	Switch $sScope
		Case 'Global', 'Local'
			;Try to find second occurrence of the var
			If UBound($aLine) >= 2 Then
				;It's used if declared var is in Global/Local
				Return 1
			EndIf
		Case 'Global_F'
			;It's used if the first occurrence of the var found and it's not set as Local
			If IsArray($aLine) And Not StringRegExp($aLine[0], '(?i)^Local\h+(?:\' & $sVName & '|.+,\h*\' & $sVName & ')\h*(?:,|=|$)') Then
				Return 1
			EndIf
	EndSwitch
	
	Return 0
EndFunc

Func _A3S_GetUnqName($sSrc, $sFuncVar)
	Local $sUnqName, $sName = '', $iChr = 97, $iNum = 1
	Local $sSgn = '\$'
	
	If $sFuncVar = 'Func' Then
		$sSgn = '\b'
	EndIf
	
	Do 
		$sUnqName = $sName & Chr($iChr) & $iNum
		
		$iNum += 1
		
		If $iNum = 11 Then
			$iNum = 1
			$iChr += 1
			
			If $iChr = 123 Then
				$sName = $sUnqName
				$iChr = 97
			EndIf
		EndIf
	Until StringRegExp($sSrc, $sSgn & $sUnqName & '\b') = 0
	
	Return $sUnqName
EndFunc

Func _A3S_GetUnqStr($sSrc)
	Local $sUnqStr, $iNum = 1
	
	Do 
		$sUnqStr = '@UNIQUE_STR_' & $iNum & '@'
		$iNum += 1
	Until StringRegExp($sSrc, $sUnqStr) = 0
	
	Return $sUnqStr
EndFunc

Func _A3S_Incld2Pth($sInclude, $sScriptDir)
	Local $aRegExp, $aRet, $sSYS, $sA3S, $sWorkDir
	Local $iError = 0, $sRet = ''
	
	$aRegExp = StringRegExp($sInclude, '(?i)^\h*#include\h*(<|"|'')([^>"'']+)(?:>|"|'')\h*(;.*?)?$', 3)
    
    If UBound($aRegExp) < 2 Then
		Return SetError(1, 0, '')
	EndIf
	
    $sInclude = $aRegExp[1]
	
	;Get User Include folders
	Local Static $aUDL = StringRegExp(RegRead('HKCU\Software\AutoIt v3\AutoIt', 'Include'), '([^;]+)(?:;|$)', 3)
	
	While 1
		;Check include type 4 (include with full path)
		If Not _WinAPI_PathIsRelative($sInclude) Then
			If FileExists($sInclude) Then
				$sRet = $sInclude
			Else
				$iError = 2
			EndIf
			
			ExitLoop
		EndIf
		
		;Set Current & AutoIt Include file
		$sA3S = $sScriptDir & '\' & $sInclude
		$sSYS = $sIncludes_Dir & '\' & $sInclude
		
		;Check include type 1 and 2 (before user includes check)
		If $aRegExp[0] == '<' Then
			If FileExists($sSYS) Then
				$sRet = $sSYS
				ExitLoop
			EndIf
		ElseIf $aRegExp[0] == '"' Or $aRegExp[0] == "'" Then
			If FileExists($sA3S) Then
				$sRet = $sA3S
				ExitLoop
			EndIf
		EndIf
		
		;Check include type 3 (search in user includes)
		For $i = 0 To UBound($aUDL) - 1
			$aUDL[$i] &= '\' & $sInclude
			
			If FileExists($aUDL[$i]) Then
				$sRet = $aUDL[$i]
				ExitLoop 2
			EndIf
		Next
		
		;Check include type 1 and 2 (after user includes check)
		If $aRegExp[0] == '<' Then
			If FileExists($sA3S) Then
				$sRet = $sA3S
				ExitLoop
			EndIf
		ElseIf $aRegExp[0] == '"' Or $aRegExp[0] == "'" Then
			If FileExists($sSYS) Then
				$sRet = $sSYS
				ExitLoop
			EndIf
		EndIf
		
		;Include file not found
		$iError = 3
		ExitLoop
	WEnd
	
	If Not $iError Then
		$sWorkDir = @WorkingDir
		
		If $sWorkDir <> $sScriptDir Then
			FileChangeDir($sScriptDir)
		EndIf
		
		$sRet = _WinAPI_GetFullPathName($sRet)
		
		If @error Or $sRet = '' Then
			$iError = 4
		EndIf
		
		If $sWorkDir <> $sScriptDir Then
			FileChangeDir($sWorkDir)
		EndIf
	Else
		$sRet = ''
	EndIf
	
    Return SetError($iError, 0, $sRet)
EndFunc

Func _A3S_StrpCmntBlcks(ByRef $sString)
	;If Not StringRegExp($sString, '(?mi)^\h*#(?:cs|comments-start)(?:[\h;].*)?$') Then
	;	Return
	;EndIf
	
	Local $aSplit = StringSplit(StringStripCR($sString), @LF)
	Local $iCmntsStart_Count
	
	$sString = ''
	
	For $i = 1 To $aSplit[0]
		If StringRegExp($aSplit[$i], '(?i)^\h*#(cs|comments-start)([\h;].*)?$') Then
			$iCmntsStart_Count = 1
			
			While 1
				If $i + 1 >= $aSplit[0] Then
					ExitLoop
				EndIf
				
				$i += 1
				
				If StringRegExp($aSplit[$i], '(?i)^\h*#(?:cs|comments-start)(?:[\h;].*)?') Then
					$iCmntsStart_Count += 1
					ContinueLoop
				EndIf
				
				If StringRegExp($aSplit[$i], '(?i)^\h*#(?:ce|comments-end)(?:[\h;].*)?') Then
					$iCmntsStart_Count -= 1
					
					If $iCmntsStart_Count <= 0 Then
						ExitLoop
					EndIf
				EndIf
			WEnd
			
			ContinueLoop
		EndIf
		
		$sString &= $aSplit[$i] & @CRLF
	Next
EndFunc

Func _A3S_StrpCmnts(ByRef $sLine)
	Local $aStrs = StringRegExp($sLine, '("[^"]*"|''[^'']*'')', 3)
	Local $sStrs = StringRegExpReplace($sLine, '("[^"]*"|''[^'']*'')', '#~_~@~_~#')
	Local $iInStr = StringInStr($sStrs, ';', 2)
	
	If $iInStr = 0 Then
		Return
	EndIf
	
	$sLine = StringStripWS(StringLeft($sStrs, $iInStr - 1), 2)
	
	For $i = 0 To UBound($aStrs) - 1
		$sLine = StringReplace($sLine, '#~_~@~_~#', $aStrs[$i], 1, 2)
	Next
EndFunc
