#include-once

#Region Header

#CS
	Name:				GroupCheckBox.au3
	Description:		Allows to create and manage group checkboxes.
	Author:				Copyright  2015 CreatoR's Lab (G.Sandler), www.creator-lab.ucoz.ru, www.autoit-script.ru. All rights reserved.
	AutoIt version:		3.3.6.1 - 3.3.12.0
	UDF version:		0.2
	
	History:
						0.2
						* _GroupCheckBox_Delete renamed to _GroupCheckBox_Destroy.
						* _GroupCheckBox_CheckAll renamed to _GroupCheckBox_SetState (changed parameters, check the function description for more details).
						* All $GCB_STATE_* constants renamed to $GCB_CHECK_STATE_*.
						* Changed return values from _GroupCheckBox_GetInfo (check the function description for more details).
						+ Added optional $iCheckBox parameter to _GroupCheckBox_GetState function (check the function description for more details).
						+ Added _GroupCheckBox_Add and _GroupCheckBox_Remove functions (check the functions description for more details).
						+ Added new user constants: $GCB_INDEX_LEFT, $GCB_INDEX_TOP, $GCB_INDEX_WIDTH, $GCB_INDEX_HEIGHT, $GCB_INDEX_MAXROWS, $GCB_INDEX_COLSINTRVL, $GCB_INDEX_ROWSINTRVL.
						
						0.1
						* First public version.
	
#CE

;Includes
#include <Array.au3>
#include <FontConstants.au3>
#include <WinAPI.au3>

#EndRegion Header

#Region Global Variables

Global Enum $iGCB_iCtrlID, $iGCB_sText, $iGCB_vParam, $iGCB_iLeft, $iGCB_iTop, $iGCB_iWidth, $iGCB_iHeight, $iGCB_iMaxRows, $iGCB_iColsIntrvl, $iGCB_iRowsIntrvl, _
	$iGCB_iTotal

#EndRegion Global Variables

#Region User Variables

Global Const $GCB_INDEX_ALL = -1
Global Const $GCB_INDEX_CTRLID = $iGCB_iCtrlID
Global Const $GCB_INDEX_TEXT = $iGCB_sText
Global Const $GCB_INDEX_PARAM = $iGCB_vParam
Global Const $GCB_INDEX_LEFT = $iGCB_iLeft
Global Const $GCB_INDEX_TOP = $iGCB_iTop
Global Const $GCB_INDEX_WIDTH = $iGCB_iWidth
Global Const $GCB_INDEX_HEIGHT = $iGCB_iHeight
Global Const $GCB_INDEX_MAXROWS = $iGCB_iMaxRows
Global Const $GCB_INDEX_COLSINTRVL = $iGCB_iColsIntrvl
Global Const $GCB_INDEX_ROWSINTRVL = $iGCB_iRowsIntrvl

Global Const $GCB_CHECK_STATE_ONE = 0
Global Const $GCB_CHECK_STATE_ALL = 1
Global Const $GCB_CHECK_STATE_NONE = 4

#EndRegion User Variables

#Region Public Functions

; #FUNCTION# ====================================================================================================================
; Name ..........: _GroupCheckBox_Add
; Description ...: Adds checkboxes to GroupCheckBox control.
; Syntax ........: _GroupCheckBox_Add($hGroupCB, $aData[, $iPos = -1])
; Parameters ....: $hGroupCB - GroupCheckBox handle as returned by _GroupCheckBox_Create. Passed ByRef.
;                  $aData    - 2D array with the following data for the checkboxes:
;																			$aData[0][0] - First checkbox text
;																			$aData[0][1] - First checkbox params (to use on checkbox action)
;																			$aData[N][0] - Nth checkbox text
;																			$aData[N][1] - Nth checkbox params (to use on checkbox action)
;                  $iPos     - [Optional] 1-based position to insert the checkbox (inserted in place of specified checkbox). Default is -1 -> add to the end.
;
; Return values .: Success - 1.
;				   Failure - 0 and sets @error as following:
;                                                             1 - unable to get info from $hGroupCB control.
;                                                             2 - $aData is not a valid array.
;
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: 
; Example .......: Yes.
; ===============================================================================================================================
Func _GroupCheckBox_Add(ByRef $hGroupCB, $aData, $iPos = -1)
	Local $iGCB_Ubnd = UBound($hGroupCB)
	Local $iData_Ubnd = UBound($aData)
	Local $aFirstInfo, $aState[$iGCB_Ubnd][2]
	Local $aTmp_Data[$iData_Ubnd + $iGCB_Ubnd][2]
	Local $iTmp_Pos = $iPos, $iC = 0
	
	;Get info for last checkbox
	$aFirstInfo = _GroupCheckBox_GetInfo($hGroupCB, 1, $GCB_INDEX_ALL)
	
	If Not IsArray($aFirstInfo) Then
		Return SetError(1, 0, 0)
	EndIf
	
	If UBound($aData, 2) <> 2 Or $iData_Ubnd < 1 Then
		Return SetError(2, 0, 0)
	EndIf
	
	If $iPos <= 0 Or $iPos > $iGCB_Ubnd - 1 Then
		$iTmp_Pos = $iGCB_Ubnd - 1
		
		If $iPos <= 0 Or $iPos > $iGCB_Ubnd Then
			$iPos = -1
		EndIf
	ElseIf $iTmp_Pos > 0 Then
		$iTmp_Pos -= 1
	EndIf
	
	For $i = 0 To $iGCB_Ubnd - 1
		$aState[$i][1] = BitOR(GUICtrlRead($hGroupCB[$i][$iGCB_iCtrlID]), GUICtrlGetState($hGroupCB[$i][$iGCB_iCtrlID]))
		GUICtrlDelete($hGroupCB[$i][$iGCB_iCtrlID])
		
		If $i = $iTmp_Pos Then
			If $iPos = -1 Then
				$aTmp_Data[$iC][0] = $hGroupCB[$i][$iGCB_sText]
				$aTmp_Data[$iC][1] = $hGroupCB[$i][$iGCB_vParam]
				
				$aState[$i][0] = $iC
				
				$i += 1
			EndIf
			
			For $j = 0 To $iData_Ubnd - 1
				$aTmp_Data[$i + $j][0] = $aData[$j][0]
				$aTmp_Data[$i + $j][1] = $aData[$j][1]
			Next
			
			If $iPos = -1 Then
				ExitLoop
			EndIf
			
			$iC += $iData_Ubnd
		EndIf
		
		$aState[$i][0] = $iC
		
		$aTmp_Data[$iC][0] = $hGroupCB[$i][$iGCB_sText]
		$aTmp_Data[$iC][1] = $hGroupCB[$i][$iGCB_vParam]
		
		$iC += 1
	Next
	
	$hGroupCB = _GroupCheckBox_Create($aTmp_Data, $aFirstInfo[$iGCB_iLeft], $aFirstInfo[$iGCB_iTop], -1, -1, $aFirstInfo[$iGCB_iMaxRows], $aFirstInfo[$iGCB_iColsIntrvl], $aFirstInfo[$iGCB_iRowsIntrvl])
	
	For $i = 0 To UBound($aState) - 1
		GUICtrlSetState($hGroupCB[$aState[$i][0]][$iGCB_iCtrlID], $aState[$i][1])
	Next
	
	Return 1
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GroupCheckBox_Create
; Description ...: Creates GroupCheckBox control.
; Syntax ........: _GroupCheckBox_Create($aData, $iLeft, $iTop[, $iWidth = -1[, $iHeight = -1[, $iMax_Rows = 10[, $iCols_Intrvl = 10[, $iRows_Intrvl = 10]]]]])
; Parameters ....: $aData               - 2D array with the following data for the checkboxes:
;																			$aData[0][0] - First checkbox text
;																			$aData[0][1] - First checkbox params (to use on checkbox action)
;																			$aData[N][0] - Nth checkbox text
;																			$aData[N][1] - Nth checkbox params (to use on checkbox action)
;                  $iLeft               - Start left position of the checkboxes group.
;                  $iTop                - Start top position of the checkboxes group.
;                  $iWidth              - [Optional] Width of the checkboxes. Default is -1.
;                  $iHeight             - [Optional] Height of the checkboxes. Default is -1.
;                  $iMax_Rows           - [Optional] Maximum checkboxes in one row. Default is 10.
;                  $iCols_Intrvl        - [Optional] Interval between columns in the checkboxes group. Default is 10.
;                  $iRows_Intrvl        - [Optional] Interval between rows in the checkboxes group. Default is 10.
;
; Return values .: Success              - GroupCheckBox handle (array actualy).
;				   Failure              - 0 and sets @error to 1 if $aData is not a valid array.
;
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: 
; Example .......: Yes.
; ===============================================================================================================================
Func _GroupCheckBox_Create($aData, $iLeft, $iTop, $iWidth = -1, $iHeight = -1, $iMax_Rows = 10, $iCols_Intrvl = 10, $iRows_Intrvl = 10)
	If UBound($aData, 2) <> 2 Or UBound($aData) < 1 Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $aStrSize = 0
	Local $aCheckBox[UBound($aData)][$iGCB_iTotal]
	
	Local $iXLeft = $iLeft
	Local $iXTop = $iTop
	Local $iXWidth = $iWidth
	Local $iXHeight = $iHeight
	
	If $iWidth = -1 Or $iHeight = -1 Then
		$aStrSize = __GCB_GetTextSize($aData[0][0])
		
		If $iWidth = -1 Then
			$iXWidth = $aStrSize[0]
		EndIf
		
		If $iHeight = -1 Then
			$iXHeight = $aStrSize[1]
		EndIf
	EndIf
	
	For $i = 0 To UBound($aData) - 1
		$aCheckBox[$i][$iGCB_iCtrlID] = GUICtrlCreateCheckbox($aData[$i][0], $iXLeft, $iXTop, $iWidth, $iHeight)
		$aCheckBox[$i][$iGCB_sText] = $aData[$i][0]
		$aCheckBox[$i][$iGCB_vParam] = $aData[$i][1]
		$aCheckBox[$i][$iGCB_iLeft] = $iXLeft
		$aCheckBox[$i][$iGCB_iTop] = $iXTop
		$aCheckBox[$i][$iGCB_iWidth] = $iXWidth
		$aCheckBox[$i][$iGCB_iHeight] = $iXHeight
		$aCheckBox[$i][$iGCB_iMaxRows] = $iMax_Rows
		$aCheckBox[$i][$iGCB_iColsIntrvl] = $iCols_Intrvl
		$aCheckBox[$i][$iGCB_iRowsIntrvl] = $iRows_Intrvl
		
		If $iWidth = -1 Or $iHeight = -1 Then
			$aStrSize = __GCB_GetTextSize($aData[$i][0])
			
			If $iWidth = -1 Then
				$iXWidth = $aStrSize[0]
			EndIf
			
			If $iHeight = -1 Then
				$iXHeight = $aStrSize[1]
			EndIf
		EndIf
		
		$iXTop += $iXHeight + $iRows_Intrvl
		
		If Mod($i + 1, $iMax_Rows) = 0 Then
			$iXTop = $iTop
			$iXLeft += ($iXWidth + $iCols_Intrvl + $iXHeight) ;$iXHeight is for the checkbox box width
		EndIf
	Next
	
	Return $aCheckBox
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _GroupCheckBox_SetState
; Description....: Check/UnCheck group checkboxes.
; Syntax.........: _GroupCheckBox_SetState($hGroupCB, $iState[, $iCheckBox = -1])
; Parameters.....: $hGroupCB - GroupCheckBox handle as returned by _GroupCheckBox_Create.
;                  $iState - State for the checkbox(es), special constants can be used to set checked state ($GCB_CHECK_STATE_ONE, $GCB_CHECK_STATE_ALL, $GCB_CHECK_STATE_NONE).
;                  $iCheckBox [Optional] - 1-based checkbox instance (number). Default is -1 (set state for all checkboxes group).
;                 
; Return values..: Success - 1.
;				   Failure - 0 and sets @error to 1 if $iCheckBox is invalid instance.
;
; Author.........: G.Sandler
; Remarks........: 
; Related........: 
; Example........: Yes.
; ===============================================================================================================
Func _GroupCheckBox_SetState($hGroupCB, $iState, $iCheckBox = -1)
	If $iCheckBox <> -1 And ($iCheckBox < 1 Or $iCheckBox > UBound($hGroupCB)) Then
		Return SetError(1, 0, 0)
	EndIf
	
	For $i = 0 To UBound($hGroupCB) - 1
		If $iCheckBox <= 0 Or $i = $iCheckBox - 1 Then
			GUICtrlSetState($hGroupCB[$i][$iGCB_iCtrlID], $iState)
			
			If $i = $iCheckBox - 1 Then
				ExitLoop
			EndIf
		EndIf
	Next
	
	Return 1
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GroupCheckBox_Destroy
; Description ...: Destroys GroupCheckBox control.
; Syntax ........: _GroupCheckBox_Destroy($hGroupCB)
; Parameters ....: $hGroupCB - GroupCheckBox handle as returned by _GroupCheckBox_Create.
;
; Return values .: Success - Number of deleted checkboxes.
;				   Failure - 0.
;
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: 
; Example .......: Yes.
; ===============================================================================================================================
Func _GroupCheckBox_Destroy($hGroupCB)
	Local $iDel = 0
	
	For $i = 0 To UBound($hGroupCB) - 1
		$iDel += GUICtrlDelete($hGroupCB[$i][$iGCB_iCtrlID])
	Next
	
	Return $iDel
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _GroupCheckBox_Remove
; Description ...: Removes checkboxes from GroupCheckBox control.
; Syntax ........: _GroupCheckBox_Remove($hGroupCB, $sCheckBoxes[, $bShift = True])
; Parameters ....: $hGroupCB - GroupCheckBox handle as returned by _GroupCheckBox_Create.
;                  $sCheckBoxes - String with 1-based checkboxes instances (numbers) list delimited with | (i.e: "1|3").
;                  $bShift - [Optional] If True (default), the remained checkboxes will be shifted to removed checkboxes positions.
;
; Return values .: Success - Number of removed checkboxes.
;				   Failure - 0 and sets @error to 1 if none of the checboxes passed as $sCheckBoxes were removed.
;
; Author ........: G.Sandler
; Remarks .......: 
; Related .......: 
; Example .......: Yes.
; ===============================================================================================================================
Func _GroupCheckBox_Remove(ByRef $hGroupCB, $sCheckBoxes, $bShift = True)
	Local $aCBs = StringSplit($sCheckBoxes, '|')
	Local $iUbnd = UBound($hGroupCB)
	Local $iCheckBox, $iRem = 0, $aInfo, $iNewLeft, $iNewTop
	
	_ArraySort($aCBs, 0, 1)
	
	For $x = $aCBs[0] To 1 Step -1
		$iCheckBox = Number($aCBs[$x])
		$iCheckBox -= 1
		
		If $iCheckBox < 0 Or $iCheckBox > $iUbnd - 1 Or $aCBs[$x] = '' Then
			ContinueLoop
		EndIf
		
		$iRem += GUICtrlDelete($hGroupCB[$iCheckBox][$iGCB_iCtrlID])
		
		If $bShift Then
			$aInfo = _GroupCheckBox_GetInfo($hGroupCB, $iCheckBox + 1, $GCB_INDEX_ALL)
			
			For $i = $iCheckBox + 1 To $iUbnd - 1
				GUICtrlSetPos($hGroupCB[$i][$iGCB_iCtrlID], $aInfo[$GCB_INDEX_LEFT], $aInfo[$GCB_INDEX_TOP])
				
				$iNewLeft = $aInfo[$GCB_INDEX_LEFT]
				$iNewTop = $aInfo[$GCB_INDEX_TOP]
				
				$aInfo = _GroupCheckBox_GetInfo($hGroupCB, $i + 1, $GCB_INDEX_ALL)
				
				$hGroupCB[$i][$iGCB_iLeft] = $iNewLeft
				$hGroupCB[$i][$iGCB_iTop] = $iNewTop
			Next
		EndIf
		
		$iUbnd = _ArrayDelete($hGroupCB, $iCheckBox)
	Next
	
	If $iRem = 0 Then
		Return SetError(1, 0, 0)
	EndIf
	
	Return $iRem
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _GroupCheckBox_GetInfo
; Description....: Get group checkboxes info.
; Syntax.........: _GroupCheckBox_GetInfo($hGroupCB[, $iCheckBox = -1[, $iIndex = -1]])
; Parameters.....: $hGroupCB - GroupCheckBox handle as returned by _GroupCheckBox_Create.
;                  $iCheckBox - [Optional] 1-based checkbox instance (number). Default is -1 (see Return values).
;                  $iIndex - [Optional] Info Index to retrieve (ignored if $iCheckBox = -1). Default is $GCB_INDEX_ALL (-1). These are supported:
;																						$GCB_INDEX_ALL    - Get 1D array of all bellow info (wich corresponding number of elements).
;																						$GCB_INDEX_CTRLID - Checkbox CtrlID.
;																						$GCB_INDEX_TEXT   - Checkbox Text.
;																						$GCB_INDEX_PARAM  - Checkbox param.
;																						$GCB_INDEX_LEFT   - Checkbox left position.
;																						$GCB_INDEX_TOP    - Checkbox top position.
;																						$GCB_INDEX_WIDTH  - Checkbox width.
;																						$GCB_INDEX_HEIGHT - Checkbox height.
;																						$GCB_INDEX_MAXROWS - Checkboxes group rows.
;																						$GCB_INDEX_COLSINTRVL - Checkboxes group columns interval.
;																						$GCB_INDEX_ROWSINTRVL - Checkboxes group rows interval.
;                  
; Return values..: Requested data. If $iCheckBox = -1 or omited, the return value is the group checkboxes count.
;
; Author.........: G.Sandler
; Remarks........: For 3 last indexes (starting fom $GCB_INDEX_MAXROWS) the $iCheckBox can be any (1-based), because the indexes are global for entire GroupCheckBox control.
; Related........: 
; Example........: Yes.
; ===============================================================================================================
Func _GroupCheckBox_GetInfo($hGroupCB, $iCheckBox = -1, $iIndex = $GCB_INDEX_ALL)
	If $iCheckBox < 0 Then
		Return UBound($hGroupCB)
	EndIf
	
	$iCheckBox -= 1
	
	If $iCheckBox < 0 Or $iCheckBox > UBound($hGroupCB) - 1 Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $aInfo[$iGCB_iTotal] = _
		[ _
			$hGroupCB[$iCheckBox][$iGCB_iCtrlID], $hGroupCB[$iCheckBox][$iGCB_sText], $hGroupCB[$iCheckBox][$iGCB_vParam], _
			$hGroupCB[$iCheckBox][$iGCB_iLeft], $hGroupCB[$iCheckBox][$iGCB_iTop], $hGroupCB[$iCheckBox][$iGCB_iWidth], $hGroupCB[$iCheckBox][$iGCB_iHeight], _
			$hGroupCB[$iCheckBox][$iGCB_iMaxRows], $hGroupCB[$iCheckBox][$iGCB_iColsIntrvl], $hGroupCB[$iCheckBox][$iGCB_iRowsIntrvl] _
		]
	
	If $iIndex >= 0 And $iIndex < $iGCB_iTotal Then
		Return $aInfo[$iIndex]
	EndIf
	
	Return $aInfo
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _GroupCheckBox_GetState
; Description....: Get group checkboxes state.
; Syntax.........: _GroupCheckBox_GetState($hGroupCB[, $iCheckBox = -1])
; Parameters.....: $hGroupCB - GroupCheckBox handle as returned by _GroupCheckBox_Create.
;                  $iCheckBox [Optional] - 1-based checkbox instance (number) to get the state for. Default is -1 (see Return values).
;                  
; Return values..: If $iCheckBox = -1, the return is the state for that specified checkbox instance, otherwise the return is a state of all checkboxes group.
;                  Theese (and standard $GUI_*) constants can be used to determine the state:
;																							$GCB_CHECK_STATE_ONE  - At least one checkbox is checked.
;																							$GCB_CHECK_STATE_ALL  - All checkboxes are checked.
;																							$GCB_CHECK_STATE_NONE - All checkboxes are unchecked.
;
; Author.........: G.Sandler
; Remarks........: 
; Related........: 
; Example........: Yes.
; ===============================================================================================================
Func _GroupCheckBox_GetState($hGroupCB, $iCheckBox = -1)
	Local $iState = -1
	
	$iCheckBox -= 1
	
	If $iCheckBox < 0 Or $iCheckBox > UBound($hGroupCB) - 1 Then
		For $i = 0 To UBound($hGroupCB) - 1
			$iState = BitAND($iState, GUICtrlRead($hGroupCB[$i][$iGCB_iCtrlID]))
		Next
		
		Return $iState
	EndIf
	
	Return GUICtrlRead($hGroupCB[$iCheckBox][$iGCB_iCtrlID])
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _GroupCheckBox_GetEvent
; Description....: Event wrapper to check group checkboxes events.
; Syntax.........: _GroupCheckBox_GetEvent($iMsg, $hGroupCB)
; Parameters.....: $hGroupCB - GroupCheckBox handle as returned by _GroupCheckBox_Create.
;                  $iMsg - MsgID as returned by GUIGetMsg()/@GUI_CtrlID.
;                  
; Return values..: Success - MsgID passed in $iMsg parameter.
;                  Failure - -1 if $iMsg <= 0.
;
; Author.........: G.Sandler
; Remarks........: 
; Related........: 
; Example........: Yes.
; ===============================================================================================================
Func _GroupCheckBox_GetEvent($hGroupCB, $iMsg)
	If $iMsg <= 0 Then
		Return -1
	EndIf
	
	For $i = 0 To UBound($hGroupCB) - 1
		If $iMsg = $hGroupCB[$i][$iGCB_iCtrlID] Then
			Return $iMsg
		EndIf
	Next
EndFunc

#EndRegion Public Functions

#Region Internal Functions

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __GCB_GetTextSize
; Description....: Get size of text for static control.
; Syntax.........: __GCB_GetTextSize($sData[, $iFontSize = 8.5[, $iFontWeight = $FW_NORMAL[, $iFontAttribs = 0[, $sFontName = 'Arial'[, $iFixTabs = 0]]]]])
; Parameters.....: $sData - Text to check. 
; Author.........: G.Sandler
; ===============================================================================================================
Func __GCB_GetTextSize($sData, $iFontSize = 8.5, $iFontWeight = $FW_NORMAL, $iFontAttribs = 0, $sFontName = 'Arial', $iFixTabs = 0)
	Local $sTabs, $sAdd, $bFItalic, $bFUnderline, $bFStrikeout, $h_GDW_GUI, $hDC, $intDeviceCap, $intFontHeight, $hFont, $tSIZE, $aRet
	
	If $iFixTabs Then
		$sData = StringReplace($sData, @TAB, '     ')
		
		If @extended Then
			For $i = 1 To @extended + 1
				$sTabs &= '    ' & $sAdd
				$sAdd &= ' '
			Next
		EndIf
		
		$sData &= $sTabs
	EndIf
	
	$bFItalic = BitAND($iFontAttribs, 2) = 2
	$bFUnderline = BitAND($iFontAttribs, 4) = 4
	$bFStrikeout = BitAND($iFontAttribs, 8) = 8
	
	$h_GDW_GUI = GUICreate('__GCB_GetTextSize', 10, 10, -100, -100, 0x80880000, 0x00000080)
	$hDC = _WinAPI_GetDC($h_GDW_GUI)
	$intDeviceCap = _WinAPI_GetDeviceCaps($hDC, $LOGPIXELSY)
	$intFontHeight = _WinAPI_MulDiv($iFontSize, $intDeviceCap, 72)
	$hFont = _WinAPI_CreateFont(-$intFontHeight, 0, 0, 0, $iFontWeight, $bFItalic, $bFUnderline, $bFStrikeout, $DEFAULT_CHARSET, $OUT_CHARACTER_PRECIS, $CLIP_DEFAULT_PRECIS, $PROOF_QUALITY, $FIXED_PITCH, $sFontName)
	_WinAPI_SelectObject($hDC, $hFont)
	$tSIZE = _WinAPI_GetTextExtentPoint32($hDC, $sData)
	
	Dim $aRet[2] = [DllStructGetData($tSIZE, 'X'), DllStructGetData($tSIZE, 'Y')]
	
	GUIDelete($h_GDW_GUI)
	Return $aRet
EndFunc

#EndRegion Internal Functions
