#Region Header

#CS --------------------------------------------------------------------------------------------------------
	Title:			TimeEvent UDF for AutoIt3
	Filename:		TimeEvent.au3
	Description:	TimeEvent functions to set on specific date/time.
	Author:			G.Sandler (a.k.a (Mr)CreatoR), www.autoit-script.ru, www.creator-lab.ucoz.ru
	Version:		1.2
	Requirements:	AutoIt v3.3.8.1
	Uses:           Constants.au3, WindowsConstants.au3, Date.au3, Timers.au3, WinAPI.au3
	Notes:          
	
	Forum Link:     
	
	History version:
						1.2
						+ [EXPERIMENTAL] Added _TimeEvent_EnableSync function, allows to enable or disable events time synchronization (with system time). Disabled by default.
						+ Added _TimeEvent_GetEventDates function to retrieve correct event dates.
						+ Added date validation to _TimeEvent_Add function.
						* Changed return @error in _TimeEvent_Add function (see docs for this function).
						* Fixed _TimeEvent_GetNextEvent function.
						* Fixed issue with rapidly called event function.
						
						1.1
						+ Added _TimeEvent_GetEvents function to get all events names.
						+ Added _TimeEvent_GetNextEvent function to get the upcoming event.
						- Removed _TimeEvent_GetCount function, the count now can be retrieved using new _TimeEvent_GetEvents function.
						* Docs updated.
						
						1.0
						* First public release.

#CE ---------------------------------------------------------------------------------------------------------

#include-once
#include <Constants.au3>
#include <WindowsConstants.au3>
#include <Date.au3>
#include <Timers.au3>
#include <WinAPI.au3>

#EndRegion

#Region Global vars & constants

Global $a_TE_Data[1][1]
Global $i_TE_MainTimer
Global $i_TE_PrevDTTimer
Global $s_TE_PrevDT
Global $s_TE_Name

Global Const $h_TE_Wnd = GUICreate('TimeEvent_Wnd')
Global Const $s_TE_Date_Sep = RegRead('HKCU\Control Panel\International', 'sDate')
Global Const $s_TE_Time_Sep = RegRead('HKCU\Control Panel\International', 'sTime')
Global Const $s_TE_TIMECHANGE_FuncName = '__TE_WM_TIMECHANGE'

Global $h_TE_TIMECHANGE_Clbk = DllCallbackRegister('__TimeEvent_TC_CallBack', 'ptr', 'hwnd;uint;long;ptr')
Global $h_TE_TIMECHANGE_WndProc = _WinAPI_SetWindowLong($h_TE_Wnd, $GWL_WNDPROC, DllCallbackGetPtr($h_TE_TIMECHANGE_Clbk))
Global $i_TE_SyncOnTimeChange = 0

#EndRegion

#Region Options

OnAutoItExitRegister('__TimeEvent_OnExit')

#EndRegion

; #FUNCTION# ====================================================================================================
; Name...........: _TimeEvent_Add
; Description....: Registers time event
; Syntax.........: _TimeEvent_Add($sEventName, $sEventDate, $sEventTime, $sEventFunc, $sEventParams = '', $bRemoveEvent = False)
; Parameters.....: $sEventName - Event name, should be unique for every event, otherwise the event with the same name will be reset.
;                  $sEventDate - Date for the event in DD.MM.YYYY format (the separator should be used the same as system's).
;                                This parameter can be also one of these values:
;                                                                                -1 - Date is ignored ("Every day").
;                                                                                1|N|... - Defines week days ("Every week"), separator is the 'GUIDataSeparatorChar'.
;                  $sEventTime - Time for the event in HH:MM:SS format (the separator should be used the same as system's).
;                  $sEventFunc - Function to call when event is triggered (only user defined function).
;                  $sEventParams [Optional] - Extra data for the $sEventFunc (%d expanded to event's day, and %t to event's time).
;                  $bRemoveEvent [Optional] - Defines if the event should be removed from the list after it's triggered.
;                 
;                 
; Return values..: Success - Returns 1 and adds event to the list. @error is set to -1 if the event is been reset ($sEventName already set as event).
;                  Failure - Returns 0 and set @error as following:
;                                                                  1 - Invalid $sEventDate parameter
;                                                                  2 - Invalid $sEventTime parameter
;                 
; Author.........: G.Sandler (CreatoR)
; Modified.......: 
; Remarks........: 
; Related........: _TimeEvent_Remove, _TimeEvent_GetData
; Link...........: 
; Example........: Yes
; ===============================================================================================================
Func _TimeEvent_Add($sEventName, $sEventDate, $sEventTime, $sEventFunc, $sEventParams = '', $bRemoveEvent = False)
	If __TimeEvent_ValidateDate($sEventDate) = 0 Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $aSplit = StringSplit($sEventTime, $s_TE_Time_Sep, 1)
	
	$sEventTime = ''
	
	For $i = 1 To $aSplit[0]
		$sEventTime &= StringFormat('%02d', $aSplit[$i])
		
		If $i < $aSplit[0] Then
			$sEventTime &= $s_TE_Time_Sep
		EndIf
	Next
	
	Switch $aSplit[0]
		Case 1
			If Not StringIsDigit($sEventTime) Then
				Return SetError(2, 0, 0)
			EndIf
			
			$sEventTime &= $s_TE_Time_Sep & '00' & $s_TE_Time_Sep & '00'
		Case 2
			$sEventTime &= $s_TE_Time_Sep & '00'
		Case Else
			If $aSplit[0] <> 3 Then
				Return SetError(2, 0, 0)
			EndIf
	EndSwitch
	
	For $i = 1 To $a_TE_Data[0][0]
		If $a_TE_Data[$i][0] = $sEventName Then
			$a_TE_Data[$i][1] = $sEventDate
			$a_TE_Data[$i][2] = $sEventTime
			$a_TE_Data[$i][3] = $sEventFunc
			$a_TE_Data[$i][4] = $sEventParams
			$a_TE_Data[$i][5] = $bRemoveEvent
			$a_TE_Data[$i][6] = 0
			
			Return SetError(-1, 0, 1)
		EndIf
	Next
	
	$a_TE_Data[0][0] += 1
	ReDim $a_TE_Data[$a_TE_Data[0][0] + 1][7]
	
	If $a_TE_Data[0][0] <= 1 Then
		$i_TE_MainTimer = _Timer_SetTimer($h_TE_Wnd, 1500, '__TimeEvent_Handler')
		$i_TE_PrevDTTimer = _Timer_SetTimer($h_TE_Wnd, 1000, '__TimeEvent_SetPrevDT')
	EndIf
	
	$a_TE_Data[$a_TE_Data[0][0]][0] = $sEventName
	$a_TE_Data[$a_TE_Data[0][0]][1] = $sEventDate
	$a_TE_Data[$a_TE_Data[0][0]][2] = $sEventTime
	$a_TE_Data[$a_TE_Data[0][0]][3] = $sEventFunc
	$a_TE_Data[$a_TE_Data[0][0]][4] = $sEventParams
	$a_TE_Data[$a_TE_Data[0][0]][5] = $bRemoveEvent
	$a_TE_Data[$a_TE_Data[0][0]][6] = 0 ;Function Call status (to avoid repeating function call)
	
	Return 1
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _TimeEvent_EnableSync
; Description....: Enable or disable events time synchronization (with system time)
; Syntax.........: _TimeEvent_EnableSync($bEnable)
; Parameters.....: $bEnable - Determines, if the events should be synchronized on system time change
;                  
;                 
; Return values..: None.
;
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
;                  1. When system time changes, every event's time data updated to be synchronized with the system time,
;                     and special function stored in $s_TE_TIMECHANGE_FuncName called (with seconds differency as a parameter) to allow time correction handling from main script.
;                  2. Synchronization will be relevant only for 2 hours range.
;
; Related........: 
; Link...........: 
; Example........: No.
; ===============================================================================================================
Func _TimeEvent_EnableSync($bEnable)
	$i_TE_SyncOnTimeChange = $bEnable
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _TimeEvent_GetData
; Description....: Get event data
; Syntax.........: _TimeEvent_GetData($sEventName)
; Parameters.....: $sEventName - The event name to get data from (the same name that was used for _TimeEvent_Add call)
;                  
;                 
; Return values..: Success - Returns array with the following data:
;                                                                   [0] - Event name
;                                                                   [1] - Event date
;                                                                   [2] - Event time
;                                                                   [3] - Function name to call on event
;                                                                   [4] - Extra prameters for the function in [3]
;                                                                   [5] - Remove event definer (True or False)
;                  Failure - Returns 0 and set @error to 1 if the event was not found in the events list.
;
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: _TimeEvent_Add
; Link...........: 
; Example........: Yes.
; ===============================================================================================================
Func _TimeEvent_GetData($sEventName)
	For $i = 1 To $a_TE_Data[0][0]
		If $a_TE_Data[$i][0] = $sEventName Then
			Local $aRet[6] = [$a_TE_Data[$i][0], $a_TE_Data[$i][1], $a_TE_Data[$i][2], $a_TE_Data[$i][3], $a_TE_Data[$i][4], $a_TE_Data[$i][5]]
			Return $aRet
		EndIf
	Next
	
	Return SetError(1, 0, 0)
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _TimeEvent_GetEvents
; Description....: Get events names
; Syntax.........: _TimeEvent_GetEvents()
; Parameters.....: None.
;                  
;                 
; Return values..: Success - Returns array with the following data:
;                                                                   [0] - Events count
;                                                                   [N] - Event name
;                  Failure - Returns 0 and set @error to 1 if the events list is empty.
;
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: _TimeEvent_Add, _TimeEvent_GetData
; Link...........: 
; Example........: Yes.
; ===============================================================================================================
Func _TimeEvent_GetEvents()
	If $a_TE_Data[0][0] = 0 Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $aRet[$a_TE_Data[0][0]+1] = [$a_TE_Data[0][0]]
	
	For $i = 1 To $a_TE_Data[0][0]
		$aRet[$i] = $a_TE_Data[$i][0]
	Next
	
	Return $aRet
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _TimeEvent_GetEvents
; Description....: Get events names
; Syntax.........: _TimeEvent_GetEventDates($sEventName)
; Parameters.....: $sEventName - The event name to get dates from (the same name that was used for _TimeEvent_Add call)
;                  
;                 
; Return values..: Success - Returns array with the following data:
;                                                                   [0] - Event dates count
;                                                                   [N] - Event date
;                  Failure - Returns 0 and set @error to 1 if the event not found in events list.
;
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: _TimeEvent_Add, _TimeEvent_GetData, _TimeEvent_GetNextEvent
; Link...........: 
; Example........: Yes.
; ===============================================================================================================
Func _TimeEvent_GetEventDates($sEventName)
	Local $aEvent_Data = _TimeEvent_GetData($sEventName)
	
	If @error Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $aDates[1]
	Local $sDate = $aEvent_Data[1]
	Local $sNowDate = _NowCalcDate()
	Local $sSep = Opt('GUIDataSeparatorChar')
	
	Select
		Case $sDate = -1
			Dim $aDates[2] = [1, $sNowDate]
		Case StringInStr($sDate, $sSep) Or StringIsDigit($sDate)
			Local $aSplit = StringSplit($sDate, $sSep, 1)
			
			For $i = 1 To $aSplit[0]
				$iDays = (7 - @WDAY) + $aSplit[$i]
				
				If $iDays > 7 Then
					$iDays -= 7
				ElseIf $iDays = 7 Then
					$iDays = 0
				EndIf
				
				$aDates[0] += 1
				ReDim $aDates[$aDates[0] + 1]
				$aDates[$aDates[0]] = __TimeEvent_FormatDate(_DateAdd('d', $iDays, $sNowDate), 0)
			Next
		Case Else
			Dim $aDates[2] = [1, $sDate]
	EndSelect
	
	Return $aDates
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _TimeEvent_GetNextEvent
; Description....: Get next upcoming event
; Syntax.........: _TimeEvent_GetNextEvent($aIgnoreEvents = 0)
; Parameters.....: $aIgnoreEvents [Optional] - Zero-based array with ignored events names (usefull for non active events)
;                  
;                 
; Return values..: Success - Returns data array of the next upcoming event, where:
;                                                                                   [0] = Event name
;                                                                                   [1] = Event date & time
;                  Failure - Returns 0 and set @error as following:
;                                                                   1 - The events list is empty.
;                                                                   2 - No upcoming events.
;
; Author.........: G.Sandler
; Modified.......: 
; Remarks........: 
; Related........: _TimeEvent_Add, _TimeEvent_GetData
; Link...........: 
; Example........: Yes.
; ===============================================================================================================
Func _TimeEvent_GetNextEvent($aIgnoreEvents = 0)
	If $a_TE_Data[0][0] = 0 Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $sTime, $aDates, $sDate, $sNextEvent = '', $sRetEvent = ''
	Local $sNowDateTime = _NowCalc()
	Local $iUbnd = UBound($aIgnoreEvents)-1
	
	For $i = 1 To $a_TE_Data[0][0]
		For $j = 0 To $iUbnd
			If $aIgnoreEvents[$j] = $a_TE_Data[$i][0] Then
				ContinueLoop 2
			EndIf
		Next
		
		$sTime = StringReplace($a_TE_Data[$i][2], $s_TE_Time_Sep, ':')
		$aDates = _TimeEvent_GetEventDates($a_TE_Data[$i][0])
		
		For $j = 1 To UBound($aDates)-1
			$sDate = __TimeEvent_FormatDate($aDates[$j], 1)
			
			;Check if the event is expired
			If _DateDiff('s', $sDate & ' ' & $sTime, $sNowDateTime) > 0 Then
				ContinueLoop
			EndIf
			
			;Get nearest date
			If $sNextEvent = '' Or _DateDiff('s', $sDate & ' ' & $sTime, $sNextEvent) > 0 Then
				$sNextEvent = $sDate & ' ' & $sTime
				$sRetEvent = $a_TE_Data[$i][0]
			EndIf
		Next
	Next
	
	If $sNextEvent = '' Then
		Return SetError(2, 0, 0)
	EndIf
	
	Local $aRet[2] = [$sRetEvent, __TimeEvent_FormatDate($sNextEvent, 0)]
	Return $aRet
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _TimeEvent_Remove
; Description....: Removes time event
; Syntax.........: _TimeEvent_Remove($sEventName)
; Parameters.....: $sEventName - The event name to remove (the same name that was used for _TimeEvent_Add call)
;                  
;                 
; Return values..: Current events count (0 in case that there is no more events).
;                 
; Author.........: G.Sandler (CreatoR)
; Modified.......: 
; Remarks........: 
; Related........: _TimeEvent_Add
; Link...........: 
; Example........: Yes.
; ===============================================================================================================
Func _TimeEvent_Remove($sEventName)
	Local $aTmp[1][1]
	
	For $i = 1 To $a_TE_Data[0][0]
		If $sEventName <> $a_TE_Data[$i][0] Then
			$aTmp[0][0] += 1
			ReDim $aTmp[$aTmp[0][0] + 1][7]
			
			$aTmp[$aTmp[0][0]][0] = $a_TE_Data[$i][0]
			$aTmp[$aTmp[0][0]][1] = $a_TE_Data[$i][1]
			$aTmp[$aTmp[0][0]][2] = $a_TE_Data[$i][2]
			$aTmp[$aTmp[0][0]][3] = $a_TE_Data[$i][3]
			$aTmp[$aTmp[0][0]][4] = $a_TE_Data[$i][4]
			$aTmp[$aTmp[0][0]][5] = $a_TE_Data[$i][5]
			$aTmp[$aTmp[0][0]][6] = $a_TE_Data[$i][6]
		EndIf
	Next
	
	If $aTmp[0][0] = 0 Then
		If $i_TE_MainTimer Then
			_Timer_KillTimer($h_TE_Wnd, $i_TE_MainTimer)
			$i_TE_MainTimer = 0
		EndIf
		
		Return 0
	EndIf
	
	$a_TE_Data = $aTmp
	Return $a_TE_Data[0][0]
EndFunc

; #FUNCTION# ====================================================================================================
; Name...........: _TimeEvent_RemoveAll
; Description....: Removes all time events
; Syntax.........: _TimeEvent_RemoveAll()
; Parameters.....: None.
;                  
;                 
; Return values..: Always returns 1 and remove all the events from list.
;                 
; Author.........: G.Sandler (CreatoR)
; Modified.......: 
; Remarks........: 
; Related........: _TimeEvent_Add, _TimeEvent_Remove
; Link...........: 
; Example........: Yes.
; ===============================================================================================================
Func _TimeEvent_RemoveAll()
	If $i_TE_MainTimer Then
		_Timer_KillTimer($h_TE_Wnd, $i_TE_MainTimer)
		$i_TE_MainTimer = 0
	EndIf
	
	If $i_TE_PrevDTTimer Then
		_Timer_KillTimer($h_TE_Wnd, $i_TE_PrevDTTimer)
		$i_TE_PrevDTTimer = 0
	EndIf
	
	$s_TE_Name = 0
	$a_TE_Data = 0
	
	Dim $a_TE_Data[1][1]
	
	Return 1
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TimeEvent_Handler
; Description....: Internal TimeEvent handler function to trigger time event (called by timer)
; Author.........: G.Sandler
; ===============================================================================================================
Func __TimeEvent_Handler($hWnd, $iMsg, $iIDTimer, $dwTime)
	Local $iHour = StringFormat('%02d', @HOUR), $iMin = StringFormat('%02d', @MIN), $iSec = StringFormat('%02d', @SEC)
	
	Local $sNowDate = @MDAY & $s_TE_Date_Sep & @MON & $s_TE_Date_Sep & @YEAR
	Local $sNowTime = $iHour & $s_TE_Time_Sep & $iMin & $s_TE_Time_Sep & $iSec
	Local $sFutureTime = $iHour & $s_TE_Time_Sep & $iMin & $s_TE_Time_Sep & StringFormat('%02d', $iSec + 1)
	Local $sPastTime = $iHour & $s_TE_Time_Sep & $iMin & $s_TE_Time_Sep & StringFormat('%02d', $iSec - 1)
	
	Local $sEName, $sEDate, $sETime, $sEParams, $bERemove, $aSplit, $iSkip, $aParams
	Local $sSep = Opt('GUIDataSeparatorChar')
	
	For $i = 1 To $a_TE_Data[0][0]
		;Prevents repeated false call
		If $a_TE_Data[$i][6] Then
			$a_TE_Data[$i][6] = 0
			ContinueLoop
		EndIf
		
		$sEName = $a_TE_Data[$i][0]
		$sEDate = $a_TE_Data[$i][1]
		$sETime = $a_TE_Data[$i][2]
		$sEParams = $a_TE_Data[$i][4]
		$bERemove = $a_TE_Data[$i][5]
		
		;-1 means "Every Day", so if it's -1 we check next (for time, bellow this condition)
		If $sEDate <> -1 Then
			;It's a date, but no match with current date
			If StringInStr($sEDate, $s_TE_Date_Sep) And $sEDate <> $sNowDate Then
				ContinueLoop
			EndIf
			
			;It's not a date (it's "Every Week"), so we need to check if the current day is matches
			If StringInStr($sEDate, $sSep) Or StringIsDigit($sEDate) Then
				$aSplit = StringSplit($sEDate, $sSep)
				$iSkip = 1
				
				For $j = 1 To $aSplit[0]
					;The day is matches, so we check next (for time, bellow)
					If $aSplit[$j] = @WDAY Then
						$iSkip = 0
						ExitLoop
					EndIf
				Next
				
				If $iSkip Then
					ContinueLoop
				EndIf
			EndIf
		EndIf
		
		If $sETime = $sNowTime Or (($sETime < $sNowTime And $sETime >= $sPastTime) Or ($sETime > $sNowTime And $sETime <= $sFutureTime)) Then
			$a_TE_Data[$i][6] = 1
			
			Dim $aParams[5] = [$sEName, $sEDate, $sETime, StringReplace(StringReplace($sEParams, '%d', $sEDate), '%t', $sETime), $bERemove]
			Local $iRet = Call($a_TE_Data[$i][3], $aParams)
			
			If @error = 0xDEAD And @extended = 0xBEEF Then
				Call($a_TE_Data[$i][3])
			EndIf
			
			;$a_TE_Data[$i][6] = 0
			
			;Remove Event?
			If $a_TE_Data[$i][5] Then
				$s_TE_Name = $sEName
				AdlibRegister('__TimeEvent_Remove', 100)
			EndIf
			
			ExitLoop
		EndIf
	Next
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TimeEvent_SetPrevDT
; Description....: Internal function to set previous date & time for time change handler (called by timer)
; Author.........: G.Sandler
; ===============================================================================================================
Func __TimeEvent_SetPrevDT($hWnd, $iMsg, $iIDTimer, $dwTime)
	$s_TE_PrevDT = _NowCalc()
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TimeEvent_TC_CallBack
; Description....: Internal function to handle time change correction (sync option)
; Author.........: G.Sandler
; ===============================================================================================================
Func __TimeEvent_TC_CallBack($hWnd, $iMsg, $wParam, $lParam)
	If $i_TE_SyncOnTimeChange And $iMsg = $WM_TIMECHANGE Then
		Static $iTIMECHANGE
		
		If Not $iTIMECHANGE Then
			Switch $hWnd
				Case $h_TE_Wnd
					Local $sTime = _NowCalc()
					
					If $sTime <> $s_TE_PrevDT Then
						Local $sHours_Diff = _DateDiff('h', $s_TE_PrevDT, $sTime)
						
						If $sHours_Diff <= 2 And $sHours_Diff >= -2 Then
							Local $iSec_Diff = _DateDiff('s', $s_TE_PrevDT, $sTime)
							
							For $i = 1 To $a_TE_Data[0][0]
								$a_TE_Data[$i][2] = StringRegExpReplace(_DateAdd('s', $iSec_Diff, _NowCalcDate() & ' ' & $a_TE_Data[$i][2]), '[^ ]+ ', '')
							Next
							
							Call($s_TE_TIMECHANGE_FuncName, $iSec_Diff)
							
							If @error = 0xDEAD Then
								Call($s_TE_TIMECHANGE_FuncName)
							EndIf
						EndIf
					EndIf
			EndSwitch
			
			$iTIMECHANGE = 1
		Else
			$iTIMECHANGE = 0
		EndIf
	EndIf
	
	Return _WinAPI_CallWindowProc($h_TE_TIMECHANGE_WndProc, $hWnd, $iMsg, $wParam, $lParam)
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TimeEvent_Remove
; Description....: Internal TimeEvent remove function
; Author.........: G.Sandler
; ===============================================================================================================
Func __TimeEvent_Remove()
	_TimeEvent_Remove($s_TE_Name)
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TimeEvent_ValidateDate
; Description....: Internal TimeEvent function to validate event date
; Author.........: G.Sandler
; ===============================================================================================================
Func __TimeEvent_ValidateDate($sDate)
	Local $sSep = Opt('GUIDataSeparatorChar')
	
	Select
		Case $sDate = -1
			Return 1
		Case StringInStr($sDate, $sSep)
			Local $aSplit = StringSplit($sDate, $sSep, 1)
			
			For $i = 1 To $aSplit[0]
				If Not StringIsDigit($aSplit[$i]) Or $aSplit[$i] > 7 Or $aSplit[$i] < 1 Then
					Return 0
				EndIf
			Next
			
			Return 1
		Case StringIsDigit($sDate)
			If $sDate > 0 And $sDate < 8 Then
				Return 1
			EndIf
		Case StringRegExp($sDate, '(?:0?[1-9]|1[0-9]|2[0-9]|3[01])' & $s_TE_Date_Sep & '(?:0?[1-9]|1[012])' & $s_TE_Date_Sep & '(?:\d{2}|\d{4})')
			Return 1
	EndSelect
	
	Return 0
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TimeEvent_FormatDate
; Description....: Internal TimeEvent function to correct event date (from YYYY/MM/DD to DD/MM/YYYY or vise versa)
; Author.........: G.Sandler
; ===============================================================================================================
Func __TimeEvent_FormatDate($sDate, $iMode = 0)
	If Not __TimeEvent_ValidateDate($sDate) Then
		Return SetError(1, 0, $sDate)
	EndIf
	
	Local $sTime
	Local $aSplit = StringRegExp($sDate, '(.*?) (\d{1,2}\' & $s_TE_Time_Sep & '\d{1,2}(?:\' & $s_TE_Time_Sep & '\d{1,2})?)', 3)
	
	If UBound($aSplit) > 1 Then
		$sDate = $aSplit[0]
		$sTime = ' ' & $aSplit[1]
	EndIf
	
	Local $aDate = StringSplit($sDate, '/-.')
	
	If $aDate[0] < 3 Then
		Return SetError(2, 0, $sDate)
	EndIf
	
	Switch $iMode
		Case 1
			Return $aDate[3] & '/' & $aDate[2] & '/' & $aDate[1] & $sTime
		Case Else
			Return $aDate[3] & $s_TE_Date_Sep & $aDate[2] & $s_TE_Date_Sep & $aDate[1] & $sTime
	EndSwitch
EndFunc

; #INTERNAL FUNCTION# ====================================================================================================
; Name...........: __TimeEvent_OnExit
; Description....: Internal TimeEvent exit function 
; Author.........: G.Sandler
; ===============================================================================================================
Func __TimeEvent_OnExit()
	_TimeEvent_RemoveAll()
	_WinAPI_SetWindowLong($h_TE_Wnd, $GWL_WNDPROC, $h_TE_TIMECHANGE_WndProc)
	DllCallbackFree($h_TE_TIMECHANGE_Clbk)
	GUIDelete($h_TE_Wnd)
EndFunc
