#include-once

#Region Header

#CS --------------------------------------------------------------------------------------------------------
	Title:			UserInput UDF
	Filename:		UserInput.au3
	Description:	Alternative UDF to the original _IsPressed() function.
	Author:			G.Sandler (MrCreatoR), initial idea/concept by FireFox.
	Version:		1.2
	Last Update:	09.02.2015
	Requirements:	AutoIt v3.2.10.0 - 3.3.12.0, Developed/Tested on WinXP SP2/3 x86, Win7 x64, Rus.
	Uses:           User32.dll
	Notes:          This library is based on _IsPressed UDFs library by Firefox: http://www.autoitscript.com/forum/index.php?showtopic=86296
	
	History:
					v1.2
					* UDF renamed to UserInput.
					* Better return handling (see examples).
					* Changed functions usage concept (see examples).
					* Many bug fixes.
					
					v1.1
					* First public version
	
	Available functions:
					_UserInput_Read
					_UserInput_GetKeyString
					_UserInput_TimeFormat
	
	Forum Link:     http://www.autoitscript.com/forum/index.php?showtopic=88562
#CE ---------------------------------------------------------------------------------------------------------

#EndRegion Header

#Region Global Variables & Initialization

Global Const $WH_MOUSE_LL 					= 14
Global Const $MOUSE_MOVE_EVENT				= 512
Global Const $MOUSE_PRIMARYDOWN_EVENT		= 513
Global Const $MOUSE_PRIMARYUP_EVENT			= 514
Global Const $MOUSE_SECONDARYDOWN_EVENT		= 516
Global Const $MOUSE_SECONDARYUP_EVENT		= 517
Global Const $MOUSE_WHEELDOWN_EVENT			= 519
Global Const $MOUSE_WHEELUP_EVENT			= 520
Global Const $MOUSE_WHEELSCROLL_EVENT		= 522
Global Const $MOUSE_EXTRABUTTONDOWN_EVENT	= 523
Global Const $MOUSE_EXTRABUTTONUP_EVENT		= 524

Global $hUInput_Callback 					= DllCallbackRegister('__UserInput_MouseEvent_Callback', 'int', 'int;ptr;ptr')
Global $iUInput_IsMouseEvent				= -1
Global $iUInput_ScrollDirection				= -1

#EndRegion

#Region Public Functions

; #FUNCTION# ===================================================================
; Name:           	_UserInput_Read
; Description:      Check user input.
;
; Parameter(s):     $nHexStrKey - Hex or String key to check.
;                       [*] This parameter supports strings as used in HotKeySet() function.
;                       [*] To make an "OR" checking operation, use "|" delimiter between the keys/classes.
;                       [*] To make an "AND" checking operation, use "+" delimiter between the keys/classes.
;                       [*] This parameter accepts special CLASS name, bellow is the list of supported classes:
;                                                  [:ALLKEYS:]     - All possible keys (any key) except mouse events (combine with [:ALLMOUSE:] to read all inputs)
;                                                  [:ALPHA:]       - Standard Alpha keys
;                                                  [:ALLNUM:]      - Standard Numeric keys or Numpad keys
;                                                  [:NUMPAD:]      - Numpad keys
;                                                  [:NUMERIC:]     - Standard Numeric keys
;                                                  [:ALLFUNC:]     - F1 -> F24
;                                                  [:FUNC:]        - F1 -> F12
;                                                  [:FUNCSPEC:]    - F13 -> F24
;                                                  [:ARROW:]       - Up, Down, Left, Right
;                                                  [:ALLMOUSE:]    - All mouse buttons + Wheel scroll + Mouse Move
;                                                  [:MOUSEMOVE:]   - Mouse move
;                                                  [:WHEELDOWN:]   - Wheel button held down (pressed)
;                                                  [:WHEELSCROLL:] - Wheel scroll
;                                                  [:SPECIAL:]     - BACKSPACE, TAB, ENTER etc.
;
;                   $iWait      - [Optional] Wait untill the pressed key is released (default is 0, no wait).
;                                 	in this case the return will include time (in milliseconds) the key has been held down.
;                        	      NOTE: $iWait will fail on "+" operation and few mouse events (Wheel scroll and Mouse movement).
;
; Return Value(s):  Always return array with 2 elements. The values in these elements depends on $iWait/$nHexStrKey parameters:
;                                  [0] - If key(s) has been pressed, returns 1 or time representation if $iWait = 1, otherwise returns 0.
;                                  [1] - If key(s) has been pressed, returns pressed key (see bellow exceptions list), otherwise returns empty string ("").
;                                  	* When using mouse classes ([:ALLMOUSE:]/[:MOUSEMOVE:]/[:WHEELDOWN:]/[:WHEELSCROLL:]),
;                                  	  there is an exception in return value, here is the list of return values depending on specified class:
;                                  	  								[:ALLMOUSE:] - Returns:
;                                  	  														"MOUSE_MOVE" string if mouse has been moved
;                                  	  														"MOUSE_BUTTON" string if mouse wheel button has been pressed
;                                  	  														"MOUSE_SCROLL" string if mouse wheel has been scrolled
;                                  	  								[:MOUSEMOVE:] - Returns "MOUSE_MOVE" string if mouse has been moved
;                                  	  								[:WHEELDOWN:] - Returns "MOUSE_BUTTON" string if mouse wheel button has been pressed
;                                  	  								[:WHEELSCROLL:] - Returns "MOUSE_SCROLL" string if mouse wheel has been scrolled
;                                  	  								
;                                  	  								
; Author(s):        G.Sandler (MrCreatoR)
; Note(s):          See examples to get idea of proper usage.
;===============================================================================
Func _UserInput_Read($nHexStrKey, $iWait = 0)
	Local $aKeysPressed, $iFlag = -1, $aPressed[2] = [0, ''], $sPressed = ''
	Local $aRet[2] = [0, '']
	Local $aSplit_Keys = StringSplit($nHexStrKey, '|')
	
	For $i = 1 To $aSplit_Keys[0] ;'OR' pressed check proc
		If $aSplit_Keys[$i] = '' Then
			ContinueLoop
		EndIf
		
		If StringInStr($aSplit_Keys[$i], '+', 2) Then ;'AND' Pressed check proc
			$iFlag = -1
			
			Local $aSplit_And_Keys = StringSplit($aSplit_Keys[$i], '+')
			
			For $j = 1 To $aSplit_And_Keys[0]
				$aPressed = _UserInput_Read($aSplit_And_Keys[$j])
				
				If Not $aPressed[0] Then
					$iFlag = 0
					ExitLoop
				Else
					$sPressed &= $aPressed[1]
					
					If $j < $aSplit_And_Keys[0] Then
						$sPressed &= '+'
					EndIf
				EndIf
			Next
			
			If $iFlag = -1 Then
				$iFlag = 1
			EndIf
			
			If $iFlag = 1 Or ($aSplit_Keys[0] = 1 And $sPressed And StringRight($sPressed, 1) <> '+') Then
				Dim $aRet[2] = [1, $sPressed]
				Return $aRet
			EndIf
			
			$sPressed = ''
			ContinueLoop
		EndIf
		
		Select
			Case $aSplit_Keys[$i] = '[:ALLKEYS:]'			;All possible keys (any key)
				$aKeysPressed = __UserInput_CheckKeysPressed(7, 222, -1, $iWait)
				If Not @error Then Return $aKeysPressed
			Case $aSplit_Keys[$i] = '[:ALPHA:]'				;Standard Alpha keys
				$aKeysPressed = __UserInput_CheckKeysPressed(58, 90, -1, $iWait)
				If Not @error Then Return $aKeysPressed
			Case $aSplit_Keys[$i] = '[:ALLNUM:]' 			;Standard Numeric keys or Numpad keys
				$aKeysPressed = __UserInput_CheckKeysPressed(48, 57, -1, $iWait) ;Standard Numeric
				If Not @error Then Return $aKeysPressed
				$aKeysPressed = __UserInput_CheckKeysPressed(96, 105, -1, $iWait) ;Numpad keys
				If Not @error Then Return $aKeysPressed
			Case $aSplit_Keys[$i] = '[:NUMPAD:]'			;Numpad keys
				$aKeysPressed = __UserInput_CheckKeysPressed(96, 105, -1, $iWait)
				If Not @error Then Return $aKeysPressed
			Case $aSplit_Keys[$i] = '[:NUMERIC:]'			;Standard numeric keys
				$aKeysPressed = __UserInput_CheckKeysPressed(48, 57, -1, $iWait)
				If Not @error Then Return $aKeysPressed
			Case $aSplit_Keys[$i] = '[:ALLFUNC:]' 			;F1 - F24
				$aKeysPressed = __UserInput_CheckKeysPressed(112, 135, -1, $iWait)
				If Not @error Then Return $aKeysPressed
			Case $aSplit_Keys[$i] = '[:FUNC:]' 				;F1 - F12
				$aKeysPressed = __UserInput_CheckKeysPressed(112, 123, -1, $iWait)
				If Not @error Then Return $aKeysPressed
			Case $aSplit_Keys[$i] = '[:FUNCSPEC:]' 			;F13 - F24
				$aKeysPressed = __UserInput_CheckKeysPressed(124, 135, -1, $iWait)
				If Not @error Then Return $aKeysPressed
			Case $aSplit_Keys[$i] = '[:ARROW:]' 			;Up, Down, Left, Right
				$aKeysPressed = __UserInput_CheckKeysPressed(37, 40, -1, $iWait)
				If Not @error Then Return $aKeysPressed
			Case $aSplit_Keys[$i] = '[:ALLMOUSE:]' 			;All mouse buttons + Wheel scroll + Mouse Move
				$aKeysPressed = __UserInput_CheckKeysPressed(1, 6, -1, $iWait) ;All standard mouse buttons
				If Not @error Then Return $aKeysPressed
				
				If __UserInput_IsMouseEvent(1) Then
					Dim $aRet[2] = [1, 'MOUSE_MOVE']
					Return $aRet
				EndIf
				
				If __UserInput_IsMouseEvent(2) Then
					Dim $aRet[2] = [1, 'WHEEL_BUTTON']
					Return $aRet
				EndIf
				
				If __UserInput_IsMouseEvent(3) Then
					If @extended Then
						Dim $aRet[2] = [1, 'WHEEL_UP_SCROLL']
					Else
						Dim $aRet[2] = [1, 'WHEEL_DOWN_SCROLL']
					EndIf
					
					Return $aRet
				EndIf
			Case $aSplit_Keys[$i] = '[:MOUSEMOVE:]'			;Mouse move
				If __UserInput_IsMouseEvent(1) Then
					Dim $aRet[2] = [1, 'MOUSE_MOVE']
					Return $aRet
				EndIf
			Case $aSplit_Keys[$i] = '[:WHEELDOWN:]'			;Wheel button held down (pressed)
				If __UserInput_IsMouseEvent(2) Then
					Dim $aRet[2] = [1, 'WHEEL_BUTTON']
					Return $aRet
				EndIf
			Case $aSplit_Keys[$i] = '[:WHEELSCROLL:]'		;Wheel scroll
				If __UserInput_IsMouseEvent(3) Then
					If @extended Then
						Dim $aRet[2] = [1, 'WHEEL_UP_SCROLL']
					Else
						Dim $aRet[2] = [1, 'WHEEL_DOWN_SCROLL']
					EndIf
					
					Return $aRet
				EndIf
			Case $aSplit_Keys[$i] = '[:SPECIAL:]'			;BACKSPACE, TAB, ENTER etc.
				$aKeysPressed = __UserInput_CheckKeysPressed(8, 36, -1, $iWait)
				If Not @error Then Return $aKeysPressed
				
				$aKeysPressed = __UserInput_CheckKeysPressed(41, 46, -1, $iWait)
				If Not @error Then Return $aKeysPressed
				
				$aKeysPressed = __UserInput_CheckKeysPressed(91, 92, -1, $iWait) ;LWin & RWin keys
				If Not @error Then Return $aKeysPressed
				
				;Multiply, Add, Separater, Substract, Decimal, Divide keys
				$aKeysPressed = __UserInput_CheckKeysPressed(106, 111, -1, $iWait) 
				If Not @error Then Return $aKeysPressed
				
				$aKeysPressed = __UserInput_CheckKeysPressed(136, 222, -1, $iWait)
				If Not @error Then Return $aKeysPressed
			Case $iFlag <> 1								;Other (seperate/group) keys
				$aKeysPressed = __UserInput_CheckKeysPressed(1, 1, $aSplit_Keys[$i], $iWait)
				If Not @error Then Return $aKeysPressed
		EndSelect
	Next
	
	$aRet[0] = $iFlag
	
	If $iFlag = -1 Then
		$aRet[0] = 0
	EndIf
	
	Return $aRet
EndFunc

; #FUNCTION# ===================================================================
; Name:           	_UserInput_GetKeyString
; Description:      Returns Hexadecimal or Aplha key for specified key.
;
; Parameter(s):     $iKeyIn   - Key to check (compare).
;					$iRetType - [Optional] Return type:
;                                                  0 Return Alpha.
;                                                  1 Return Hexadecimal.
;
; Requirement(s):   None.
;
; Return Value(s):  On Success - Returns Key type.
;                   On Failure - Returns $iKeyIn.
;
; Author(s):        Valuater, FireFox, MrCreatoR
; Note(s):          Thanks Valuater... 8)
;===============================================================================
Func _UserInput_GetKeyString($iKeyIn, $iRetType = 0)
	If $iKeyIn = '' Then
		Return SetError(1, 0, '')
	EndIf
	
	Local $s_String = '|01{LMouse}|02{RMouse}|04{MMouse}|05{X1Mouse}|06{X2Mouse}|08{BACKSPACE}|09{TAB}|0C{CLEAR}|' & _
			'0D{ENTER}|10{SHIFT}|11{CTRL}|12{ALT}|13{PAUSE}|14{CAPSLOCK}|1B{ESC}|20{SPACE}|21{PGUP}|22{PGDN}|' & _
			'23{END}|24{HOME}|25{LEFT}|26{UP}|27{RIGHT}|28{DOWN}|29{SELECT}|2A{PRINT}|2B{EXECUTE}|2C{PRINTSCREEN}|2D{INS}|' & _
			'2E{DEL}|30{0}|31{1}|32{2}|33{3}|34{4}|35{5}|36{6}|37{7}|38{8}|39{9}|41{A}|42{B}|43{C}|44{D}|45{E}|46{F}|47{G}|' & _
			'48{H}|49{I}|4A{J}|4B{K}|4C{L}|4D{M}|4E{N}|4F{O}|50{P}|51{Q}|52{R}|53{S}|54{T}|55{U}|56{V}|57{W}|58{X}|59{Y}|5A{Z}|' & _
			'5B{LWindows}|5C{RWindows}|60{Numpad0}|61{Numpad1}|62{Numpad2}|63{Numpad3}|64{Numpad4}|65{Numpad5}|' & _
			'66{Numpad6}|67{Numpad7}|68{Numpad8}|69{Numpad9}|' & _
			'6A{Multiply}|6B{Add}|6C{Separator}|6D{Subtract}|6E{Decimal}|6F{Divide}|' & _
			'70{F1}|71{F2}|72{F3}|73{F4}|74{F5}|75{F6}|76{F7}|77{F8}|78{F9}|' & _
			'79{F10}|7A{F11}|7B{F12}|7C{F13}|7D{F14}|7E{F15}|7F{F16}|80H{F17}|81H{F18}|82H{F19}|83H{F20}|84H{F21}|85H{F22}|' & _
			'86H{F23}|87H{F24}|90{NUMLOCK}|91{SCROLLLOCK}|A0{LSHIFT}|A1{RSHIFT}|A2{LCTRL}|A3{RCTRL}|' & _
			'A4{LMENU}|A5{RMENU}|BA{;}|BB{=}|BC{,}|BD{-}|BE{.}|BF{/}|C0{`}|DB{[}|DC{\}|DD{]}|DE{''}|'
	
	If $iRetType = 1 Then
		;Return Hex
		$s_String = StringRegExpReplace($s_String, '(?i).*\|(.*?)\{' & $iKeyIn & '\}\|.*', '\1')
	Else
		;Return string
		$s_String = StringRegExpReplace($s_String, '(?i).*\|' & $iKeyIn & '\{(.*?)\}\|.*', '\1')
	EndIf
	
	If @extended > 0 Then
		Return $s_String
	EndIf
	
	Return $iKeyIn
EndFunc

; #FUNCTION# ===================================================================
; Name:      		_UserInput_TimeFormat
; Description:      Return time formated string from milliseconds.
;
; Parameter(s):     $iMSecs - Number that include amount of milliseconds to transform to time format.
;					$sDelim - [Optional] Delimeter for the return time format (default is ':').
;
; Requirement(s):   None.
; Return Value(s):  On Success - Returns formatted string time.
;                   On Failure - Returns 0 and set @error to 1 if $iSecs param is invalid.
;
; Author(s):       	MrCreatoR
; Note(s):          
;===============================================================================
Func _UserInput_TimeFormat($iMSecs, $sDelim = ':')
	If Not IsNumber($iMSecs) Or $iMSecs < 0 Then
		Return SetError(1, 0, 0)
	EndIf
	
	Local $iHours, $iMins, $iSecs
	
	$iMSecs /= 1000
	$iHours = Int($iMSecs / 3600)
	$iMSecs = Mod($iMSecs, 3600)
	$iMins = Int($iMSecs / 60)
	$iSecs = Round(Mod($iMSecs, 60))
	
	Return StringFormat('%02i' & $sDelim & '%02i' & $sDelim & '%02i', $iHours, $iMins, $iSecs)
EndFunc

#EndRegion Public Functions

#Region Internal Functions

; #INTERNAL FUNCTION# ===================================================================
; Name: 			__UserInput_CheckKeysPressed
; Description:      Check if specified keys are pressed
;
; Parameter(s):     $iStart   - Start char key to check (this one ignored if $iHexKey is specified).
;                   $iFinish  - End char key to check (this one ignored if $iHexKey is specified).
;                   $iHexKey  - [Optional] Key to check for (default is -1, check keys by $iStart and $iFinish chars).
;                   $iWait    - [Optional] If this param is > 0, then $aRet[0] will contain time in milliseconds.
;
; Requirement(s):	None.
; Return Value(s):  On Success - Returns an array with 2 elements:
;																	[0] - if $iWait = 0, returns 0, otherwise returns the time milliseconds
;																	[1] - pressed hex key
;                   On Failure - Set @error to 1 if invalid $iHexKey, and @error to 2 if key not pressed.
; Author(s):        Valuater, mod. by MrCreatoR
; Note(s):			
;===============================================================================
Func __UserInput_CheckKeysPressed($iStart, $iFinish, $iHexKey = -1, $iWait = 0)
	Local $iKey, $ia_R, $nHex_Key, $aDelim_Keys
	Local $aRet[2] = [0, '']
	
	If $iHexKey <> -1 Then
		$iStart = 1
		$iFinish = 1
		
		$iHexKey = __UserInput_KeyStrToVkCode($iHexKey)
		
		If @error Then
			$iHexKey = _UserInput_GetKeyString($iHexKey, 1)
			
			If @error Then
				Return SetError(1, 0, $aRet)
			EndIf
		EndIf
	EndIf
	
	For $iKey = $iStart To $iFinish
		If $iHexKey <> -1 Then
			$nHex_Key = $iHexKey
		Else
			$nHex_Key = Hex($iKey, 2)
			
			If $iKey > 127 And $iKey < 136 Then
				$nHex_Key &= 'H' ;F17 - F24 keys have the lenght of 3 chars, "H" is added.
			EndIf
		EndIf
		
		If StringInStr($nHex_Key, '|', 2) Then
			$aDelim_Keys = StringSplit($nHex_Key, '|')
			
			For $jKey = 1 To $aDelim_Keys[0]
				If __UserInput_IsPressed($aDelim_Keys[$jKey]) Then
					$aRet[0] = 1
					$aRet[1] = $aDelim_Keys[$jKey]
					
					If $iWait Then
						$aRet[0] = __UserInput_WaitGetTime($aRet[1])
					EndIf
					
					Return $aRet
				EndIf
			Next
		Else
			If __UserInput_IsPressed($nHex_Key) Then
				$aRet[0] = 1
				$aRet[1] = $nHex_Key
				
				If $iWait Then
					$aRet[0] = __UserInput_WaitGetTime($aRet[1])
				EndIf
				
				Return $aRet
			EndIf
		EndIf
	Next
	
	Return SetError(2, 0, $aRet)
EndFunc

; #INTERNAL FUNCTION# ===================================================================
; Name:			    _UserInput_WaitGetTime
; Description:      Returns pressed time of specified key (the time the key has been held down)
; Parameter(s):     $sHexKey    - Hexadecimal/string key.
;
; Requirement(s):	None.
; Return Value(s):  On Success - Returns the time in milliseconds the key has been held down (pressed)
;                   On Failure - Returns 0
; Author(s):        FireFox, Mod. by MrCreatoR
; Note(s):			
;===============================================================================
Func __UserInput_WaitGetTime($sHexKey)
	If __UserInput_IsPressed($sHexKey) Then
		Local $iTimerInit = TimerInit()
		
		While __UserInput_IsPressed($sHexKey)
			Sleep(10)
		WEnd
		
		Local $iDiffKey = TimerDiff($iTimerInit) - 10 ;Remove the Sleep mseconds
		Return $iDiffKey
	EndIf
	
	Return 0
EndFunc

; #INTERNAL FUNCTION# ===================================================================
; Name:      		__UserInput_KeyStrToVkCode
; Description:      Return hex-code for passed string (in format of HotKeySet() function).
;
; Parameter(s):     $sKeyStr - String in format of HotKeySet() function.
;
; Requirement(s):   None.
; Return Value(s):  On Success - Returns "_IsPressed compatible" hex-code.
;                   On Failure - Returns 0 and set @error to 1.
; Author(s):       	MrCreatoR
; Note(s):          
;===============================================================================
Func __UserInput_KeyStrToVkCode($sKeyStr)
	Local $sRet_Keys = '', $aDelim_Keys[1]
	
	Local $aKeys = StringSplit('{LMouse}|{RMouse}|{}|(MMouse}|{}|{}|{}|{BACKSPACE}|{TAB}|{}|{}|{}|{ENTER}|{}|{}|{SHIFT}|{CTRL}|{ALT}|{PAUSE}|{CAPSLOCK}|{}|{}|{}|{}|{}|{}|{ESC}|{}|{}|{}|{}|{SPACE}|{PGUP}|{PGDN}|{END}|{HOME}|{LEFT}|{UP}|{RIGHT}|{DOWN}|{SELECT}|{PRINTSCREEN}|{}|{PRINTSCREEN}|{INSERT}|{DEL}|{}|0|1|2|3|4|5|6|7|8|9|{}|{}|{}|{}|{}|{}|{}|a|b|c|d|e|f|g|h|i|j|k|l|m|n|o|p|q|r|s|t|u|v|w|x|y|z|{LWIN}|{RWIN}|{APPSKEY}|{}|{SLEEP}|{Numpad0}|{Numpad1}|{Numpad2}|{Numpad3}|{Numpad4}|{Numpad5}|{Numpad6}|{Numpad7}|{Numpad8}|{Numpad9}|{NUMPADMULT}|{NUMPADADD}|{}|{NUMPADSUB}|{NUMPADDOT}|{NUMPADDIV}|{F1}|{F2}|{F3}|{F4}|{F5}|{F6}|{F7}|{F8}|{F9}|{F10}|{F11}|{F12}|{F13}|{F14}|{F15}|{F16}|{F17}|{F18}|{F19}|{F20}|{F21}|{F22}|{F23}|{F24}|{}|{}|{}|{}|{}|{}|{}|{}|{NUMLOCK}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{SHIFT}|{SHIFT}|{CTRL}|{CTRL}|{ALT}|{ALT}|{BROWSER_BACK}|{BROWSER_FORWARD}|{BROWSER_REFRESH}|{BROWSER_STOP}|{BROWSER_SEARCH}|{BROWSER_FAVORITES}|{BROWSER_HOME}|{VOLUME_MUTE}|{VOLUME_DOWN}|{VOLUME_UP}|{MEDIA_NEXT}|{MEDIA_PREV}|{MEDIA_STOP}|{MEDIA_PLAY_PAUSE}|{LAUNCH_MAIL}|{LAUNCH_MEDIA}|{LAUNCH_APP1}|{LAUNCH_APP2}|{}|{}|;|{+}|,|{-}|.|/|`|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|{}|[|\|]|''', '|')
	
	If StringRegExp($sKeyStr, '\A\[[^\[\]]+\]\z') Then
		$sKeyStr = StringRegExpReplace($sKeyStr, '\A\[|\]\z', '')
		$sKeyStr = StringRegExpReplace($sKeyStr, '(.)', '\1|')
		$sKeyStr = StringRegExpReplace($sKeyStr, '\|+$', '')
		
		$aDelim_Keys = StringSplit($sKeyStr, '')
	EndIf
	
	For $i = 1 To $aKeys[0]
		If $aDelim_Keys[0] > 1 Then
			For $j = 1 To $aDelim_Keys[0]
				If $aKeys[$i] = $aDelim_Keys[$j] Then
					If $sRet_Keys Then
						$sRet_Keys &= '|'
					EndIf
					
					$sRet_Keys &= Hex($i, 2)
				EndIf
			Next
		Else
			If $aKeys[$i] = $sKeyStr Then
				Return Hex($i, 2)
			EndIf
		EndIf
	Next
	
	If $sRet_Keys = '' Then
		Return SetError(1, 0, $sKeyStr)
	EndIf
	
	Return $sRet_Keys
EndFunc

; #INTERNAL FUNCTION# ===================================================================
; Name:      		__UserInput_IsPressed
; Description:      IsPressed function from standard UDFs.
;
; Parameter(s):     $sHexKey - Hexadecimal/string key.
;					$vDLL - [Optional] Dll filename (default is 'user32.dll').
;
; Return Value(s):  On Success - Returns True.
;                   On Failure - Returns False and set @error to 1.
;
; Author(s):       	ezzetabi and Jon
;===============================================================================
Func __UserInput_IsPressed($sHexKey, $vDLL = 'user32.dll')
	Local $a_R = DllCall($vDLL, 'short', 'GetAsyncKeyState', 'int', '0x' & $sHexKey)
	Local $iError = @error
	Local $aLastErr = DllCall("kernel32.dll", "dword", "GetLastError")
	
	If $iError Or (Not $a_R[0] And $aLastErr[0]) Then
		Return SetError(1, 0, False)
	EndIf
	
	Return BitAND($a_R[0], 0x8000) = 0x8000
EndFunc

; #INTERNAL FUNCTION# ===================================================================
; Name:           	__UserInput_IsMouseEvent
; Description:      Returns wheel mouse button scrolled up or down.
;
; Parameter(s):     $iEvent - The event to check, these are currently supported:
;                                                                   1 - Mouse Move event.
;                                                                   2 - Mouse Wheel button held down (pressed).
;                                                                   3 - Mouse Wheel button scrolled.
;
; Requirement(s):   __UserInput_MouseEvent_Callback.
; Return Value(s):  On Success - Returns 1.
;                   On Failure - Returns 0. @error will be set to 1 if the event is not supported.
; Author(s):        MrCreatoR
; Note(s):          This function includes a sleep of 100 ms, and uses callbacks.
;===============================================================================
Func __UserInput_IsMouseEvent($iEvent)
	Local $hModule, $hHook, $iTimer, $iRet_Event, $iExtended
	
	$hModule = DllCall('Kernel32.dll', 'hwnd', 'GetModuleHandle', 'ptr', 0)
	$hHook = DllCall('User32.dll', 'hwnd', 'SetWindowsHookEx', 'int', $WH_MOUSE_LL, 'ptr', DllCallbackGetPtr($hUInput_Callback), 'hwnd', $hModule[0], 'dword', 0)
	
	If IsArray($hHook) And $hHook[0] > 0 Then
		Local $iTimer = TimerInit()
		
		;Wait untill our $iUInput_IsMouseEvent is set
		While $iUInput_IsMouseEvent = -1 And $iUInput_ScrollDirection = -1
			;Sleep(10)
			
			;We exit the loop after 10 msecond
			If TimerDiff($iTimer) >= 10 Then
				ExitLoop
			EndIf
		WEnd
		
		DllCall('user32.dll', 'int', 'UnhookWindowsHookEx', 'hwnd', $hHook[0])
	Else
		Return SetError(1, 0, 0)
	EndIf
	
	$iRet_Event = $iUInput_IsMouseEvent
	$iExtended = Number($iUInput_ScrollDirection > 0)
	
	$iUInput_IsMouseEvent = -1
	$iUInput_ScrollDirection = -1
	
	Switch $iEvent
		Case 1 ;Mose Move
			Return ($iRet_Event = $MOUSE_MOVE_EVENT)
		Case 2 ;Mouse Wheel Button held down (pressed)
			Return ($iRet_Event = $MOUSE_WHEELDOWN_EVENT)
		Case 3 ;Mouse Wheel Button Scroll
			Return SetExtended($iExtended, ($iRet_Event = $MOUSE_WHEELSCROLL_EVENT))
	EndSwitch
	
	Return SetError(2, 0, 0)
EndFunc

;Callback function for __UserInput_IsMouseEvent
Func __UserInput_MouseEvent_Callback($iCode, $wParam, $lParam)
	Local $tStruct = DllStructCreate('struct;long X;long Y;endstruct;dword mouseData;dword flags;dword time;ulong_ptr dwExtraInfo', $lParam)
	
	$iUInput_IsMouseEvent = BitAND($wParam, 0xFFFF)
	$iUInput_ScrollDirection = BitShift(DllStructGetData($tStruct, 3), 16)
	
	Return 0
EndFunc

#EndRegion Internal Functions
