#include-once
#include <GUIConstantsEx.au3>
#include <StaticConstants.au3>
#include <ScreenCapture.au3>
#include <Misc.au3>
#include <File.au3>

#CS Library Info/Header

	        Name/Title:         Opera.au3 / Opera Browser Automation UDF Library
	        Description:        Functions to handle some of the common routines for Opera Browser!
	        Version:            1.08
	        Requirements:       AutoIt 3.2.10.0 or higher.
	        Note(s):            This UDF Library tested on Opera versions: 8.54-9.5, 10.63.3516, few critical functions tested also on 11.x-12.x.
	        Support Link:       
	        Author:             Copyright  2007 - 2013 CreatoR's Lab (G.Sandler), www.autoit-script.ru
	
	        ToDo:
	                        * Add UDFs:
	                            _OperaFindHotkeyAction (search for hotkey by action)
	                            _OperaFindActionHotkey (search for action by hotkey)
	                            _OperaAddToolbarButton()
	                            _OperaRemoveToolbarButton()
	                            _OperaAddSkinIcon() -> Required 7zip/WinRar, or use system zipfolder.
	                            _OperaRemoveSkinIcon() -> Required 7zip/WinRar, or use system zipfolder.
	                            _OperaItemGetType() -> Returns IsItem, IsCommentedItem, IsSeperator, IsCommentedSeperator, IsCommentedLine
	
	        History Version:
								v1.08 [21.Jun.2013]
								* Fixed issue with getting opera profile path (mostly related to _OperaGetMailDir).
								
								v1.07 [04.Jun.2013]
								+ Added _OperaGetDownloadsDir function.
								+ Added _OperaGetSessionFiles function.
								+ Added global $i_Opera_SearchAllDrives variable to allow profile dir detection on other drives. Usefull when installed more than one OS.
								+ Added global $s_Opera_ProfileDir variable to specify custom profile dir (this variable should be empty if you need to use _OperaGetProfileDir).
								* Fixed few functions to work better with new Opera versions.
								* Fixed internal __Opera_GetOperaPageHandle function (used in public _OperaScreenCapture function).
								
								v1.06 [12.Jul.2012]
								* Fixed _OperaGetDir function:
								  - To be compatible with x64 system.
								  - Changed/expanded parameters, now used $iFlag to determine the function behavior, see function header for more details.
								* _OperaIsRuning function renamed to _OperaIsRunning. Old function name still supported.
								* Renamed internal functions to __Opera_*.
								
								v1.05 [08.Jul.2012]
								* Fixed _OperaGetTitle function, the title was incorrect if $sOpDir was specified and $iRet <> 2.
								* Fixed __Correct_Opera_Path internal function to replace back slashes (/).
								
			                    v1.04 [26.Jun.2011]
								* Fixed following functions to be more compatible with Opera 11.x:
								  _OperaGetProfileDir
								  _OperaMultiUserModeIsOn
								  _OperaGetUserJSDir
								  __Correct_Opera_Path
								
								* Fixed major bugs with recognizing Opera Dir (if $sOpDir parameter is passed as wrong Opera dir path).
								* Fixed _OperaGetUserJSDir function (now working better with relative pathes).
								* Renamed and changed _Opera_Integration_Example function. Now it's considered active function and named _Opera_Integration_Module.
								* Changed _OperaSelectFolder function. Added new optional parameter $hParent.
								* Changed _OperaAddMenuItem and _OperaRemoveMenuItem functions. Added new optional parameter $iBackup.
								* Fixed _OperaAddMenuItem function. Encoding issues.
								+ Added _OperaGetCacheDir function.
								+ Added optional $iGetLastInstallPath parameter to _OperaGetDir function.
								   If this parameter is 1, then function will try to detect the last installed Opera path, 
                                   instead of the path that Opera.exe was last time launched from.
								
								v1.03 [16.May.2011]
								* UDF renamed to Opera.au3.
								* Fixed _OperaScreenCapture function (now compatible with Opera 10.6, but still having an issue with capturing panels).
								* Fixed few major issues with getting preference file's pathes.
								* Fixed _OperaGetDir() function.
								* Fixed _OperaGetProfileDir() function (errors when getting profile dir on multiuser mode).
								* Changed _OperaFindDir function.
								  - Now the $sInitPath parameter can accept pathes (delimited with "|").
								  - Now -1 as second parameter ($iRetCount) tels to the function to return all found Opera dirs.
								  - Also the function now works faster.
								+ Opera 10.6 support.
								+ Added _OperaToggleMenuItemState function.
								+ Added optional $sStandard_Name parameter to _OperaSetConfigFile function:
									If this parameter is a string (default is -1), then it's used as standard file name
									(instead of "standard_" & $sConfig_Prefix & ".ini").
								
								v1.02 [03.Jul.2009]
								* Now compatible with AutoIt 3.3.0.0.
								+ Opera 10 support.
								+ Added _OperaGetBookmarksFile() function.
								+ Added _OperaScreenCapture() function...
									ATTENTION: supported only for v3.2.10.0 or higher, due to (lack of) callback support.
								+ Added _OperaAddMenuItem()/_OperaRemoveMenuItem() function.
								* Fixed few function to read relative paths in opera prefs file.
	
	                            v1.01 [5.Jan.2008]
								* Now compatible with AutoIt 3.2.10.0.
								* Variable names and spell corrections.
								+ Added 2 more UDFs:
														_OperaRestart()
														_OperaMultiUserModeIsOn()
	
	                            v1.0 [28.Dec.2007]
								* First Release.
	
	        Functions List:
	           User's funcs...
	                           1)_OperaFindDir
	                           2)_OperaGetDir
	                           3)_OperaGetDownloadsDir
							   4)_OperaGetVersion
	                           5)_OperaGetTitle
	                           6)_OperaGetProfileDir
	                           7)_OperaGetMailDir
	                           8)_OperaGetUserJSDir
	                           9)_OperaGetPluginsDir
	                           10)_OperaGetSessionFiles
	                           11)_OperaGetSkinFile
	                           12)_OperaGetLangFile
	                           13)_OperaGetBookmarksFile
							   14)_OperaGetCacheDir
	                           15)_OperaGetConfigFile
	                           16)_OperaSetConfigFile
	                           17)_OperaSetSkinFile
	                           18)_OperaSetUserJSDir
	                           19)_OperaSetLangFile
	                           20)_OperaSelectFolder
	                           21)_OperaAddMenuItem
	                           22)_OperaRemoveMenuItem
	                           23)_OperaIsRunning
	                           24)_OperaRun
	                           25)_OperaClose
	                           26)_OperaKill
	                           27)_OperaRestart
	                           28)_OperaScreenCapture
	                           29)_OperaMultiUserModeIsOn
	                           30)_Opera_Integration_Example
	                           31)_OpenGetURL
	                           32)_OperaToggleMenuItemState
	
	           internal funcs...
	                           1)__Opera_WinGetTitleEx
	                           2)__Opera_WinGetTitleByPID
	                           3)__Opera_ProcessPathExists
	                           4)__Opera_ProcessGetPath
	                           5)__Opera_PathIsFolder
	                           6)__Opera_PathIsOperaDir
	                           7)__Opera_PathIsOperaProfileDir
	                           8)__Opera_PathIsOperaConfigFile
	                           9)__Opera_PathIsOperaLangFile
	                           10)__Opera_PathIsOperaSkinFile
	                           11)__Opera_PathIsOperaBookmarksFile
	                           12)__Opera_PrefixStrIsOperaConfig
	                           13)__Opera_ConfirmExitOperaWinExists
	                           14)__Opera_Correct_Opera_Path
	                           15)__Opera_GetOperaPIDs
	                           16)__Opera_GetOperaPrefsFileName
	                           17)__Opera_GetOperaPageHandle
	                           18)__Opera_EnumOperaChildWindowsProc
	                           19)__Opera_ExpandOperaMenuSectClassName
	                           20)__Opera_Backup_Config
	                           21)__Opera_FileNewName
	                           22)__Opera_UTF8ToAnsi
	                           23)__Opera_StringIsUTF8Format
	                           24)__Opera_StringToUTF
	                           25)__Opera_IniWriteToPos
	                           26)__Opera_IniDeleteRegExp
	                           27)__Opera_WinAPI_FindWindowEx
	                           28)__Opera_WinAPI_GetClassName
	                           29)__Opera_WinAPI_PathRemoveFileSpec
	                           30)__Opera_WinAPI_AssocQueryString

#CE

Global $a_OperaChildWindows[1]
Global $i_Opera_SearchAllDrives = 0
Global $s_Opera_ProfileDir = ''
Global Const $s_Opera_Exe = "Opera.exe"

;Usage:
;~ #include <Array.au3>
;~ $aHotKeys = _OperaGetActionHotkey("(?i)Close page.*", 1, "C:\Opera_Tests\Opera_12")
;~ _ArrayDisplay($aHotKeys)
Func _OperaGetActionHotkey($sAction, $iUseRegExp = 0, $sOpDir = -1, $sKeyboard_File = '')
	If $sKeyboard_File = '' Then
		$sKeyboard_File = _OperaGetConfigFile("Keyboard", $sOpDir)
	EndIf
	
	If @error Or Not FileExists($sKeyboard_File) Then Return SetError(1, 0, 0)
	
	Local $aHotKeys_Section = IniReadSection($sKeyboard_File, "Application")
	If @error Then Return SetError(2, 0, 0)
	
	Local $aRet_HotKeys[$aHotKeys_Section[0][0]+1][2] = [[0]]
	
	For $i = 1 To $aHotKeys_Section[0][0]
		If ($iUseRegExp And StringRegExp($aHotKeys_Section[$i][1], $sAction)) Or (Not $iUseRegExp And $aHotKeys_Section[$i][1] = $sAction) Then
			$aHotKeys_Section[$i][0] = StringRegExpReplace($aHotKeys_Section[$i][0], "(?i)platform.*?,|feature.*?,", "")
			$aHotKeys_Section[$i][0] = StringRegExpReplace($aHotKeys_Section[$i][0], "(?i)(f\d+)", "{\1}")
			$aHotKeys_Section[$i][0] = StringReplace($aHotKeys_Section[$i][0], "ctrl", "^")
			$aHotKeys_Section[$i][0] = StringReplace($aHotKeys_Section[$i][0], "shift", "+")
			$aHotKeys_Section[$i][0] = StringReplace($aHotKeys_Section[$i][0], "alt", "!")
			$aHotKeys_Section[$i][0] = StringStripWS($aHotKeys_Section[$i][0], 8)
			$aHotKeys_Section[$i][0] = StringRegExpReplace($aHotKeys_Section[$i][0], "(?i)(.*?)([\^\+!]+)(.*?)", "\2\1\3")
			
			$aRet_HotKeys[0][0] += 1
			$aRet_HotKeys[$aRet_HotKeys[0][0]][0] = $aHotKeys_Section[$i][1]
			$aRet_HotKeys[$aRet_HotKeys[0][0]][1] = $aHotKeys_Section[$i][0]
		EndIf
	Next
	
	If $aRet_HotKeys[0][0] = 0 Then Return SetError(3, 0, 0)
	
	ReDim $aRet_HotKeys[$aRet_HotKeys[0][0]+1][2]
	Return $aRet_HotKeys
EndFunc

;===============================================================================
;
; Function Name:  		   _OperaFindDir()
;
; Function Description:    Performs a search for Opera Browser Installed Dir(s).
;
; Parameter(s):            $sInitPath - [Optional] Initial Path(s) to start search from,
;                                          if used -1 (default) then the search performed in all fixed drives (C:\, D:\, etc.).
;                                          To specify more than one path, separate them with |.
;                          $iRetCount - [Optional] Defines how many Opera Dirs to return:
;                                          -1 - (default) will return all found dirs.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return array with full pathes to Opera Dir, where:
;                                                $aOperaPaths[0] is the total found Opera dirs.
;                                                $aOperaPaths[N] is the full path to Opera Installed dir.
;                          On Failure -  If no Opera dirs found, Set @error to 1 and return "" (empty string).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaFindDir($sInitPath = -1, $iRetCount = -1)
	Local $aInitPaths = StringSplit($sInitPath, "|")
	Local $aOperaPaths[1], $sOutBin, $sOut, $aOut, $sRead, $hDir, $sAttrib
	
	$sAttrib = ' /A-D' ;Only files
	$sOut = StringToBinary('0' & @CRLF, 2)
	
	If Not __Opera_PathIsFolder($aInitPaths[1]) Then
		$aInitPaths = DriveGetDrive("FIXED")
		
		If @error Then
			Return SetError(-1, 0, "")
		EndIf
	EndIf
	
	For $x = 1 To UBound($aInitPaths) - 1
		$aInitPaths[$x] = StringRegExpReplace($aInitPaths[$x], '\\+$', '')
		
		$hDir = Run(@ComSpec & ' /U /C DIR "' & $aInitPaths[$x] & '\' & $s_Opera_Exe & '" /S /B' & $sAttrib, @SystemDir, @SW_HIDE, 6)
		
		While 1
			$sRead = StdoutRead($hDir, False, True)
			
			If @error Then
				ExitLoop
			EndIf
			
			If $sRead <> "" Then
				$sOut &= $sRead
			EndIf
			
			If $iRetCount > 0 Then
				StringReplace(BinaryToString($sOut), @LF, "")
				
				If @extended > $iRetCount Then
					ExitLoop 2
				EndIf
			EndIf
		Wend
	Next
	
	$sOut = BinaryToString($sOut, 2)
	$sOut = StringRegExpReplace($sOut, "(?m)\\[^\\]*$", "")
	$aOut = StringRegExp($sOut, '[^\r\n]+', 3)
	
	$aOut[0] = UBound($aOut)-1
	$aOperaPaths = $aOut
	
	If $aOperaPaths[0] < 1 Then
		Return SetError(1, 0, "")
	EndIf
	
	Return $aOperaPaths
EndFunc   ;==>_OperaFindDir

;===============================================================================
;
; Function Name:  		   _OperaGetDir()
;
; Function Description:    Returns full path to Opera Browser Installed Dir.
;
; Parameter(s):            $iFlags - [Optional] Combination of flags:
;                                                                    1 - The function will try to detect the last installed Opera path, 
;                                                                        instead of the path that Opera.exe was last time launched from.
;                                                                    2 - The function will try to detect the Opera dir using .html files association.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return full path to Opera Dir, i.e. C:\Program Files\Opera 11
;                          On Failure -  Set @error to 1 and return "" (empty string).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaGetDir($iFlags = 0)
	Local $sOperaDirRegRead
	Local $sOperaDir = @ScriptDir
	If __Opera_PathIsOperaDir($sOperaDir) Then Return $sOperaDir
	
	If BitAND($iFlags, 2) Then
		$sOperaDir = __Opera_WinAPI_PathRemoveFileSpec(__Opera_WinAPI_AssocQueryString(".html", 2)) ;$ASSOCSTR_EXECUTABLE = 2
		If __Opera_PathIsOperaDir($sOperaDir) Then Return $sOperaDir
	EndIf
	
	Local $x64 = _Iif(@OSArch = "x64", "64", "")
	Local $sHKCU_RegKey = "HKEY_CURRENT_USER" & $x64
	
	If BitAND($iFlags, 1) Then
		$sOperaDir = __Opera_WinAPI_PathRemoveFileSpec(RegRead($sHKCU_RegKey & "\Software\Opera Software", "Last Install Path"))
	EndIf
	
	If Not __Opera_PathIsOperaDir($sOperaDir) Then
		$sOperaDir = __Opera_WinAPI_PathRemoveFileSpec(RegRead($sHKCU_RegKey & "\Software\Opera Software", "Last CommandLine v2"))
	EndIf
	
	If Not __Opera_PathIsOperaDir($sOperaDir) Then $sOperaDir = @ProgramFilesDir & "\Opera" & $x64
	If Not __Opera_PathIsOperaDir($sOperaDir) Then $sOperaDir = @ProgramFilesDir & "\Opera"
	
	If Not __Opera_PathIsOperaDir($sOperaDir) Then
		DllCall("kernel32.dll", "int", "Wow64DisableWow64FsRedirection", "int", 1)
		$sOperaDir = @ProgramFilesDir & "\Opera" & $x64
	EndIf
	
	If Not __Opera_PathIsOperaDir($sOperaDir) Then Return SetError(1, 0, "")
	
	Return StringRegExpReplace($sOperaDir, "\\+$", "")
EndFunc   ;==>_OperaGetDir

;===============================================================================
;
; Function Name:  		   _OperaGetVersion()
;
; Function Description:    Returns Opera.exe file Version.
;
; Parameter(s):            $sOpDir - [Optional] Full path to Opera dir,
;                                    so the returned Opera.exe version will be taken from specific Opera installation.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  String with Opera Version, i.e. 10.0.1355.0
;                          On Failure -  Returns 0 and set @error to:
;                                                                      1 - $sOpDir (Opera Dir) not found.
;                                                                      2 - Unable to get Opera version (FileGetVersion() fails).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaGetVersion($sOpDir = -1)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, 0)
	
	Local $sOperaVersion = FileGetVersion($sOpDir & "\" & $s_Opera_Exe)
	If @error Then Return SetError(2, 0, 0)
	
	Return $sOperaVersion
EndFunc   ;==>_OperaGetVersion

;===============================================================================
;
; Function Name:  		   _OperaGetTitle()
;
; Function Description:    Returns Title(s)/Win handle(s) for Opera Window.
;
; Parameter(s):            $sOpDir - [Optional] Full path to Opera dir,
;                                     so the returned Title will be related to specific Opera installation.
;                          $iRet   - [Optional] Defins return type:
;                                     -1 (default) - Will return only Opera Title.
;                                     0 - Will return only Opera Window Handle.
;                                     1 - Will return 2-elements array where [0] is Opera Title, and [1] is Opera Win Handle.
;                                     2 - Will return 2-dimentional array with all founded Opera Windows, where...
;                                              $aOpWinList[0][0] is number of total windows found.
;                                              $aOpWinList[N][0] is Opera Window Title.
;                                              $aOpWinList[N][1] is Opera Window Handle.
;                                           $sOpDir is ignored with this mode.
;                          $iUseDDE - [Optional] Get Opera Title method, if > 0 (0 is the default),
;                                               then the Opera Title will be returned using DDE
;                                                  (for last executed Opera's Window).
;                                               When this method used, $iRet can be only as -1 (default) or as 1
;                                                  (other values are ignored).
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return Opera Title(s)/Window Handle(s) according to passed parameters.
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                              1 - $iUseDDE > 0 and there was an error to get access to Opera DDE.
;                                              2 - $iRet = 2 and there was an error to get Opera Windows list.
;                                              3 - Unable to find Opera Title/Win Handle.
;                                              4 - Other error, such as incorrect passed parameter values or unable to find window.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaGetTitle($sOpDir = -1, $iRet = -1, $iUseDDE = 0)
	Local $sOpTitle = ""
	
	If $iUseDDE Then
		$sOpTitle = _OpenGetURL("", -1, 1)
		If Not IsArray($sOpTitle) Then Return SetError(1, 0, "")
		
		$sOpTitle = StringRegExpReplace($sOpTitle[0], '\A".*?\[(.*)\]"$', '\1')
		
		If $iRet = 1 Then
			Local $aRetArr[2] = [$sOpTitle, WinGetHandle($sOpTitle) ]
			Return $aRetArr
		EndIf
		
		Return $sOpTitle
	EndIf
	
	If $iRet = 2 Then
		Local $aOpWinList = WinList("[REGEXPTITLE:(.*?)(- Opera.*|$);[REGEXPCLASS:(OpWindow|OperaWindowClass)]")
		If Not IsArray($aOpWinList) Then Return SetError(2, 0, "")
		Return $aOpWinList
	EndIf
	
	If __Opera_PathIsOperaDir($sOpDir) Then
		$sOpTitle = __Opera_WinGetTitleEx($sOpDir & "\" & $s_Opera_Exe)
		If Not WinExists($sOpTitle) Or $sOpTitle = "" Then Return SetError(3, 0, "")
		;Return $sOpTitle
	Else
		$sOpDir = _OperaGetDir()
	EndIf
	
	If __Opera_PathIsOperaDir($sOpDir) Then $sOpTitle = __Opera_WinGetTitleEx($sOpDir & "\" & $s_Opera_Exe)
	
	If Not WinExists($sOpTitle) Or $sOpTitle = "" Then
		$sOpTitle = WinGetTitle("[REGEXPTITLE:(.*?)(- Opera.*|$);REGEXPCLASS:(OpWindow|OperaWindowClass)]")
		$sOpTitle = StringLeft($sOpTitle, StringInStr($sOpTitle, "-", 0, -1))
		
		If StringRight($sOpTitle, 1) = "-" Then $sOpTitle = StringStripWS(StringTrimRight($sOpTitle, 1), 3)
		If Not WinExists($sOpTitle) Or $sOpTitle = "" Then Return SetError(4, 0, "") ;$sOpTitle = "Opera"
	EndIf
	
	Local $hOphWnd = WinGetHandle($sOpTitle)
	
	Switch $iRet
		Case -1
			Return $sOpTitle
		Case 0
			Return $hOphWnd
		Case 1
			Local $aRetArr[2] = [$sOpTitle, $hOphWnd]
			Return $aRetArr
	EndSwitch
	
	Return SetError(4, 0, "")
EndFunc   ;==>_OperaGetTitle

;===============================================================================
;
; Function Name:  		   _OperaGetProfileDir()
;
; Function Description:    Returns full path to Opera Profile Dir.
;
; Parameter(s):            $sOpDir       - [Optional] Full path to Opera dir,
;                                           so the returned profile dir will be related to specific Opera installed dir.
;                          $sAppDataPath - [Optional] Application Data dir.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return full path to Opera Profile Dir,
;                                        i.e. C:\Documents and Settings\User\Application Data\Opera\Opera 9.25\profile
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                                          1 - $sOpDir (Opera Dir) not found.
;                                                                          2 - Opera6.ini (main Opera's config file) not found.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaGetProfileDir($sOpDir = -1, $sAppDataPath = @AppDataDir)
	If $s_Opera_ProfileDir <> '' Then Return $s_Opera_ProfileDir
	
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, "")
	
	Local $sOperaPrefs_FileName = __Opera_GetOperaPrefsFileName($sOpDir, 0)
	
	If _OperaMultiUserModeIsOn($sOpDir) Then
		Local $sAD_OpPrflDir = "Opera\" & StringRegExpReplace($sOpDir, "^.*\\", "")
		Local $sOpProfileDir = $sAppDataPath & "\" & $sAD_OpPrflDir
		
		If Not __Opera_PathIsOperaConfigFile($sOpProfileDir & "\" & $sOperaPrefs_FileName) And Not FileExists($sOpProfileDir & "\mail") Then
			If $i_Opera_SearchAllDrives Then
				Local $aDrives = DriveGetDrive('FIXED')
				
				For $i = 1 To UBound($aDrives)-1
					$sOpProfileDir = $aDrives[$i] & StringTrimLeft(@AppDataDir, 2) & "\" & $sAD_OpPrflDir
					
					If __Opera_PathIsOperaConfigFile($sOpProfileDir & "\" & $sOperaPrefs_FileName) Then
						Return $sOpProfileDir
					EndIf
					
					$sOpProfileDir = $aDrives[$i] & "\Users\" & @UserName & "\AppData\Roaming\" & $sAD_OpPrflDir
					
					If __Opera_PathIsOperaConfigFile($sOpProfileDir & "\" & $sOperaPrefs_FileName) Then
						Return $sOpProfileDir
					EndIf
					
					$sOpProfileDir = $aDrives[$i] & "\Users\" & @UserName & "\Application Data\" & $sAD_OpPrflDir
					
					If __Opera_PathIsOperaConfigFile($sOpProfileDir & "\" & $sOperaPrefs_FileName) Then
						Return $sOpProfileDir
					EndIf
				Next
			EndIf
			
			Return SetError(2, 0, "")
		EndIf
		
		Return $sOpProfileDir
	Else
		If Not __Opera_PathIsOperaConfigFile($sOpDir & "\profile\" & $sOperaPrefs_FileName) Then
			Return SetError(2, 0, "")
		EndIf
		
		Return $sOpDir & "\profile"
	EndIf
EndFunc   ;==>_OperaGetProfileDir

;===============================================================================
;
; Function Name:  		   _OperaGetDownloadsDir()
;
; Function Description:    Returns full path to Opera main Downloads Dir.
;
; Parameter(s):            $sOpDir    - [Optional] Full path to Opera dir,
;                                        so the returned Downloads dir will be related to specific Opera installed dir.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return full path to Opera Downloads Dir,
;                                         i.e. C:\Documents and Settings\User\Application Data\Opera\Opera 9.25\profile\Downloads
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                 1 - $sOpDir (Opera Dir) not found.
;                                                 2 - Downloads dir not found.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaGetDownloadsDir($sOpDir = -1)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, "")
	
	Local $sOpProfileDir1 = _OperaGetProfileDir($sOpDir, @AppDataDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sOpProfileDir2 = _OperaGetProfileDir($sOpDir, __Opera_GetLocalSettingsPath())
	
	Local $sOperaPrefs_FileName = __Opera_GetOperaPrefsFileName($sOpDir, 0)
	Local $sOpDownloadsDir = IniRead($sOpProfileDir1 & "\" & $sOperaPrefs_FileName, "User Prefs", "Download Directory", "")
	
	__Opera_Correct_Opera_Path($sOpDownloadsDir, $sOpDir, $sOpProfileDir1, $sOpProfileDir2)
	
	If Not __Opera_PathIsFolder($sOpDownloadsDir) Then $sOpDownloadsDir = _PathFull($sOpDownloadsDir, $sOpDir)
	If Not __Opera_PathIsFolder($sOpDownloadsDir) Then Return SetError(2, 0, "")
	
	Return $sOpDownloadsDir
EndFunc

;===============================================================================
;
; Function Name:  		   _OperaGetMailDir()
;
; Function Description:    Returns full path to Opera Mail Dir.
;
; Parameter(s):            $sOpDir    - [Optional] Full path to Opera dir,
;                                        so the returned Mail dir will be related to specific Opera installed dir.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return full path to Opera Mail Dir,
;                                         i.e. C:\Documents and Settings\User\Application Data\Opera\Opera 9.25\mail
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                                     1 - $sOpDir (Opera Dir) not found.
;                                                                     2 - Index.ini (main Opera's Mail config file) not found.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaGetMailDir($sOpDir = -1)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, "")
	
	Local $sOpProfileDir1 = _OperaGetProfileDir($sOpDir, @AppDataDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sOpProfileDir2 = _OperaGetProfileDir($sOpDir, __Opera_GetLocalSettingsPath())
	
	Local $sOperaPrefs_FileName = __Opera_GetOperaPrefsFileName($sOpDir, 0)
	Local $sOpMailDir = IniRead($sOpProfileDir1 & "\" & $sOperaPrefs_FileName, "Mail", "Mail Root Directory", "Def_ERROR")
	
	__Opera_Correct_Opera_Path($sOpMailDir, $sOpDir, $sOpProfileDir1, $sOpProfileDir2)
	
	If $sOpMailDir == "Def_ERROR" And _VersionCompare(_OperaGetVersion($sOpDir), "10") <> -1 Then
		$sOpMailDir = $sOpProfileDir1 & "\mail"
	EndIf
	
	If Not __Opera_PathIsOperaConfigFile($sOpMailDir & "\Index.ini") And Not __Opera_PathIsOperaConfigFile($sOpMailDir & "\accounts.ini") Then
		$sOpMailDir = StringRegExpReplace($sOpProfileDir1, "\\[^\\]*$", "") & "\mail"
	EndIf
	
	If Not __Opera_PathIsOperaConfigFile($sOpMailDir & "\Index.ini") And Not __Opera_PathIsOperaConfigFile($sOpMailDir & "\accounts.ini") Then
		$sOpMailDir = $sOpProfileDir2 & "\mail"
	EndIf
	
	If Not __Opera_PathIsOperaConfigFile($sOpMailDir & "\Index.ini") And Not __Opera_PathIsOperaConfigFile($sOpMailDir & "\accounts.ini") Then
		$sOpMailDir = StringRegExpReplace($sOpProfileDir2, "\\[^\\]*$", "") & "\mail"
	EndIf
	
	If Not __Opera_PathIsOperaConfigFile($sOpMailDir & "\Index.ini") And Not __Opera_PathIsOperaConfigFile($sOpMailDir & "\accounts.ini") Then
		Return SetError(2, 0, "")
	EndIf
	
	Return $sOpMailDir
EndFunc   ;==>_OperaGetMailDir

;===============================================================================
;
; Function Name:  		   _OperaGetUserJSDir()
;
; Function Description:    Returns full path to Opera main UserJS Dir.
;
; Parameter(s):            $sOpDir    - [Optional] Full path to Opera dir,
;                                        so the returned UserJS dir will be related to specific Opera installed dir.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return full path to Opera UserJS Dir,
;                                         i.e. C:\Documents and Settings\User\Application Data\Opera\Opera 9.25\profile\UserJS
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                 1 - $sOpDir (Opera Dir) not found.
;                                                 2 - UserJS dir not found.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaGetUserJSDir($sOpDir = -1)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, "")
	
	Local $sOpProfileDir1 = _OperaGetProfileDir($sOpDir, @AppDataDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sOpProfileDir2 = _OperaGetProfileDir($sOpDir, __Opera_GetLocalSettingsPath())
	
	Local $sOperaPrefs_FileName = __Opera_GetOperaPrefsFileName($sOpDir, 0)
	Local $sOpUserJSDir = IniRead($sOpProfileDir1 & "\" & $sOperaPrefs_FileName, "User Prefs", "User JavaScript File", "")
	
	__Opera_Correct_Opera_Path($sOpUserJSDir, $sOpDir, $sOpProfileDir1, $sOpProfileDir2)
	
	If Not __Opera_PathIsFolder($sOpUserJSDir) Then $sOpUserJSDir = _PathFull($sOpUserJSDir, $sOpDir)
	If Not __Opera_PathIsFolder($sOpUserJSDir) Then Return SetError(2, 0, "")
	
	Return $sOpUserJSDir
EndFunc   ;==>_OperaGetUserJSDir

;===============================================================================
;
; Function Name:  		   _OperaGetPluginsDir()
;
; Function Description:    Returns used paths to Opera main Plugins Dir(s).
;
; Parameter(s):            $sOpDir    - [Optional] Full path to Opera dir,
;                                        so the returned Plugins dir will be related to specific Opera installed dir.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return array with full paths to Opera Plugins Dir.
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                                                     1 - $sOpDir (Opera Dir) not found.
;                                                                                     2 - Plugins dir(s) not found.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaGetPluginsDir($sOpDir = -1)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, "")
	
	Local $sOperaPrefs_DefFileName = __Opera_GetOperaPrefsFileName($sOpDir, 1)
	
	Local $sOpPluginsDir = IniRead($sOpDir & "\" & $sOperaPrefs_DefFileName, "User Prefs", "Plugin Path", "")
	Local $aOpPlugsDir, $aRetOpPlugsDir[1]
	$sOpPluginsDir = StringRegExpReplace($sOpPluginsDir, ";+$", "")
	
	If StringInStr($sOpPluginsDir, ";") Then
		$aOpPlugsDir = StringSplit($sOpPluginsDir, ";")
		
		For $i = 1 To UBound($aOpPlugsDir) - 1
			If __Opera_PathIsFolder($aOpPlugsDir[$i]) Then
				$aRetOpPlugsDir[0] += 1
				ReDim $aRetOpPlugsDir[$aRetOpPlugsDir[0] + 1]
				$aRetOpPlugsDir[$aRetOpPlugsDir[0]] = $aOpPlugsDir[$i]
			EndIf
		Next
		
		If $aRetOpPlugsDir[0] > 0 Then Return $aRetOpPlugsDir
	EndIf
	
	If Not __Opera_PathIsFolder($sOpPluginsDir) Then $sOpPluginsDir = RegRead("HKEY_CURRENT_USER\Software\Opera Software", "Plugin Path")
	If Not __Opera_PathIsFolder($sOpPluginsDir) Then $sOpPluginsDir = $sOpDir & "\program\plugins"
	If Not __Opera_PathIsFolder($sOpPluginsDir) Then Return SetError(2, 0, "")
	
	Dim $aRetOpPlugsDir[2] = [1, $sOpPluginsDir]
	Return $aRetOpPlugsDir
EndFunc   ;==>_OperaGetPluginsDir

;===============================================================================
;
; Function Name:  		   _OperaGetSessionFiles()
;
; Function Description:    Returns used Opera Session files.
;
; Parameter(s):            $sOpDir    - [Optional] Full path to Opera dir,
;                                        so the returned Session files will be related to specific Opera installation.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return an array with Opera session files, where $aArray[1] is the "Automatic Window Storage File".
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                                                     1 - $sOpDir (Opera Dir) not found.
;                                                                                     2 - Session files not found.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaGetSessionFiles($sOpDir = -1)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, "")
	
	Local $sOpProfileDir1 = _OperaGetProfileDir($sOpDir, @AppDataDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sOpProfileDir2 = _OperaGetProfileDir($sOpDir, __Opera_GetLocalSettingsPath())
	
	Local $sOperaPrefs_FileName = __Opera_GetOperaPrefsFileName($sOpDir, 0)
	Local $sOpAutoSessionFile = IniRead($sOpProfileDir1 & "\" & $sOperaPrefs_FileName, "User Prefs", "Automatic Window Storage File", "")
	
	__Opera_Correct_Opera_Path($sOpAutoSessionFile, $sOpDir, $sOpProfileDir1, $sOpProfileDir2)
	
	If Not __Opera_PathIsOperaConfigFile($sOpAutoSessionFile, "win") Then
		If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
		If Not @error Then $sOpAutoSessionFile = $sOpProfileDir1 & "\sessions\autosave.win"
	EndIf
	
	Local $sOpSessionsDir = $sOpProfileDir1 & "\sessions"
	Local $aFiles = _FileListToArray($sOpSessionsDir, '*.win', 1)
	Local $aOpSessionFiles[2] = [1, $sOpAutoSessionFile]
	
	If Not __Opera_PathIsOperaConfigFile($sOpAutoSessionFile, "win") Then
		Dim $aOpSessionFiles[1] = [0]
	EndIf
	
	For $i = 1 To UBound($aFiles)-1
		If $aFiles[$i] <> "autosave.win" Then
			$aOpSessionFiles[0] += 1
			ReDim $aOpSessionFiles[$aOpSessionFiles[0]+1]
			$aOpSessionFiles[$aOpSessionFiles[0]] = $sOpSessionsDir & "\" & $aFiles[$i]
		EndIf
	Next
	
	If $aOpSessionFiles[0] = 0 Then
		Return SetError(2, 0, 0)
	EndIf
	
	Return $aOpSessionFiles
EndFunc   ;==>_OperaGetSessionFiles

;===============================================================================
;
; Function Name:  		   _OperaGetSkinFile()
;
; Function Description:    Returns used Opera Skin file.
;
; Parameter(s):            $sOpDir    - [Optional] Full path to Opera dir,
;                                        so the returned Skin file will be related to specific Opera installation.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return full path to Opera Skin file.
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                                                     1 - $sOpDir (Opera Dir) not found.
;                                                                                     2 - Skin file not found.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaGetSkinFile($sOpDir = -1)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, "")
	
	Local $sOpProfileDir1 = _OperaGetProfileDir($sOpDir, @AppDataDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sOpProfileDir2 = _OperaGetProfileDir($sOpDir, __Opera_GetLocalSettingsPath())
	
	Local $sOperaPrefs_FileName = __Opera_GetOperaPrefsFileName($sOpDir, 0)
	Local $sOpSkinFile = IniRead($sOpProfileDir1 & "\" & $sOperaPrefs_FileName, "User Prefs", "Button Set", "")
	
	__Opera_Correct_Opera_Path($sOpSkinFile, $sOpDir, $sOpProfileDir1, $sOpProfileDir2)
	
	If Not __Opera_PathIsOperaSkinFile($sOpSkinFile) Then
		If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
		If Not @error Then $sOpSkinFile = $sOpDir & "\skin\standard_skin.zip"
	EndIf
	
	If Not __Opera_PathIsOperaSkinFile($sOpSkinFile) Then Return SetError(2, 0, "")
	
	Return $sOpSkinFile
EndFunc   ;==>_OperaGetSkinFile

;===============================================================================
;
; Function Name:  		   _OperaGetLangFile()
;
; Function Description:    Returns used Opera Language file.
;
; Parameter(s):            $sOpDir    - [Optional] Full path to Opera dir,
;                                        so the returned Language file will be related to specific Opera installation.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return full path to Opera Language file.
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                                                    1 - $sOpDir (Opera Dir) not found.
;                                                                                    2 - Language file not found.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaGetLangFile($sOpDir = -1)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, "")
	
	Local $sOpProfileDir1 = _OperaGetProfileDir($sOpDir, @AppDataDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sOpProfileDir2 = _OperaGetProfileDir($sOpDir, __Opera_GetLocalSettingsPath())
	
	Local $sOperaPrefs_FileName = __Opera_GetOperaPrefsFileName($sOpDir, 0)
	Local $sOpLangFile = IniRead($sOpProfileDir1 & "\" & $sOperaPrefs_FileName, "User Prefs", "Language File", "")
	
	__Opera_Correct_Opera_Path($sOpLangFile, $sOpDir, $sOpProfileDir1, $sOpProfileDir2)
	
	If Not __Opera_PathIsOperaLangFile($sOpLangFile) Then
		If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
		If Not @error Then $sOpLangFile = $sOpDir & "\English.lng"
	EndIf
	
	If Not __Opera_PathIsOperaLangFile($sOpLangFile) Then $sOpLangFile = $sOpDir & "\locale\en\en.lng"
	If Not __Opera_PathIsOperaLangFile($sOpLangFile) Then Return SetError(2, 0, "")
	
	Return $sOpLangFile
EndFunc   ;==>_OperaGetLangFile

;===============================================================================
;
; Function Name:  		   _OperaGetBookmarksFile()
;
; Function Description:    Return Opera's Bookmarks file.
;
; Parameter(s):            $sOpDir    - [Optional] Full path to Opera dir,
;                                        so the returned Bookmarks file will be related to specific Opera installation.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return full path to Opera Bookmarks file.
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                                                    1 - $sOpDir (Opera Dir) not found.
;                                                                                    2 - Bookmarks file not found.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaGetBookmarksFile($sOpDir = -1)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, "")
	
	Local $sOpProfileDir = _OperaGetProfileDir($sOpDir, @AppDataDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sOperaPrefs_FileName = __Opera_GetOperaPrefsFileName($sOpDir, 0)
	Local $sOpBkmrksFile = IniRead($sOpProfileDir & "\" & $sOperaPrefs_FileName, "User Prefs", "Hot List File Ver2", "")
	
	If Not FileExists($sOpBkmrksFile) Then $sOpBkmrksFile = $sOpProfileDir & "\Opera6.adr"
	If Not FileExists($sOpBkmrksFile) Then $sOpBkmrksFile = $sOpProfileDir & "\Bookmarks.adr"
	
	If Not __Opera_PathIsOperaBookmarksFile($sOpBkmrksFile) Then Return SetError(2, 0, "")
	
	Return $sOpBkmrksFile
EndFunc   ;==>_OperaGetBookmarksFile

;===============================================================================
;
; Function Name:  		   _OperaGetConfigFile()
;
; Function Description:    Returns used Opera Configuration file based on given prefix (see Parameter(s)).
;
; Parameter(s):            $sConfig_Prefix - Prefix string that indicate what configurtion file to return, these are supported:
;                                             Menu, Toolbar, Keyboard, Mouse
;                          $sOpDir         - [Optional] Full path to Opera dir,
;                                             so the returned Config file will be related to specific Opera installed dir.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success      - Return full path to specific Opera Configuration file.
;                          On Failure      - Returns "" (empty string) and set @error to:
;                                                    1 - $sConfig_Prefix is unsupported prefix string (in this case returns -1).
;                                                    2 - If Opera/profile dir not found.
;                                                    3 - Configuration file not found,
;                                                        or founded configuration file is not an Opera Configuration file.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaGetConfigFile($sConfig_Prefix, $sOpDir = -1)
	If Not __Opera_PrefixStrIsOperaConfig($sConfig_Prefix) Then Return SetError(1, 0, -1)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(2, 0, "")
	
	Local $sOpProfileDir1 = _OperaGetProfileDir($sOpDir, @AppDataDir)
	If @error Then Return SetError(2, 0, "")
	
	Local $sOpProfileDir2 = _OperaGetProfileDir($sOpDir, __Opera_GetLocalSettingsPath())
	
	Local $sOperaPrefs_FileName = __Opera_GetOperaPrefsFileName($sOpDir, 0)
	Local $sOpConfigFile = IniRead($sOpProfileDir1 & "\" & $sOperaPrefs_FileName, "User Prefs", $sConfig_Prefix & " Configuration", "")
	
	__Opera_Correct_Opera_Path($sOpConfigFile, $sOpDir, $sOpProfileDir1, $sOpProfileDir2)
	
	If Not __Opera_PathIsOperaConfigFile($sOpConfigFile) Then
		If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
		
		If Not @error Then
			$sOpConfigFile = $sOpDir & "\" & __Opera_GetOperaDefaultsDirName($sOpDir) & "\standard_" & $sConfig_Prefix & ".ini"
		EndIf
	EndIf
	
	If Not __Opera_PathIsOperaConfigFile($sOpConfigFile) Then Return SetError(3, 0, "")
	
	Return $sOpConfigFile
EndFunc   ;==>_OperaGetConfigFile

;===============================================================================
;
; Function Name:  		   _OperaSetConfigFile()
;
; Function Description:    Set specific Opera Configuration file based on given prefix (see Parameter(s)).
;
; Parameter(s):            $sConfig_Prefix - Prefix string that indicate what configurtion file to set, these are supported:
;                                             Menu, Toolbar, Keyboard, Mouse
;                          $sConfig_Name   - Name for the configuration file, used as file name and as menu info name.
;                          $sSetConfigFile - [Optional] Full path to external configuration file,
;                                             this file will be used as main configuration file,
;                                             if not found, standard file will be cloned to config dir with name = $sConfig_Name.
;                          $sOpDir         - [Optional] Full path to Opera dir,
;                                             so the returned Config file will be related to specific Opera installed dir.
;                          $iRepRenFlag    - [Optional] Defines replace mode:
;                                             -1 (default) or 0 will backup existing file in case the standard file is copied.
;                                                (*Without* replacing the existing backup file).
;                                             1 will backup existing file in case the standard file is copied.
;                                                (*Will* replace the existing backup file).
;                                             2 auto-rename existing file in case the standard file is copied.
;                                             In other cases existing file will be replaced.
;
;                          $sStandard_Name - [Optional] If this is a string (default is -1), then it's used as standard file name
;                                              (instead of "standard_" & $sConfig_Prefix & ".ini").
;
;
; Requirement(s):          Opera Browser should be closed; otherwise the written values can be erased by Opera.
;
; Return Value(s):         On Success - Set specific Opera Configuration file for usage in Opera Browser,
;                                         and return full path to "old" (used) Opera specific Configuration file.
;                          On Failure - Returns "" (empty string) and set @error to:
;                                                    1 - $sConfig_Prefix is unsupported prefix string (in this case returns -1).
;                                                    2 - Opera profile dir not found.
;                                                    3 - Current Configuration file = $sSetConfigFile.
;                                                    4 - $sSetConfigFile is not found (or it is not an Opera config file),
;                                                         and Opera Dir not found.
;                                                    5 - Unable to copy Configuration file (if standard config file cloned).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaSetConfigFile($sConfig_Prefix, $sConfig_Name, $sSetConfigFile = -1, $sOpDir = -1, $iRepRenFlag = -1, $sStandard_Name=-1)
	If Not __Opera_PrefixStrIsOperaConfig($sConfig_Prefix) Then Return SetError(1, 0, -1)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(2, 0, "")
	
	Local $sOpProfileDir = _OperaGetProfileDir($sOpDir)
	If @error Then Return SetError(2, 0, "")
	
	Local $sCurrConfigFile = _OperaGetConfigFile($sConfig_Prefix, $sOpDir)
	If Not @error And $sSetConfigFile = $sCurrConfigFile Then Return SetError(3, 0, $sCurrConfigFile)
	
	If StringRight($sConfig_Name, 4) <> ".ini" Then $sConfig_Name &= ".ini"
	
	If Not __Opera_PathIsOperaConfigFile($sSetConfigFile) Then
		If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
		If @error Then Return SetError(4, 0, "")
		
		Local $sDefaults_Path = $sOpDir & "\" & __Opera_GetOperaDefaultsDirName($sOpDir)
		
		$sStandard_ConfigFile = $sDefaults_Path & "\standard_" & $sConfig_Prefix & ".ini"
		If FileExists($sDefaults_Path & "\" & $sStandard_Name) Then $sStandard_ConfigFile = $sDefaults_Path & "\" & $sStandard_Name
		
		If $iRepRenFlag = 2 Then $sConfig_Name = __Opera_FileNewName($sConfig_Name, $sOpProfileDir & "\" & $sConfig_Prefix)
		$sSetConfigFile = $sOpProfileDir & "\" & $sConfig_Prefix & "\" & $sConfig_Name
		
		Switch $iRepRenFlag
			Case Default, -1, 0
				__Opera_Backup_Config($sSetConfigFile, 0)
			Case 1
				__Opera_Backup_Config($sSetConfigFile, 1)
		EndSwitch
		
		If Not FileCopy($sStandard_ConfigFile, $sSetConfigFile, 9) Then Return SetError(5, 0, "")
	EndIf
	
	Local $sOperaPrefs_FileName = __Opera_GetOperaPrefsFileName($sOpDir, 0)
	IniWrite($sOpProfileDir & "\" & $sOperaPrefs_FileName, "User Prefs", $sConfig_Prefix & " Configuration", $sSetConfigFile)
	IniWrite($sSetConfigFile, "Info", "NAME", StringRegExpReplace($sConfig_Name, "(?i).ini+$", ""))
	
	Return $sCurrConfigFile
EndFunc   ;==>_OperaSetConfigFile

;===============================================================================
;
; Function Name:  		   _OperaGetCacheDir()
;
; Function Description:    Returns full path to Opera Cache Dir.
;
; Parameter(s):            $sOpDir    - [Optional] Full path to Opera dir,
;                                        so the returned Cache dir will be related to specific Opera installed dir.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return full path to Opera Cache Dir,
;                                         i.e. C:\Documents and Settings\User\Local Settings\Application Data\Opera\Opera 11\cache
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                 1 - $sOpDir (Opera Dir) not found.
;                                                 2 - Cache dir not found.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaGetCacheDir($sOpDir = -1)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, "")
	
	Local $sOpProfileDir1 = _OperaGetProfileDir($sOpDir, @AppDataDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sOpProfileDir2 = _OperaGetProfileDir($sOpDir, __Opera_GetLocalSettingsPath())
	
	Local $sOperaPrefs_FileName = __Opera_GetOperaPrefsFileName($sOpDir, 0)
	Local $sOpCacheDir = IniRead($sOpProfileDir1 & "\" & $sOperaPrefs_FileName, "User Prefs", "Cache Directory4", "Def_ERROR")
	
	If $sOpCacheDir == "Def_ERROR" Then
		$sOpCacheDir = $sOpProfileDir1 & "\cache"
		If Not __Opera_PathIsFolder($sOpCacheDir) Then $sOpCacheDir = $sOpProfileDir1 & "\cache4"
	Else
		__Opera_Correct_Opera_Path($sOpCacheDir, $sOpDir, $sOpProfileDir1, $sOpProfileDir2)
	EndIf
	
	If Not __Opera_PathIsFolder($sOpCacheDir) Then $sOpCacheDir = _PathFull($sOpCacheDir, $sOpDir)
	If Not __Opera_PathIsFolder($sOpCacheDir) Then Return SetError(2, 0, "")
	
	Return $sOpCacheDir
EndFunc   ;==>_OperaGetUserJSDir

;===============================================================================
;
; Function Name:  		   _OperaSetSkinFile()
;
; Function Description:    Set Opera Skin file.
;
; Parameter(s):            $sOpSkinFile  - Full path to skin file that needs to be set.
;                          $sOpDir       - [Optional] Full path to Opera dir,
;                                           so the Skin file will be set for specific Opera installation.
;                          $iCopyToSkins - [Optional] If this is <> 0 then skin file will be copied to Opera's Skin folder.
;                          $iRepRenFlag  - [Optional] Defines replace mode:
;                                           -1 (default) or 0 will backup existing file in case the skin file is copied.
;                                              (*Without* replacing the existing backup file).
;                                           1 will backup existing file in case the skin file is copied.
;                                              (*Will* replace the existing backup file).
;                                           2 auto-rename existing file in case the skin file is copied.
;                                           In other cases existing file will be replaced.
;
; Requirement(s):          Opera Browser should be closed; otherwise the written values can be erased by Opera.
;
; Return Value(s):         On Success -  Set Opera Skin file, and return full path to "old" (used) Opera Skin file.
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                                   1 - Opera profile dir not found.
;                                                                   2 - Current Skin file = $sOpSkinFile.
;                                                                   3 - $sOpSkinFile not found, or it's not an Opera Skin file.
;                                                                   4 - Unable to copy skin file (with $iCopyToSkins mode on).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaSetSkinFile($sOpSkinFile, $sOpDir = -1, $iCopyToSkins = 0, $iRepRenFlag = -1)
	Local $sOpProfileDir = _OperaGetProfileDir($sOpDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sCurrOpSkinFile = _OperaGetSkinFile($sOpDir)
	If Not @error And $sCurrOpSkinFile = $sOpSkinFile Then Return SetError(2, 0, "")
	
	If Not __Opera_PathIsOperaSkinFile($sOpSkinFile) Then Return SetError(3, 0, "")
	
	If $iCopyToSkins <> 0 Then
		Local $sOpSkinName = StringRegExpReplace($sOpSkinFile, "^.*\\", "")
		Local $sOpSkinPath = $sOpProfileDir & "\Skin"
		Local $sOpDestSkinFile = $sOpSkinPath & "\" & $sOpSkinName
		
		Switch $iRepRenFlag
			Case Default, -1, 0
				__Opera_Backup_Config($sOpDestSkinFile, 0)
			Case 1
				__Opera_Backup_Config($sOpDestSkinFile, 9)
			Case 2
				$sOpDestSkinFile = $sOpSkinPath & "\" & __Opera_FileNewName($sOpSkinName, $sOpSkinPath)
		EndSwitch
		
		If Not FileCopy($sOpSkinFile, $sOpDestSkinFile, 9) Then SetError(4, 0, "")
		$sOpSkinFile = $sOpDestSkinFile
	EndIf
	
	Local $sOperaPrefs_FileName = __Opera_GetOperaPrefsFileName($sOpDir, 0)
	IniWrite($sOpProfileDir & "\" & $sOperaPrefs_FileName, "User Prefs", "Button Set", $sOpSkinFile)
	
	Return $sCurrOpSkinFile
EndFunc   ;==>_OperaSetSkinFile

;===============================================================================
;
; Function Name:  		   _OperaSetUserJSDir()
;
; Function Description:    Set Opera UserJS directory.
;
; Parameter(s):            $sOpUserJSDir  - Full path to User JavaScript directory (that include UserJS scripts).
;                          $sOpDir        - [Optional] Full path to Opera dir,
;                                            so the UserJS directory will be set for specific Opera installation.
;                          $iEnableUserJS - [Optional] If this is = 1 (default) then UserJS will be enabled for Opera Browser.
;
; Requirement(s):          Opera Browser should be closed; otherwise the written values can be erased by Opera.
;
; Return Value(s):         On Success -  Set Opera UserJS directory, and return full path to "old" (used) Opera UserJS directory.
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                                             1 - Opera profile dir not found.
;                                                                             2 - Current UserJS dir = $sOpUserJSDir.
;                                                                             3 - $sOpUserJSDir is not an existing directory.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaSetUserJSDir($sOpUserJSDir, $sOpDir = -1, $iEnableUserJS = 1)
	Local $sOpProfileDir = _OperaGetProfileDir($sOpDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sCurrOpUserJSDir = _OperaGetUserJSDir($sOpDir)
	If Not @error And $sCurrOpUserJSDir = $sOpUserJSDir Then Return SetError(2, 0, "")
	
	If Not __Opera_PathIsFolder($sOpUserJSDir) Then Return SetError(3, 0, "")
	
	Local $sOperaPrefs_FileName = __Opera_GetOperaPrefsFileName($sOpDir, 0)
	IniWrite($sOpProfileDir & "\" & $sOperaPrefs_FileName, "User Prefs", "User JavaScript File", $sOpUserJSDir)
	
	If $iEnableUserJS = 1 Then
		IniWrite($sOpProfileDir & "\" & $sOperaPrefs_FileName, "User Prefs", "Browser JavaScript", 2)
		IniWrite($sOpProfileDir & "\" & $sOperaPrefs_FileName, "User Prefs", "User JavaScript", 1)
		IniWrite($sOpProfileDir & "\" & $sOperaPrefs_FileName, "User Prefs", "Always Load User JavaScript", 1)
	EndIf
	
	Return $sCurrOpUserJSDir
EndFunc   ;==>_OperaSetUserJSDir

;===============================================================================
;
; Function Name:  		   _OperaSetLangFile()
;
; Function Description:    Set Opera Language file.
;
; Parameter(s):            $sOpLangFile  - Full path to Language file that needs to be set.
;                          $sOpDir       - [Optional] Full path to Opera dir,
;                                           so the Language file will be set for specific Opera installation.
;                          $iCopyToOpDir - [Optional] If this is <> 0 then language file will be copied to Opera's folder.
;                          $iRepRenFlag  - [Optional] Defines replace mode:
;                                           -1 - (default) or 0 will backup existing file in case the lang file is copied.
;                                                 (*Without* replacing the existing backup file).
;                                           1  - will backup existing file in case the lang file is copied.
;                                              (*Will* replace the existing backup file).
;                                           2  - auto-rename existing file in case the lang file is copied.
;                                           * In other cases existing file will be replaced.
;
; Requirement(s):          Opera Browser should be closed; otherwise the written values can be erased by Opera.
;
; Return Value(s):         On Success -  Set Opera Language file, and return full path to "old" (used) Opera Language file.
;                          On Failure -  Returns "" (empty string),
;                                           and set @error to:
;                                                              1 - Opera/profile dir not found.
;                                                              2 - Current Language file = $sOpLangFile.
;                                                              3 - $sOpLangFile not found, or it's not an Opera Language file.
;                                                              4 - $sOpDir (Opera dir) not found.
;                                                              5 - Unable to copy language file (with $iCopyToOpDir mode on).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaSetLangFile($sOpLangFile, $sOpDir = -1, $iCopyToOpDir = 0, $iRepRenFlag = -1)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, "")
	
	Local $sOpProfileDir = _OperaGetProfileDir($sOpDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sCurrOpLangFile = _OperaGetLangFile($sOpDir)
	If Not @error And $sCurrOpLangFile = $sOpLangFile Then Return SetError(2, 0, "")
	
	If Not __Opera_PathIsOperaLangFile($sOpLangFile) Then Return SetError(3, 0, "")
	
	If $iCopyToOpDir <> 0 Then
		If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
		If @error Then Return SetError(4, 0, "")
		
		Local $sOpLangName = StringRegExpReplace($sOpLangFile, "^.*\\", "")
		Local $sOpDestLangFile = $sOpDir & "\" & $sOpLangName
		
		Switch $iRepRenFlag
			Case Default, -1, 0
				__Opera_Backup_Config($sOpDestLangFile, 0)
			Case 1
				__Opera_Backup_Config($sOpDestLangFile, 9)
			Case 2
				$sOpDestLangFile = $sOpDir & "\" & __Opera_FileNewName($sOpLangName, $sOpDir)
		EndSwitch
		
		If Not FileCopy($sOpLangFile, $sOpDestLangFile, 9) Then SetError(5, 0, "")
		$sOpLangFile = $sOpDestLangFile
	EndIf
	
	Local $sOperaPrefs_FileName = __Opera_GetOperaPrefsFileName($sOpDir, 0)
	IniWrite($sOpProfileDir & "\" & $sOperaPrefs_FileName, "User Prefs", "Language File", $sOpLangFile)
	
	Return $sCurrOpLangFile
EndFunc   ;==>_OperaSetLangFile

;===============================================================================
;
; Function Name:  		   _OperaSelectFolder()
;
; Function Description:    Open Browse Dialog to select Opera install dir.
;
; Parameter(s):            $sTitle      - Title for FileSelectFolder dialog.
;                          $sErrMsgText - Message string to show when selected wrong path to Opera Browser.
;                          $sOpDir      - [Optional] Full path to initial Opera dir, to start FileSelectFolder dialog with.
;                          $iInputID    - [Optional] Identifier to GUI Edit control, so the selected path will be updated in it.
;                          $hParent     - [Optional] Parent window for the dialogs.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success   -  Return new selected Opera Dir.
;                          On Failure   -  Set @error to 1 and return old initial Opera dir ($sOpDir).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaSelectFolder($sTitle, $sErrMsgText, $sOpDir = "", $iInputID = 0, $hParent = 0)
	Local $sSelectOpDir, $sInitDir = $sOpDir
	
	While 1
		$sSelectOpDir = FileSelectFolder($sTitle, "", 2, $sInitDir, $hParent)
		If @error Then Return SetError(1, 0, $sOpDir)
		
		If Not __Opera_PathIsOperaDir($sSelectOpDir) Then
			MsgBox(262144 + 48, "Attention", $sErrMsgText, 0, $hParent)
			$sInitDir = $sSelectOpDir
			ContinueLoop
		EndIf
		
		If $iInputID > 0 Then GUICtrlSetData($iInputID, $sSelectOpDir)
		Return $sSelectOpDir
	WEnd
EndFunc   ;==>_OperaSelectFolder

;===============================================================================
;
; Function Name:  		   _OperaAddMenuItem()
;
; Function Description:    Adds Item to Opera menu file.
;
; Parameter(s):            $sMenuSection  - Menu section name to write the $sItemName.
;                          $sItemName     - Menu item name (user should pass all the item, 
;                                             including "Item," and similar prefixes used in Opera).
;                          $sItemValue    - Menu item value.
;                          $iItemPosition - [Optional] Defines in what position in section the function should write the item.
;                                             * Negative number indicating that position should be checked from the bottom.
;                                             * Default is -1 -> Write to the end of section.
;                          $sMenuFile     - [Optional] Menu file wich is used to write the item
;                                            (default is -1 -> Get current file, see next parameter).
;                          $sOpDir        - [Optional] Opera dir. Used only to search the menu file if $sMenuFile not found.
;                                            (default is -1 -> Get current Opera Dir from registry).
;                          $iBackup       - [Optional] Backup menu file (default is 1).
;
; Requirement(s):          Opera Browser should be closed; otherwise the written values can be erased by Opera.
;
; Return Value(s):         On Success - Returns 1.
;                          On Failure - Returns 0 and set @error to 1 if unable to add the item.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaAddMenuItem($sMenuSection, $sItemName, $sItemValue, $iItemPosition = -1, $sMenuFile = -1, $sOpDir = -1, $iBackup = 1)
	If Not __Opera_PathIsOperaConfigFile($sMenuFile) Then $sMenuFile = _OperaGetConfigFile("Menu", $sOpDir)
	
	__Opera_ExpandOperaMenuSectClassName($sMenuSection)
	
	If $iBackup Then __Opera_Backup_Config($sMenuFile, 1)
	Local $iRet = __Opera_IniWriteToPos($sMenuFile, $sMenuSection, __Opera_StringToUTF($sItemName), __Opera_StringToUTF($sItemValue), $iItemPosition)
	
	If @error Then SetError(1)
	Return $iRet
EndFunc   ;==>_OperaAddMenuItem

;===============================================================================
;
; Function Name:  		   _OperaRemoveMenuItem()
;
; Function Description:    Removes Item from Opera menu file.
;
; Parameter(s):            $sMenuSection - Menu section name to search for $sItemName.
;                          $sItemName    - Menu item name (user should pass all the item, 
;                                            including "Item," and similar prefixes used in Opera).
;                          $sItemValue   - [Optional] Menu item value, 
;                                            to search item by it's value (default is empty string ("") - do not search by value).
;                          $iRegExpItem  - [Optional] Defines whether the deletion should be performed using RegExp pattern or not.
;                          $sMenuFile    - [Optional] Menu file wich is used to delete the item
;                                            (default is -1 -> Get current file, see next parameter).
;                          $sOpDir       - [Optional] Opera dir. Used only to search the menu file if $sMenuFile not found.
;                                            (default is -1 -> Get current Opera Dir from registry).
;                          $iBackup      - [Optional] Backup menu file (default is 1).
;
; Requirement(s):          Opera Browser should be closed; otherwise the written values can be erased by Opera.
;
; Return Value(s):         On Success - Returns 1.
;                          On Failure - Returns 0 and set @error to 1 if unable to remove the item.
;
; Note(s):                 If $sItemName or $sItemValue will include non latin characters (not ANSI),
;                           the function will use case-sensitive comparision when deleting key
;                           (even with $iRegExpItem = 1 and "(?i)" in the pattern).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaRemoveMenuItem($sMenuSection, $sItemName, $sItemValue = "", $iRegExpItem = 0, $sMenuFile = -1, $sOpDir = -1, $iBackup = 1)
	If Not __Opera_PathIsOperaConfigFile($sMenuFile) Then $sMenuFile = _OperaGetConfigFile("Menu", $sOpDir)
	
	__Opera_ExpandOperaMenuSectClassName($sMenuSection)
	
	If Not __Opera_StringIsUTF8Format($sItemName) Then $sItemName = __Opera_StringToUTF($sItemName)
	If Not __Opera_StringIsUTF8Format($sItemValue) Then $sItemValue = __Opera_StringToUTF($sItemValue)
	
	If $iBackup Then __Opera_Backup_Config($sMenuFile, 1)
	
	Local $iRet = 0
	
	If $iRegExpItem Then
		$iRet = __Opera_IniDeleteRegExp($sMenuFile, $sMenuSection, $sItemName, $sItemValue, 1)
	Else
		$iRet = IniDelete($sMenuFile, $sMenuSection, $sItemName)
	EndIf
	
	If @error Then SetError(1)
	Return $iRet
EndFunc   ;==>_OperaAddMenuItem

;===============================================================================
;
; Function Name:  		   _OperaIsRunning()
;
; Function Description:    Check if Opera Browser process is currently runing.
;
; Parameter(s):            $sOpDir    - [Optional] Full path to Opera dir,
;                                        to check the run state for the specific Opera executable file.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success -  Return True.
;                          On Failure -  Returns False and set @error to 1 (even if unable check Opera's "Run State").
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaIsRunning($sOpDir = -1)
	If Not ProcessExists($s_Opera_Exe) Then Return False
	
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, False)
	
	If __Opera_ProcessPathExists($sOpDir & "\" & $s_Opera_Exe) Then Return True
	
	Local $sOpProfileDir = _OperaGetProfileDir($sOpDir)
	If @error Then Return SetError(1, 0, "")
	
	Local $sOperaPrefs_FileName = __Opera_GetOperaPrefsFileName($sOpDir, 0)
	Local $nRunState = IniRead($sOpProfileDir & "\" & $sOperaPrefs_FileName, "State", "Run", "Def_ERROR")
	If $nRunState == "Def_ERROR" Then Return SetError(1, 0, False)
	
	Return Not Number($nRunState) = 0
EndFunc   ;==>_OperaIsRunning

;===============================================================================
;
; Function Name:  		   _OperaRun()
;
; Function Description:    Run Opera Browser.
;
; Parameter(s):            $sOpExec      - [Optional] Full path to Opera executable file.
;                          $vArguments  -  [Optional] Arguments to pass when executing Opera.exe, can be a *one*-dimentional array.
;                          $sWorkDir     - [Optional] Set Working directory when executing Opera.exe
;                                            (-1 is the default, will set "Opera Dir" as working dir).
;                          $iState       - [Optional] The "show" flag of the executed program, default is @SW_SHOWNORMAL.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success    - Returns 1 if Opera.exe was executed.
;                          On Failure    - Returns 0 (zero) and set @error to:
;                                                                              1 - $sOpDir (Opera Dir) not found.
;                                                                              2 - Unable to execute Opera.exe.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaRun($sOpExec = -1, $vArguments = "", $sWorkDir = -1, $iState = @SW_SHOWNORMAL)
	If Not FileExists($sOpExec) Or StringRight($sOpExec, 4) <> ".exe" Then $sOpExec = _OperaGetDir() & "\" & $s_Opera_Exe
	If @error Then Return SetError(1, 0, "")
	
	Local $sOpt_REF = "RunErrorsFatal"
	Local $iOld_Opt_REF = -1
	
	;If @AutoItVersion < "3.2.12.0" Then ....
	If _VersionCompare(@AutoItVersion, "3.2.12.0") = -1 Then $iOld_Opt_REF = Opt($sOpt_REF, 0)
	
	If IsArray($vArguments) And UBound($vArguments, 0) = 1 Then
		Local $aSwitchArr = $vArguments
		Local $sSpace = " ", $iUbound = UBound($aSwitchArr) - 1
		
		$vArguments = ""
		
		For $i = 1 To $iUbound
			If $i = $iUbound Then $sSpace = ""
			$vArguments &= $aSwitchArr[$i] & $sSpace
		Next
	EndIf
	
	If $sWorkDir = -1 Or $sWorkDir = Default Then $sWorkDir = StringRegExpReplace($sOpExec, "\\[^\\]*$", "")
	
	ShellExecute($sOpExec, $vArguments, $sWorkDir, "open", $iState)
	Local $iError = @error
	
	If $iOld_Opt_REF <> -1 Then Opt($sOpt_REF, $iOld_Opt_REF)
	
	If $iError Then Return SetError(2, 0, 0)
	Return 1
EndFunc   ;==>_OperaRun

;===============================================================================
;
; Function Name:  		   _OperaClose()
;
; Function Description:    Close Opera Browser.
;
; Parameter(s):            $sOpDir       - [Optional] Full path to Opera dir,
;                                           so the specific Opera window(s) will be closed.
;                          $iInstance    - [Optional] Defins how many Opera instances to close:
;                                           0 (default) - will close instanse of Opera related to given/found Opera Dir.
;                                           -1  - will close all found Opera instanses.
;                                           > 0 - will close this much Opera instanses.
;                          $iForceClose  - [Optional] If <> 0 then it will force to close Opera window(s),
;                                           even if there will be a confirm close message (0 is the default).
;
; Requirement(s):          None.
;
; Return Value(s):         On Success    - Return 1 regardless if Opera was closed or not.
;                          On Failure    - Return 0 and set @error to:
;                                                                      1 - $sOpDir (Opera Dir) not found.
;                                                                      2 - Unable to get Opera title(s).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaClose($sOpDir = -1, $iInstance = 0, $iForceClose = 0)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, 0)
	
	Switch $iInstance
		Case 0
			Local $hOphWnd = _OperaGetTitle($sOpDir, 0)
			If @error Then Return SetError(2, 0, 0)
			
			Local $sConfirmExitWinExists
			
			While WinExists($hOphWnd) Or __Opera_ProcessPathExists($sOpDir & "\" & $s_Opera_Exe)
				If Not WinExists($hOphWnd) Then
					$hOphWnd = _OperaGetTitle($sOpDir, 0)
					If @error Then Return 1
				EndIf
				
				WinActivate($hOphWnd)
				WinClose($hOphWnd)
				
				If $iForceClose <> 0 Then WinWaitClose($hOphWnd, "", 1)
				
				If $iForceClose <> 0 And WinExists($hOphWnd) Then
					$sConfirmExitWinExists = __Opera_ConfirmExitOperaWinExists($sOpDir)
					
					If $sConfirmExitWinExists <> False Then
						WinActivate($sConfirmExitWinExists)
						WinWaitActive($sConfirmExitWinExists, "", 1)
						ControlSend($sConfirmExitWinExists, "", "", "{ENTER}")
						Sleep(200)
					Else
						WinClose($hOphWnd)
					EndIf
				EndIf
			WEnd
		Case Else
			Local $sConfirmExitWinExists, $iWinProc, $sProcPath
			Local $aOpTitles = _OperaGetTitle("", 2)
			If @error Then Return SetError(2, 0, 0)
			
			If $iInstance < 0 Then $iInstance = $aOpTitles[0][0]
			
			For $i = 1 To $iInstance
				If $i > $aOpTitles[0][0] Then ExitLoop
				WinClose($aOpTitles[$i][1])
				If $iForceClose <> 0 Then WinWaitClose($aOpTitles[$i][1], "", 1)
				
				If $iForceClose <> 0 And WinExists($aOpTitles[$i][1]) Then
					$iWinProc = WinGetProcess($aOpTitles[$i][1])
					$sProcPath = StringRegExpReplace(__Opera_ProcessGetPath($iWinProc), "\\[^\\]*$", "")
					
					While WinExists($aOpTitles[$i][1]) Or ProcessExists($iWinProc)
						$sConfirmExitWinExists = __Opera_ConfirmExitOperaWinExists($sProcPath)
						
						If $sConfirmExitWinExists <> False Then
							WinActivate($sConfirmExitWinExists)
							ControlSend($sConfirmExitWinExists, "", "", "{ENTER}")
							Sleep(200)
						Else
							WinClose($aOpTitles[$i][1])
						EndIf
					WEnd
				EndIf
			Next
	EndSwitch
	
	Return 1
EndFunc   ;==>_OperaClose

;===============================================================================
;
; Function Name:  		   _OperaKill()
;
; Function Description:    Kill Opera Browser.
;
; Parameter(s):            $sOpDir       - [Optional] Full path to Opera dir,
;                                           so the specific Opera process (that executed from Opera dir) will be killed.
;                          $iInstance    - [Optional] Defins how many Opera instances to kill:
;                                           < 1 - (0 is the default) will kill instanse of Opera related to given/found Opera Dir.
;                                           > 0 - will kill this much Opera instanses (processes).
;                          $iWaitClose   - [Optional] If > 0 then function will wait this much time (seconds)
;                                                            until Opera process is closed.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success    - Return 1 regardless if Opera process was killed or not.
;                          On Failure    - Returns 0
;                                             and set @error to:
;                                                                 1 - If $iInstance >= 1 and unable to get Opera process(es).
;                                                                 2 - If $sOpDir (Opera Dir) not found (when $iInstance < 1).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaKill($sOpDir = -1, $iInstance = 0, $iWaitClose = 0)
	Select
		Case $iInstance >= 1
			Local $aOperaPIDs = __Opera_GetOperaPIDs()
			If @error Then Return SetError(1, 0, 0)
			
			For $i = 1 To $aOperaPIDs[0]
				ProcessClose($aOperaPIDs[$i])
				If $iWaitClose > 0 Then ProcessWaitClose($aOperaPIDs[$i], $iWaitClose)
				If $iInstance = $i Then Return 1
			Next
		Case Else
			If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
			If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, 0)
			
			Local $iOperaPID = __Opera_ProcessPathExists($sOpDir & "\" & $s_Opera_Exe)
			If @error Then Return SetError(2, 0, 0)
			
			ProcessClose($iOperaPID)
			If $iWaitClose > 0 Then ProcessWaitClose($iOperaPID, $iWaitClose)
	EndSelect
	
	Return 1
EndFunc   ;==>_OperaKill

;===============================================================================
;
; Function Name:  		   _OperaRestart()
;
; Function Description:    Restarts Opera Browser.
;
; Parameter(s):            $sOpDir       - [Optional] Full path to Opera dir, to restart specific Opera instance.
;                          $iForceClose  - [Optional] If <> 0 then it will force to close Opera window(s),
;                                           even if there will be a confirm close message (0 is the default).
;
; Requirement(s):          None.
;
; Return Value(s):         On Success    - Return 1 if Opera restarted.
;                          On Failure    - Returns 0 (zero) and set @error to:
;                                                                     1 - $sOpDir (Opera Dir) not found.
;                                                                     2 - Unable to close runing Opera.exe process.
;                                                                     3 - Unable to execute Opera, error by _OperaRun() function.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaRestart($sOpDir = -1, $iForceClose = 0)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, 0)
	
	If ProcessExists($s_Opera_Exe) Then
		_OperaClose($sOpDir, 0, $iForceClose)
		If @error Then Return SetError(2, 0, 0)
	EndIf
	
	If __Opera_ProcessPathExists($sOpDir & "\" & $s_Opera_Exe) Then Return SetError(2, 0, 0)
	
	_OperaRun($sOpDir & "\" & $s_Opera_Exe)
	If @error Then Return SetError(3, 0, 0)
	
	Return 1
EndFunc   ;==>_OperaRestart

;===============================================================================
;
; Function Name:  		   _OperaScreenCapture()
;
; Function Description:    Make a screenshot of Opera page or entire Opera window.
;
; Parameter(s):            $sFileName   - [Optional] Full path and extension of the image file to save the screenshot.
;                                           * If FileName is empty string (""), this function will capture the screen and return
;                                             a HBITMAP handle to the bitmap image. In this case, after you are finished with the
;                                             bitmap you must call _WinAPI_DeleteObject to delete the bitmap handle.
;                          $sOpTitle    - [Optional] Opera Window Title to get the screenshot.
;                          $sOpDir      - [Optional] Full path to Opera dir 
;                                            (used only to get Opera title of specific runing Opera instance 
;                                             in case that $sOpTitle = -1).
;                          $iCaptCursor - [Optional] If > 0 the cursor will be captured with the image (default is 0).
;                          $iFullWindow - [Optional] If <> 0 then the function will capture a screenshot for entire Opera window.
;
; Requirement(s):          GDI+: GDI+ requires a redistributable for applications that
;                           run on the Microsoft Windows NT 4.0 SP6, Windows 2000, Windows 98, and Windows Me operating systems.
;
; Return Value(s):         On Success   - Depending on $sFileName (1 if it's <> "", otherwise HBITMAP handle to the bitmap image).
;                          On Failure   - Returns @error and values as following:
;                                                     * Set @error to 1 and returns 0 - unable to activate Opera window.
;                                                     * Set @error on 2 and returns __Opera_GetOperaPageHandle error:
;                                                          -1 - Unable to get Opera page handle, probably popup message appear.
;                                                          -2 - Unable to get Opera page handle, other unknown error.
;                                                     * Set @error on 3 and returns 0 - unable to capture (or file writing issue).
;                                                     * Set @error on 4 and returns 0 - GDI+ error (probably GDI+ not installed).
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaScreenCapture($sFileName = "", $sOpTitle = -1, $sOpDir = -1, $iCaptCursor = 0, $iFullWindow = 0)
	If $sOpTitle = -1 Or Not WinExists($sOpTitle) Then $sOpTitle = _OperaGetTitle($sOpDir)
	If @error Or Not StringRegExp(__Opera_WinAPI_GetClassName($sOpTitle), '(OpWindow|OperaWindowClass)') Then Return SetError(1, 0, 0)
	
	Local $hOp_ScreenCapture
	Local $iLeft = 0, $iTop = 0, $iRight = -1, $Bottom = -1
	
	WinActivate($sOpTitle)
	WinWaitActive($sOpTitle, "", 3)
	
	If Not WinActive($sOpTitle) Then Return SetError(1, 0, 0)
	
	If $iFullWindow = 0 Then
		$hOp_ScreenCapture = __Opera_GetOperaPageHandle($sOpTitle, $sOpDir)
		If @error <> 0 Then Return SetError(2, 0, @error)
	Else
		$hOp_ScreenCapture = WinGetHandle($sOpTitle)
		
		Local $aCaptureWnd_Pos = WinGetPos($hOp_ScreenCapture)
		Local $iCaption_Height = _WinAPI_GetSystemMetrics(4) ;$SM_CYCAPTION = 4
		Dim $iLeft = 4, $iTop = $iCaption_Height, $iRight = $aCaptureWnd_Pos[2]-4, $Bottom = $aCaptureWnd_Pos[3]-$iCaption_Height
	EndIf
	
	Local $nRet = _ScreenCapture_CaptureWnd($sFileName, $hOp_ScreenCapture, $iLeft, $iTop, $iRight, $Bottom, $iCaptCursor)
	Local $iError = @error
	
	If $sFileName <> "" Then $nRet = 1
	
	If $iError Then
		$iError = 3
		$nRet = 0
		
		If StringRegExp(@OSVersion, "WIN_2000|WIN_ME|WIN_98|WIN_95") Or _
			(@OSVersion = "WIN_NT4" And @OSServicePack = "Service Pack 6") Then $iError = 4
	EndIf
	
	Return SetError($iError, 0, $nRet)
EndFunc   ;==>_OperaScreenCapture

;===============================================================================
;
; Function Name:  		   _OperaMultiUserModeIsOn()
;
; Function Description:    Check if Opera installed with Multi User profile.
;
; Parameter(s):            $sOpDir         - [Optional] Full path to Opera dir,
;                                             to check for specific Opera installed dir.
;
; Requirement(s):          None.
;
; Return Value(s):         On Success      - Returns True if Opera installed with Multi User profile, otherwise returns False.
;                          On Failure      - Returns "" (empty string) and set @error to:
;                                                    1 - Opera dir not found.
;                                                    2 - Unable to read Opera's configuration file.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaMultiUserModeIsOn($sOpDir = -1)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, "")
	
	Local $sOperaPrefs_DefFileName = __Opera_GetOperaPrefsFileName($sOpDir, 1)
	Local $sReadMultiUser = IniRead($sOpDir & "\" & $sOperaPrefs_DefFileName, "System", "Multi User", "Def_ERROR")
	
	If $sReadMultiUser == "Def_ERROR" Then
		If Not __Opera_PathIsOperaProfileDir($sOpDir & "\profile") Then
			Return True ;The latest Opera versions not always uses "Multi User" key, and if mssing then it's multi user mode.
		Else
			Return SetError(2, 0, "")
		EndIf
	EndIf
	
	Return Number($sReadMultiUser) = 1
EndFunc   ;==>_OperaMultiUserModeIsOn

;===============================================================================
;
; Function Name:  		   _Opera_Integration_Module()
;
; Function Description:    Opera integrate module for your app.
;
; Parameter(s):            $aMessages       - An array of dialog messages, should contain [12] elements, otherwise the function will use default strings.
;                          $sIntgrFuncName  - Function name to call when all ready for the integration,
;                                              this function should perform the integration process itself,
;                                              and it should recieve one parameter wich is the selected Opera Path.
;                          $hParent         - [Optional] Parent window for the dialogs.
;
; Requirement(s):          #include <GUIConstantsEx.au3>.
;
; Return Value(s):         None.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _Opera_Integration_Module($aMessages, $sIntgrFuncName, $hParent = 0)
	Local $iError = 0
	Local $OperaDir, $OperaProfileDir, $OpTitle
	Local $AppName, $AskIntegr, $OpPathReadInput
	Local $Intgr_GUI, $OperaPath_Input, $Browse_Button, $OK_Button, $Cancel_Button
	
	Local $aDef_Messages[13] = [ _
		"Opera Integrator", _
		"Integration to Opera", _
		"Would you like to integrate this program into Opera Browser?", _
		"Path to browser Opera", _
		"Enter Opera Browser installation path:", _
		"OK", _
		"Cancel", _
		"Please Select Opera dir", _
		"Can not find Opera files (%s), please select folder with Opera Browser...", _
		"Attention", _
		"Finishing..", _
		"Integration to Opera is successfully completed!", _
		"There was an error (%i) while trying to integrate." _
		]
	
	If UBound($aMessages) < 13 Then
		$aMessages = $aDef_Messages
	EndIf
	
	;----- Set Opera Preferences -----
	$OperaDir = _OperaGetDir()
	$OperaProfileDir = _OperaGetProfileDir($OperaDir)
	$AppName = $aMessages[0]
	$OpTitle = _OperaGetTitle($OperaDir)
	
	;----- Integration GUI -----
	$AskIntegr = MsgBox(262144 + 36, $AppName & " - " & $aMessages[1], $aMessages[2], 0, $hParent)
	If $AskIntegr <> 6 Then Return 0
	
	WinSetState($hParent, "", @SW_DISABLE)
	$Intgr_GUI = GUICreate($AppName & " - " & $aMessages[1], 380, 190, -1, -1, -1, -1, $hParent)
	
	$OperaPath_Input = GUICtrlCreateInput($OperaDir, 20, 120, 320, -1, -1, $SS_BLACKFRAME)
	GUICtrlSetTip(-1, $aMessages[3])
	
	$Browse_Button = GUICtrlCreateButton("...", 345, 120, 20, 20)
	
	GUICtrlCreateLabel($aMessages[4], 77, 50, -1, 50, $SS_CENTER)
	GUICtrlSetFont(-1, 10, 400, 0, "Georgia")
	
	GUICtrlCreateIcon($OperaDir & "\" & $s_Opera_Exe, 0, 175, 13)
	
	$OK_Button = GUICtrlCreateButton($aMessages[5], 20, 150, 80, 25)
	$Cancel_Button = GUICtrlCreateButton($aMessages[6], 120, 150, 80, 25)
	
	GUISetState(@SW_SHOW, $Intgr_GUI)
	
	While 1
		Switch GUIGetMsg()
			Case $Browse_Button
				$OpPathReadInput = GUICtrlRead($OperaPath_Input)
				$OperaDir = _OperaSelectFolder($aMessages[7], StringFormat($aMessages[8], $s_Opera_Exe), $OpPathReadInput, $OperaPath_Input, $Intgr_GUI)
			Case $OK_Button
				$OpPathReadInput = GUICtrlRead($OperaPath_Input)
				
				If Not __Opera_PathIsOperaDir($OpPathReadInput) Then
					MsgBox(262144 + 48, $aMessages[9], StringFormat($aMessages[8], $s_Opera_Exe), 0, $Intgr_GUI)
					$OpPathReadInput = _OperaSelectFolder($aMessages[7], StringFormat($aMessages[8], $s_Opera_Exe), $OpPathReadInput, $OperaPath_Input, $Intgr_GUI)
				EndIf
				
				If __Opera_PathIsOperaDir($OpPathReadInput) Then
					WinSetState($hParent, "", @SW_ENABLE)
					GUIDelete($Intgr_GUI)
					
					Call($sIntgrFuncName, $OpPathReadInput)
					$iError = @error
					
					Switch $iError
						Case -1 ;Should indicate that integration process was internaly canceled by $sIntgrFuncName function.
							ExitLoop
						Case 0
							MsgBox(262144 + 64, $AppName & " - " & $aMessages[10], $aMessages[11], 0, $hParent)
						Case Else
							MsgBox(262144 + 48, $AppName & " - " & $aMessages[9], StringFormat($aMessages[12], $iError), 0, $hParent)
					EndSwitch
					
					ExitLoop
				EndIf
			Case $GUI_EVENT_CLOSE, $Cancel_Button
				WinSetState($hParent, "", @SW_ENABLE)
				GUIDelete($Intgr_GUI)
				$iError = -1
				ExitLoop
		EndSwitch
	WEnd
	
	Return SetError($iError)
EndFunc   ;==>_Opera_Integration_Module

;===============================================================================
;
; Function Name:           OpenGetURL()
; Function Description:    Open/Get URL in/from certain browser using DDE.
;
; Parameter(s):            $sURL     - [Optional] Address of page to open,
;                                       if this is empty string (default), then returned Url from address field of particular tab.
;                          $iWin     - [Optional] Number of tab:
;                                       -1 - Current tab.
;                                       0  - New tab (when opening).
;                          $iRetType - [Optional] Definds returned value:
;                                        0  - String with Title and URL address.
;                                        1  - Array with 3 elements...
;                                             $aRetArr[0] = Title
;                                             $aRetArr[1] = URL address
;                                             $aRetArr[2] = String with Title And URL address.
;                          $sServer  - [Optional] Serever to open/get Url in/from.
;                          $iWait    - [Optional] Waiting Timeout in milliseconds, on overload will return an error.
;
; Requirement(s):          None
; Return Value(s):         On Success -  See 'Parameter(s)'.
;                          On Failure -  Returns "" (empty string) and set @error to:
;                                                      1 - Error to open Dll (user32.dll)
;                                                      2 - Error Initializing DDE
;                                                       (@extended includes more details about returned value from DllCall).
;                                                      3 - Other DDE Errors
;                                                       (@extended macro include more details about returned value from DllCall).
; Author(s):               amel27 (A.Melnichuk)
;
;=====================================================================
Func _OpenGetURL($sURL = "", $iWin = -1, $iRetType = 0, $sServer = "Opera", $iWait = 10000)
	Local $aRet, $uIdInst = DllStructCreate("int")
	Local $hServer[1], $hTopic[1], $hItem[1], $hConv[1], $hData[1], $sData[1]
	Local $sTopic = "WWW_OpenURL", $sItem = '"' & $sURL & '",,0x' & Hex($iWin)
	
	If $sURL = '' Then
		$sTopic = "WWW_GetWindowInfo"
		$sItem = "0x" & Hex($iWin)
	EndIf
	
	Local $hDll = DllOpen("user32.dll")
	If $hDll = -1 Then Return SetError(1, 0, "") ; Error to open Dll
	$aRet = DllCall("user32.dll", "int", "DdeInitialize", "ptr", DllStructGetPtr($uIdInst), "ptr", 0, "int", 0, "int", 0)
	If $aRet[0] Then Return SetError(2, $aRet[0], "") ; Error Initializing DDE
	$hServer = DllCall($hDll, "int", "DdeCreateStringHandle", "int", DllStructGetData($uIdInst, 1), "str", $sServer, "int", 1004)
	
	If $hServer[0] Then
		$hTopic = DllCall($hDll, "int", "DdeCreateStringHandle", "int", DllStructGetData($uIdInst, 1), "str", $sTopic, "int", 1004)
		If $hTopic[0] Then
			$hItem = DllCall($hDll, "int", "DdeCreateStringHandle", "int", DllStructGetData($uIdInst, 1), "str", $sItem, "int", 1004)
			If $hItem[0] Then
				$hConv = DllCall($hDll, "int", "DdeConnect", "int", DllStructGetData($uIdInst, 1), "int", $hServer[0], "int", $hTopic[0], "int", 0)
				
				If $hConv[0] Then
					$hData = DllCall($hDll, "int", "DdeClientTransaction", "ptr", 0, "int", 0, "int", $hConv[0], "int", $hItem[0], "int", 1, "int", 0x20B0, "int", $iWait, "ptr", 0)
					If $hData[0] Then $sData = DllCall($hDll, "str", "DdeAccessData", "int", $hData[0], "ptr", 0)
				EndIf
			EndIf
		EndIf
	EndIf
	
	$iErr = DllCall($hDll, "int", "DdeGetLastError", "int", DllStructGetData($uIdInst, 1))
	If $hData[0] Then DllCall($hDll, "int", "DdeFreeDataHandle", "int", $hData[0])
	If $hConv[0] Then DllCall($hDll, "int", "DdeFreeDataHandle", "int", $hConv[0])
	If $hItem[0] Then DllCall($hDll, "int", "DdeFreeDataHandle", "int", $hItem[0])
	If $hTopic[0] Then DllCall($hDll, "int", "DdeFreeDataHandle", "int", $hTopic[0])
	If $hServer[0] Then DllCall($hDll, "int", "DdeFreeDataHandle", "int", $hServer[0])
	If $iErr[0] Then Return SetError(3, $iErr[0], "") ; Othe DDE Errors
	DllCall($hDll, "int", "DdeUninitialize", "int", DllStructGetData($uIdInst, 1))
	DllClose($hDll)
	
	If StringRight($sData[0], 3) = ',""' Then $sData[0] = StringTrimRight($sData[0], 3)
	If $sURL = '' Then $sURL = StringRegExpReplace($sData[0], '^"([^"]*?)".*', '"\1"')
	
	If $iRetType = 1 Then
		Local $sRetTitle = StringReplace(StringTrimLeft($sData[0], StringLen($sURL) + 1), '\"', '"')
		Local $aRetArr[3] = [$sRetTitle, $sURL, StringReplace($sData[0], '\"', '"') ]
		Return $aRetArr
	EndIf
	
	Return $sURL
EndFunc   ;==>_OpenGetURL

;===============================================================================
;
; Function Name:           _OperaToggleMenuItemState()
; Function Description:    Toggles (disabling/enabling) menu item or any line in menu file (*.ini structure).
;
; Parameter(s):            $sMenuFile      - Menu file to use.
;                          $sItem          - Item string that should be toggled in menu file.
;                          $iCheckEncoding - [Optional] If this parameter is <> 0 (default is 0),
;                                             then file will be written using Unicode encoding (usualy used on non-unicode files).
;                          $iUseRegExp     - [Optional] If this parameter is <> 0 (default is 0), 
;                                             then $sItem is checked as Regular Expression, 
;                                             any special characters should be escaped (i.e: "\chr").
;                          $sSectName      - [Optional] Used to restrict the toggling of $sItem by one specific section in the file.
;
; Requirement(s):          None
; Return Value(s):         On Success -  Returns 1.
;                          On Failure -  Returns 0 and set @error to:
;                                                      1 - Error to open $sMenuFile in read mode.
;                                                      2 - Error to open $sMenuFile in write mode.
;
; Author(s):               G.Sandler (a.k.a CreatoR) - CreatoR's Lab (http://creator-lab.ucoz.ru, http://autoit-script.ru)
;
;=====================================================================
Func _OperaToggleMenuItemState($sMenuFile, $sItem, $iCheckEncoding=0, $iUseRegExp=0, $sSectName="")
	Local $hFileW, $hFileR, $sFileRead, $aFileRead, $iWriteMode = 2, $iReadMode = 128, $sNew_Content = "", $sSect_Content = ""
	
	$hFileR = FileOpen($sMenuFile, $iReadMode)
	If $hFileR = -1 Then Return SetError(1, 0, 0)
	$sFileRead = FileRead($hFileR)
	FileClose($hFileR)
	
	If $iCheckEncoding = 1 Then $iWriteMode += 128
	
	$aFileRead = StringSplit(StringStripCR($sFileRead), @LF)
	
	If $sSectName = "" Or ($sSectName <> "" And Not StringInStr($sFileRead, "[" & $sSectName & "]")) Then
		$aFileRead = StringSplit(StringStripCR($sFileRead), @LF)
		
		For $i = 1 To $aFileRead[0]
			If ($iUseRegExp And StringRegExp($aFileRead[$i], $sItem)) Or (Not $iUseRegExp And $aFileRead[$i] = $sItem) Then
				If StringLeft($aFileRead[$i], 1) = ";" Then
					$sNew_Content &= StringTrimLeft($aFileRead[$i], 1) & @CRLF
				Else
					$sNew_Content &= ";" & $aFileRead[$i] & @CRLF
				EndIf
			Else
				$sNew_Content &= $aFileRead[$i] & @CRLF
			EndIf
		Next
		
		$hFileW = FileOpen($sMenuFile, $iWriteMode)
		If $hFileW = -1 Then Return SetError(2, 0, 0)
		FileWrite($hFileW, StringTrimRight($sNew_Content, 2))
	Else
		For $i = 1 To $aFileRead[0]
			If StringRegExp($aFileRead[$i], "(?i)\A\[" & $sSectName & "\].*?$") Then
				For $j = $i+1 To $aFileRead[0]
					If StringLeft($aFileRead[$j], 1) = "[" Then ExitLoop
					
					If ($iUseRegExp And StringRegExp($aFileRead[$j], $sItem)) Or (Not $iUseRegExp And $aFileRead[$j] = $sItem) Then
						If StringLeft($aFileRead[$j], 1) = ";" Then
							$sSect_Content &= StringTrimLeft($aFileRead[$j], 1) & @CRLF
						Else
							$sSect_Content &= ";" & $aFileRead[$j] & @CRLF
						EndIf
					Else
						$sSect_Content &= $aFileRead[$j] & @CRLF
					EndIf
				Next
				
				$sNew_Content &= "[" & $sSectName & "]" & @CRLF & $sSect_Content
				$i = $j-1
			Else
				$sNew_Content &= $aFileRead[$i] & @CRLF
			EndIf
		Next
		
		$hFileW = FileOpen($sMenuFile, $iWriteMode)
		If $hFileW = -1 Then Return SetError(2, 0, 0)
		FileWrite($hFileW, StringTrimRight($sNew_Content, 2))
	EndIf
	
	Return FileClose($hFileW)
EndFunc

#Region Backward compatibility functions

Func _OperaIsRuning($sOpDir = -1)
	Return _OperaIsRunning($sOpDir)
EndFunc

#EndRegion Backward compatibility functions

#Region Internal Functions

;Extended function to return Title
Func __Opera_WinGetTitleEx($sExTitle)
	If IsHWnd($sExTitle) Then Return WinGetTitle($sExTitle)
	Local $iOld_Opt_WTMM = Opt("WinTitleMatchMode", 2)
	Local $sRetTitle = __Opera_ProcessPathExists($sExTitle)
	If $sRetTitle Then $sRetTitle = __Opera_WinGetTitleByPID($sRetTitle)
	If Not $sRetTitle Then $RetTitle = WinGetTitle($sExTitle)
	If Not WinExists($sRetTitle) Or Not $sRetTitle Then
		Opt("WinTitleMatchMode", 4)
		$sRetTitle = WinGetTitle($sExTitle)
		If Not $sRetTitle Then $sRetTitle = WinGetTitle("[REGEXPTITLE:(?i)(.*)" & $sExTitle & "(.*)]")
	EndIf
	Opt("WinTitleMatchMode", $iOld_Opt_WTMM)
	Return $sRetTitle
EndFunc   ;==>__Opera_WinGetTitleEx

;Retrieves Title base on given PID (Process ID)
;Based on Smoke_N's _WinGetHandleByPID()
Func __Opera_WinGetTitleByPID($iPID, $iRetMode = 1)
	If IsString($iPID) Then $iPID = ProcessExists($iPID)
	Local $aWList = WinList(), $sHold
	For $iCC = 1 To $aWList[0][0]
		If WinGetProcess($aWList[$iCC][1]) = $iPID And BitAND(WinGetState($aWList[$iCC][1]), 2) Then
			If $iRetMode = 1 Then Return $aWList[$iCC][0]
			$sHold &= $aWList[$iCC][0] & Chr(1)
		EndIf
	Next
	If $sHold Then Return StringSplit(StringTrimRight($sHold, 1), Chr(1))
	Return SetError(1, 0, 0)
EndFunc   ;==>__Opera_WinGetTitleByPID

;Check if given path is a runing process and return PID of that process.
Func __Opera_ProcessPathExists($sPath)
	If Not FileExists($sPath) Then Return SetError(1, 0, False)
	
	Local $sPathName = StringRegExpReplace($sPath, "^.*\\", "")
	Local $aProcList = ProcessList($sPathName)
	
	If Not ProcessExists($sPathName) Then Return SetError(2, 0, False)
	Local $iUbound = UBound($aProcList) - 1
	
	Local $aProc, $aPath, $iError = 0, $sReturn = False
	Local $hKernel32DllOpen = DllOpen('kernel32.dll')
	Local $hPsapiDllOpen = DllOpen('Psapi.dll')
	Local $vStruct = DllStructCreate('int[1024]')
	
	For $i = 1 To $iUbound
		$aProc = DllCall($hKernel32DllOpen, 'hwnd', 'OpenProcess', 'int', BitOR(0x0400, 0x0010), 'int', 0, 'int', $aProcList[$i][1])
		
		If Not IsArray($aProc) Or Not $aProc[0] Then
			$iError = 3
			ExitLoop
		EndIf
		
		DllCall($hPsapiDllOpen, 'int', 'EnumProcessModules', _
				'hwnd', $aProc[0], _
				'ptr', DllStructGetPtr($vStruct), _
				'int', DllStructGetSize($vStruct), _
				'int*', 0)
		
		$aPath = DllCall($hPsapiDllOpen, 'int', 'GetModuleFileNameEx', _
				'hwnd', $aProc[0], _
				'int', DllStructGetData($vStruct, 1), _
				'str', '', _
				'int', 2048)
		
		If IsArray($aPath) And $aPath[3] = $sPath Then
			$sReturn = $aProcList[$i][1]
			ExitLoop
		EndIf
	Next
	
	DllClose($hKernel32DllOpen)
	DllClose($hPsapiDllOpen)
	
	Return SetError($iError, $iUbound, $sReturn)
EndFunc   ;==>__Opera_ProcessPathExists

;Retrieves Path based on given PID (Process ID)
Func __Opera_ProcessGetPath($iPID)
	If StringRight($iPID, 3) = "exe" Then $iPID = ProcessExists($iPID)
	If Not ProcessExists($iPID) Then Return SetError(1, 0, "")
	
	Local $aProc = DllCall('kernel32.dll', 'hwnd', 'OpenProcess', 'int', BitOR(0x0400, 0x0010), 'int', 0, 'int', $iPID)
	If Not IsArray($aProc) Or Not $aProc[0] Then Return SetError(1, 0, '')
	
	Local $vStruct = DllStructCreate('int[1024]')
	
	Local $hPsapi_Dll = DllOpen('Psapi.dll')
	If $hPsapi_Dll = -1 Then $hPsapi_Dll = DllOpen(@SystemDir & '\Psapi.dll')
	If $hPsapi_Dll = -1 Then $hPsapi_Dll = DllOpen(@WindowsDir & '\Psapi.dll')
	If $hPsapi_Dll = -1 Then Return SetError(2, 0, '')
	
	DllCall($hPsapi_Dll, 'int', 'EnumProcessModules', _
			'hwnd', $aProc[0], _
			'ptr', DllStructGetPtr($vStruct), _
			'int', DllStructGetSize($vStruct), _
			'int*', 0)
	
	Local $aRet = DllCall($hPsapi_Dll, 'int', 'GetModuleFileNameEx', _
			'hwnd', $aProc[0], _
			'int', DllStructGetData($vStruct, 1), _
			'str', '', _
			'int', 2048)
	
	DllClose($hPsapi_Dll)
	
	If Not IsArray($aRet) Or StringLen($aRet[3]) = 0 Then Return SetError(3, 0, '')
	Return $aRet[3]
EndFunc   ;==>__Opera_ProcessGetPath

;Check if given path (or pathes) is an existing directory
Func __Opera_PathIsFolder($sPath, $iPathes_Mode=0)
	If StringInStr($sPath, "|") Then
		Local $aPathes = StringSplit($sPath, "|")
		Local $iRet = 0, $iStart = 1
		
		Switch $iPathes_Mode
			Case 1
				For $i = 2 To UBound($aPathes)-1
					$iRet += Number(StringInStr(FileGetAttrib($aPathes[1] & "\" & $aPathes[$i]), "D") > 0)
				Next
			Case Else
				For $i = 1 To UBound($aPathes)-1
					$iRet += Number(StringInStr(FileGetAttrib($aPathes[$i]), "D") > 0)
				Next
		EndSwitch
		
		Return $iRet
	EndIf
	
	Return Number(StringInStr(FileGetAttrib($sPath), "D") > 0)
EndFunc   ;==>__Opera_PathIsFolder

;Check if given path is an Opera's Install Dir
Func __Opera_PathIsOperaDir($sPath)
	Return FileExists($sPath & "\" & $s_Opera_Exe) = 1
EndFunc   ;==>__Opera_PathIsOperaDir

;Check if given path is an Opera's Profile Dir
Func __Opera_PathIsOperaProfileDir($sPath)
	Return __Opera_PathIsOperaConfigFile($sPath & "\opera6.ini") Or __Opera_PathIsOperaConfigFile($sPath & "\operaprefs.ini")
EndFunc   ;==>__Opera_PathIsOperaProfileDir

;Check if given path is an Opera's Configuration file
Func __Opera_PathIsOperaConfigFile($sPath, $sExt = "ini")
	Local $hFileOpen = FileOpen($sPath, 0)
	If $hFileOpen = -1 Then Return False
	
	Local $sReadLines = FileReadLine($hFileOpen, 1) & FileReadLine($hFileOpen, 2) & FileReadLine($hFileOpen, 3)
	
	FileClose($hFileOpen)
	
	Return FileExists($sPath) And StringRight($sPath, 4) = "." & $sExt And StringInStr($sReadLines, "Opera Preferences")
EndFunc   ;==>__Opera_PathIsOperaConfigFile

;Check if given path is an Opera's Language file
Func __Opera_PathIsOperaLangFile($sPath)
	Local $hFileOpen = FileOpen($sPath, 0)
	If $hFileOpen = -1 Then Return False
	
	Local $sReadLines = FileReadLine($hFileOpen, 1) & FileReadLine($hFileOpen, 2) & FileReadLine($hFileOpen, 3)
	
	FileClose($hFileOpen)
	
	Return FileExists($sPath) And StringRight($sPath, 4) = ".lng" And StringInStr($sReadLines, "Opera Language")
EndFunc   ;==>__Opera_PathIsOperaLangFile

;Check if given path is an Opera's Skin file
Func __Opera_PathIsOperaSkinFile($sPath)
	Return FileExists($sPath) = 1 And StringRight($sPath, 4) = ".zip"
EndFunc   ;==>__Opera_PathIsOperaSkinFile

;Check if given path is an Opera's Bookmarks file
Func __Opera_PathIsOperaBookmarksFile($sPath)
	Return FileExists($sPath) = 1 And StringRight($sPath, 4) = ".adr"
EndFunc   ;==>__Opera_PathIsOperaSkinFile

;Check if Prefix string is a supported string to use as Opera's Configuration file
Func __Opera_PrefixStrIsOperaConfig($sConfig_Prefix)
	Return StringRegExp($sConfig_Prefix, "(?i)\A(Menu|Toolbar|Keyboard|Mouse)$") = 1
EndFunc   ;==>__Opera_PrefixStrIsOperaConfig

;Check the existing of Opera's dialog that prompt about closing Opera Browser, if exists Title of that dialog is returned
Func __Opera_ConfirmExitOperaWinExists($sOpDir = -1)
	Local $sLangFile = _OperaGetLangFile($sOpDir)
	If @error Then Return SetError(1, 0, False)
	
	Local $sOpExitTitle = IniRead($sLangFile, "Translation", "1534399195", "")
	Local $sOpActiveTransTitle = IniRead($sLangFile, "Translation", "603725896", "")
	
	If __Opera_StringIsUTF8Format($sOpExitTitle) Then $sOpExitTitle = __Opera_UTF8ToAnsi($sOpExitTitle)
	If __Opera_StringIsUTF8Format($sOpActiveTransTitle) Then $sOpActiveTransTitle = __Opera_UTF8ToAnsi($sOpActiveTransTitle)
	
	If $sOpExitTitle <> "" And WinExists($sOpExitTitle) Then Return "[TITLE:" & $sOpExitTitle & ";REGEXPCLASS:(OpWindow|OperaWindowClass)]"
	
	If $sOpActiveTransTitle <> "" And WinExists($sOpActiveTransTitle) Then
		Return "[TITLE:" & $sOpActiveTransTitle & ";REGEXPCLASS:(OpWindow|OperaWindowClass)]"
	EndIf
	
	Return False
EndFunc   ;==>__Opera_ConfirmExitOperaWinExists

;Correct strange path classes used in Opera 10
Func __Opera_Correct_Opera_Path(ByRef $sPath, $sOpDir, $sOpProfileDir1, $sOpProfileDir2)
	$sPath = StringReplace($sPath, "{Resources}", $sOpDir & "\")
	$sPath = StringReplace($sPath, "{SmallPreferences}", $sOpProfileDir1 & "\")
	$sPath = StringReplace($sPath, "{LargePreferences}", $sOpProfileDir2 & "\")
	$sPath = StringReplace($sPath, "{SmallPrefs}", $sOpProfileDir1 & "\")
	$sPath = StringReplace($sPath, "{LargePrefs}", $sOpProfileDir2 & "\")
	$sPath = StringReplace($sPath, "/", "\", 0, 2)
EndFunc   ;==>__Opera_Correct_Opera_Path

;Retrieves all runing Opera's PIDs (Process IDs)
Func __Opera_GetOperaPIDs()
	Local $aProcList = ProcessList($s_Opera_Exe)
	If @error Then Return SetError(1, 0, 0)
	
	Local $aRetPIDs[1]
	
	For $i = 1 To UBound($aProcList) - 1
		$aRetPIDs[0] += 1
		ReDim $aRetPIDs[$aRetPIDs[0] + 1]
		$aRetPIDs[$aRetPIDs[0]] = $aProcList[$i][1]
	Next
	
	Return $aRetPIDs
EndFunc   ;==>__Opera_GetOperaPIDs

;Returns OperaPrefs file name (to be compatible with different Opera versions)
Func __Opera_GetOperaPrefsFileName($sOpDir = -1, $iDefault = 0)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, "")
	
	Local $sOperaPrefs_FileName = "Opera6.ini"
	Local $sOperaPrefs_DefFileName = "Operadef6.ini"
	
	If (Not FileExists($sOpDir & "\" & $sOperaPrefs_DefFileName) Or FileExists($sOpDir & "\operaprefs_default.ini")) And _
		_VersionCompare(_OperaGetVersion($sOpDir), "10") <> -1 Then
		
		$sOperaPrefs_FileName = "Operaprefs.ini"
		$sOperaPrefs_DefFileName = "operaprefs_default.ini"
	EndIf
	
	If $iDefault Then Return $sOperaPrefs_DefFileName
	Return $sOperaPrefs_FileName 
EndFunc   ;==>__Opera_GetOperaPrefsFileName

;Returns Defaults dir name (to be compatible with different Opera versions)
Func __Opera_GetOperaDefaultsDirName($sOpDir = -1)
	If $sOpDir = -1 Then $sOpDir = _OperaGetDir()
	If Not __Opera_PathIsOperaDir($sOpDir) Then Return SetError(1, 0, "")
	
	Local $sDefaults_DirName = "Defaults"
	If _VersionCompare(_OperaGetVersion($sOpDir), "10") <> -1 Then $sDefaults_DirName = "ui"
	
	Return $sDefaults_DirName
EndFunc   ;==>__Opera_GetOperaDefaultsDirName

;Retrieves Opera page's body control handle
Func __Opera_GetOperaPageHandle($sTitle, $sOpDir = -1)
	Local $sPart_OpTitle = StringRegExpReplace($sTitle, "(.*) -.*$", "\1")
	Local $hOpWnd = WinGetHandle($sTitle)
	
	Local $hCB_ChildWnds = DllCallbackRegister("__Opera_EnumOperaChildWindowsProc", "int", "hwnd;lparam")
	DllCall("User32.dll", "int", "EnumChildWindows", "hwnd", $hOpWnd, "int", DllCallbackGetPtr($hCB_ChildWnds), "int", 1)
	DllCallbackFree($hCB_ChildWnds)
	
	Local $iOperaIs1060Version = (_VersionCompare(_OperaGetVersion($sOpDir), "10.6") <> -1)
	
	For $hWndChild In $a_OperaChildWindows
		If Not $iOperaIs1060Version And Not IsHWnd($hWndChild) Then ContinueLoop
		
		$sGetChildTitle = WinGetTitle($hWndChild)
		
		If $sGetChildTitle = $sPart_OpTitle Or $sGetChildTitle = $sTitle Or ($iOperaIs1060Version And $sGetChildTitle = "") Then
			Local $hLast_OpWnd = 0, $sOpWindowClass = 'OpWindow'
			
			If $iOperaIs1060Version Then
				$sOpWindowClass = 'OperaWindowClass'
			EndIf
			
			$hLast_OpWnd = __Opera_WinAPI_FindWindowEx($hWndChild, 0, 'OperaWindowClass')
			$hLast_OpWnd = __Opera_WinAPI_FindWindowEx($hLast_OpWnd, 0, $sOpWindowClass)
			$hLast_OpWnd = __Opera_WinAPI_FindWindowEx($hLast_OpWnd, 0, 'OperaWindowClass')
			$hLast_OpWnd = __Opera_WinAPI_FindWindowEx($hLast_OpWnd, 0, 'OperaWindowClass')
			
			If $hLast_OpWnd = 0 Then
				$hLast_OpWnd = __Opera_WinAPI_FindWindowEx($hWndChild, 0, 'OperaWindowClass')
			EndIf
			
			$a_OperaChildWindows = 0
			
			If $hLast_OpWnd = 0 Then
				Return SetError(-1, 0, 0)
			EndIf
			
			Return $hLast_OpWnd
		EndIf
	Next
	
	$a_OperaChildWindows = 0
	Return SetError(-2, 0, 0)
EndFunc   ;==>__Opera_GetOperaPageHandle

;Retrieves AppDataDir from Local Settings
Func __Opera_GetLocalSettingsPath()
	;Return @UserProfileDir & "\Local Settings\Application Data" or @UserProfileDir & "\AppData\Local"
	
	$sLocalSettingsPath = StringReplace(StringReplace(@AppDataDir, "Application Data", "Local Settings\Application Data", 0, 2), 'AppData\Roaming', 'AppData\Local', 0, 2)
	
	Return $sLocalSettingsPath
EndFunc   ;==>__Opera_GetLocalSettingsPath

;CallBack function to enumerate opera child windows
Func __Opera_EnumOperaChildWindowsProc($hWnd, $lParam)
	Local $i_Ubound = UBound($a_OperaChildWindows)
	
	ReDim $a_OperaChildWindows[$i_Ubound+1]
	$a_OperaChildWindows[$i_Ubound] = $hWnd
	
	Return True ; To get next window
EndFunc   ;==>__Opera_EnumOperaChildWindowsProc

;Converts Menu ClassName into Opera Menu Section Name
Func __Opera_ExpandOperaMenuSectClassName(ByRef $sMenuSectClassName)
	$sMenuSectClassName = StringRegExpReplace($sMenuSectClassName, "\A\[+|\]+\z", "")
	
	Local $aMenu_Classes = StringSplit( _
		"Main|File|Edit|View|Bookmarks|Widgets|Feeds|Mail|Chat|Tools|Window|Help|" & _
		"Documents|HotClick|Link|Selected.Link|Image|Image.Link", "|")
	
	Local $aMenu_Classes_Expand = StringSplit( _
		"Browser Menu Bar|Browser File Menu|Browser Edit Menu|Browser View Menu|Browser Bookmarks Menu|" & _
		"Browser Widgets Menu|Browser Feeds Menu|Browser Mail Menu|Browser Chat Menu|Browser Tools Menu|" & _
		"Browser Window Menu|Browser Help Menu|Documents Popup Menu|Hotclick Popup Menu|Link Popup Menu|" & _
		"Link Selection Popup Menu|Image Popup Menu|Image Link Popup Menu", "|")
	
	For $i = 1 To $aMenu_Classes[0]
		$sMenuSectClassName = StringRegExpReplace($sMenuSectClassName, "(?i)\A" & $aMenu_Classes[$i] & ".Menu\z", $aMenu_Classes_Expand[$i])
	Next
EndFunc   ;==>__Opera_ExpandOperaMenuSectClassName

;Perform a backup operation for specific file
Func __Opera_Backup_Config($sSourceFile, $iReplaceFlag = 0)
	Local $iBackupMode = 0
	
	Switch $iReplaceFlag
		Case 1, 8, 1 + 8
			$iBackupMode = $iReplaceFlag
		Case Else
			$iBackupMode = 0
	EndSwitch
	
	FileCopy($sSourceFile, $sSourceFile & ".bak", $iBackupMode)
EndFunc   ;==>__Opera_Backup_Config

;Retrieves new file name for existing file (File (1).txt, File (2).txt etc.)
;by amel27 (A.Melnichuk), mod. by MsCreatoR (G.Sandler)
Func __Opera_FileNewName($sFileName, $sDir, $sDelim1 = -1, $sDelim2 = -2)
	If Not __Opera_PathIsFolder($sDir) Then Return $sFileName
	If Not FileExists($sDir & "\" & $sFileName) Then Return $sFileName
	
	$sDir = StringRegExpReplace($sDir, "\\ *$", "")
	
	Local $sName = StringRegExpReplace($sFileName, "\.[^.]*$", "")
	Local $sExtn = StringMid($sFileName, StringLen($sName) + 1)
	Local $iCount = 1, $sRet_FileName = $sFileName
	
	While FileExists($sDir & "\" & $sRet_FileName)
		If $sDelim1 = -1 And $sDelim2 = -2 Then
			$sRet_FileName = $sName & " (" & $iCount & ")" & $sExtn
		ElseIf $sDelim1 <> -1 And $sDelim2 <> -2 Then
			$sRet_FileName = $sName & $sDelim1 & $iCount & $sDelim2 & $sExtn
		ElseIf $sDelim1 <> -1 And $sDelim2 = -2 Then
			$sRet_FileName = $sName & $sDelim1 & $iCount & $sExtn
		EndIf
		$iCount += 1
	WEnd
	
	Return $sRet_FileName
EndFunc   ;==>__Opera_FileNewName

;Convert UTF-8 String to ANSI string
;by amel27 (A.Melnichuk)
Func __Opera_UTF8ToAnsi($sUTF8)
	Local $iLen = StringLen($sUTF8)
	Local $stBuf = DllStructCreate("byte[" & $iLen * 2 & "];byte[2]")
	Local $aRet = DllCall("Kernel32.dll", "int", "MultiByteToWideChar", _
			"int", 65001, "int", 0, _
			"str", $sUTF8, "int", -1, _
			"ptr", DllStructGetPtr($stBuf), "int", $iLen * 2 + 2)
	Local $stOut = DllStructCreate("char[" & $iLen & "];char")
	$aRet = DllCall("kernel32.dll", "int", "WideCharToMultiByte", _
			"int", 0, "int", 0, _
			"ptr", DllStructGetPtr($stBuf), "int", -1, _
			"ptr", DllStructGetPtr($stOut), "int", $iLen + 1, _
			"int", 0, "int", 0)
	Return DllStructGetData($stOut, 1)
EndFunc   ;==>__Opera_UTF8ToAnsi

;Check if string contain a UTF-8 format characters
;by amel27 (A.Melnichuk)
Func __Opera_StringIsUTF8Format($sString)
	Local $sAsc, $sLen = StringLen($sString), $sExt = $sLen
	For $i = 1 To $sLen
		$sAsc = Asc(StringMid($sString, $i, 1))
		If Not BitAND($sAsc, 0x80) Then
			$sExt = 0
		ElseIf Not BitXOR(BitAND($sAsc, 0xE0), 0xC0) Then
			$sExt = 1
		ElseIf Not (BitXOR(BitAND($sAsc, 0xF0), 0xE0)) Then
			$sExt = 2
		ElseIf Not BitXOR(BitAND($sAsc, 0xF8), 0xF0) Then
			$sExt = 3
		EndIf
		
		If $i + $sExt > $sLen Then Return False
		
		For $j = $i + 1 To $i + $sExt
			$sAsc = Asc(StringMid($sString, $j, 1))
			If BitXOR(BitAND($sAsc, 0xC0), 0x80) Then Return False
		Next
		$i += $sExt
	Next
	Return True
EndFunc   ;==>__Opera_StringIsUTF8Format

;Convert string to UTF-8 format
;by LEX1 (A.Ruzanov)
Func __Opera_StringToUTF($sString)
	Local $sResult = "", $iCode
	Local $aUTF_Split = StringSplit($sString, "")
	
	For $i = 1 To $aUTF_Split[0]
		$iCode = Asc($aUTF_Split[$i])
		
		Select
			Case $iCode >= 192 And $iCode <= 239
				$aUTF_Split[$i] = Chr(208) & Chr($iCode - 48)
			Case $iCode >= 240 And $iCode <= 255
				$aUTF_Split[$i] = Chr(209) & Chr($iCode - 112)
			Case $iCode = 168
				$aUTF_Split[$i] = Chr(208) & Chr(129)
			Case $iCode = 184
				$aUTF_Split[$i] = Chr(209) & Chr(145)
			Case Else
				$aUTF_Split[$i] = Chr($iCode)
		EndSelect
		
		$sResult &= $aUTF_Split[$i]
	Next
	
	Return $sResult
EndFunc   ;==>__Opera_StringToUTF

;IniWrite function allows to write to certain position in the section (-N supported to reverse positioning)
Func __Opera_IniWriteToPos($sIniFile, $sSection, $sKey, $sValue, $iPos=-1)
	If $iPos = -1 Then Return IniWrite($sIniFile, $sSection, $sKey, $sValue)
	If IniRead($sIniFile, $sSection, $sKey, "Def_Error") <> "Def_Error" Then Return IniWrite($sIniFile, $sSection, $sKey, $sValue)
	
	Local $hFileOpen, $aFileRead, $iCounter = 0, $iNew_Counter = 0, $iPos_Mark = 0, $iPos_Check = 0
	Local $iKeyWritten = 0, $sIni_Content = ""
	
	$aFileRead = StringSplit(StringStripCR(FileRead($sIniFile)), @LF)
	If @error Then Return SetError(1, 0, 0)
	
	For $i = 1 To $aFileRead[0]
		If $aFileRead[$i] = "[" & $sSection & "]" Then
			$sIni_Content &= $aFileRead[$i] & @CRLF
			$i += 1
			
			If $iPos < 0 Then $iPos_Mark = $i
			
			While 1
				If $i > $aFileRead[0] Or StringRegExp($aFileRead[$i], "\A(;|)\[.*?\]\z") Then ExitLoop
				If $aFileRead[$i] <> "" Then $iCounter += 1
				
				If StringRegExpReplace($aFileRead[$i], "\A(.*?)(=.*|$)", "\1") = $sKey Then _
					Return IniWrite($sIniFile, $sSection, $sKey, $sValue)
				
				If $iPos > 0 Then
					If $iCounter = $iPos Then
						$iKeyWritten = 1
						$sIni_Content &= $sKey & "=" & $sValue & @CRLF & $aFileRead[$i] & @CRLF
						ExitLoop
					EndIf
					
					$sIni_Content &= $aFileRead[$i] & @CRLF
				EndIf
				
				$i += 1
			WEnd
			
			If $iPos < 0 Then
				$iPos_Check = BitNOT($iPos-1)
				
				For $j = $iPos_Mark To $i
					If $j > $aFileRead[0] Then ExitLoop
					If $aFileRead[$j] <> "" Then $iNew_Counter += 1
					
					If $iCounter-$iNew_Counter = $iPos_Check-1 Then
						$iKeyWritten = 1
						$sIni_Content &= $aFileRead[$j] & @CRLF & $sKey & "=" & $sValue & @CRLF
					Else
						$sIni_Content &= $aFileRead[$j] & @CRLF
					EndIf
				Next
			EndIf
		Else
			$sIni_Content &= $aFileRead[$i] & @CRLF
		EndIf
	Next
	
	If Not $iKeyWritten Then
		If $iPos < 0 Then Return __Opera_IniWriteToPos($sIniFile, $sSection, $sKey, $sValue, 1)
		Return IniWrite($sIniFile, $sSection, $sKey, $sValue)
	EndIf
	
	$hFileOpen = FileOpen($sIniFile, 2)
	If $hFileOpen = -1 Then Return SetError(2, 0, 0)
	
	FileWrite($hFileOpen, StringStripWS($sIni_Content, 2))
	FileClose($hFileOpen)
	
	Return 1
EndFunc   ;==>__Opera_IniWriteToPos

;IniDelete function allows to delete certain key(s) using RegExp engine + delete by key value
Func __Opera_IniDeleteRegExp($sIniFile, $sSectionName, $sKeyPattern, $sValue="", $iDelCount=1)
	If Not FileExists($sIniFile) Then Return SetError(1, 0, -1)
	
	Local $aReadIniSectArr = IniReadSection($sIniFile, $sSectionName)
	If @error Then Return SetError(2, 0, 0)
	If Not IsArray($aReadIniSectArr) Then Return SetError(3, 0, 0)
	
	Local $sCurrentKey, $sCurrentValue, $iDel_Counter = 0
	
	For $i = 1 To $aReadIniSectArr[0][0]
		$sCurrentKey = $aReadIniSectArr[$i][0]
		$sCurrentValue = $aReadIniSectArr[$i][1]
		
		If StringRegExp($sCurrentKey, $sKeyPattern) And StringRegExp($sCurrentValue, $sValue) Then
			IniDelete($sIniFile, $sSectionName, $sCurrentKey)
			
			$iDel_Counter += 1
			If $iDel_Counter = $iDelCount Then Return 1
		EndIf
	Next
EndFunc   ;==>__Opera_IniDeleteRegExp

;Extended function to find window (by parent/child handle, class name and window name
Func __Opera_WinAPI_FindWindowEx($hWnd_Parent, $hWnd_Child, $sClass_Name="", $sWindow_Name="")
	Local $aResult = DllCall("User32.dll", "hwnd", "FindWindowEx", _
		"hwnd", $hWnd_Parent, "hwnd", $hWnd_Child, "str", $sClass_Name, "str", $sWindow_Name)
	
	Return $aResult[0]
EndFunc   ;==>__Opera_WinAPI_FindWindowEx

Func __Opera_WinAPI_GetClassName($hWnd)
	If Not IsHWnd($hWnd) Then
		If IsString($hWnd) Then
			$hWnd = WinGetHandle($hWnd)
		Else
			$hWnd = GUICtrlGetHandle($hWnd)
		EndIf
	EndIf
	
	Local $aResult = DllCall("User32.dll", "int", "GetClassName", "hwnd", $hWnd, "str", "", "int", 4096)
	If @error Then Return SetError(@error, 0, "")
	Return $aResult[2]
EndFunc

Func __Opera_WinAPI_PathRemoveFileSpec($sPath)
	If Not StringInStr($sPath, $s_Opera_Exe) Then
		Return $sPath
	EndIf
	
	Local $Ret = DllCall('shlwapi.dll', 'none', 'PathRemoveFileSpecW', 'wstr', $sPath)
	
	If @error Then
		Return SetError(1, 0, StringRegExpReplace($sPath, '(?i)^"?(.*?)\\' & $s_Opera_Exe & '"? ?([a-z]:\\|"?%1"?)?.*?$', '\1'))
	EndIf
	
	Return StringRegExpReplace($Ret[1], '^"*|"*$', '')
EndFunc

Func __Opera_WinAPI_AssocQueryString($sAssoc, $iType, $iFlags = 0, $sExtra = '')
	Local $TypeOfExtra = 'wstr'
	
	If Not StringStripWS($sExtra, 3) Then
		$TypeOfExtra = 'ptr'
		$sExtra = 0
	EndIf
	
	Local $Ret = DllCall('shlwapi.dll', 'uint', 'AssocQueryStringW', 'dword', $iFlags, 'dword', $iType, 'wstr', $sAssoc, $TypeOfExtra, $sExtra, 'wstr', '', 'dword*', 4096)
	
	If @error Then
		Return SetError(1, 0, '')
	Else
		If $Ret[0] Then
			Return SetError(1, $Ret[0], '')
		EndIf
	EndIf
	
	Return $Ret[5]
EndFunc

#EndRegion Internal Functions
