#Region Header
#cs
	
	Title:			Hotkeys Input Control UDF Library for AutoIt3
	Filename:		HotKeyInput.au3
	Description:	Creates and manages an Hotkey Input control for the GUI
	(see "Shortcut key" input control in shortcut properties dialog box for example)
	Author:			Yashied
	Version:		1.1
	Requirements:	AutoIt v3.3 +, Developed/Tested on WindowsXP Pro Service Pack 2
	Uses:			StructureConstants.au3, WinAPI.au3, WindowsConstants.au3, vkArray.au3
	Notes:			-
	
	Available functions:
	
	_GUICtrlCreateHotKeyInput
	_GUICtrlDeleteHotKeyInput
	_GUICtrlReadHotKeyInput
	_GUICtrlSetHotKeyInput
	_GUICtrlReleaseHotKeyInput
	
	Additional features:
	
	_KeyLock
	_KeyLoadName
	_KeyToStr
	
	Example:
	
	#Include <GUIConstantsEx.au3>
	#Include <HotKeyInput.au3>
	
	local $Form, $ButtonOk, $HotkeyInput1, $HotkeyInput2, $GUIMsg
	local $t
	
	$Form = GUICreate('Test', 300, 160)
	GUISetFont(8.5, 400, 0, 'Tahoma', $Form)
	
	$HotkeyInput1 = _GUICtrlCreateHotKeyInput(0, 56, 55, 230, 20)
	$HotkeyInput2 = _GUICtrlCreateHotKeyInput(0, 56, 89, 230, 20)
	
	_KeyLock(0x062E) ; Lock CTRL-ALT-DEL for Hotkey Input control, but not for Windows
	
	GUICtrlCreateLabel('Hotkey1:', 10, 58, 44, 14)
	GUICtrlCreateLabel('Hotkey2:', 10, 92, 44, 14)
	GUICtrlCreateLabel('Click on Input box and hold a combination of keys.' & @CR & 'Press OK to view the code.', 10, 10, 280, 28)
	$ButtonOk = GUICtrlCreateButton('OK', 110, 124, 80, 23)
	GUICtrlSetState(-1, BitOR($GUI_DEFBUTTON, $GUI_FOCUS))
	GUISetState()
	
	while 1
	$GUIMsg = GUIGetMsg()
	
	select
	case $GUIMsg = $GUI_EVENT_CLOSE
	exit
	case $GUIMsg = $ButtonOk
	$t = '   Hotkey1:  0x' & StringRight(Hex(_GUICtrlReadHotKeyInput($HotkeyInput1)), 4) & '  (' & GUICtrlRead($HotkeyInput1) & ')   ' & @CR & @CR & _
	'   Hotkey2:  0x' & StringRight(Hex(_GUICtrlReadHotKeyInput($HotkeyInput2)), 4) & '  (' & GUICtrlRead($HotkeyInput2) & ')   '
	MsgBox(0, 'Code', $t, 0, $Form)
	endselect
	wend
#ce

#include-once

#include <StructureConstants.au3>
#include <WinAPI.au3>
#include <WindowsConstants.au3>

#include <vkArray.au3>
#EndRegion Header
;

#Region Local Variables and Constants
Dim $hkId[1][8] = [[0, 0, 0, 0, 0, 0, 0, 0]]

#cs
	
	DO NOT USE THIS ARRAY IN THE SCRIPT, INTERNAL USE ONLY!
	
	$hkId[0][0]   - Count item of array
	[0][1]   - Interruption control flag (need to set this flag before changing $hkId array)
	[0][2]   - Last key pressed (16-bit code)
	[0][3]   - System key pressed control flag
	[0][4]   - Index in array of the last control with the keyboard focus (don`t change it)
	[0][5]   - Handle to the hook procedure
	[0][6]   - Lock key control flag
	[0][7]   - Release key control flag
	
	$hkId[i][0]   - The control identifier (controlID) as returned by GUICtrlCreateInput()
	[i][1]   - Handle of the given controlID (GUICtrlGetHandle($hkId[i][0]))
	[i][2]   - Last hotkey code for Hotkey Input control
	[i][3]   - Separating characters
	[i][4-7] - Reserved
	
#ce

Dim $hkLock[1] = [0]

If @AutoItVersion >= "3.3.2.0" Then
	OnAutoItExitRegister('OnHotKeyInputExit')
Else
	$sOnExitFunc_Opt = 'OnExitFunc'
	$OnHotKeyInputExit = Opt($sOnExitFunc_Opt, 'OnHotKeyInputExit')
EndIf
#EndRegion Local Variables and Constants
;

#Region Public Functions
; #FUNCTION# ========================================================================================================================
; Function Name:	_GUICtrlCreateHotKeyInput
; Description:		Creates a Hotkey Input control for the GUI.
; Syntax:			_GUICtrlCreateHotKeyInput ( $iKey, $iLeft, $iTop [, $iWidth [, $iHeight [, $iStyle [, $iExStyle]]]] )
; Parameter(s):		$iKey      - Combined 16-bit hotkey code, which consists of upper and lower bytes. Value of bits shown in the following table.
;
;								 Hotkey code bits:
;
;								 0-7   - Specifies the virtual-key (VK) code of the key. Codes for the mouse buttons (0x01 - 0x06) are not supported.
;										 (http://msdn.microsoft.com/en-us/library/dd375731(VS.85).aspx)
;
;								 8     - SHIFT key
;								 9     - CONTROL key
;								 10    - ALT key
;								 11    - WIN key
;								 12-15 - Don`t used
;
;					$iLeft, $iTop, $iWidth, $iHeight, $iStyle, $iExStyle - See description for the GUICtrlCreateInput() function.
;					(http://www.autoitscript.com/autoit3/docs/functions/GUICtrlCreateInput.htm)
;
;					$sSeparator - Separating characters. Default is "-".
;
; Return Value(s):	Success: Returns the identifier (controlID) of the new control.
;					Failure: Returns 0.
; Author(s):		Yashied
;
; Note(s):			Use _GUICtrlDeleteHotKeyInput() to delete Hotkey Input control. (Do not use GUICtrlDelete()). To work with the Hotkey Input
;					control used functions designed to Input control. If you set the GUI_DISABLE state for control then Hotkey Input control
;					will not work until the state will be set to GUI_ENABLE. Before calling GUIDelete() remove all created Hotkey Input controls
;					by _GUICtrlReleaseHotKeyInput()
;==========================================================================================================================
Func _GUICtrlCreateHotKeyInput($iKey, $iLeft, $iTop, $iWidth = -1, $iHeight = -1, $iStyle = -1, $iExStyle = -1, $sSeparator = '-')
	Local $iID
	
	If BitAND($iKey, 0x00FF) = 0 Then $iKey = 0
	If $iStyle < 0 Then $iStyle = 0x0080

	$hkId[0][1] = 1
	
	If ($hkId[0][0] = 0) And ($hkId[0][5] = 0) Then
		$hkId[0][5] = _WinAPI_SetWindowsHookEx($WH_KEYBOARD_LL, _
			DllCallbackGetPtr(DllCallbackRegister('__hook', 'long', 'int;wparam;lparam')), _WinAPI_GetModuleHandle(0), 0)
		
		If (@error) Or ($hkId[0][5] = 0) Then
			$hkId[0][1] = 0
			Return 0
		EndIf
	EndIf
	
	$iID = GUICtrlCreateInput('', $iLeft, $iTop, $iWidth, $iHeight, BitOR($iStyle, 0x0800), $iExStyle)
	
	If $iID = 0 Then
		If $hkId[0][0] = 0 And _WinAPI_UnhookWindowsHookEx($hkId[0][5]) Then $hkId[0][5] = 0
		
		$hkId[0][1] = 0
		Return 0
	EndIf
	
	GUICtrlSetBkColor($iID, 0xFFFFFF)
	GUICtrlSetData($iID, _KeyToStr($iKey, $sSeparator))
	
	$hkId[0][0] += 1
	ReDim $hkId[$hkId[0][0] + 1][8]
	
	$hkId[$hkId[0][0]][0] = $iID
	$hkId[$hkId[0][0]][1] = GUICtrlGetHandle($iID)
	$hkId[$hkId[0][0]][2] = $iKey
	$hkId[$hkId[0][0]][3] = $sSeparator
	$hkId[$hkId[0][0]][4] = 0
	$hkId[$hkId[0][0]][5] = 0
	$hkId[$hkId[0][0]][6] = 0
	$hkId[$hkId[0][0]][7] = 0
	
	$hkId[0][1] = 0
	
	Return $iID
EndFunc ; _GUICtrlCreateHotKeyInput

; #FUNCTION# ========================================================================================================================
; Function Name:	_GUICtrlDeleteHotKeyInput
; Description:		Deletes a Hotkey Input control.
; Syntax:			_GUICtrlDeleteHotKeyInput ( $iCtrlID )
; Parameter(s):		$iCtrlID - The control identifier (controlID) as returned by a _GUICtrlCreateHotKeyInput() function.
; Return Value(s):	Success: Returns 1.
;					Failure: Returns 0.
; Author(s):		Yashied
; Note(s):			-
;=======================================================================================================================
Func _GUICtrlDeleteHotKeyInput($iCtrlID)
	Local $i, $j, $k
	
	For $i = 1 To $hkId[0][0]
		If $iCtrlID = $hkId[$i][0] Then
			$hkId[0][1] = 1
			
			If Not GUICtrlDelete($hkId[$i][0]) Then
				;				$hkId[0][1] = 1
				;				return 0
			EndIf
			
			For $j = $i To $hkId[0][0] - 1
				For $k = 0 To 7
					$hkId[$i][$k] = $hkId[$i + 1][$k]
				Next
			Next
			
			ReDim $hkId[$hkId[0][0]][8]
			$hkId[0][0] -= 1
			
			If $hkId[0][0] = 0 And _WinAPI_UnhookWindowsHookEx($hkId[0][5])Then $hkId[0][5] = 0
			If $hkId[0][4] = $i Then $hkId[0][4] = 0
			
			$hkId[0][1] = 0
			Return 1
		EndIf
	Next
	
	Return 0
EndFunc ; _GUICtrlDeleteHotKeyInput

; #FUNCTION# ========================================================================================================================
; Function Name:	_GUICtrlReadHotKeyInput
; Description:		Reads a hotkey code from Hotkey Input control.
; Syntax:			_GUICtrlReadHotKeyInput ( $iCtrlID )
; Parameter(s):		$iCtrlID - The control identifier (controlID) as returned by a _GUICtrlCreateHotKeyInput() function.
; Return Value(s):	Success: Returns combined 16-bit hotkey code (see _GUICtrlCreateHotKeyInput()).
;					Failure: Returns 0.
; Author(s):		Yashied
; Note(s):			Use the GUICtrlRead() to obtain the string of hotkey.
;=======================================================================================================================
Func _GUICtrlReadHotKeyInput($iCtrlID)
	Local $i, $iRet = 0
	
	For $i = 1 To $hkId[0][0]
		If $iCtrlID = $hkId[$i][0] Then
			$iRet = $hkId[$i][2]
			If BitAND($iRet, 0x00FF) = 0 Then $iRet = 0
			
			ExitLoop
		EndIf
	Next
	
	Return $iRet
EndFunc ; _GUICtrlReadHotKeyInput

; #FUNCTION# ========================================================================================================================
; Function Name:	_GUICtrlSetHotKeyInput
; Description:		Modifies a data for a Hotkey Input control.
; Syntax:			_GUICtrlSetHotKeyInput ( $iCtrlID, $iKey )
; Parameter(s):		$iCtrlID - The control identifier (controlID) as returned by a _GUICtrlCreateHotKeyInput() function.
;					$iKey      - Combined 16-bit hotkey code (see _GUICtrlCreateHotKeyInput()).
; Return Value(s):	Success: Returns 1.
;					Failure: Returns 0.
; Author(s):		Yashied
; Note(s):			-
;=======================================================================================================================
Func _GUICtrlSetHotKeyInput($iCtrlID, $iKey)
	Local $i
	
	If BitAND($iKey, 0x00FF) = 0 Then $iKey = 0
	
	For $i = 1 To $hkId[0][0]
		If $iCtrlID = $hkId[$i][0] Then
			$hkId[0][1] = 1
			
			If Not GUICtrlSetData($hkId[$i][0], _KeyToStr($iKey, $hkId[$i][3])) Then
				$hkId[0][1] = 1
				Return 0
			EndIf
			
			$hkId[$i][2] = $iKey
			$hkId[0][1] = 0
			
			Return 1
		EndIf
	Next
	
	Return 0
EndFunc ; _GUICtrlSetHotKeyInput

; #FUNCTION# ========================================================================================================================
; Function Name:	_GUICtrlReleaseHotKeyInput
; Description:		Deletes all Hotkey Input control, which were created by a _GUICtrlCreateHotKeyInput() function.
; Syntax:			_GUICtrlReleaseHotKeyInput (  )
; Parameter(s):		None.
; Return Value(s):	Success: Returns 1.
;					Failure: Returns 0 and sets the @error flag to non-zero.
; Author(s):		Yashied
; Note(s):			-
;=======================================================================================================================
Func _GUICtrlReleaseHotKeyInput()
	Local $i
	
	While $hkId[0][0] > 0
		$i = $hkId[0][0]
		_GUICtrlDeleteHotKeyInput($hkId[$hkId[0][0]][0])
		
		If $i = $hkId[0][0] Then Return SetError(1, 0, 0)
	WEnd
	
	Return SetError(0, 0, 1)
EndFunc ; _GUICtrlReleaseHotKeyInput

; #FUNCTION# ========================================================================================================================
; Function Name:	_KeyLock
; Description:		Locks a specified key combination for a Hotkey Input control.
; Syntax:			_KeyLock ( $iKey )
; Parameter(s):		$iKey - Combined 16-bit hotkey code (see _GUICtrlCreateHotKeyInput()). If $iKey has a negative value (-1), then all
;							previous settings will be cleared.
;
; Return Value(s):	Success: Returns 1.
;					Failure: Returns 0 and sets the @error flag to non-zero.
; Author(s):		Yashied
;
; Note(s):			This function is independent and can be called at any time. _KeyLock() locked keys for a Hotkey Input control only,
;					but not blocked a Windows hotkeys, such as CTRL-ALT-DEL etc.
;=========================================================================================================================
Func _KeyLock($iKey)
	Local $i
	
	If Not IsInt($iKey) Then Return SetError(1, 0, 0)
	
	Select
		Case $iKey > 0
			For $i = 1 To $hkLock[0]
				If $hkLock[$i] = $iKey Then Return SetError(0, 0, 1)
			Next
			
			ReDim $hkLock[$hkLock[0] + 2]
			$hkLock[$hkLock[0] + 1] = $iKey
			$hkLock[0] += 1
		Case $iKey < 0
			$hkLock[0] = 0
			ReDim $hkLock[1]
		Case Else
			
	EndSelect
	
	Return SetError(0, 0, 1)
EndFunc ; _KeyLock

; #FUNCTION# ========================================================================================================================
; Function Name:	_KeyLoadName
; Description:		Loads a names of keys.
; Syntax:			_KeyLoadName ( $aKeyName )
; Parameter(s):		$aKeyName - 256-string array that receives the name for each virtual key (see vkCodes.au3). If the name is not
;								specified in array then the key will be ignored.
;
; Return Value(s):	Success: Returns 1.
;					Failure: Returns 0 and sets the @error flag to non-zero.
; Author(s):		Yashied
; Note(s):			Function can be called at any time.
;=======================================================================================================================
Func _KeyLoadName(ByRef $aKeyName)
	Local $i
	
	If (Not IsArray($aKeyName)) Or (UBound($aKeyName) < 256) Then Return SetError(1, 0, 0)
	
	$hkId[0][1] = 1
	
	For $i = 0 To 255
		$VK[$i] = $aKeyName[$i]
	Next
	
	For $i = 0 To $hkId[0][0]
		GUICtrlSetData($hkId[$i][0], _KeyToStr($hkId[$i][2], $hkId[$i][3]))
	Next
	
	$hkId[0][1] = 0
	
	Return SetError(1, 0, 0)
EndFunc ; _KeyLoadName

; #FUNCTION# ========================================================================================================================
; Function Name:	_KeyToStr
; Description:		Places the key names of an hotkey into a single string, separated by the specified characters.
; Syntax:			_KeyToStr ( $iKey [, $sSeparator] )
;					$iKey       - Combined 16-bit hotkey code (see _GUICtrlCreateHotKeyInput()).
;					$sSeparator - Separating characters. Default is "-".
; Return Value(s):	Returns a string containing of a combination of the key names and separating characters, eg. "Alt-Shift-D".
; Author(s):		Yashied
; Note(s):			-
;=======================================================================================================================
Func _KeyToStr($iKey, $sSeparator = '-')
	Local $sRet = '', $n = StringLen($sSeparator)
	
	If BitAND($iKey, 0x0200) = 0x0200 Then $sRet &= $VK[0xA2] & $sSeparator
	If BitAND($iKey, 0x0100) = 0x0100 Then $sRet &= $VK[0xA0] & $sSeparator
	If BitAND($iKey, 0x0400) = 0x0400 Then $sRet &= $VK[0xA4] & $sSeparator
	If BitAND($iKey, 0x0800) = 0x0800 Then $sRet &= $VK[0x5B] & $sSeparator
	If BitAND($iKey, 0x00FF) > 0 Then $sRet &= $VK[BitAND($iKey, 0x00FF)]
	
	If StringRight($sRet, $n) = $sSeparator Then $sRet = StringTrimRight($sRet, $n)
	
	If $sRet = '' Then $sRet = $VK[0x00]
	
	Return $sRet
EndFunc ; _KeyToStr
#EndRegion Public Functions
;

#Region Internal Functions
Func __hold($vkCode)
	Local $aRet = DllCall('user32.dll', 'int', 'GetAsyncKeyState', 'int', $vkCode)
	
	If (Not @error) And (BitAND($aRet[0], 0x8000) = 0x8000) Then Return 1
	Return 0
EndFunc ; __hold

Func __hook($iCode, $wParam, $lParam)
	If $iCode > -1 Then
		
		If $hkId[0][1] = 1 Then
			Switch $wParam
				Case $WM_KEYDOWN, $WM_SYSKEYDOWN
					Return -1
				Case Else
					Return _WinAPI_CallNextHookEx($hkId[0][5], $iCode, $wParam, $lParam)
			EndSwitch
		EndIf
		
		Local $vkCode = DllStructGetData(DllStructCreate($tagKBDLLHOOKSTRUCT, $lParam), 'vkCode')
		Local $i, $iIndex = 0, $iFocus = _WinAPI_GetFocus()
		
		For $i = 1 To $hkId[0][0]
			If $iFocus = $hkId[$i][1] Then
				$iIndex = $i
				ExitLoop
			EndIf
		Next
		
		Switch $wParam
			Case $WM_KEYDOWN, $WM_SYSKEYDOWN
				If $iIndex = $hkId[0][4] Then
					
				Else
					If $hkId[0][4] > 0 Then
						If (BitAND($hkId[$hkId[0][4]][2], 0x00FF) = 0) And (BitAND($hkId[$hkId[0][4]][2], 0xFF00) > 0) Then
							$hkId[$hkId[0][4]][2] = 0
							GUICtrlSetData($hkId[$hkId[0][4]][0], $VK[0x00])
						EndIf
						
						$hkId[0][6] = 0
						$hkId[0][7] = 0
					EndIf
				EndIf
				
				If $iIndex > 0 Then
					$hkId[0][4] = $iIndex
					
					If Not ($vkCode = $hkId[0][2]) Then
						$hkId[0][2] = $vkCode
						
						Switch $vkCode
							Case 0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0x5B, 0x5C
								If ($hkId[0][6] = 0) Then
									If $hkId[0][3] = 0 Then
										$hkId[$iIndex][2] = 0
										$hkId[0][3] = 1
									EndIf
									
									Switch $vkCode
										Case 0xA0, 0xA1
											$hkId[$iIndex][2] = BitOR($hkId[$iIndex][2], 0x0100)
										Case 0xA2, 0xA3
											$hkId[$iIndex][2] = BitOR($hkId[$iIndex][2], 0x0200)
										Case 0xA4, 0xA5
											$hkId[$iIndex][2] = BitOR($hkId[$iIndex][2], 0x0400)
										Case 0x5B, 0x5C
											$hkId[$iIndex][2] = BitOR($hkId[$iIndex][2], 0x0800)
									EndSwitch
									
									GUICtrlSetData($hkId[$iIndex][0], _KeyToStr($hkId[$iIndex][2], $hkId[$iIndex][3]))
								EndIf
								
								Return -1
							Case Else
								Switch $vkCode
									Case 0x08, 0x1B
										If $hkId[0][3] = 0 Then
											If BitAND($hkId[$iIndex][2], 0x00FF) > 0 Then
												$hkId[$iIndex][2] = 0
												GUICtrlSetData($hkId[$iIndex][0], $VK[0x00])
											EndIf
											
											Return -1
										EndIf
								EndSwitch
								
								If ($hkId[0][6] = 0) And ($VK[$vkCode] > '') Then
									If $hkId[0][3] = 0 Then $hkId[$iIndex][2] = 0
									
									$hkId[0][6] = 1
									$hkId[0][7] = 1
									$hkId[$iIndex][2] = BitOR(BitAND($hkId[$iIndex][2], 0xFF00), $vkCode)
									
									For $i = 1 To $hkLock[0]
										If $hkId[$iIndex][2] = $hkLock[$i] Then
											$hkId[0][7] = 0
											
											If BitAND($hkId[$iIndex][2], 0xFF00) = 0 Then
												$hkId[$iIndex][2] = $vkCode
												Return -1
											Else
												$hkId[$iIndex][2] = 0
												ExitLoop
											EndIf
										EndIf
									Next
									
									GUICtrlSetData($hkId[$iIndex][0], _KeyToStr($hkId[$iIndex][2], $hkId[$iIndex][3]))
								EndIf
								
								Return -1
						EndSwitch
					Else
						Return -1
					EndIf
				EndIf
			Case $WM_KEYUP, $WM_SYSKEYUP
				If $iIndex > 0 Then
					$hkId[0][4] = $iIndex
					If BitAND($hkId[$iIndex][2], 0x00FF) = 0 Then
						Switch $vkCode
							Case 0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0x5B, 0x5C
								Switch $vkCode
									Case 0xA0, 0xA1
										$hkId[$iIndex][2] = BitAND($hkId[$iIndex][2], 0xFEFF)
									Case 0xA2, 0xA3
										$hkId[$iIndex][2] = BitAND($hkId[$iIndex][2], 0xFDFF)
									Case 0xA4, 0xA5
										$hkId[$iIndex][2] = BitAND($hkId[$iIndex][2], 0xFBFF)
									Case 0x5B, 0x5C
										$hkId[$iIndex][2] = BitAND($hkId[$iIndex][2], 0xF7FF)
								EndSwitch
								
								GUICtrlSetData($hkId[$iIndex][0], _KeyToStr($hkId[$iIndex][2], $hkId[$iIndex][3]))
						EndSwitch
					EndIf
				EndIf
				
				$hkId[0][2] = 0
				
				If $vkCode = BitAND($hkId[$iIndex][2], 0x00FF) Then $hkId[0][7] = 0
				
				If (Not __hold(0x10)) And (Not __hold(0x11)) And (Not __hold(0x12)) And _
					(Not __hold(0x5B)) And (Not __hold(0x5C)) Then
					
					$hkId[0][3] = 0
					If $hkId[0][7] = 0 Then $hkId[0][6] = 0
				EndIf
		EndSwitch
	EndIf

	Return _WinAPI_CallNextHookEx($hkId[0][5], $iCode, $wParam, $lParam)
EndFunc ; __hook

Func OnHotKeyInputExit()
	_WinAPI_UnhookWindowsHookEx($hkId[0][5])
	If IsDeclared("OnHotKeyInputExit") Then Call(Eval("OnHotKeyInputExit"))
EndFunc ; OnHotKeyInputExit
#EndRegion Internal Functions
;